/**
*   @file    pSIP_Sai.h
*   @version
*
*   @brief   AUTOSAR Sai register map
*   @details Register map for SAI
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Adc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V1.0.0.HF001
*
*   (c) Copyright 2020-2025 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_SAI_H
#define PSIP_SAI_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"

/*==================================================================================================
*                              SAI REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup SAI_Peripheral_Access_Layer SAI Peripheral Access Layer
 * @{
 */


/** SAI - Size of Registers Arrays */
#define SAI_TDATA_COUNT                 (4)
#define SAI_TFR_COUNT                   (4)
#define SAI_RDATA_COUNT                 (4)
#define SAI_RFR_COUNT                   (4)

/*! @brief SAI channel count */
#define SAI0_CHANNEL_COUNT              (4U)
#define SAI1_CHANNEL_COUNT              (1U)
#define SAI_CHANNEL_COUNT_DEFINITION    {SAI0_CHANNEL_COUNT, SAI1_CHANNEL_COUNT}
#define SAI_MAX_CHANNEL_VALUE           {0U, 0U, 0U, 0U}

#define SAI_FIFO_SIZE                   (4U)
#define FIFO_SIZE                       (8U)
#define SAI_FPACK_8_BIT                 ((uint8)2U) /*!< 8-bit FIFO packing is enabled */
#define SAI_FPACK_16_BIT                ((uint8)3U) /*!< 16-bit FIFO packing is enabled */

/* SAI Register Layout Typedef */
typedef struct
{
    uint32 TCTRL;                    /**< Transmit Control Register, offset: 0x0000 */
    uint32 TINTE;                    /**< Transmit Interrupt Enable Register, offset: 0x0004 */
    uint32 TSTS;                     /**< Transmit Status Register, offset: 0x0008 */
    uint32 TFCR;                     /**< Transmit FIFO Control Register, offset: 0x000C */
    uint32 TCFG0;                    /**< Transmit Configuration 0 Register, offset: 0x0010 */
    uint32 TCFG1;                    /**< Transmit Configuration 1 Register, offset: 0x0014 */
    uint32 TCFG2;                    /**< Transmit Configuration 2 Register, offset: 0x0018 */
    uint32 TCFG3;                    /**< Transmit Configuration 3 Register, offset: 0x001C */
    uint32 TDATA[SAI_TDATA_COUNT];   /**< Transmit Data Register, offset: 0x0020 */
    uint32 Reserved0[4];             /**< Reserved0, offset: 0x0024 */
    uint32 TFR[SAI_TFR_COUNT];       /**< Transmit FIFO Register, offset: 0x0040 */
    uint32 Reserved1[4];             /**< Reserved1, offset: 0x0044 */
    uint32 TMR;                      /**< Transmit Word Mask, offset: 0x0060 */
    uint32 Reserved2[7];             /**< Reserved2, offset: 0x0064 */
    uint32 RCTRL;                    /**< Receive Control Register, offset: 0x0080 */
    uint32 RINTE;                    /**< Receive Interrupt Enable Register, offset: 0x0084 */
    uint32 RSTS;                     /**< Receive Status Register, offset: 0x0088 */
    uint32 RFCR;                     /**< Receive FIFO Control Register, offset: 0x008C */
    uint32 RCFG0;                    /**< Receive Configuration 0 Register, offset: 0x0090 */
    uint32 RCFG1;                    /**< Receive Configuration 1 Register, offset: 0x0094 */
    uint32 RCFG2;                    /**< Receive Configuration 2 Register, offset: 0x0098 */
    uint32 RCFG3;                    /**< Receive Configuration 3 Register, offset: 0x009C */
    uint32 RDATA[SAI_RDATA_COUNT];   /**< Receive Data Register, offset: 0x00A0 */
    uint32 Reserved3[4];             /**< Reserved3, offset: 0x00A4 */
    uint32 RFR[SAI_RFR_COUNT];       /**< Receive FIFO Register, offset: 0x00C0 */
    uint32 Reserved4[4];             /**< Reserved4, offset: 0x00C4 */
    uint32 RMR;                      /**< Receive Word Mask, offset: 0x00E0 */
    uint32 Reserved5[7];             /**< Reserved5, offset: 0x00E4 */
    uint32 MCLK_CTRL;                /**< MCLK Control Register, offset: 0x0100 */
    uint32 MCLK_FRAC_DIV;            /**< MCLK Fractional Divider Register, offset: 0x0104 */

} SAI_Type, *SAI_MemMapPtr;

/** Number of instances of the SAI module. */
#define SAI_INSTANCE_COUNT                                 (2u)

/* SAI0  base address */
#define SAI0_BASE_ADDR32                                   (0x401CC000U)
#define SAI0                                               ((volatile SAI_Type *)(SAI0_BASE_ADDR32))
/* SAI1  base address */
#define SAI1_BASE_ADDR32                                   (0x401D0000U)
#define SAI1                                               ((volatile SAI_Type *)(SAI1_BASE_ADDR32))

/** Array initializer of SAI peripheral base addresses */
#define SAI_BASE_ADDRS                                     { SAI0_BASE_ADDR32,  SAI1_BASE_ADDR32 }
/** Array initializer of SAI peripheral base pointers */
#define SAI_BASE_PTRS                                      { SAI0,  SAI1 }

/* Sai module reset via IPC Register Layout Typedef */
typedef struct
{
    uint32 CTRL[SAI_INSTANCE_COUNT];  /**< Control Register, offset: 0x0000 */
} Sai_ResetType;

/* Spi rest via IPC base address */
#define SAI_REST_VIA_IPC_BASE_ADDR32                       (0x401F41CCU)
#define SAI_REST_VIA_IPC                                   ((volatile Sai_ResetType *)(SAI_REST_VIA_IPC_BASE_ADDR32))
#define IPC_CTRL_SWRESET_MASK                              (0x2U)

#define SAI_TCTRL_OFFSET32                                 (0x0000U)                    /**< Offset for Transmit Control Register */
#define SAI_TINTE_OFFSET32                                 (0x0004U)                    /**< Offset for Transmit Interrupt Enable Register */
#define SAI_TSTS_OFFSET32                                  (0x0008U)                    /**< Offset for Transmit Status Register */
#define SAI_TFCR_OFFSET32                                  (0x000CU)                    /**< Offset for Transmit FIFO Control Register */
#define SAI_TCFG0_OFFSET32                                 (0x0010U)                    /**< Offset for Transmit Configuration 0 Register */
#define SAI_TCFG1_OFFSET32                                 (0x0014U)                    /**< Offset for Transmit Configuration 1 Register */
#define SAI_TCFG2_OFFSET32                                 (0x0018U)                    /**< Offset for Transmit Configuration 2 Register */
#define SAI_TCFG3_OFFSET32                                 (0x001CU)                    /**< Offset for Transmit Configuration 3 Register */
#define SAI_TDATA_OFFSET32(x)                              (0x0020U + ((x) * (4U)))     /**< Offset for Transmit Data Register */
#define SAI_TFR_OFFSET32(x)                                (0x0040U + ((x) * (4U)))     /**< Offset for Transmit FIFO Register */
#define SAI_TMR_OFFSET32                                   (0x0060U)                    /**< Offset for Transmit Word Mask */
#define SAI_RCTRL_OFFSET32                                 (0x0080U)                    /**< Offset for Receive Control Register */
#define SAI_RINTE_OFFSET32                                 (0x0084U)                    /**< Offset for Receive Interrupt Enable Register */
#define SAI_RSTS_OFFSET32                                  (0x0088U)                    /**< Offset for Receive Status Register */
#define SAI_RFCR_OFFSET32                                  (0x008CU)                    /**< Offset for Receive FIFO Control Register */
#define SAI_RCFG0_OFFSET32                                 (0x0090U)                    /**< Offset for Receive Configuration 0 Register */
#define SAI_RCFG1_OFFSET32                                 (0x0094U)                    /**< Offset for Receive Configuration 1 Register */
#define SAI_RCFG2_OFFSET32                                 (0x0098U)                    /**< Offset for Receive Configuration 2 Register */
#define SAI_RCFG3_OFFSET32                                 (0x009CU)                    /**< Offset for Receive Configuration 3 Register */
#define SAI_RDATA_OFFSET32(x)                              (0x00A0U + ((x) * (4U)))     /**< Offset for Receive Data Register */
#define SAI_RFR_OFFSET32(x)                                (0x00C0U + ((x) * (4U)))     /**< Offset for Receive FIFO Register */
#define SAI_RMR_OFFSET32                                   (0x00E0U)                    /**< Offset for Receive Word Mask */
#define SAI_MCLK_CTRL_OFFSET32                             (0x0100U)                    /**< Offset for MCLK Control Register */
#define SAI_MCLK_FRAC_DIV_OFFSET32                         (0x0104U)                    /**< Offset for MCLK Fractional Divider Register */


/**
 * @addtogroup SAI Register Mask
 * @{
 */
/* SAI_TCTRL Register */
#define SAI_TCTRL_DSEN_MASK                                (0x8U)
#define SAI_TCTRL_DSEN_SHIFT                               (3U)
#define SAI_TCTRL_DSEN(x)                                  (((uint32)(((uint32)(x)) << SAI_TCTRL_DSEN_SHIFT)) & SAI_TCTRL_DSEN_MASK)
#define SAI_TCTRL_SR_MASK                                  (0x4U)
#define SAI_TCTRL_SR_SHIFT                                 (2U)
#define SAI_TCTRL_SR(x)                                    (((uint32)(((uint32)(x)) << SAI_TCTRL_SR_SHIFT)) & SAI_TCTRL_SR_MASK)
#define SAI_TCTRL_BCE_MASK                                 (0x2U)
#define SAI_TCTRL_BCE_SHIFT                                (1U)
#define SAI_TCTRL_BCE(x)                                   (((uint32)(((uint32)(x)) << SAI_TCTRL_BCE_SHIFT)) & SAI_TCTRL_BCE_MASK)
#define SAI_TCTRL_TE_MASK                                  (0x1U)
#define SAI_TCTRL_TE_SHIFT                                 (0U)
#define SAI_TCTRL_TE(x)                                    (((uint32)(((uint32)(x)) << SAI_TCTRL_TE_SHIFT)) & SAI_TCTRL_TE_MASK)
/* SAI_TINTE Register */
#define SAI_TINTE_WSIE_MASK                                (0x10U)
#define SAI_TINTE_WSIE_SHIFT                               (4U)
#define SAI_TINTE_WSIE(x)                                  (((uint32)(((uint32)(x)) << SAI_TINTE_WSIE_SHIFT)) & SAI_TINTE_WSIE_MASK)
#define SAI_TINTE_SEIE_MASK                                (0x8U)
#define SAI_TINTE_SEIE_SHIFT                               (3U)
#define SAI_TINTE_SEIE(x)                                  (((uint32)(((uint32)(x)) << SAI_TINTE_SEIE_SHIFT)) & SAI_TINTE_SEIE_MASK)
#define SAI_TINTE_FEIE_MASK                                (0x4U)
#define SAI_TINTE_FEIE_SHIFT                               (2U)
#define SAI_TINTE_FEIE(x)                                  (((uint32)(((uint32)(x)) << SAI_TINTE_FEIE_SHIFT)) & SAI_TINTE_FEIE_MASK)
#define SAI_TINTE_FWIE_MASK                                (0x2U)
#define SAI_TINTE_FWIE_SHIFT                               (1U)
#define SAI_TINTE_FWIE(x)                                  (((uint32)(((uint32)(x)) << SAI_TINTE_FWIE_SHIFT)) & SAI_TINTE_FWIE_MASK)
#define SAI_TINTE_FRIE_MASK                                (0x1U)
#define SAI_TINTE_FRIE_SHIFT                               (0U)
#define SAI_TINTE_FRIE(x)                                  (((uint32)(((uint32)(x)) << SAI_TINTE_FRIE_SHIFT)) & SAI_TINTE_FRIE_MASK)
/* SAI_TSTS Register */
#define SAI_TSTS_WSF_MASK                                  (0x10U)
#define SAI_TSTS_WSF_SHIFT                                 (4U)
#define SAI_TSTS_WSF(x)                                    (((uint32)(((uint32)(x)) << SAI_TSTS_WSF_SHIFT)) & SAI_TSTS_WSF_MASK)
#define SAI_TSTS_SEF_MASK                                  (0x8U)
#define SAI_TSTS_SEF_SHIFT                                 (3U)
#define SAI_TSTS_SEF(x)                                    (((uint32)(((uint32)(x)) << SAI_TSTS_SEF_SHIFT)) & SAI_TSTS_SEF_MASK)
#define SAI_TSTS_FEF_MASK                                  (0x4U)
#define SAI_TSTS_FEF_SHIFT                                 (2U)
#define SAI_TSTS_FEF(x)                                    (((uint32)(((uint32)(x)) << SAI_TSTS_FEF_SHIFT)) & SAI_TSTS_FEF_MASK)
#define SAI_TSTS_FWF_MASK                                  (0x2U)
#define SAI_TSTS_FWF_SHIFT                                 (1U)
#define SAI_TSTS_FWF(x)                                    (((uint32)(((uint32)(x)) << SAI_TSTS_FWF_SHIFT)) & SAI_TSTS_FWF_MASK)
#define SAI_TSTS_FRF_MASK                                  (0x1U)
#define SAI_TSTS_FRF_SHIFT                                 (0U)
#define SAI_TSTS_FRF(x)                                    (((uint32)(((uint32)(x)) << SAI_TSTS_FRF_SHIFT)) & SAI_TSTS_FRF_MASK)
/* SAI_TFCR Register */
#define SAI_TFCR_FR_MASK                                   (0x80000000U)
#define SAI_TFCR_FR_SHIFT                                  (31U)
#define SAI_TFCR_FR(x)                                     (((uint32)(((uint32)(x)) << SAI_TFCR_FR_SHIFT)) & SAI_TFCR_FR_MASK)
#define SAI_TFCR_FCONT_MASK                                (0x10000000U)
#define SAI_TFCR_FCONT_SHIFT                               (28U)
#define SAI_TFCR_FCONT(x)                                  (((uint32)(((uint32)(x)) << SAI_TFCR_FCONT_SHIFT)) & SAI_TFCR_FCONT_MASK)
#define SAI_TFCR_FCOMB_MASK                                (0xC000000U)
#define SAI_TFCR_FCOMB_SHIFT                               (26U)
#define SAI_TFCR_FCOMB(x)                                  (((uint32)(((uint32)(x)) << SAI_TFCR_FCOMB_SHIFT)) & SAI_TFCR_FCOMB_MASK)
#define SAI_TFCR_FPACK_MASK                                (0x3000000U)
#define SAI_TFCR_FPACK_SHIFT                               (24U)
#define SAI_TFCR_FPACK(x)                                  (((uint32)(((uint32)(x)) << SAI_TFCR_FPACK_SHIFT)) & SAI_TFCR_FPACK_MASK)
#define SAI_TFCR_TFW_MASK                                  (0x70000U)
#define SAI_TFCR_TFW_SHIFT                                 (16U)
#define SAI_TFCR_TFW(x)                                    (((uint32)(((uint32)(x)) << SAI_TFCR_TFW_SHIFT)) & SAI_TFCR_TFW_MASK)
#define SAI_TFCR_CFR_MASK                                  (0xF00U)
#define SAI_TFCR_CFR_SHIFT                                 (8U)
#define SAI_TFCR_CFR(x)                                    (((uint32)(((uint32)(x)) << SAI_TFCR_CFR_SHIFT)) & SAI_TFCR_CFR_MASK)
#define SAI_TFCR_FWDE_MASK                                 (0xF0U)
#define SAI_TFCR_FWDE_SHIFT                                (4U)
#define SAI_TFCR_FWDE(x)                                   (((uint32)(((uint32)(x)) << SAI_TFCR_FWDE_SHIFT)) & SAI_TFCR_FWDE_MASK)
#define SAI_TFCR_FRDE_MASK                                 (0xFU)
#define SAI_TFCR_FRDE_SHIFT                                (0U)
#define SAI_TFCR_FRDE(x)                                   (((uint32)(((uint32)(x)) << SAI_TFCR_FRDE_SHIFT)) & SAI_TFCR_FRDE_MASK)
/* SAI_TCFG0 Register */
#define SAI_TCFG0_SYNC_MASK                                (0xC0000000U)
#define SAI_TCFG0_SYNC_SHIFT                               (30U)
#define SAI_TCFG0_SYNC(x)                                  (((uint32)(((uint32)(x)) << SAI_TCFG0_SYNC_SHIFT)) & SAI_TCFG0_SYNC_MASK)
#define SAI_TCFG0_BCS_MASK                                 (0x20000000U)
#define SAI_TCFG0_BCS_SHIFT                                (29U)
#define SAI_TCFG0_BCS(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG0_BCS_SHIFT)) & SAI_TCFG0_BCS_MASK)
#define SAI_TCFG0_BCI_MASK                                 (0x10000000U)
#define SAI_TCFG0_BCI_SHIFT                                (28U)
#define SAI_TCFG0_BCI(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG0_BCI_SHIFT)) & SAI_TCFG0_BCI_MASK)
#define SAI_TCFG0_BCP_MASK                                 (0x2000000U)
#define SAI_TCFG0_BCP_SHIFT                                (25U)
#define SAI_TCFG0_BCP(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG0_BCP_SHIFT)) & SAI_TCFG0_BCP_MASK)
#define SAI_TCFG0_BCD_MASK                                 (0x1000000U)
#define SAI_TCFG0_BCD_SHIFT                                (24U)
#define SAI_TCFG0_BCD(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG0_BCD_SHIFT)) & SAI_TCFG0_BCD_MASK)
#define SAI_TCFG0_DIV_MASK                                 (0xFFU)
#define SAI_TCFG0_DIV_SHIFT                                (0U)
#define SAI_TCFG0_DIV(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG0_DIV_SHIFT)) & SAI_TCFG0_DIV_MASK)
/* SAI_TCFG1 Register */
#define SAI_TCFG1_TCE_MASK                                 (0xF0000U)
#define SAI_TCFG1_TCE_SHIFT                                (16U)
#define SAI_TCFG1_TCE(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG1_TCE_SHIFT)) & SAI_TCFG1_TCE_MASK)
#define SAI_TCFG1_WDFL_MASK                                (0xFU)
#define SAI_TCFG1_WDFL_SHIFT                               (0U)
#define SAI_TCFG1_WDFL(x)                                  (((uint32)(((uint32)(x)) << SAI_TCFG1_WDFL_SHIFT)) & SAI_TCFG1_WDFL_MASK)
/* SAI_TCFG2 Register */
#define SAI_TCFG2_FRSZ_MASK                                (0xF0000U)
#define SAI_TCFG2_FRSZ_SHIFT                               (16U)
#define SAI_TCFG2_FRSZ(x)                                  (((uint32)(((uint32)(x)) << SAI_TCFG2_FRSZ_SHIFT)) & SAI_TCFG2_FRSZ_MASK)
#define SAI_TCFG2_SYWD_MASK                                (0x1F00U)
#define SAI_TCFG2_SYWD_SHIFT                               (8U)
#define SAI_TCFG2_SYWD(x)                                  (((uint32)(((uint32)(x)) << SAI_TCFG2_SYWD_SHIFT)) & SAI_TCFG2_SYWD_MASK)
#define SAI_TCFG2_CHMOD_MASK                               (0x20U)
#define SAI_TCFG2_CHMOD_SHIFT                              (5U)
#define SAI_TCFG2_CHMOD(x)                                 (((uint32)(((uint32)(x)) << SAI_TCFG2_CHMOD_SHIFT)) & SAI_TCFG2_CHMOD_MASK)
#define SAI_TCFG2_MF_MASK                                  (0x10U)
#define SAI_TCFG2_MF_SHIFT                                 (4U)
#define SAI_TCFG2_MF(x)                                    (((uint32)(((uint32)(x)) << SAI_TCFG2_MF_SHIFT)) & SAI_TCFG2_MF_MASK)
#define SAI_TCFG2_FSE_MASK                                 (0x8U)
#define SAI_TCFG2_FSE_SHIFT                                (3U)
#define SAI_TCFG2_FSE(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG2_FSE_SHIFT)) & SAI_TCFG2_FSE_MASK)
#define SAI_TCFG2_ONDEM_MASK                               (0x4U)
#define SAI_TCFG2_ONDEM_SHIFT                              (2U)
#define SAI_TCFG2_ONDEM(x)                                 (((uint32)(((uint32)(x)) << SAI_TCFG2_ONDEM_SHIFT)) & SAI_TCFG2_ONDEM_MASK)
#define SAI_TCFG2_FSP_MASK                                 (0x2U)
#define SAI_TCFG2_FSP_SHIFT                                (1U)
#define SAI_TCFG2_FSP(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG2_FSP_SHIFT)) & SAI_TCFG2_FSP_MASK)
#define SAI_TCFG2_FSD_MASK                                 (0x1U)
#define SAI_TCFG2_FSD_SHIFT                                (0U)
#define SAI_TCFG2_FSD(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG2_FSD_SHIFT)) & SAI_TCFG2_FSD_MASK)
/* SAI_TCFG3 Register */
#define SAI_TCFG3_WNW_MASK                                 (0x1F000000U)
#define SAI_TCFG3_WNW_SHIFT                                (24U)
#define SAI_TCFG3_WNW(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG3_WNW_SHIFT)) & SAI_TCFG3_WNW_MASK)
#define SAI_TCFG3_W0W_MASK                                 (0x1F0000U)
#define SAI_TCFG3_W0W_SHIFT                                (16U)
#define SAI_TCFG3_W0W(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG3_W0W_SHIFT)) & SAI_TCFG3_W0W_MASK)
#define SAI_TCFG3_FBT_MASK                                 (0x1FU)
#define SAI_TCFG3_FBT_SHIFT                                (0U)
#define SAI_TCFG3_FBT(x)                                   (((uint32)(((uint32)(x)) << SAI_TCFG3_FBT_SHIFT)) & SAI_TCFG3_FBT_MASK)
/* SAI_TDATA Register */
#define SAI_TDATA_TDATA_MASK                               (0xFFFFFFFFU)
#define SAI_TDATA_TDATA_SHIFT                              (0U)
#define SAI_TDATA_TDATA(x)                                 (((uint32)(((uint32)(x)) << SAI_TDATA_TDATA_SHIFT)) & SAI_TDATA_TDATA_MASK)
/* SAI_TFR Register */
#define SAI_TFR_WCS_MASK                                   (0x80000000U)
#define SAI_TFR_WCS_SHIFT                                  (31U)
#define SAI_TFR_WCS(x)                                     (((uint32)(((uint32)(x)) << SAI_TFR_WCS_SHIFT)) & SAI_TFR_WCS_MASK)
#define SAI_TFR_WFP_MASK                                   (0xF0000U)
#define SAI_TFR_WFP_SHIFT                                  (16U)
#define SAI_TFR_WFP_WIDTH                                  (4U)
#define SAI_TFR_WFP(x)                                     (((uint32)(((uint32)(x)) << SAI_TFR_WFP_SHIFT)) & SAI_TFR_WFP_MASK)
#define SAI_TFR_RFP_MASK                                   (0xFU)
#define SAI_TFR_RFP_SHIFT                                  (0U)
#define SAI_TFR_RFP(x)                                     (((uint32)(((uint32)(x)) << SAI_TFR_RFP_SHIFT)) & SAI_TFR_RFP_MASK)
/* SAI_TMR Register */
#define SAI_TMR_TWM_MASK                                   (0xFFFFU)
#define SAI_TMR_TWM_SHIFT                                  (0U)
#define SAI_TMR_TWM(x)                                     (((uint32)(((uint32)(x)) << SAI_TMR_TWM_SHIFT)) & SAI_TMR_TWM_MASK)
/* SAI_RCTRL Register */
#define SAI_RCTRL_DSEN_MASK                                (0x8U)
#define SAI_RCTRL_DSEN_SHIFT                               (3U)
#define SAI_RCTRL_DSEN(x)                                  (((uint32)(((uint32)(x)) << SAI_RCTRL_DSEN_SHIFT)) & SAI_RCTRL_DSEN_MASK)
#define SAI_RCTRL_SR_MASK                                  (0x4U)
#define SAI_RCTRL_SR_SHIFT                                 (2U)
#define SAI_RCTRL_SR(x)                                    (((uint32)(((uint32)(x)) << SAI_RCTRL_SR_SHIFT)) & SAI_RCTRL_SR_MASK)
#define SAI_RCTRL_BCE_MASK                                 (0x2U)
#define SAI_RCTRL_BCE_SHIFT                                (1U)
#define SAI_RCTRL_BCE(x)                                   (((uint32)(((uint32)(x)) << SAI_RCTRL_BCE_SHIFT)) & SAI_RCTRL_BCE_MASK)
#define SAI_RCTRL_RE_MASK                                  (0x1U)
#define SAI_RCTRL_RE_SHIFT                                 (0U)
#define SAI_RCTRL_RE(x)                                    (((uint32)(((uint32)(x)) << SAI_RCTRL_RE_SHIFT)) & SAI_RCTRL_RE_MASK)
/* SAI_RINTE Register */
#define SAI_RINTE_WSIE_MASK                                (0x10U)
#define SAI_RINTE_WSIE_SHIFT                               (4U)
#define SAI_RINTE_WSIE(x)                                  (((uint32)(((uint32)(x)) << SAI_RINTE_WSIE_SHIFT)) & SAI_RINTE_WSIE_MASK)
#define SAI_RINTE_SEIE_MASK                                (0x8U)
#define SAI_RINTE_SEIE_SHIFT                               (3U)
#define SAI_RINTE_SEIE(x)                                  (((uint32)(((uint32)(x)) << SAI_RINTE_SEIE_SHIFT)) & SAI_RINTE_SEIE_MASK)
#define SAI_RINTE_FEIE_MASK                                (0x4U)
#define SAI_RINTE_FEIE_SHIFT                               (2U)
#define SAI_RINTE_FEIE(x)                                  (((uint32)(((uint32)(x)) << SAI_RINTE_FEIE_SHIFT)) & SAI_RINTE_FEIE_MASK)
#define SAI_RINTE_FWIE_MASK                                (0x2U)
#define SAI_RINTE_FWIE_SHIFT                               (1U)
#define SAI_RINTE_FWIE(x)                                  (((uint32)(((uint32)(x)) << SAI_RINTE_FWIE_SHIFT)) & SAI_RINTE_FWIE_MASK)
#define SAI_RINTE_FRIE_MASK                                (0x1U)
#define SAI_RINTE_FRIE_SHIFT                               (0U)
#define SAI_RINTE_FRIE(x)                                  (((uint32)(((uint32)(x)) << SAI_RINTE_FRIE_SHIFT)) & SAI_RINTE_FRIE_MASK)
/* SAI_RSTS Register */
#define SAI_RSTS_WSF_MASK                                  (0x10U)
#define SAI_RSTS_WSF_SHIFT                                 (4U)
#define SAI_RSTS_WSF(x)                                    (((uint32)(((uint32)(x)) << SAI_RSTS_WSF_SHIFT)) & SAI_RSTS_WSF_MASK)
#define SAI_RSTS_SEF_MASK                                  (0x8U)
#define SAI_RSTS_SEF_SHIFT                                 (3U)
#define SAI_RSTS_SEF(x)                                    (((uint32)(((uint32)(x)) << SAI_RSTS_SEF_SHIFT)) & SAI_RSTS_SEF_MASK)
#define SAI_RSTS_FEF_MASK                                  (0x4U)
#define SAI_RSTS_FEF_SHIFT                                 (2U)
#define SAI_RSTS_FEF(x)                                    (((uint32)(((uint32)(x)) << SAI_RSTS_FEF_SHIFT)) & SAI_RSTS_FEF_MASK)
#define SAI_RSTS_FWF_MASK                                  (0x2U)
#define SAI_RSTS_FWF_SHIFT                                 (1U)
#define SAI_RSTS_FWF(x)                                    (((uint32)(((uint32)(x)) << SAI_RSTS_FWF_SHIFT)) & SAI_RSTS_FWF_MASK)
#define SAI_RSTS_FRF_MASK                                  (0x1U)
#define SAI_RSTS_FRF_SHIFT                                 (0U)
#define SAI_RSTS_FRF(x)                                    (((uint32)(((uint32)(x)) << SAI_RSTS_FRF_SHIFT)) & SAI_RSTS_FRF_MASK)
/* SAI_RFCR Register */
#define SAI_RFCR_FR_MASK                                   (0x80000000U)
#define SAI_RFCR_FR_SHIFT                                  (31U)
#define SAI_RFCR_FR(x)                                     (((uint32)(((uint32)(x)) << SAI_RFCR_FR_SHIFT)) & SAI_RFCR_FR_MASK)
#define SAI_RFCR_FCONT_MASK                                (0x10000000U)
#define SAI_RFCR_FCONT_SHIFT                               (28U)
#define SAI_RFCR_FCONT(x)                                  (((uint32)(((uint32)(x)) << SAI_RFCR_FCONT_SHIFT)) & SAI_RFCR_FCONT_MASK)
#define SAI_RFCR_FCOMB_MASK                                (0xC000000U)
#define SAI_RFCR_FCOMB_SHIFT                               (26U)
#define SAI_RFCR_FCOMB(x)                                  (((uint32)(((uint32)(x)) << SAI_RFCR_FCOMB_SHIFT)) & SAI_RFCR_FCOMB_MASK)
#define SAI_RFCR_FPACK_MASK                                (0x3000000U)
#define SAI_RFCR_FPACK_SHIFT                               (24U)
#define SAI_RFCR_FPACK(x)                                  (((uint32)(((uint32)(x)) << SAI_RFCR_FPACK_SHIFT)) & SAI_RFCR_FPACK_MASK)
#define SAI_RFCR_TFW_MASK                                  (0x70000U)
#define SAI_RFCR_TFW_SHIFT                                 (16U)
#define SAI_RFCR_TFW(x)                                    (((uint32)(((uint32)(x)) << SAI_RFCR_TFW_SHIFT)) & SAI_RFCR_TFW_MASK)
#define SAI_RFCR_CFR_MASK                                  (0xF00U)
#define SAI_RFCR_CFR_SHIFT                                 (8U)
#define SAI_RFCR_CFR(x)                                    (((uint32)(((uint32)(x)) << SAI_RFCR_CFR_SHIFT)) & SAI_RFCR_CFR_MASK)
#define SAI_RFCR_FWDE_MASK                                 (0xF0U)
#define SAI_RFCR_FWDE_SHIFT                                (4U)
#define SAI_RFCR_FWDE(x)                                   (((uint32)(((uint32)(x)) << SAI_RFCR_FWDE_SHIFT)) & SAI_RFCR_FWDE_MASK)
#define SAI_RFCR_FRDE_MASK                                 (0xFU)
#define SAI_RFCR_FRDE_SHIFT                                (0U)
#define SAI_RFCR_FRDE(x)                                   (((uint32)(((uint32)(x)) << SAI_RFCR_FRDE_SHIFT)) & SAI_RFCR_FRDE_MASK)
/* SAI_RCFG0 Register */
#define SAI_RCFG0_SYNC_MASK                                (0xC0000000U)
#define SAI_RCFG0_SYNC_SHIFT                               (30U)
#define SAI_RCFG0_SYNC(x)                                  (((uint32)(((uint32)(x)) << SAI_RCFG0_SYNC_SHIFT)) & SAI_RCFG0_SYNC_MASK)
#define SAI_RCFG0_BCS_MASK                                 (0x20000000U)
#define SAI_RCFG0_BCS_SHIFT                                (29U)
#define SAI_RCFG0_BCS(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG0_BCS_SHIFT)) & SAI_RCFG0_BCS_MASK)
#define SAI_RCFG0_BCI_MASK                                 (0x10000000U)
#define SAI_RCFG0_BCI_SHIFT                                (28U)
#define SAI_RCFG0_BCI(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG0_BCI_SHIFT)) & SAI_RCFG0_BCI_MASK)
#define SAI_RCFG0_BCP_MASK                                 (0x2000000U)
#define SAI_RCFG0_BCP_SHIFT                                (25U)
#define SAI_RCFG0_BCP(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG0_BCP_SHIFT)) & SAI_RCFG0_BCP_MASK)
#define SAI_RCFG0_BCD_MASK                                 (0x1000000U)
#define SAI_RCFG0_BCD_SHIFT                                (24U)
#define SAI_RCFG0_BCD(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG0_BCD_SHIFT)) & SAI_RCFG0_BCD_MASK)
#define SAI_RCFG0_DIV_MASK                                 (0xFFU)
#define SAI_RCFG0_DIV_SHIFT                                (0U)
#define SAI_RCFG0_DIV(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG0_DIV_SHIFT)) & SAI_RCFG0_DIV_MASK)
/* SAI_RCFG1 Register */
#define SAI_RCFG1_RCE_MASK                                 (0xF0000U)
#define SAI_RCFG1_RCE_SHIFT                                (16U)
#define SAI_RCFG1_RCE(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG1_RCE_SHIFT)) & SAI_RCFG1_RCE_MASK)
#define SAI_RCFG1_WDFL_MASK                                (0xFU)
#define SAI_RCFG1_WDFL_SHIFT                               (0U)
#define SAI_RCFG1_WDFL(x)                                  (((uint32)(((uint32)(x)) << SAI_RCFG1_WDFL_SHIFT)) & SAI_RCFG1_WDFL_MASK)
/* SAI_RCFG2 Register */
#define SAI_RCFG2_FRSZ_MASK                                (0xF0000U)
#define SAI_RCFG2_FRSZ_SHIFT                               (16U)
#define SAI_RCFG2_FRSZ(x)                                  (((uint32)(((uint32)(x)) << SAI_RCFG2_FRSZ_SHIFT)) & SAI_RCFG2_FRSZ_MASK)
#define SAI_RCFG2_SYWD_MASK                                (0x1F00U)
#define SAI_RCFG2_SYWD_SHIFT                               (8U)
#define SAI_RCFG2_SYWD(x)                                  (((uint32)(((uint32)(x)) << SAI_RCFG2_SYWD_SHIFT)) & SAI_RCFG2_SYWD_MASK)
#define SAI_RCFG2_MF_MASK                                  (0x10U)
#define SAI_RCFG2_MF_SHIFT                                 (4U)
#define SAI_RCFG2_MF(x)                                    (((uint32)(((uint32)(x)) << SAI_RCFG2_MF_SHIFT)) & SAI_RCFG2_MF_MASK)
#define SAI_RCFG2_FSE_MASK                                 (0x8U)
#define SAI_RCFG2_FSE_SHIFT                                (3U)
#define SAI_RCFG2_FSE(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG2_FSE_SHIFT)) & SAI_RCFG2_FSE_MASK)
#define SAI_RCFG2_ONDEM_MASK                               (0x4U)
#define SAI_RCFG2_ONDEM_SHIFT                              (2U)
#define SAI_RCFG2_ONDEM(x)                                 (((uint32)(((uint32)(x)) << SAI_RCFG2_ONDEM_SHIFT)) & SAI_RCFG2_ONDEM_MASK)
#define SAI_RCFG2_FSP_MASK                                 (0x2U)
#define SAI_RCFG2_FSP_SHIFT                                (1U)
#define SAI_RCFG2_FSP(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG2_FSP_SHIFT)) & SAI_RCFG2_FSP_MASK)
#define SAI_RCFG2_FSD_MASK                                 (0x1U)
#define SAI_RCFG2_FSD_SHIFT                                (0U)
#define SAI_RCFG2_FSD(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG2_FSD_SHIFT)) & SAI_RCFG2_FSD_MASK)
/* SAI_RCFG3 Register */
#define SAI_RCFG3_WNW_MASK                                 (0x1F000000U)
#define SAI_RCFG3_WNW_SHIFT                                (24U)
#define SAI_RCFG3_WNW(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG3_WNW_SHIFT)) & SAI_RCFG3_WNW_MASK)
#define SAI_RCFG3_W0W_MASK                                 (0x1F0000U)
#define SAI_RCFG3_W0W_SHIFT                                (16U)
#define SAI_RCFG3_W0W(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG3_W0W_SHIFT)) & SAI_RCFG3_W0W_MASK)
#define SAI_RCFG3_FBT_MASK                                 (0x1FU)
#define SAI_RCFG3_FBT_SHIFT                                (0U)
#define SAI_RCFG3_FBT(x)                                   (((uint32)(((uint32)(x)) << SAI_RCFG3_FBT_SHIFT)) & SAI_RCFG3_FBT_MASK)
/* SAI_RDATA Register */
#define SAI_RDATA_RDATA_MASK                               (0xFFFFFFFFU)
#define SAI_RDATA_RDATA_SHIFT                              (0U)
#define SAI_RDATA_RDATA(x)                                 (((uint32)(((uint32)(x)) << SAI_RDATA_RDATA_SHIFT)) & SAI_RDATA_RDATA_MASK)
/* SAI_RFR Register */
#define SAI_RFR_RCS_MASK                                   (0x80000000U)
#define SAI_RFR_RCS_SHIFT                                  (31U)
#define SAI_RFR_RCS(x)                                     (((uint32)(((uint32)(x)) << SAI_RFR_RCS_SHIFT)) & SAI_RFR_RCS_MASK)
#define SAI_RFR_WFP_MASK                                   (0xF0000U)
#define SAI_RFR_WFP_SHIFT                                  (16U)
#define SAI_RFR_WFP(x)                                     (((uint32)(((uint32)(x)) << SAI_RFR_WFP_SHIFT)) & SAI_RFR_WFP_MASK)
#define SAI_RFR_RFP_MASK                                   (0xFU)
#define SAI_RFR_RFP_SHIFT                                  (0U)
#define SAI_RFR_RFP(x)                                     (((uint32)(((uint32)(x)) << SAI_RFR_RFP_SHIFT)) & SAI_RFR_RFP_MASK)
/* SAI_RMR Register */
#define SAI_RMR_RWM_MASK                                   (0xFFFFU)
#define SAI_RMR_RWM_SHIFT                                  (0U)
#define SAI_RMR_RWM(x)                                     (((uint32)(((uint32)(x)) << SAI_RMR_RWM_SHIFT)) & SAI_RMR_RWM_MASK)
/* SAI_MCLK_CTRL Register */
#define SAI_MCLK_CTRL_CLKSEL_MASK                          (0x30U)
#define SAI_MCLK_CTRL_CLKSEL_SHIFT                         (4U)
#define SAI_MCLK_CTRL_CLKSEL(x)                            (((uint32)(((uint32)(x)) << SAI_MCLK_CTRL_CLKSEL_SHIFT)) & SAI_MCLK_CTRL_CLKSEL_MASK)
#define SAI_MCLK_CTRL_MCKEN_MASK                           (0x1U)
#define SAI_MCLK_CTRL_MCKEN_SHIFT                          (0U)
#define SAI_MCLK_CTRL_MCKEN(x)                             (((uint32)(((uint32)(x)) << SAI_MCLK_CTRL_MCKEN_SHIFT)) & SAI_MCLK_CTRL_MCKEN_MASK)
/* SAI_MCLK_FRAC_DIV Register */
#define SAI_MCLK_FRAC_DIV_NUME_MASK                        (0xFF000U)
#define SAI_MCLK_FRAC_DIV_NUME_SHIFT                       (12U)
#define SAI_MCLK_FRAC_DIV_NUME(x)                          (((uint32)(((uint32)(x)) << SAI_MCLK_FRAC_DIV_NUME_SHIFT)) & SAI_MCLK_FRAC_DIV_NUME_MASK)
#define SAI_MCLK_FRAC_DIV_DENO_MASK                        (0xFFFU)
#define SAI_MCLK_FRAC_DIV_DENO_SHIFT                       (0U)
#define SAI_MCLK_FRAC_DIV_DENO(x)                          (((uint32)(((uint32)(x)) << SAI_MCLK_FRAC_DIV_DENO_SHIFT)) & SAI_MCLK_FRAC_DIV_DENO_MASK)


/**
 * @}
 */ /* end of group SAI_Register_Masks */


/**
 * @}
 */ /* end of group SAI_Peripheral_Access_Layer */

#endif /* PSIP_SAI_H */

