export * from './parser.ts';

import { LDF, Frame, SchTable } from './parser.ts'
import Schema from 'async-validator';
import { Rules } from 'async-validator'

Schema.warning = function () { null };

export async function globalValid(ldf: LDF, hwList:string[]) {
    const ret: Record<string, string> = {}
        const schErrors = await schsValid(ldf)
    for (const key of Object.keys(schErrors)) {
        for (const skey of Object.keys(schErrors[key])) {
            const id = `Sch-${key}-${skey}`
            ret[id] = schErrors[key][skey]
        }
    }
        const signalsErrors = await signalsValid(ldf)
    for (const key of Object.keys(signalsErrors)) {
       for(const skey of Object.keys(signalsErrors[key])){
        const id = `Signal-${key}-${skey}`
        ret[id] = signalsErrors[key][skey]
       }
    }
        const framesErrors = await framesValid(ldf)
    for (const key of Object.keys(framesErrors)) {
        for(const skey of Object.keys(framesErrors[key])){
            const id = `Frame-${key}-${skey}`
            ret[id] = framesErrors[key][skey]
        }
    }
        const nodesErrors = await nodesAttrValid(ldf)
    for (const key of Object.keys(nodesErrors)) {
        for(const skey of Object.keys(nodesErrors[key])){
            const id = `Node-${key}-${skey}`
            ret[id] = nodesErrors[key][skey]
        }
    }
        const generalErrors = await generalValid(ldf,hwList)
    for(const key of Object.keys(generalErrors)){
        const id = `General-${key}`
        ret[id] = generalErrors[key]
    }
        return ret
    
}

export function getFrameSignalsBits(ldfObj: LDF, frameName: string): number {
    if (frameName in ldfObj.frames) {
        const frame = ldfObj.frames[frameName]
        if (frame.signals.length == 0) {
            return 0
        }
        let maxbits = 0
        for (const signal of frame.signals) {
            
            const bit=signal.offset + ldfObj.signals[signal.name].signalSizeBits||0
            if (bit > maxbits) {
                maxbits = bit
            }
            
        }
        return maxbits
    }

    if (frameName in ldfObj.eventTriggeredFrames) {
        const frames = ldfObj.eventTriggeredFrames[frameName].frameNames
        let maxbit = 0;
        for (const fn of frames) {
            const bits = getFrameSignalsBits(ldfObj, fn)
            if (bits > maxbit) {
                maxbit = bits
            }
        }

        return maxbit
    }

    if (frameName in ldfObj.sporadicFrames) {
        const frames = ldfObj.sporadicFrames[frameName].frameNames
        let maxbit = 0;
        for (const fn of frames) {
            const bits = getFrameSignalsBits(ldfObj, fn)
            if (bits > maxbit) {
                maxbit = bits
            }
        }

        return maxbit
    }
    return 0
    // if(frameName in ldfObj.value.)

}



export function getFrameSize(ldfObj: LDF, frameName: string): number {
    if (frameName in ldfObj.frames) {
        const frame = ldfObj.frames[frameName]
        if (frame.signals.length == 0) {
            return 0
        }
        return frame.frameSize
    }

    if (frameName in ldfObj.eventTriggeredFrames) {
        const frames = ldfObj.eventTriggeredFrames[frameName].frameNames
        let maxbit = 0;
        for (const fn of frames) {
            const bits = getFrameSize(ldfObj, fn)
            if (bits > maxbit) {
                maxbit = bits
            }
        }

        return maxbit
    }

    if (frameName in ldfObj.sporadicFrames) {
        const frames = ldfObj.sporadicFrames[frameName].frameNames
        let maxbit = 0;
        for (const fn of frames) {
            const bits = getFrameSize(ldfObj, fn)
            if (bits > maxbit) {
                maxbit = bits
            }
        }

        return maxbit
    }
    return 0
    // if(frameName in ldfObj.value.)

}


export function getConfigFrames(ldfObj: LDF, nodeName: string): string[] {
    const list: string[] = []
    for (const sig of Object.keys(ldfObj.signals)) {
        if (ldfObj.signals[sig].punishedBy == nodeName) {
            list.push(sig)
        } else if (ldfObj.signals[sig].subscribedBy.indexOf(nodeName) != -1) {
            list.push(sig)
        }
    }
    const frames: string[] = []
    for (const frame of Object.keys(ldfObj.frames)) {
        const sigs = ldfObj.frames[frame].signals
        for (const s of list) {
            sigs.forEach((sig) => {
                if (sig.name == s) {
                    frames.push(frame)
                }
            });
        }
    }

    const lastFrames = [...new Set(frames)]
    for (const e of Object.keys(ldfObj.eventTriggeredFrames)) {
        for (const f of ldfObj.eventTriggeredFrames[e].frameNames) {
            lastFrames.forEach((frame) => {
                if (frame == f) {
                    lastFrames.push(e)
                }
            })
        }

    }
    return [...new Set(lastFrames)]
}


export async function nodeAttibuteValid(ldfObj: LDF, nodeName: string, nodeAttr: any) {
    const ret: Record<string, string> = {}
    function faultStateSignalsCheck(rule, value, callback) {
        if (value.length == 0) {
            callback();
        } else {
            const signals = Object.keys(ldfObj.signals)
            //make sure all value in signals
            for (const v of value) {
                if (signals.indexOf(v) == -1) {
                    callback(new Error('signal not defined: ' + v));
                    return
                }
            }
        }
        callback();
    }
    function configFramesCheck(rule, value, callback) {
        if (value.length == 0) {
            callback();
        } else {
            const frames = getConfigFrames(ldfObj, nodeName)
            //make sure all value in frames
            for (const v of value) {
                if (frames.indexOf(v) == -1) {
                    callback(new Error('frame not defined: ' + v));
                    return
                }
            }
        }
        callback();
    }
    const MasterRules: Rules = {
        LIN_protocol: { type: 'enum', required: true, enum: ["2.2"] },
        configured_NAD: { type: 'number', required: true, min: 1, max: 125, transform: (v) => Number(v) }, /* 0 is sleep,126,127 for dia*/
        initial_NAD: { type: 'number', required: false, min: 1, max: 125, transform: (v) => v ? Number(v) : 1 },
        supplier_id: { type: 'number', required: true, min: 0, max: 65535, transform: (v) => Number(v) },
        function_id: { type: 'number', required: true, min: 0, max: 65535, transform: (v) => Number(v) },
        variant: { type: 'number', required: false, min: 0, max: 255, transform: (v) => Number(v) },
        response_error: { type: 'enum', required: true, enum: Object.keys(ldfObj.signals) },
        fault_state_signals: { type: 'array', required: false, validator: faultStateSignalsCheck },
        P2_min: { type: 'number', required: false, },
        ST_min: { type: 'number', required: false, },
        N_As_timeout: { type: 'number', required: false, },
        N_Cr_timeout: { type: 'number', required: false, },
        configFrames: { type: 'array', required: false, validator: configFramesCheck },
    }
    const masterValidator = new Schema(MasterRules);
    try {
        await masterValidator.validate(nodeAttr)
    } catch ({ errors, fields }) {
        for (const e of errors) {
            ret[e.field] = e.message
        }
    }
    return ret
}
export async function nodesAttrValid(ldfObj: LDF){
    const ret: Record<string, Record<string, string>> = {}
    for (const nodeName of ldfObj.node.salveNode) {
        ret[nodeName] = await nodeAttibuteValid(ldfObj, nodeName, ldfObj.nodeAttrs[nodeName])
    }
    return ret
}

export async function frameValid(ldfObj: LDF, frame: Frame) {
    const ret: Record<string, string> = {}
    function signalCheck(rule, value, callback) {
        if (value.length == 0) {
            callback(new Error('frame must have at least one signal'));
        } else {
            /* check all signals belong the frame, make sure all signals size <= 64, and check overlap*/
            const signals = Object.keys(ldfObj.signals)
            const signalBits: number[][] = []
            for (const v of value) {
                if (signals.indexOf(v.name) == -1) {
                    callback(new Error('signal not defined: ' + v.name));
                    return
                }
                if (v.offset + ldfObj.signals[v.name].signalSizeBits > 64) {
                    callback(new Error('signal size > 64: ' + v.name));
                    return
                }
                for (const bit of signalBits) {
                    if (v.offset < bit[1] && v.offset + ldfObj.signals[v.name].signalSizeBits > bit[0]) {
                        callback(new Error('signal overlap: ' + v.name));
                        return
                    }
                }
                signalBits.push([v.offset, v.offset + ldfObj.signals[v.name].signalSizeBits])
            }
        }
        callback();
    }

    function frameSizeCheck(rule, value, callback) {
        if (value > 8) {
            callback(new Error('frame size > 8: ' + value));
            return
        }
        /* make sure frame size >= all signals size belong the frame */
        const bits = getFrameSignalsBits(ldfObj, frame.name)
        if (bits > value * 8) {
            callback(new Error('frame size < signals size: ' + value));
            return
        }
        callback();
    }

    const MasterRules: Rules = {
        name: { type: 'string', required: true },
        id: { type: 'number', required: true, min: 0, max: 0x3D, transform: (v) => Number(v) },
        frameSize: { type: 'number', required: true, validator: frameSizeCheck },
        publishedBy: { type: 'enum', required: true, enum: [ldfObj.node.master.nodeName, ...ldfObj.node.salveNode] },
        signals: { type: 'array', required: true, validator: signalCheck },

    }
    const masterValidator = new Schema(MasterRules);
    try {
        await masterValidator.validate(frame)
    } catch ({ errors, fields }) {
        for (const e of errors) {
            ret[e.field] = e.message
        }
    }
    return ret
}

export async function framesValid(ldfObj: LDF) {
    const frames = ldfObj.frames
    const ret: Record<string, Record<string, string>> = {}
    for (const key of Object.keys(frames)) {
        ret[key] = await frameValid(ldfObj, frames[key])
    }
    return ret

}

export async function signalValid(ldfObj: LDF, signalDef: any) {
    const ret: Record<string, string> = {}
    function publishCheck(rule, value, callback) {
        if(ldfObj.node.master.nodeName){
            const nodeList = [ldfObj.node.master.nodeName]
            for (const node of ldfObj.node.salveNode) {
                nodeList.push(node)
            }
            /* make sure all value in nodeList */
            if (nodeList.indexOf(value) == -1) {
                callback(new Error('node not defined: ' + value));
                return
            }
            /* make sure value not in subscribedBy */
            if (signalDef.subscribedBy.indexOf(value) != -1) {
                callback(new Error('node already subscribed: ' + value));
                return
            }
        }else{
            callback(new Error('master node name is empty'));
        }
        callback();
    }
    function subscribedCheck(rule, value, callback) {
        if(value.length==0){
            callback(new Error('signal must have at least one subscribed node'));
        }
        if(ldfObj.node.master.nodeName){
            const nodeList = [ldfObj.node.master.nodeName]
            for (const node of ldfObj.node.salveNode) {
                nodeList.push(node)
            }
            /* make sure all value in nodeList */
            for (const v of value) {
                if (nodeList.indexOf(v) == -1) {
                    callback(new Error('node not defined: ' + v));
                    return
                }
            }
            /* make sure all value not in punishedBy */
            for (const v of value) {
                if (signalDef.punishedBy == v) {
                    callback(new Error('node already punished: ' + v));
                    return
                }
            }
        }else{
            callback(new Error('master node name is empty'));
        }
        callback();
    }
    function sizeCheck(rule, value, callback) {
        if (value > 64) {
            callback(new Error('signal size > 64: ' + value));
            return
        }
        if (signalDef.singleType == 'Scalar') {
            if (value > 16) {
                callback(new Error('signal size > 16: ' + value));
                return
            }
        }
        callback();
    }
    function initValueCheck(rule, value, callback) {
        if (Array.isArray(value)) {
            /* make sure all value 0-255 */
            for (const [index, v] of value.entries()) {
                const nv = Number(v)
                if (nv < 0 || nv > 255) {
                    callback(new Error(`${index},init value from 0-255`));
                    return
                }
            }
        } else {
            value = Number(value)
            if (value < 0 || value > 255) {
                callback(new Error('init value from 0-255'));
                return
            }
        }
        callback();
    }

    const MasterRules: Rules = {
        encode: { type: 'enum', required: false, enum: [...Object.keys(ldfObj.signalRep), ''] },
        signalName: { type: 'string', required: true },
        signalSizeBits: { type: 'number', required: true, validator: sizeCheck },
        initValue: { required: true, validator: initValueCheck },
        punishedBy: { type: 'string', required: true, validator: publishCheck },
        subscribedBy: { type: 'array',required: true, validator: subscribedCheck },
        singleType: { type: 'enum', required: true, enum: ["Scalar", "ByteArray"] },
    }
    const masterValidator = new Schema(MasterRules);
    try {
        await masterValidator.validate(signalDef)
    } catch ({ errors, fields }) {
        for (const e of errors) {
            ret[e.field] = e.message
        }
    }
    return ret
}

export async function signalsValid(ldfObj: LDF) {
    const signals = ldfObj.signals
    const ret: Record<string, Record<string, string>> = {}
    for (const key of Object.keys(signals)) {
        ret[key] = await signalValid(ldfObj, signals[key])
    }
    return ret
}



export async function schValid(ldfObj: LDF, sch: SchTable) {

    const ret: Record<string, string> = {}
    function entryCheck(rule, value, callback) {
        for (const [index, e] of value.entries()) {
            const buadrate = ldfObj.global.LIN_speed
            
            let bytes=0;
            if(e.isCommand){
                bytes = 8
            }else{
                bytes = getFrameSize(ldfObj, e.name)
            }
            const frameMinTime = (bytes * 10 + 44) * (1 / buadrate)
            const frameMaxTime = frameMinTime * 1.4
            /* make sure frame size >= all signals size belong the frame */
            if (e.delay < frameMaxTime) {
                callback(new Error(`entry ${index},delay < frameMaxTime: ${e.delay}`));
            }

            if(e.isCommand){
                if(e.name=='AssignNAD'){
                    if(e.AssignNAD.nodeName){
                        /* must one of slavenode */
                        if(ldfObj.node.salveNode.indexOf(e.AssignNAD.nodeName)==-1){
                            callback(new Error(`entry ${index},AssignNAD.nodeName not in slavenode: ${e.name}`));
                        }
                    }else{
                        callback(new Error(`entry ${index},AssignNAD.nodeName is empty: ${e.name}`));
                    }
                }
                else if(e.name=='ConditionalChangeNAD'){
                    if(e.ConditionalChangeNAD.nad==undefined||e.ConditionalChangeNAD.nad<1||e.ConditionalChangeNAD.nad>125){
                        callback(new Error(`entry ${index},ConditionalChangeNAD.nad not in range: ${e.name}`));
                    }
                    if(e.ConditionalChangeNAD.id==undefined||e.ConditionalChangeNAD.id<0||e.ConditionalChangeNAD.id>255){
                        callback(new Error(`entry ${index},ConditionalChangeNAD.id not in range: ${e.name}`));
                    }
                    if(e.ConditionalChangeNAD.byte==undefined||e.ConditionalChangeNAD.byte<0||e.ConditionalChangeNAD.byte>255){
                        callback(new Error(`entry ${index},ConditionalChangeNAD.byte not in range: ${e.name}`));
                    }
                    if(e.ConditionalChangeNAD.mask==undefined||e.ConditionalChangeNAD.mask<0||e.ConditionalChangeNAD.mask>255){
                        callback(new Error(`entry ${index},ConditionalChangeNAD.mask not in range: ${e.name}`));
                    }
                    if(e.ConditionalChangeNAD.inv==undefined||e.ConditionalChangeNAD.inv<0||e.ConditionalChangeNAD.inv>255){
                        callback(new Error(`entry ${index},ConditionalChangeNAD.inv not in range: ${e.name}`));
                    }
                    if(e.ConditionalChangeNAD.newNad==undefined||e.ConditionalChangeNAD.newNad<1||e.ConditionalChangeNAD.newNad>125){
                        callback(new Error(`entry ${index},ConditionalChangeNAD.newNad not in range: ${e.name}`));
                    }
                }
                else if(e.name=='DataDump'){
                    if(e.DataDump.nodeName){
                        /* must one of slavenode */
                        if(ldfObj.node.salveNode.indexOf(e.DataDump.nodeName)==-1){
                            callback(new Error(`entry ${index},DataDump.nodeName not in slavenode: ${e.name}`));
                        }
                    }else{
                        callback(new Error(`entry ${index},DataDump.nodeName is empty: ${e.name}`));
                    }
                    for(const v of ['D1','D2','D3','D4','D5']){
                        if(e.DataDump[v]==undefined||e.DataDump[v]<0||e.DataDump[v]>255){
                            callback(new Error(`entry ${index},DataDump.${v} not in range: ${e.name}`));
                        }
                    }
                }
                else if(e.name=='SaveConfiguration'){
                    if(e.SaveConfiguration.nodeName){
                        /* must one of slavenode */
                        if(ldfObj.node.salveNode.indexOf(e.SaveConfiguration.nodeName)==-1){
                            callback(new Error(`entry ${index},SaveConfiguration.nodeName not in slavenode: ${e.name}`));
                        }
                    }else{
                        callback(new Error(`entry ${index},SaveConfiguration.nodeName is empty: ${e.name}`));
                    }
                }
                else if(e.name=='AssignFrameIdRange'){
                    if(e.AssignFrameIdRange.nodeName){
                        /* must one of slavenode */
                        if(ldfObj.node.salveNode.indexOf(e.AssignFrameIdRange.nodeName)==-1){
                            callback(new Error(`entry ${index},AssignFrameIdRange.nodeName not in slavenode: ${e.name}`));
                        }
                    }else{
                        callback(new Error(`entry ${index},AssignFrameIdRange.nodeName is empty: ${e.name}`));
                    }

                    if(e.AssignFrameIdRange.frameIndex==undefined||e.AssignFrameIdRange.frameIndex<0||e.AssignFrameIdRange.frameIndex>255){
                        callback(new Error(`entry ${index},AssignFrameIdRange.frameIndex not in range: ${e.name}`));
                    }

                    if(e.AssignFrameIdRange.framePID){
                        for(const pid of e.AssignFrameIdRange.framePID){
                            /* pid check valid 0-0x3b or 0xff, others report error*/
                            if(pid==0xff){
                                continue
                            }
                            if(pid<0||pid>0x3b){
                                callback(new Error(`entry ${index},AssignFrameIdRange.framePID not in range: ${e.name}`));
                            }
                            
                        }
                    }
                    
                }
                else if(e.name=='FreeFormat'){
                    for(const d of e.FreeFormat.D){
                        if(d==undefined||d<0||d>255){
                            callback(new Error(`entry ${index},FreeFormat.D not in range: ${e.name}`));
                        }
                    }
                }
                else if(e.name=='AssignFrameId'){
                    if(e.AssignFrameId.nodeName){
                        /* must one of slavenode */
                        if(ldfObj.node.salveNode.indexOf(e.AssignFrameId.nodeName)==-1){
                            callback(new Error(`entry ${index},AssignFrameId.nodeName not in slavenode: ${e.name}`));
                        }
                    }else{
                        callback(new Error(`entry ${index},AssignFrameId.nodeName is empty: ${e.name}`));
                    }

                    if(e.AssignFrameId.frameName){
                        if(!ldfObj.frames.hasOwnProperty(e.AssignFrameId.frameName)){
                            if(!ldfObj.eventTriggeredFrames.hasOwnProperty(e.AssignFrameId.frameName)){
                                callback(new Error(`entry ${index},AssignFrameId.frameName not defined: ${e.name}`));
                            }
                        }   
                    }else{
                        callback(new Error(`entry ${index},AssignFrameId.frameName is empty: ${e.name}`));
                    }
                }
            }

            /* duplicate frame name check */
            if(e.name in ldfObj.frames){
                /* then can't in event and sporadic frames */
                if(e.name in ldfObj.eventTriggeredFrames){
                    callback(new Error(`entry ${index},frame name duplicate: ${e.name}`));
                }
                if(e.name in ldfObj.sporadicFrames){
                    callback(new Error(`entry ${index},frame name duplicate: ${e.name}`));
                }
            }
            /* duplicate event and sporadic frames name check */
            if(e.name in ldfObj.eventTriggeredFrames){
                /* then can't in frames */
                if(e.name in ldfObj.frames){
                    callback(new Error(`entry ${index},frame name duplicate: ${e.name}`));
                }
                if(e.name in ldfObj.sporadicFrames){
                    callback(new Error(`entry ${index},frame name duplicate: ${e.name}`));
                }
            }

            if(e.name in ldfObj.sporadicFrames){
                /* then can't in frames */
                if(e.name in ldfObj.frames){
                    callback(new Error(`entry ${index},frame name duplicate: ${e.name}`));
                }
                if(e.name in ldfObj.eventTriggeredFrames){
                    callback(new Error(`entry ${index},frame name duplicate: ${e.name}`));
                }
            }


            /* event frame check */
            if(e.name in ldfObj.eventTriggeredFrames){
                const eventFrame = ldfObj.eventTriggeredFrames[e.name]
                /* make sure event frame is not empty */
                if(eventFrame.name==''){
                    callback(new Error(`entry ${index},event frame is empty: ${e.name}`));
                }
               
                /* frame id check */
                if(eventFrame.frameId<0||eventFrame.frameId>0x3b){
                    callback(new Error(`entry ${index},event frame id not in range: ${e.name}`));
                }

                if(eventFrame.frameNames.length == 0){
                    callback(new Error(`entry ${index},event frame is empty: ${e.name}`));
                }
                /* frameName check, make sure all value in frames, all frames has same size, and published by different slave nodes*/
                const frameNames = eventFrame.frameNames;
                let frameSize=0;
                const slaveNodes = new Set<string>();
                for (const frameName of frameNames) {
                    if (!ldfObj.frames.hasOwnProperty(frameName)) {
                        callback(new Error(`entry ${index},frame not defined: ${frameName}`));
                    }
                    if(frameSize==0){
                        frameSize = ldfObj.frames[frameName].frameSize
                    } else{
                        if(ldfObj.frames[frameName].frameSize!=frameSize){
                            callback(new Error(`entry ${index},frames size not same: ${frameName}`));
                        }
                    }
                    const frame = ldfObj.frames[frameName];
                    slaveNodes.add(frame.publishedBy);
                }
               
                if (slaveNodes.size < frameNames.length) {
                    callback(new Error(`entry ${index},frames published by same slave node: ${frameNames}`));
                }
                

                
            }

            /* sporadic frame check */
            if(e.name in ldfObj.sporadicFrames){
                /* make sure sporadic frame is not empty */
                const sporadicFrame = ldfObj.sporadicFrames[e.name]
                if(sporadicFrame.name==''){
                    callback(new Error(`entry ${index},sporadic frame is empty: ${e.name}`));
                }
                if(sporadicFrame.frameNames.length == 0){
                    callback(new Error(`entry ${index},sporadic frame is empty: ${e.name}`));
                }
                /* make sure frameNames publisd is master node */
                const frameNames = sporadicFrame.frameNames;
                for (const frameName of frameNames) {
                    if (!ldfObj.frames.hasOwnProperty(frameName)) {
                        callback(new Error(`entry ${index},frame not defined: ${frameName}`));
                    }
                    const frame = ldfObj.frames[frameName];
                    if(frame.publishedBy!=ldfObj.node.master.nodeName){
                        callback(new Error(`entry ${index},frame not published by master node: ${frameName}`));
                    }
                }
            }
            
        }
        callback();
    }

    const MasterRules: Rules = {
        name: { type: 'string', required: true },
        entries: { type: 'array', validator: entryCheck },

    }
    const masterValidator = new Schema(MasterRules);
    try {
        await masterValidator.validate(sch)
    } catch ({ errors, fields }) {
        for (const e of errors) {
            ret[e.field] = e.message
        }
    }
    return ret
}


export async function schsValid(ldfObj: LDF){
  
    const ret: Record<string, Record<string, string>> = {}
    for (const sch of ldfObj.schTables) {
        ret[sch.name] = await schValid(ldfObj, sch)
    }
    return ret
}


export function getDefLdf():LDF{
    return {
        global: {
          LIN_protocol_version: "2.2",
          LIN_language_version: "2.2",
          LIN_speed: 19.2,
          hwInst: '',
          maxIdleTimeoutMs:4000,
          targetNode:'',
          diagClass:1,
          supportId:[],
          coreErrCb:'NULL',
          tpEnable:false,
          tpRxQueueSize:10,
          tpTxQueueSize:10,
        },
        node: {
          master: {
            nodeName: "Master",
            timeBase: 1,
            jitter: 0.1
          },
          salveNode: []
        },
        nodeAttrs: {
      
        },
        signals: {
      
        },
        signalGroups: [],
        frames: {},
        sporadicFrames: {},
        eventTriggeredFrames: {},
        schTables: [],
        signalEncodTypes: {},
        signalRep: {}
      }
}

export async function generalValid(ldfObj: LDF, hwList:string[]){
    const ret: Record<string, string> = {}
    
    function entryCheck(rule, value, callback) {
        /* value must be one of hwList*/ 
        if(hwList.indexOf(value)==-1){
            callback(new Error(`hwInst not in hwList: ${hwList}`));
        }
        /* make sure hwList does not have duplicate value */
        const hwListSet = new Set<string>(hwList)
        if(hwListSet.size!=hwList.length){
            callback(new Error(`hwList has duplicate value`));
        }
        callback()
    }

    function schTimeoutCheck(rule, value, callback) {
        /* value must be one of hwList*/ 
        if(ldfObj.global.targetNode!=ldfObj.node.master.nodeName||ldfObj.global.tpEnable==false){
            callback()
        }else{
          
            if(value){
                /* value should more than all slave node st min */
                let maxStMin=0
                for(const attr of Object.values(ldfObj.nodeAttrs)){
                    if(attr.ST_min)
                        if(attr.ST_min>maxStMin){
                            maxStMin=attr.ST_min
                        }
                }
                if(value<maxStMin){
                    callback(new Error(`schTimeout < maxStMin: ${maxStMin}`));
                }
            }else{
                callback(new Error(`schTimeout is empty`));
            }
        }
        callback()
    }

    // function delayCheck(rule, value, callback) {
    //     /* value must be one of hwList*/ 
    //     const bytes=8;
    //     const buadrate = ldfObj.global.LIN_speed
    //     const frameMinTime = (bytes * 10 + 44) * (1 / buadrate)
    //     const frameMaxTime = frameMinTime * 1.4
    //     if(value<frameMaxTime){
    //         callback(new Error(`delay < frameMaxTime: ${frameMaxTime}`));
    //     }
    //     callback()
    // }
    const MasterRules: Rules = {
        hwInst: { type: 'string', required: true, validator: entryCheck },
        targetNode: { type: 'enum', required: true,enum:[ldfObj.node.master.nodeName,...ldfObj.node.salveNode]},
        tpEnable: {type:'boolean',required:true},
        tpTxQueueSize: {type:'number',required:true,min:2},
        tpRxQueueSize: {type:'number',required:true,min:2},
        schTimeout:{type:'number',validator:schTimeoutCheck},
        maxIdleTimeoutMs:{type:'number',required:true,min:100},
        // masterReqDelay: { type: 'number', required: true,validator: delayCheck },
        // slaveRespDelay: { type: 'number', required: true,validator: delayCheck },
    }
    const masterValidator = new Schema(MasterRules);
    try {
        await masterValidator.validate(ldfObj.global)
    } catch ({ errors, fields }) {
        for (const e of errors) {
            ret[e.field] = e.message
        }
    }
    return ret 
}

