/**
 * NVIC (Nested Vectored Interrupt Controller) Viewer Plugin for SEGGER Ozone
 * 
 * This plugin reads and displays ARM Cortex-M NVIC interrupt status including:
 * - Interrupt names from sdk.json
 * - Enable/disable status from ISER registers
 * - Pending status from ISPR registers
 * - Priority levels from IPR registers
 * 
 * @author Run Ma
 * @version 1.0
 */

// NVIC Register Base Addresses (ARM Cortex-M)
var _NVIC_BASE                = 0xE000E000;
var _NVIC_ISER_BASE           = 0xE000E100;  // Interrupt Set Enable Registers
var _NVIC_ICER_BASE           = 0xE000E180;  // Interrupt Clear Enable Registers
var _NVIC_ISPR_BASE           = 0xE000E200;  // Interrupt Set Pending Registers
var _NVIC_IPR_BASE            = 0xE000E400;  // Interrupt Priority Registers

// SCB (System Control Block) Register Base Addresses
var _SCB_BASE                 = 0xE000ED00;  // System Control Block Base
var _SCB_ICSR                 = 0xE000ED04;  // Interrupt Control and State Register
var _SCB_SHCSR                = 0xE000ED24;  // System Handler Control and State Register
var _SCB_SHPR1                = 0xE000ED18;  // System Handler Priority Register 1
var _SCB_SHPR2                = 0xE000ED1C;  // System Handler Priority Register 2
var _SCB_SHPR3                = 0xE000ED20;  // System Handler Priority Register 3
var _SCB_DEMCR                = 0xE000EDFC;  // Debug Exception and Monitor Control Register

// SCB SHCSR bit definitions for system exception enable status
var _SHCSR_MEMFAULTENA        = (1 << 16);  // MemManage fault enable
var _SHCSR_BUSFAULTENA        = (1 << 17);  // Bus fault enable
var _SHCSR_USGFAULTENA        = (1 << 18);  // Usage fault enable

// SCB DEMCR bit definitions for debug monitor enable status
var _DEMCR_MON_EN             = (1 << 0);   // Debug monitor enable

// SCB ICSR bit definitions for system exception pending status
var _ICSR_NMIPENDSET          = (1 << 31);  // NMI pending
var _ICSR_PENDSVSET           = (1 << 28);  // PendSV pending
var _ICSR_PENDSTSET           = (1 << 26);  // SysTick pending

// SysTick Register Base Addresses
var _SYSTICK_BASE             = 0xE000E010;  // SysTick Base Address
var _SYSTICK_CTRL             = 0xE000E010;  // SysTick Control and Status Register
var _SYSTICK_CTRL_ENABLE      = (1 << 0);   // SysTick enable bit


// Plugin state variables
var _PRIORITY_BITS            = 3
var _InterruptData = [
    { name: "NMI_IRQn", number: -14 },
    { name: "HardFault_IRQn", number: -13 },
    { name: "MemManage_IRQn", number: -12 },
    { name: "BusFault_IRQn", number: -11 },
    { name: "UsageFault_IRQn", number: -10 },
    { name: "SVC_IRQn", number: -5 },
    { name: "DebugMon_IRQn", number: -4 },
    { name: "PendSV_IRQn", number: -2 },
    { name: "SysTick_IRQn", number: -1 },
    { name: "DMA0_IRQn", number: 0 },
    { name: "DMA1_IRQn", number: 1 },
    { name: "DMA2_IRQn", number: 2 },
    { name: "DMA3_IRQn", number: 3 },
    { name: "DMA4_IRQn", number: 4 },
    { name: "DMA5_IRQn", number: 5 },
    { name: "DMA6_IRQn", number: 6 },
    { name: "DMA7_IRQn", number: 7 },
    { name: "DMA8_IRQn", number: 8 },
    { name: "DMA9_IRQn", number: 9 },
    { name: "DMA10_IRQn", number: 10 },
    { name: "DMA11_IRQn", number: 11 },
    { name: "DMA12_IRQn", number: 12 },
    { name: "DMA13_IRQn", number: 13 },
    { name: "DMA14_IRQn", number: 14 },
    { name: "DMA15_IRQn", number: 15 },
    { name: "DMA_Error_IRQn", number: 16 },
    { name: "FPU_IRQn", number: 17 },
    { name: "EFM_IRQn", number: 18 },
    { name: "EFM_Error_IRQn", number: 19 },
    { name: "PCU_IRQn", number: 20 },
    { name: "EFM_Ecc_IRQn", number: 21 },
    { name: "WDG0_IRQn", number: 22 },
    { name: "I2C0_Master_IRQn", number: 24 },
    { name: "I2C0_Slave_IRQn", number: 25 },
    { name: "SPI0_IRQn", number: 26 },
    { name: "SPI1_IRQn", number: 27 },
    { name: "SPI2_IRQn", number: 28 },
    { name: "I2C1_Master_IRQn", number: 29 },
    { name: "LINFlexD0_IRQn", number: 31 },
    { name: "LINFlexD1_IRQn", number: 33 },
    { name: "LINFlexD2_IRQn", number: 35 },
    { name: "ADC0_IRQn", number: 39 },
    { name: "ACMP0_IRQn", number: 41 },
    { name: "EMU0_SB_IRQn", number: 44 },
    { name: "EMU0_DB_IRQn", number: 45 },
    { name: "RTC_IRQn", number: 46 },
    { name: "RTC_Seconds_IRQn", number: 47 },
    { name: "pTMR0_Ch0_IRQn", number: 48 },
    { name: "pTMR0_Ch1_IRQn", number: 49 },
    { name: "pTMR0_Ch2_IRQn", number: 50 },
    { name: "pTMR0_Ch3_IRQn", number: 51 },
    { name: "PTU0_IRQn", number: 52 },
    { name: "SCU_IRQn", number: 57 },
    { name: "lpTMR0_IRQn", number: 58 },
    { name: "GPIOA_IRQn", number: 59 },
    { name: "GPIOB_IRQn", number: 60 },
    { name: "GPIOC_IRQn", number: 61 },
    { name: "GPIOD_IRQn", number: 62 },
    { name: "GPIOE_IRQn", number: 63 },
    { name: "CAN0_ORed_IRQn", number: 78 },
    { name: "CAN0_Error_IRQn", number: 79 },
    { name: "CAN0_Wake_Up_IRQn", number: 80 },
    { name: "CAN0_ORed_0_15_MB_IRQn", number: 81 },
    { name: "CAN0_ORed_16_31_MB_IRQn", number: 82 },
    { name: "CAN0_ORed_32_47_MB_IRQn", number: 83 },
    { name: "CAN0_ORed_48_63_MB_IRQn", number: 84 },
    { name: "CAN1_ORed_IRQn", number: 85 },
    { name: "CAN1_Error_IRQn", number: 86 },
    { name: "CAN1_Wake_Up_IRQn", number: 87 },
    { name: "CAN1_ORed_0_15_MB_IRQn", number: 88 },
    { name: "CAN1_ORed_16_31_MB_IRQn", number: 89 },
    { name: "CAN2_ORed_IRQn", number: 92 },
    { name: "CAN2_Error_IRQn", number: 93 },
    { name: "CAN2_Wake_Up_IRQn", number: 94 },
    { name: "CAN2_ORed_0_15_MB_IRQn", number: 95 },
    { name: "CAN2_ORed_16_31_MB_IRQn", number: 96 },
    { name: "eTMR0_Ch0_Ch1_IRQn", number: 99 },
    { name: "eTMR0_Ch2_Ch3_IRQn", number: 100 },
    { name: "eTMR0_Ch4_Ch5_IRQn", number: 101 },
    { name: "eTMR0_Ch6_Ch7_IRQn", number: 102 },
    { name: "eTMR0_Fault_IRQn", number: 103 },
    { name: "eTMR0_Ovf_IRQn", number: 104 },
    { name: "eTMR1_Ch0_Ch1_IRQn", number: 105 },
    { name: "eTMR1_Ch2_Ch3_IRQn", number: 106 },
    { name: "eTMR1_Ch4_Ch5_IRQn", number: 107 },
    { name: "eTMR1_Ch6_Ch7_IRQn", number: 108 },
    { name: "eTMR1_Fault_IRQn", number: 109 },
    { name: "eTMR1_Ovf_IRQn", number: 110 },
    { name: "eTMR2_Ch0_Ch1_IRQn", number: 111 },
    { name: "eTMR2_Ch2_Ch3_IRQn", number: 112 },
    { name: "eTMR2_Ch4_Ch5_IRQn", number: 113 },
    { name: "eTMR2_Ch6_Ch7_IRQn", number: 114 },
    { name: "eTMR2_Fault_IRQn", number: 115 },
    { name: "eTMR2_Ovf_IRQn", number: 116 },
    { name: "eTMR3_Ch0_Ch1_IRQn", number: 117 },
    { name: "eTMR3_Ch2_Ch3_IRQn", number: 118 },
    { name: "eTMR3_Ch4_Ch5_IRQn", number: 119 },
    { name: "eTMR3_Ch6_Ch7_IRQn", number: 120 },
    { name: "eTMR3_Fault_IRQn", number: 121 },
    { name: "eTMR3_Ovf_IRQn", number: 122 },
    { name: "TRNG_IRQn", number: 156 },
    { name: "HCU_IRQn", number: 157 },
    { name: "TMR0_Ch0_IRQn", number: 159 },
    { name: "TMR0_Ch1_IRQn", number: 160 },
    { name: "TMR0_Ch2_IRQn", number: 161 },
    { name: "TMR0_Ch3_IRQn", number: 162 },
    { name: "SPI3_IRQn", number: 168 },
    { name: "SENT0_IRQn", number: 180 },
    { name: "WKU_IRQn", number: 190 },
    { name: "ALIGN_0_IRQn", number: 191 }
];
var _CurrentRowIndex          = 0;
var _IsInitialized            = false;

/**
 * Helper function to read 32-bit value from target memory
 * @param {number} address - Memory address to read from
 * @returns {number} 32-bit value at the address
 */
function _ReadU32(address) {
    return TargetInterface.peekWord(address);
}

/**
 * Helper function to write 32-bit value to target memory
 * @param {number} address - Memory address to write to
 * @param {number} value - 32-bit value to write
 */
function _WriteU32(address, value) {
    TargetInterface.pokeWord(address, value);
}


/**
 * Helper function to read 8-bit value from target memory
 * @param {number} address - Memory address to read from
 * @returns {number} 8-bit value at the address
 */
function _ReadU8(address) {
    return TargetInterface.peekBytes(address, 1);
}

/**
 * Convert boolean to Yes/No string
 * @param {boolean} value - Boolean value to convert
 * @returns {string} "Yes" or "No"
 */
function _ToStrBoolYesNo(value) {
    return value ? "Yes" : "No";
}


/**
 * Check if an interrupt is enabled
 * @param {number} irqNumber - Interrupt number (negative for system exceptions, 0+ for external interrupts)
 * @returns {boolean} True if interrupt is enabled
 */
function _IsInterruptEnabled(irqNumber) {
    // System exceptions (negative IRQ numbers) use SCB registers
    if (irqNumber < 0) {
        var shcsrValue = _ReadU32(_SCB_SHCSR);
        
        switch (irqNumber) {
            case -14: // NMI
                return true; // NMI is always enabled
            case -13: // HardFault
                return true; // HardFault is always enabled
            case -12: // MemManage
                return (shcsrValue & _SHCSR_MEMFAULTENA) != 0;
            case -11: // BusFault
                return (shcsrValue & _SHCSR_BUSFAULTENA) != 0;
            case -10: // UsageFault
                return (shcsrValue & _SHCSR_USGFAULTENA) != 0;
            case -5:  // SVC
                return true; // SVC cannot be disabled
            case -4:  // DebugMon
                var demcrValue = _ReadU32(_SCB_DEMCR);
                return (demcrValue & _DEMCR_MON_EN) != 0;
            case -2:  // PendSV
                return true; // PendSV cannot be disabled
            case -1:  // SysTick
                var systickCtrl = _ReadU32(_SYSTICK_CTRL);
                return (systickCtrl & _SYSTICK_CTRL_ENABLE) != 0;
            default:
                return false;
        }
    }
    
    if (irqNumber >= 240) {
        return false;
    }
    
    var registerIndex = Math.floor(irqNumber / 32);
    var bitPosition = irqNumber % 32;
    var iserAddress = _NVIC_ISER_BASE + (registerIndex * 4);
    
    var registerValue = _ReadU32(iserAddress);
    return (registerValue & (1 << bitPosition)) != 0;
}

/**
 * Check if an interrupt is pending
 * @param {number} irqNumber - Interrupt number (negative for system exceptions, 0+ for external interrupts)
 * @returns {boolean} True if interrupt is pending
 */
function _IsInterruptPending(irqNumber) {
    // System exceptions (negative IRQ numbers) use SCB registers
    if (irqNumber < 0) {
        var icsrValue = _ReadU32(_SCB_ICSR);
        var shcsrValue = _ReadU32(_SCB_SHCSR);
        
        switch (irqNumber) {
            case -14: // NMI
                return (icsrValue & _ICSR_NMIPENDSET) != 0;
            case -13: // HardFault
                return (shcsrValue & (1 << 2)) != 0; // SHCSR.HARDFAULTPENDED
            case -12: // MemManage
                return (shcsrValue & (1 << 0)) != 0; // SHCSR.MEMFAULTPENDED
            case -11: // BusFault
                return (shcsrValue & (1 << 1)) != 0; // SHCSR.BUSFAULTPENDED
            case -10: // UsageFault
                return (shcsrValue & (1 << 3)) != 0; // SHCSR.USGFAULTPENDED
            case -5:  // SVC
                return (shcsrValue & (1 << 15)) != 0; // SHCSR.SVCALLPENDED
            case -4:  // DebugMon
                return (shcsrValue & (1 << 8)) != 0; // SHCSR.MONITORPENDED
            case -2:  // PendSV
                return (icsrValue & _ICSR_PENDSVSET) != 0;
            case -1:  // SysTick
                return (icsrValue & _ICSR_PENDSTSET) != 0;
            default:
                return false;
        }
    }
    
    if (irqNumber >= 240) {
        return false;
    }
    
    var registerIndex = Math.floor(irqNumber / 32);
    var bitPosition = irqNumber % 32;
    var isprAddress = _NVIC_ISPR_BASE + (registerIndex * 4);
    
    var registerValue = _ReadU32(isprAddress);
    return (registerValue & (1 << bitPosition)) != 0;
}

/**
 * Get interrupt priority
 * @param {number} irqNumber - Interrupt number (negative for system exceptions, 0+ for external interrupts)
 * @returns {number} Actual configured priority value
 */
function _GetInterruptPriority(irqNumber) {
    // System exceptions (negative IRQ numbers) use SCB priority registers
    if (irqNumber < 0) {
        var priorityBits = _PRIORITY_BITS;
        var priorityShift = 8 - priorityBits;
        
        switch (irqNumber) {
            case -14: // NMI
            case -13: // HardFault
                return -3; // Fixed highest priority (non-configurable)
            case -12: // MemManage (SHPR1[7:0])
                var shpr1 = _ReadU32(_SCB_SHPR1);
                return (shpr1 & 0xFF) >> priorityShift;
            case -11: // BusFault (SHPR1[15:8])
                var shpr1 = _ReadU32(_SCB_SHPR1);
                return ((shpr1 >> 8) & 0xFF) >> priorityShift;
            case -10: // UsageFault (SHPR1[23:16])
                var shpr1 = _ReadU32(_SCB_SHPR1);
                return ((shpr1 >> 16) & 0xFF) >> priorityShift;
            case -5:  // SVC (SHPR2[31:24])
                var shpr2 = _ReadU32(_SCB_SHPR2);
                return ((shpr2 >> 24) & 0xFF) >> priorityShift;
            case -4:  // DebugMon (SHPR3[7:0])
                var shpr3 = _ReadU32(_SCB_SHPR3);
                return (shpr3 & 0xFF) >> priorityShift;
            case -2:  // PendSV (SHPR3[23:16])
                var shpr3 = _ReadU32(_SCB_SHPR3);
                return ((shpr3 >> 16) & 0xFF) >> priorityShift;
            case -1:  // SysTick (SHPR3[31:24])
                var shpr3 = _ReadU32(_SCB_SHPR3);
                return ((shpr3 >> 24) & 0xFF) >> priorityShift;
            default:
                return 0;
        }
    }
    
    if (irqNumber >= 240) {
        return 0;
    }
    
    var iprAddress = _NVIC_IPR_BASE + (irqNumber & ~0x03);
    var rawPriority = (_ReadU32(iprAddress) >> ((irqNumber & 0x03) * 8)) & 0xFF;

    // Get the number of implemented priority bits
    var priorityBits = _PRIORITY_BITS;
    
    // Calculate the shift amount (8 - priority_bits)
    var priorityShift = 8 - priorityBits;
    
    // Return the actual configured priority
    return rawPriority >> priorityShift;
}

/**
 * Initialize plugin data
 */
function _InitializePlugin() {
    if (_IsInitialized) {
        return;
    }
    
    _CurrentRowIndex = 0;
    _IsInitialized = true;
}

/**
 * Plugin initialization function
 * Called by Ozone when the plugin is loaded
 */
function init() {
    _InitializePlugin();
}

/**
 * Get plugin name
 * @returns {string} Plugin name
 */
function getName() {
    return "NVIC Viewer";
}

/**
 * Get available pages for this plugin
 * @returns {Array} Array of page names
 */
function getPages() {
    return ["Interrupts"];
}

/**
 * Get column headers for the specified page
 * @param {string} pageName - Name of the page
 * @returns {Array} Array of column header strings
 */
function getColHeaders(pageName) {
    if (pageName == "Interrupts") {
        return [
            "IRQ#",
            "Name", 
            "Enabled",
            "Pending",
            "Priority"
        ];
    }
    return [];
}

/**
 * Get the first row of data for the specified page
 * @param {string} pageName - Name of the page
 * @returns {Array} Array of column values for the first row, or undefined if no data
 */
function getFirstRow(pageName) {
    if (pageName != "Interrupts") {
        return undefined;
    }
    
    _InitializePlugin();
    
    if (_InterruptData.length == 0) {
        return undefined;
    }
    
    _CurrentRowIndex = 0;
    return _GetRowData(_CurrentRowIndex);
}

/**
 * Get the next row of data for the specified page
 * @param {string} pageName - Name of the page
 * @returns {Array} Array of column values for the next row, or undefined if no more data
 */
function getNextRow(pageName) {
    if (pageName != "Interrupts") {
        return undefined;
    }
    
    _CurrentRowIndex++;
    
    if (_CurrentRowIndex >= _InterruptData.length) {
        return undefined;
    }
    
    return _GetRowData(_CurrentRowIndex);
}

/**
 * Get row data for the specified index
 * @param {number} index - Row index
 * @returns {Array} Array of column values
 */
function _GetRowData(index) {
    if (index < 0 || index >= _InterruptData.length) {
        return undefined;
    }
    
    var irq = _InterruptData[index];
    var isEnabled = _IsInterruptEnabled(irq.number);
    var isPending = _IsInterruptPending(irq.number);
    var priority = _GetInterruptPriority(irq.number);
    
    return [
        irq.number.toString(),
        irq.name,
        _ToStrBoolYesNo(isEnabled),
        _ToStrBoolYesNo(isPending),
        priority.toString()
    ];
}

/**
 * Called when target state changes (e.g., target halted/running)
 * Refreshes NVIC data to ensure consistency after target state changes
 */
function onTargetChanged() {
    // Reset initialization state to force data refresh
    _IsInitialized = false;
}