/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may
 * only be used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */
/*!
 * @file flash_ecc_fault.c
 */

#include "flash_ecc_fault.h"
#include "interrupt_manager.h"

/*******************************************************************************
  * Functions
  ******************************************************************************/
void FLASH_Fault_Handler_C(uint32_t *fault_stack);
void FLASH_Fault_Handler(void) __attribute__((naked));

void FLASH_Fault_Handler(void)
{
    /* This is a naked function. */
    __asm volatile(
        "MOVS R0, #4                   \n" // Load immediate value 4 into R0 to test the LR bit.
        "MOV R1, LR                    \n" // Copy Link Register (LR) into R1.
        "TST R1, R0                    \n" // Test if bit2 of LR is set.
        "BNE get_psp                   \n" // If bit2 is set, use Process Stack Pointer (PSP).
        "MRS R0, MSP                   \n" // Otherwise, load Main Stack Pointer (MSP) into R0.
        "B c_handler                   \n" // Branch to call the C fault handler.
    "get_psp:                          \n"
        "MRS R0, PSP                   \n" // Load Process Stack Pointer (PSP) into R0.
    "c_handler:                        \n"
        "B FLASH_Fault_Handler_C       \n" // Call the C fault handler.
    );
}

// New C function that performs fault processing.
// It calculates the instruction length and adjusts the program counter if a Flash ECC error is detected.
void FLASH_Fault_Handler_C(uint32_t *fault_stack)
{
    // fault_stack[6] holds the original PC when fault occurred.
    uint32_t *pInstructionAddress = fault_stack + 6;
    uint32_t pc = *pInstructionAddress;
    uint16_t opcode = *((uint16_t *)(*pInstructionAddress));
    uint8_t instrSize = 0;

    // Determine the instruction size based on the opcode.
    if ((opcode & 0xE800) == 0xE800 || // 32-bit instruction check (pattern 11101x...)
        (opcode & 0xF000) == 0xF000 || // 32-bit instruction check (pattern 11110x...)
        (opcode & 0xF800) == 0xF800)   // 32-bit instruction check (pattern 11111x...)
    {
        instrSize = 4;
    }
    else
    {
        instrSize = 2;
    }

    // If the Flash ECC unrecovery error flag is set, skip the faulting instruction.
    // if (EFM->STS & EFM_STS_UNRECOVERR_MASK)
    {
        EFM->STS |= EFM_STS_UNRECOVERR_MASK; // Clear the unrecovery error flag.
        *pInstructionAddress = pc + instrSize;
#ifdef SCB_SHCSR_BUSFAULTENA_Msk
        SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk | SCB_CFSR_PRECISERR_Msk; /*PRQA S 0306*/
#endif                                                                /* SCB_SHCSR_BUSFAULTENA_Msk */
    }
}

#ifdef SCB_SHCSR_BUSFAULTENA_Msk
/**
  * @brief Enable and install the bus fault exception
  *
  * @return void
  */
void FLASH_DRV_ECC_Fault_Install(void)
{
    INT_SYS_InstallHandler(BusFault_IRQn, FLASH_Fault_Handler, NULL);
    /*Enable BusFault exception*/
    SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk;
    SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk;
}
#else
/**
  * @brief Install the default HardFault exception to jump 2bit ecc fault address.
  *
  * @return void
  */
void FLASH_DRV_ECC_Fault_Install(void)
{
    /* For Cortex-M0, only jump it in HardFault due to no BusFault */
    INT_SYS_InstallHandler(HardFault_IRQn, FLASH_Fault_Handler, NULL);
}
#endif /* SCB_SHCSR_BUSFAULTENA_Msk */
