/**
*   @file    pSIP_Hcu.h
*   @version
*
*   @brief   AUTOSAR Hcu register map
*   @details Register map for HCU
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Adc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_HCU_H
#define PSIP_HCU_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"

/*==================================================================================================
*                              HCU REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup HCU_Peripheral_Access_Layer HCU Peripheral Access Layer
 * @{
 */


/** HCU - Size of Registers Arrays */
#define HCU_KEY_COUNT           (4)
#define HCU_CONTEXT_4_7_COUNT           (4)
#define HCU_CONTEXT_8_11_COUNT           (4)

/* HCU Register Layout Typedef */
typedef struct {
    uint32 GO;                                 /**< Go Register, offset: 0x0000 */
    uint32 CR;                                 /**< Configuration Register, offset: 0x0004 */
    uint8 RESERVED2[ 4];                      /**< RESERVED2, offset: 0x0008 */
    uint32 MSGCUR;                             /**< Message Current number Register, offset: 0x000C */
    uint8 RESERVED4[ 8];                      /**< RESERVED4, offset: 0x0010 */
    uint32 INTE;                               /**< Interrupt Enable Register, offset: 0x0018 */
    uint32 SR;                                 /**< Status Register, offset: 0x001C */
    uint32 KEY[ 4];                            /**< Key Register, offset: 0x0020 */
    uint8 RESERVED8[48];                      /**< RESERVED8, offset: 0x0030 */
    uint32 CONTEXT_4_7[ 4];                    /**< Context data 4~7 Register, offset: 0x0060 */
    uint32 CONTEXT_8_11[ 4];                   /**< Context data 8~11 Register, offset: 0x0070 */
    uint8 RESERVED11[48];                     /**< RESERVED11, offset: 0x0080 */
    uint32 INMSG;                              /**< Input Message Register, offset: 0x00B0 */
    uint32 OUTMSG;                             /**< Output Message Register, offset: 0x00B4 */

} HCU_Type, *HCU_MemMapPtr;

/** Number of instances of the HCU module. */
#define HCU_INSTANCE_COUNT                        (1u)

/* HCU base address */
#define HCU_BASE_ADDR32                                   (0x40067000U)
#define HCU                                               ((volatile HCU_Type *)(HCU_BASE_ADDR32))

/** Array initializer of HCU peripheral base addresses */
#define HCU_BASE_ADDRS                                     { HCU_BASE_ADDR32 }
/** Array initializer of HCU peripheral base pointers */
#define HCU_BASE_PTRS                                      { HCU }

#define HCU_GO_OFFSET32                                    (0x0000U)                    /**< Offset for Go Register */
#define HCU_CR_OFFSET32                                    (0x0004U)                    /**< Offset for Configuration Register */
#define HCU_MSGCUR_OFFSET32                                (0x000CU)                    /**< Offset for Message Current number Register */
#define HCU_INTE_OFFSET32                                  (0x0018U)                    /**< Offset for Interrupt Enable Register */
#define HCU_SR_OFFSET32                                    (0x001CU)                    /**< Offset for Status Register */
#define HCU_KEY_OFFSET32(x)                                (0x0020U + ((x) * (4U)))     /**< Offset for Key Register */
#define HCU_CONTEXT_4_7_OFFSET32(x)                        (0x0060U + ((x) * (4U)))     /**< Offset for Context data 4~7 Register */
#define HCU_CONTEXT_8_11_OFFSET32(x)                       (0x0070U + ((x) * (4U)))     /**< Offset for Context data 8~11 Register */
#define HCU_INMSG_OFFSET32                                 (0x00B0U)                    /**< Offset for Input Message Register */
#define HCU_OUTMSG_OFFSET32                                (0x00B4U)                    /**< Offset for Output Message Register */


/**
 * @addtogroup HCU Register Mask
 * @{
 */
/* HCU_GO Register */
#define HCU_GO_GO_MASK                                     (0x1U)
#define HCU_GO_GO_SHIFT                                    (0U)
#define HCU_GO_GO(x)                                       (((uint32)(((uint32)(x)) << HCU_GO_GO_SHIFT)) & HCU_GO_GO_MASK)
/* HCU_CR Register */
#define HCU_CR_OUTDMAE_MASK                                (0x20000000U)
#define HCU_CR_OUTDMAE_SHIFT                               (29U)
#define HCU_CR_OUTDMAE(x)                                  (((uint32)(((uint32)(x)) << HCU_CR_OUTDMAE_SHIFT)) & HCU_CR_OUTDMAE_MASK)
#define HCU_CR_INDMAE_MASK                                 (0x10000000U)
#define HCU_CR_INDMAE_SHIFT                                (28U)
#define HCU_CR_INDMAE(x)                                   (((uint32)(((uint32)(x)) << HCU_CR_INDMAE_SHIFT)) & HCU_CR_INDMAE_MASK)
#define HCU_CR_MACLEN_MASK                                 (0xF000000U)
#define HCU_CR_MACLEN_SHIFT                                (24U)
#define HCU_CR_MACLEN(x)                                   (((uint32)(((uint32)(x)) << HCU_CR_MACLEN_SHIFT)) & HCU_CR_MACLEN_MASK)
#define HCU_CR_CS_MASK                                     (0x40000U)
#define HCU_CR_CS_SHIFT                                    (18U)
#define HCU_CR_CS(x)                                       (((uint32)(((uint32)(x)) << HCU_CR_CS_SHIFT)) & HCU_CR_CS_MASK)
#define HCU_CR_MSGB_MASK                                   (0x20000U)
#define HCU_CR_MSGB_SHIFT                                  (17U)
#define HCU_CR_MSGB(x)                                     (((uint32)(((uint32)(x)) << HCU_CR_MSGB_SHIFT)) & HCU_CR_MSGB_MASK)
#define HCU_CR_MSGE_MASK                                   (0x10000U)
#define HCU_CR_MSGE_SHIFT                                  (16U)
#define HCU_CR_MSGE(x)                                     (((uint32)(((uint32)(x)) << HCU_CR_MSGE_SHIFT)) & HCU_CR_MSGE_MASK)
#define HCU_CR_ALGSEL_MASK                                 (0xF000U)
#define HCU_CR_ALGSEL_SHIFT                                (12U)
#define HCU_CR_ALGSEL(x)                                   (((uint32)(((uint32)(x)) << HCU_CR_ALGSEL_SHIFT)) & HCU_CR_ALGSEL_MASK)
#define HCU_CR_ENGSEL_MASK                                 (0xF0U)
#define HCU_CR_ENGSEL_SHIFT                                (4U)
#define HCU_CR_ENGSEL(x)                                   (((uint32)(((uint32)(x)) << HCU_CR_ENGSEL_SHIFT)) & HCU_CR_ENGSEL_MASK)
#define HCU_CR_DATSWP_MASK                                 (0xCU)
#define HCU_CR_DATSWP_SHIFT                                (2U)
#define HCU_CR_DATSWP(x)                                   (((uint32)(((uint32)(x)) << HCU_CR_DATSWP_SHIFT)) & HCU_CR_DATSWP_MASK)
#define HCU_CR_ENC_MASK                                    (0x2U)
#define HCU_CR_ENC_SHIFT                                   (1U)
#define HCU_CR_ENC(x)                                      (((uint32)(((uint32)(x)) << HCU_CR_ENC_SHIFT)) & HCU_CR_ENC_MASK)
/* HCU_MSGCUR Register */
#define HCU_MSGCUR_MSGLEN_MASK                             (0xFFFFU)
#define HCU_MSGCUR_MSGLEN_SHIFT                            (0U)
#define HCU_MSGCUR_MSGLEN(x)                               (((uint32)(((uint32)(x)) << HCU_MSGCUR_MSGLEN_SHIFT)) & HCU_MSGCUR_MSGLEN_MASK)
/* HCU_INTE Register */
#define HCU_INTE_INMSGWIE_MASK                             (0x80000000U)
#define HCU_INTE_INMSGWIE_SHIFT                            (31U)
#define HCU_INTE_INMSGWIE(x)                               (((uint32)(((uint32)(x)) << HCU_INTE_INMSGWIE_SHIFT)) & HCU_INTE_INMSGWIE_MASK)
#define HCU_INTE_OUTMSGRIE_MASK                            (0x40000000U)
#define HCU_INTE_OUTMSGRIE_SHIFT                           (30U)
#define HCU_INTE_OUTMSGRIE(x)                              (((uint32)(((uint32)(x)) << HCU_INTE_OUTMSGRIE_SHIFT)) & HCU_INTE_OUTMSGRIE_MASK)
#define HCU_INTE_KEYWIE_MASK                               (0x20000000U)
#define HCU_INTE_KEYWIE_SHIFT                              (29U)
#define HCU_INTE_KEYWIE(x)                                 (((uint32)(((uint32)(x)) << HCU_INTE_KEYWIE_SHIFT)) & HCU_INTE_KEYWIE_MASK)
#define HCU_INTE_AESIVWIE_MASK                             (0x10000000U)
#define HCU_INTE_AESIVWIE_SHIFT                            (28U)
#define HCU_INTE_AESIVWIE(x)                               (((uint32)(((uint32)(x)) << HCU_INTE_AESIVWIE_SHIFT)) & HCU_INTE_AESIVWIE_MASK)
#define HCU_INTE_AESMACRIE_MASK                            (0x8000000U)
#define HCU_INTE_AESMACRIE_SHIFT                           (27U)
#define HCU_INTE_AESMACRIE(x)                              (((uint32)(((uint32)(x)) << HCU_INTE_AESMACRIE_SHIFT)) & HCU_INTE_AESMACRIE_MASK)
#define HCU_INTE_AESMACWIE_MASK                            (0x4000000U)
#define HCU_INTE_AESMACWIE_SHIFT                           (26U)
#define HCU_INTE_AESMACWIE(x)                              (((uint32)(((uint32)(x)) << HCU_INTE_AESMACWIE_SHIFT)) & HCU_INTE_AESMACWIE_MASK)
#define HCU_INTE_AESIVRIE_MASK                             (0x800000U)
#define HCU_INTE_AESIVRIE_SHIFT                            (23U)
#define HCU_INTE_AESIVRIE(x)                               (((uint32)(((uint32)(x)) << HCU_INTE_AESIVRIE_SHIFT)) & HCU_INTE_AESIVRIE_MASK)
#define HCU_INTE_ODIE_MASK                                 (0x1U)
#define HCU_INTE_ODIE_SHIFT                                (0U)
#define HCU_INTE_ODIE(x)                                   (((uint32)(((uint32)(x)) << HCU_INTE_ODIE_SHIFT)) & HCU_INTE_ODIE_MASK)
/* HCU_SR Register */
#define HCU_SR_INMSGW_MASK                                 (0x80000000U)
#define HCU_SR_INMSGW_SHIFT                                (31U)
#define HCU_SR_INMSGW(x)                                   (((uint32)(((uint32)(x)) << HCU_SR_INMSGW_SHIFT)) & HCU_SR_INMSGW_MASK)
#define HCU_SR_OUTMSGR_MASK                                (0x40000000U)
#define HCU_SR_OUTMSGR_SHIFT                               (30U)
#define HCU_SR_OUTMSGR(x)                                  (((uint32)(((uint32)(x)) << HCU_SR_OUTMSGR_SHIFT)) & HCU_SR_OUTMSGR_MASK)
#define HCU_SR_KEYW_MASK                                   (0x20000000U)
#define HCU_SR_KEYW_SHIFT                                  (29U)
#define HCU_SR_KEYW(x)                                     (((uint32)(((uint32)(x)) << HCU_SR_KEYW_SHIFT)) & HCU_SR_KEYW_MASK)
#define HCU_SR_AESIVW_MASK                                 (0x10000000U)
#define HCU_SR_AESIVW_SHIFT                                (28U)
#define HCU_SR_AESIVW(x)                                   (((uint32)(((uint32)(x)) << HCU_SR_AESIVW_SHIFT)) & HCU_SR_AESIVW_MASK)
#define HCU_SR_AESMACR_MASK                                (0x8000000U)
#define HCU_SR_AESMACR_SHIFT                               (27U)
#define HCU_SR_AESMACR(x)                                  (((uint32)(((uint32)(x)) << HCU_SR_AESMACR_SHIFT)) & HCU_SR_AESMACR_MASK)
#define HCU_SR_AESMACW_MASK                                (0x4000000U)
#define HCU_SR_AESMACW_SHIFT                               (26U)
#define HCU_SR_AESMACW(x)                                  (((uint32)(((uint32)(x)) << HCU_SR_AESMACW_SHIFT)) & HCU_SR_AESMACW_MASK)
#define HCU_SR_AESIVR_MASK                                 (0x800000U)
#define HCU_SR_AESIVR_SHIFT                                (23U)
#define HCU_SR_AESIVR(x)                                   (((uint32)(((uint32)(x)) << HCU_SR_AESIVR_SHIFT)) & HCU_SR_AESIVR_MASK)
#define HCU_SR_AESVAL_MASK                                 (0x2U)
#define HCU_SR_AESVAL_SHIFT                                (1U)
#define HCU_SR_AESVAL(x)                                   (((uint32)(((uint32)(x)) << HCU_SR_AESVAL_SHIFT)) & HCU_SR_AESVAL_MASK)
#define HCU_SR_OD_MASK                                     (0x1U)
#define HCU_SR_OD_SHIFT                                    (0U)
#define HCU_SR_OD(x)                                       (((uint32)(((uint32)(x)) << HCU_SR_OD_SHIFT)) & HCU_SR_OD_MASK)
/* HCU_KEY Register */
#define HCU_KEY_KEY_MASK                                   (0xFFFFFFFFU)
#define HCU_KEY_KEY_SHIFT                                  (0U)
#define HCU_KEY_KEY(x)                                     (((uint32)(((uint32)(x)) << HCU_KEY_KEY_SHIFT)) & HCU_KEY_KEY_MASK)
/* HCU_CONTEXT_4_7 Register */
#define HCU_CONTEXT_4_7_CONTEXT_MASK                       (0xFFFFFFFFU)
#define HCU_CONTEXT_4_7_CONTEXT_SHIFT                      (0U)
#define HCU_CONTEXT_4_7_CONTEXT(x)                         (((uint32)(((uint32)(x)) << HCU_CONTEXT_4_7_CONTEXT_SHIFT)) & HCU_CONTEXT_4_7_CONTEXT_MASK)
/* HCU_CONTEXT_8_11 Register */
#define HCU_CONTEXT_8_11_CONTEXT_MASK                      (0xFFFFFFFFU)
#define HCU_CONTEXT_8_11_CONTEXT_SHIFT                     (0U)
#define HCU_CONTEXT_8_11_CONTEXT(x)                        (((uint32)(((uint32)(x)) << HCU_CONTEXT_8_11_CONTEXT_SHIFT)) & HCU_CONTEXT_8_11_CONTEXT_MASK)
/* HCU_INMSG Register */
#define HCU_INMSG_INMSG_MASK                               (0xFFFFFFFFU)
#define HCU_INMSG_INMSG_SHIFT                              (0U)
#define HCU_INMSG_INMSG(x)                                 (((uint32)(((uint32)(x)) << HCU_INMSG_INMSG_SHIFT)) & HCU_INMSG_INMSG_MASK)
/* HCU_OUTMSG Register */
#define HCU_OUTMSG_OUTMSG_MASK                             (0xFFFFFFFFU)
#define HCU_OUTMSG_OUTMSG_SHIFT                            (0U)
#define HCU_OUTMSG_OUTMSG(x)                               (((uint32)(((uint32)(x)) << HCU_OUTMSG_OUTMSG_SHIFT)) & HCU_OUTMSG_OUTMSG_MASK)


/**
 * @}
 */ /* end of group HCU_Register_Masks */


/**
 * @}
 */ /* end of group HCU_Peripheral_Access_Layer */

#endif /* PSIP_HCU_H */

