/*
 * Copyright 2020-2025 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 * 
 * @file Fee_Cfg.h
 * @brief 
 * 
 */


#ifndef FEE_CFG_H
#define FEE_CFG_H

#ifdef __cplusplus
extern "C"{
#endif

/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/

/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define FEE_VENDOR_ID_CFG                      (180)
#define FEE_AR_RELEASE_MAJOR_VERSION_CFG       (4)
#define FEE_AR_RELEASE_MINOR_VERSION_CFG       (4)
#define FEE_AR_RELEASE_REVISION_VERSION_CFG    (0)
#define FEE_SW_MAJOR_VERSION_CFG               (1)
#define FEE_SW_MINOR_VERSION_CFG               (0)
#define FEE_SW_PATCH_VERSION_CFG               (0)

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
                    
/*==================================================================================================
 *                                       FeeGeneral
==================================================================================================*/
/**
 * @brief   Switches the development error detection and notification on or off.
 * @details true: detection and notification is enabled.
            false: detection and notification is disabled. 
 * @ref     ECUC_Fee_00111
 */
#define FEE_DEV_ERROR_DETECT                (STD_OFF)

/**
 * @brief   Compile switch to enable / disable the version information API
 * @details true: API enabled
            false: API disabled 
 * @ref     ECUC_Fee_00115
 */
#define FEE_VERSION_INFO_API                (STD_ON)

/**
 * @brief   Pre-processor switch to enable and disable the polling mode for this module.
 * @details unused
 *          true: Polling mode enabled, callback functions (provided to FLS module) disabled.
            false: Polling mode disabled, callback functions (provided to FLS module) enabled.
 * @ref     ECUC_Fee_00114
 */
#define FEE_POLLING_MODE                    (STD_OFF)

/**
 * @brief   Compiler switch to enable/disable the 'SetMode' functionality of the FEE module.
 * @details true: SetMode functionality supported / code present
            false: SetMode functionality not supported / code not present.
 * @ref     ECUC_Fee_00119
 */
#define FEE_SETMODE_API_SUPPORTED           (STD_ON)

/**
 * @brief   Compiler switch to enable/disable the Fls_Cancel function of the FEE module.
 * @details true: API enabled
            false: API disabled 
 */
#define FEE_CANCEL_API                      (STD_ON)

/**
 * @brief   If reset, power loss etc. occurs here, neither newly nor previously written data is available.
 */
#define FEE_BLOCK_ALWAYS_AVAILABLE          (STD_ON)

/**
 * @brief   Behavior of Fee_ImmediateBlockErase.
 */
#define FEE_LEGACY_IMM_ERASE_MODE           (STD_ON)

/**
 * @brief   This configuration defines whether Fee should swap foreign blocks found in flash at swap or not.
 */
#define FEE_SWAP_FOREIGN_BLOCKS_ENABLED     (STD_OFF)

/**
 * @brief   This configuration defines for which project the Fee configuration is used.
 */
#if (FEE_SWAP_FOREIGN_BLOCKS_ENABLED == STD_ON)
#define FEE_BOOTLOADER_CONFIG               (STD_OFF)
#endif

/**
 * @brief   This configuration defines if the status for the never written blocks must be INVALID or INCONSISTENT.
 */
#define FEE_MARK_EMPTY_BLOCKS_INVALID       (STD_ON)

/**
 * @brief This configuration defines the schm is supported or not.
*/

#define FEE_SCHM_SUPPORT                    (STD_ON)

/* This configuration enables the sector retirement feature */
#define FEE_SECTOR_RETIREMENT               (STD_OFF)

/**
 * @brief   Number of configured Fee blocks.
 */
#define FEE_CRT_CFG_NR_OF_BLOCKS            (15U)

/**
 * @brief   Maximum number of Fee blocks in all project versions and configurations.
 */
#define FEE_MAX_NR_OF_BLOCKS                (15U)

/**
 * @brief   The size in bytes to which logical blocks shall be aligned.
 * @details 0-65535
 * @ref     ECUC_Fee_00116
 */
#define FEE_VIRTUAL_PAGE_SIZE               (8U)

/**
 * @brief   Size of the data buffer in bytes.
 */
#define FEE_DATA_BUFFER_SIZE                (512U)
/**
 * @brief   The period between successive calls to the main function in seconds.
 * @details unused
 * @ref     ECUC_Fee_00153
 */
#define FEE_MAIN_FUNCTION_PERIOD            (0.001)

/**
 * @brief   Mapped to the job end notification routine provided by the upper layer module (NvM_JobEndNotification).
 * @ref     ECUC_Fee_00112
 */
#define FEE_NVM_JOB_END_NOTIFICATION        NvM_JobEndNotification()

/**
 * @brief   Mapped to the job error notification routine provided by the upper layer module (NvM_JobErrorNotification).
 * @ref     ECUC_Fee_00113
 */
#define FEE_NVM_JOB_ERROR_NOTIFICATION      NvM_JobErrorNotification()


/*==================================================================================================
 *                                       FeePublishedInformation
==================================================================================================*/

/**
 * @brief   Management overhead per logical block in bytes.
 */
#define FEE_BLOCK_OVERHEAD                  (32U)

/**
 * @brief   Management overhead per logical cluster in bytes.
 */
#define FEE_CLUSTER_OVERHEAD                (32U)

/*==================================================================================================
 *                                       FeeClusterGroup
==================================================================================================*/

/**
 * @brief   Number of configured Fee cluster group.
 */
#define FEE_NUMBER_OF_CLUSTER_GROUPS        (5U)

/*==================================================================================================
 *                                       FeeDefault (DO NOT MODIFY MANUALLY!)
==================================================================================================*/

/**
 * @brief   The contents of an erased flash memory cell.
 */
#define FEE_ERASED_VALUE                    (0xFFU)

/**
 * @brief   Value of the block and cluster validation flag.
 */
#define FEE_VALIDATED_VALUE                 (0x81U)

/**
 * @brief   Value of the block and cluster invalidation flag.
 */
#define FEE_INVALIDATED_VALUE               (0x18U)

/*==================================================================================================
 *                                       FeeBlockConfiguration
==================================================================================================*/

/**
 * @brief   Symbolic names of configured Fee block's number and block's size.
 */
#define FeeConf_FeeBlockConfiguration_FEE_BOOT_SA_FLG   (1U)

#define FeeConf_FeeBlockConfiguration_FEE_BOOT_SA_FLG_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_BOOT_BACK_UP_FLG   (2U)

#define FeeConf_FeeBlockConfiguration_FEE_BOOT_BACK_UP_FLG_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_BOOT_RESERVE_BLOCK   (3U)

#define FeeConf_FeeBlockConfiguration_FEE_BOOT_RESERVE_BLOCK_BlockSize   (16U)

#define FeeConf_FeeBlockConfiguration_FEE_TPMS_ID   (5U)

#define FeeConf_FeeBlockConfiguration_FEE_TPMS_ID_BlockSize   (24U)

#define FeeConf_FeeBlockConfiguration_FEE_FOB_ID   (7U)

#define FeeConf_FeeBlockConfiguration_FEE_FOB_ID_BlockSize   (24U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_MANUFACTURE_DATE   (9U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_MANUFACTURE_DATE_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_SERIAL_NUM   (10U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_SERIAL_NUM_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_HW_NUM   (11U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_HW_NUM_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_HW_VER   (12U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_ECU_HW_VER_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_EOL_RESULT   (13U)

#define FeeConf_FeeBlockConfiguration_FEE_NAEN_EOL_RESULT_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_DEBUG_BLOCK   (14U)

#define FeeConf_FeeBlockConfiguration_FEE_DEBUG_BLOCK_BlockSize   (32U)

#define FeeConf_FeeBlockConfiguration_NVM_BOOT_SA_CNT   (4U)

#define FeeConf_FeeBlockConfiguration_NVM_BOOT_SA_CNT_BlockSize   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_TPMS_ID_REDUNDANT   (6U)

#define FeeConf_FeeBlockConfiguration_FEE_TPMS_ID_REDUNDANT_BlockSize   (24U)

#define FeeConf_FeeBlockConfiguration_FEE_FOB_ID_REDUNDANT   (8U)

#define FeeConf_FeeBlockConfiguration_FEE_FOB_ID_REDUNDANT_BlockSize   (24U)

#define FeeConf_FeeBlockConfiguration_FEE_APP_SA_CNT   (15U)

#define FeeConf_FeeBlockConfiguration_FEE_APP_SA_CNT_BlockSize   (8U)


/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   FUNCTION PROTOTYPES
==================================================================================================*/

/**
 * @brief   Job end notification routine provided by the upper layer module (declaration).
 */
extern void NvM_JobEndNotification(void);

/**
 * @brief   Job error notification routine provided by the upper layer module (declaration).
 */
extern void NvM_JobErrorNotification(void);

#ifdef __cplusplus
}
#endif

#endif /* FEE_CFG_H */

/** @} */

