/*
 * Copyright 2020-2022 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/*!
 * @file YTM32B1HA0_features.h
 * @brief Chip specific module features
 */

#ifndef YTM32B1HA0_FEATURES_H
#define YTM32B1HA0_FEATURES_H

/* ERRATA sections*/

/* @brief Number of cores. */
#define NUMBER_OF_CORES                     (1u)

/* SOC module features */

/* @brief PORT availability on the SoC. */
#define FEATURE_SOC_PORT_COUNT              (5)

#define FEATURE_SOC_IPC_COUNT               (1)
/* @brief Power down mode availability on the SoC. */
#define FEATURE_SOC_SUPPORT_POWERDOWN_MODE  (1)

/* @brief Slow IRC clock frequency(12MHz). */
#define FEATURE_SCU_SIRC_FREQ               (12000000U)
/* @brief Fast IRC trimmed clock frequency(96MHz). */
#define FEATURE_SCU_FIRC_FREQ               (96000000U)
/* @brief FXOSC clock frequency(24MHz). */
#define FEATURE_SCU_FXOSC_FREQ              (24000000U)
/* @brief SCU FXOSC support GAIN configuration. */
#define FEATURE_SCU_FXOSC_SUPPORT_GAIN_SEL  (1)
/* @brief SCU support PLL clock. */
#define FEATURE_SCU_SUPPORT_PLL             (1)
/* @brief SCU support CMU reference clock selection. */
#define FEATURE_SCU_SUPPORT_CMU_REFCLK_SEL  (1)
/* @brief SCU support SXOSC 32.768K clock. */
#define FEATURE_SCU_SUPPORT_SXOSC           (1)
#if FEATURE_SCU_SUPPORT_SXOSC
/* @brief SXOSC clock frequency(32.768K). */
#define FEATURE_SCU_SXOSC_FREQ              (32768U)
/* @brief SCU SXOSC support GAIN configuration. */
#define FEATURE_SCU_SXOSC_SUPPORT_GAIN_SEL  (1)
#endif /* FEATURE_SCU_SUPPORT_SXOSC */
/* @brief Clock switch PLL lock timeout loop count. */
#define FEATURE_CLOCK_TIMEOUT_LOOP_CNT      (0x1FFFFF)

/* FLASH module features */

/* @brief Flash block count. */
#define FEATURE_EFM_BLOCK_COUNT             (3u)
/* @brief Flash program flash start address. */
#define FEATURE_EFM_MAIN_ARRAY_START_ADDRESS  (0x02000000U)
/* @brief Flash program flash end address. */
#define FEATURE_EFM_MAIN_ARRAY_END_ADDRESS  (0x021FFFFFU)
/* @brief Flash sector size. */
#define FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE  (0x800u)
/* @brief Flash has data flash. */
#define FEATURE_EFM_HAS_DATA_FLASH          (1)
#if FEATURE_EFM_HAS_DATA_FLASH
/* @brief Flash data flash start address. */
#define FEATURE_EFM_DATA_ARRAY_START_ADDRESS (0x06000000U)
/* @brief Flash data flash end address. */
#define FEATURE_EFM_DATA_ARRAY_END_ADDRESS  (0x0603FFFFU)
/* @brief Flash data flash sector size. */
#define FEATURE_EFM_DATA_ARRAY_SECTOR_SIZE  (0x400u)
#endif /* FEATURE_EFM_HAS_DATA_FLASH */
#define FEATURE_EFM_HAS_NVR_FLASH          (1)
#if FEATURE_EFM_HAS_NVR_FLASH
/* @brief Flash NVR start address. */
#define FEATURE_EFM_NVR_ARRAY_START_ADDRESS (0x10000000U)
/* @brief Flash NVR end address. */
#define FEATURE_EFM_NVR_ARRAY_END_ADDRESS   (0x10040400U)
/* @brief Flash NVR sector size. */
#define FEATURE_EFM_NVR_ARRAY_SECTOR_SIZE   (0x800u)
#endif /* FEATURE_EFM_HAS_NVR_FLASH */

/* @brief Flash sector minimum size. */
#define FEATURE_EFM_FLASH_MIN_SECTOR_SIZE   (0x400u)

/* @brief Flash BOOT NVR start address. */
#define FEATURE_EFM_BOOT_NVR_START_ADDRESS (0x10003800U)

/* @brief Flash write unit size. */
#define FEATURE_EFM_WRITE_UNIT_SIZE                    (32u)
/* @brief Flash write unit size in words. */
#define FEATURE_EFM_WRITE_UNIT_WORD_SIZE               (FEATURE_EFM_WRITE_UNIT_SIZE >> 2)
/* @brief Flash write unit size. */
#define FEATURE_EFM_WRITE_DATA_UNIT_SIZE               (8u)
/* @brief Flash write unit size in words. */
#define FEATURE_EFM_WRITE_DATA_UNIT_WORD_SIZE          (FEATURE_EFM_WRITE_DATA_UNIT_SIZE >> 2)
/* @brief Flash Program unit command. */
#define FEATURE_EFM_PROGRAM_DATA_CMD_CODE              (0x01u)
/* @brief Flash Program unit command. */
#define FEATURE_EFM_PROGRAM_CMD_CODE                   (0x02u)
/* @brief Flash Erase Sector command. */
#define FEATURE_EFM_ERASE_SECTOR_CMD_CODE              (0x10u)
/* @brief Erase Flash Block command. */
#define FEATURE_EFM_ERASE_BLOCK_CMD_CODE               (0x11u)
/* @brief Erase Sector Retry Flash command. */
#define FEATURE_EFM_ERASE_SECTOR_RETRY_CMD_CODE        (0x14u)
/* @brief Verify Read for Sector Erase Retry command. */
#define FEATURE_EFM_ERASE_SECTOR_VREAD_RETRY_CMD_CODE  (0x15u)
/* @brief Flash Load AES KEY command. */
#define FEATURE_EFM_LOAD_AES_KEY_CMD_CODE              (0x20u)
/* @brief Flash Boot Swap command. */
#define FEATURE_EFM_BOOT_SWAP_CMD_CODE                 (0x30u)
/* @brief Flash Boot Swap tag address. */
#define FEATURE_EFM_BOOT_SWAP_TAG_ADDR                 (0x10003800u)
/* @brief Flash Status Error Code Mask. */
#define FEATURE_EFM_CMD_ERROR_MASK                                                                               \
    (EFM_STS_FAIL_MASK | EFM_STS_ACCERR_MASK | EFM_STS_UNRECOVERR_MASK | EFM_STS_RECOVERR_MASK |                 \
     EFM_STS_CI_RECOVERR_MASK | EFM_STS_CI_UNRECOVERR_MASK | EFM_STS_CI_RDVIOLERR_MASK | EFM_STS_CI_WEERR_MASK | \
     EFM_STS_CI_BUSYERR_MASK)
/* @brief Flash Unlock command register. */
#define EFM_UNLOCK_CMD_REGISTER() do {                 \
    EFM->CMD_UNLOCK = 0xfd9573f5;                      \
    __asm("ISB");                                      \
    __asm("DSB");                                      \
} while (0)
/* @brief Flash enable flash write commands. */
#define EFM_ENABLE_WE_COMMAND() do {                   \
    EFM->CTRL |= EFM_CTRL_WE_MASK;                     \
    __asm("ISB");                                      \
    __asm("DSB");                                      \
} while (0)
/* @brief Flash disable flash write commands. */
#define EFM_DISABLE_WE_COMMAND() do {                  \
    __asm("ISB");                                      \
    __asm("DSB");                                      \
    EFM->CTRL &= ~EFM_CTRL_WE_MASK;                    \
} while (0)
/* @brief boot swap erase unlock command. */
#define EFM_BOOT_SWAP_UNLOCK()                                 \
    do                                                         \
    {                                                          \
        *(volatile uint32_t *)(EFM_BASE + 0x0304) = 0x484130U; \
    } while (0)
/* @brief boot swap erase lock command. */
#define EFM_BOOT_SWAP_LOCK()                              \
    do                                                    \
    {                                                     \
        *(volatile uint32_t *)(EFM_BASE + 0x0304) = 0x0U; \
    } while (0)

/* RCU module features */
/* @brief Reset pin output feature */
#define FEATURE_RCU_HAS_PIN_OUTPUT                  (1U)

/* WDG module features */

/*!
 * @brief WDG clock source.
 * Implements : wdg_clock_source_t_Class
 */
typedef enum
{
    WDG_IPC_CLOCK = 0x00U, /*!< IPC clock */
    WDG_BUS_CLOCK = 0x01U, /*!< BUS clock */
} wdg_clock_source_t;
/* @brief The 32-bit value used for unlocking the WDG. */
#define FEATURE_WDG_UNLOCK_VALUE_1                      (0xB631)
#define FEATURE_WDG_UNLOCK_VALUE_2                      (0xC278)
/* @brief The 32-bit value used for resetting the WDG counter. */
#define FEATURE_WDG_TRIGGER_VALUE_1                     (0xA518)
#define FEATURE_WDG_TRIGGER_VALUE_2                     (0xD826)
/* @brief The reset value of the timeout register. */
#define FEATURE_WDG_TOVR_RESET_VALUE                    (0x300U)
/* @brief The value minimum of the timeout register. */
#define FEATURE_WDG_MINIMUM_TIMEOUT_VALUE               (0x100U)
/* @brief The reset value of the window register. */
#define FEATURE_WDG_WVR_RESET_VALUE                     (0x0U)
/* @brief The mask of the reserved bit in the CS register. */
#define FEATURE_WDG_CS_RESERVED_MASK                    (0x2000U)
/* @brief The value used to set WDG source clock from LPO. */
#define FEATURE_WDG_CLK_FROM_LPO                        (0x1UL)

/* @brief Default reset value of the CS register. */
#define FEATURE_WDG_CR_RESET_VALUE                      (0x82U)

/* EWDG module features */
/* @brief First byte of the EWM Service key        */
#define FEATURE_EWDG_KEY_FIRST_BYTE                      (0xA8U)
/* @brief Second byte of the EWM Service key       */
#define FEATURE_EWDG_KEY_SECOND_BYTE                     (0x6CU)
/* @brief EWDG maximum compare high value. */
#define FEATURE_EWDG_CMPH_MAX_VALUE                      (0xFEU)
/* @brief EWDG minimum compare low value. */
#define FEATURE_EWDG_CMPL_MIN_VALUE                      (0x00U)


/* Interrupt module features */

/* @brief Lowest interrupt request number. */
#define FEATURE_INTERRUPT_IRQ_MIN                       (NMI_IRQn)
/* @brief Highest interrupt request number. */
#define FEATURE_INTERRUPT_IRQ_MAX                       (MPWM2_Ch8_Ch15_IRQn)
/**< Number of priority bits implemented in the NVIC */
#define FEATURE_NVIC_PRIO_BITS                          (3U)
/* @brief Has software interrupt. */
#define FEATURE_INTERRUPT_HAS_SOFTWARE_IRQ              (0u)
/* @brief Has pending interrupt state. */
#define FEATURE_INTERRUPT_HAS_PENDING_STATE             (1u)
/* @brief Has active interrupt state. */
#define FEATURE_INTERRUPT_HAS_ACTIVE_STATE              (0u)
/* @brief Multicore support for interrupts */
#define FEATURE_INTERRUPT_MULTICORE_SUPPORT             (0u)
/* @brief Registers in which the start of interrupt vector table needs to be configured */
#define FEATURE_INTERRUPT_INT_VECTORS                   {&SCB->VTOR}

/* MPWM module features */

#define MPWM_CLKS                                       {FAST_BUS_CLK, FAST_BUS_CLK, FAST_BUS_CLK}
/* @brief MPWM support combination capture */
#define FEATURE_MPWM_HAS_COMBINATION_CAPTURE            (0U)
/* @brief MPWM support ldok/cldok simultaneously */
#define FEATURE_MPWM_HAS_SIMULTANEOUS_UPDATE            (0U)



/* eTMR module features */

/* @brief Number of PWM channels */
#define FEATURE_eTMR_CHANNEL_COUNTS                     {8U, 8U, 8U, 8U, 8U, 8U}
#define FEATURE_eTMR_CHANNEL_INTERRUPT_COUNTS           (8U)
#define FEATURE_eTMR_CHANNEL_MAX_COUNT                  (8U)
#define FEATURE_eTMR0_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR1_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR2_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR3_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR4_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR5_CHANNEL_MAX_COUNT                 (8U)
/* @brief Number of fault channels */
#define FEATURE_eTMR_FAULT_CHANNELS                     (4U)
/* @brief Output channel offset */
#define FEATURE_eTMR_OUTPUT_CHANNEL_OFFSET              (16U)
/* @brief Max counter value */
#define FEATURE_eTMR_CNT_MAX_VALUE_U32                  (0xFFFFFFFFU)
/* @brief Input capture for single shot */
#define FEATURE_eTMR_INPUT_CAPTURE_SINGLE_SHOT          (2U)
/* @brief Number of interrupt vector for channels of the eTMR module. */
#define FEATURE_eTMR_HAS_NUM_IRQS_CHANS                 (4U)
/* @brief Deadtime prescaler */
#define FEATURE_eTMR_HAS_DEAD_TIME_PRESCALER            (0U)
/* @brief Double switch */
#define FEATURE_eTMR_HAS_DOUBLE_SWITCH                  (1U)
/* @brief Combination capture mode */
#define FEATURE_eTMR_HAS_COMBINATION_CAPTURE            (0U)
/* @brief Hardware capture method */
#define FEATURE_eTMR_HAS_HARDWARE_CAPTURE               (1U)
/* @brief Capture hold */
#define FEATURE_eTMR_HAS_CAPTURE_HOLD                   (0U)
/* @brief Output trigger width */
#define FEATURE_eTMR_HAS_OUTPUT_TRIGGER_WIDTH           (1U)
/* @brief MID feature */
#define FEATURE_eTMR_HAS_MID                            (1U)
/* @brief Support to configure counter initial value source */
#define FEATURE_eTMR_HAS_CNT_INIT_SRC                   (1U)
/* @brief Counter initial value configurable */
#define FEATURE_eTMR_HAS_INIT_REG                       (1U)
/* @brief CHINIT bit is in Channel CTRL register */
#define FEATURE_eTMR_HAS_CHINIT_BIT                     (0U)
/* @brief Support quad decoder mod register */
#define FEATURE_eTMR_HAS_QUAD_DECODER_MOD               (1U)
/* @brief Support configure dead time alone */
#define FEATURE_eTMR_DEADTIME_CONFIG_EACH_CHANNEL       (1U)
/* @brief Counter count from INIT+1 */
#define FEATURE_eTMR_COUNT_FROM_INIT_PLUS_ONE           (1U)
/* @brief eTMR internal clock source */
#define eTMR_INTERNAL_CLOCK                             (FAST_BUS_CLK)
/* @brief eTMR external clock source frequency */
#define FEATURE_eTMR_EXTERNAL_CLOCK_FREQ                (1000000U)
/* @brief eTMR Dither MOD configuration */
#define FEATURE_eTMR_HAS_DITHERING                      (1U)
/* @brief eTMR support center align calculation by hardware */
#define FEATURE_eTMR_HAS_HW_CENTER_ALIGN_CALCULATION    (1U)
/* @brief eTMR support channel val0, val1 match interrupt in PWM mode */
#define FEATURE_eTMR_HAS_CHANNEL_VALUE_MATCH_INTERRUPT_IN_PWM_MODE  (1U)

/* @brief eTMR  */

/* I2C module features */

/* @brief DMA requests for I2C module. */
#define I2C_DMA_REQ                                     {{(uint8_t)DMA_REQ_I2C0_TX, (uint8_t)DMA_REQ_I2C0_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C1_TX, (uint8_t)DMA_REQ_I2C1_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C2_TX, (uint8_t)DMA_REQ_I2C2_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C3_TX, (uint8_t)DMA_REQ_I2C3_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C4_TX, (uint8_t)DMA_REQ_I2C4_RX}}
/* @brief IPC clocks for I2C module. */
#define I2C_CLOCK_NAMES                                  {I2C0_CLK, I2C1_CLK, I2C2_CLK, I2C3_CLK, I2C4_CLK}
/* @brief IPC index for I2C module */
#define I2C_IPC_INDEX                                   {IPC_I2C0_INDEX, IPC_I2C1_INDEX, IPC_I2C2_INDEX, \
                                                        IPC_I2C3_INDEX, IPC_I2C4_INDEX}
/* @brief Disable high-speed and ultra-fast operating modes */
#define I2C_HAS_FAST_PLUS_MODE                          (1U)
#define I2C_HAS_HIGH_SPEED_MODE                         (1U)
#define I2C_HAS_ULTRA_FAST_MODE                         (1U)
#define I2C_SLAVE_SUPPORT                               {true, true, true, true, true}

/* PCRC module features */

/* @brief PCRC module */
#define FEATURE_PCRC_DRIVER_SOFT_POLYNOMIAL
/* @brief Default PCRC bit width */
#define FEATURE_PCRC_DEFAULT_WIDTH                       PCRC_BITS_16
/* @brief Default PCRC read transpose */
#define FEATURE_PCRC_DEFAULT_READ_TRANSPOSE              PCRC_TRANSPOSE_NONE
/* @brief Default PCRC write transpose */
#define FEATURE_PCRC_DEFAULT_WRITE_TRANSPOSE             PCRC_TRANSPOSE_NONE
/* @brief Default polynomial 0x1021U */
#define FEATURE_PCRC_DEFAULT_POLYNOMIAL                  (0x1021U)
/* @brief Default seed value is 0xFFFFU */
#define FEATURE_PCRC_DEFAULT_SEED                        (0xFFFFU)

/* PCTRL module features */
/* @brief Has control lock. */
#define FEATURE_PCTRL_HAS_PIN_CONTROL_LOCK              (1)
/* @brief Has ALT15. */
#define FEATURE_PCTRL_HAS_ALT_15                        (1)
/* @brief Has open drain control. */
#define FEATURE_PINS_HAS_OPEN_DRAIN                     (0)
/* @brief Has digital filter. */
#define FEATURE_PINS_HAS_DIGITAL_FILTER                 (0)
/* @brief Has digital filter clock selection. */
#define FEATURE_PINS_HAS_FILTER_CLOCK_SEL               (0)
/* @brief Has digital filter. */
#define FEATURE_PCTRL_HAS_DIGITAL_FILTER                (1)
/* @brief Has trigger output to trigger other peripherals. */
#define FEATURE_PINS_HAS_TRIGGER_OUT                    (0)
/* @brief Has setting flag only. */
#define FEATURE_PINS_HAS_FLAG_SET_ONLY                  (0)
/* @brief Has pull resistor selection available. */
#define FEATURE_PINS_HAS_PULL_SELECTION                 (1)
/* @brief Has slew rate control. */
#define FEATURE_PINS_HAS_SLEW_RATE                      (1)
/* @brief Has filter clock selection control. */
#define FEATURE_PINS_HAS_FILTER_CLOCK_SELECTION         (0)
/* @brief Has passive filter. */
#define FEATURE_PINS_HAS_PASSIVE_FILTER                 (1)
/* @brief Has drive strength. */
#define FEATURE_PINS_HAS_DRIVE_STRENGTH                 (1)
/* @brief Has drive strength control bits*/
#define FEATURE_PINS_HAS_DRIVE_STRENGTH_CONTROL         (1)
/* @brief Has port input disable control bits*/
#define FEATURE_PINS_HAS_INPUT_DISABLE                  (1)


/* ACMP module features */

/* @brief Comparator has internal DAC module */
#define FEATURE_ACMP_HAS_DAC                            (1U)
/* @brief Comparator has power mode control */
#define FEATURE_ACMP_HAS_POWER_MODE                     (1U)
/* @brief Comparator has pin output enable */
#define FEATURE_ACMP_HAS_PINOUT_ENABLE                  (0U)
/* @brief Comparator has mode config */
#define FEATURE_ACMP_HAS_EN_REG                         (0U)
/* @brief Continuous mode is in CTRL MODE bit field */
#define FEATURE_ACMP_HAS_CONTINUOUS_MODE_IN_CTRL_REG    (0U)
/* @brief ACMP Function Clock Selection */
#define FEATURE_ACMP_HAS_CLK_SRC_SEL                    (1U)
/* @brief ACMP has filter bypass control */
#define FEATURE_ACMP_HAS_FILTER_BYPASS                  (0U)
/* @brief ACMP filter clock source is in the CTRL register */
#define FEATURE_ACMP_HAS_FILTER_CLK_SRC_IN_CTRL_REG     (0U)
/* @brief ACMP positive/negative input source selected in MUX register */
#define FEATURE_ACMP_HAS_INPUT_SRC_SEL_IN_MUX_REG       (0U)
/* @brief ACMP DAC reference voltage selection */
#define FEATURE_ACMP_HAS_DAC_VOLTAGE_REF_SRC            (0U)
/* @brief ACMP DAC output enable */
#define FEATURE_ACMP_HAS_DAC_OUTPUT                     (1U)
/* @brief ACMP Auto disable hard block */
#define FEATURE_ACMP_HAS_AUTODIS                        (1U)
/* @brief ACMP comparator offset control */
#define FEATURE_ACMP_HAS_OFFSET_CONTROL                 (1U)
/* @brief ACMP internal current control */
#define FEATURE_ACMP_INTERNAL_CURRENT_CONTROL           (1U)
/* @brief ACMP support sxosc clock */
#define FEATURE_ACMP_SUPPORT_SXOSC_CLK                  (1U)
/* @brief ACMP has EXP register */
#define FEATURE_ACMP_HAS_EXP_REG                        (1U)
/* @brief ACMP has trigger mode gate */
#define FEATURE_ACMP_HAS_TRIG_MODE_GATE                 (0U)

#define FEATURE_ACMP_CTRL_RESET_VALUE                   (0x00000000)
#define FEATURE_ACMP_STS_CLEAR_MASK                     (0x00FF0000)
#define FEATURE_ACMP_STS_CH_OUT_MASK                    (0xFF000000)
#define FEATURE_ACMP_STS_CH_OUT_SHIFT                   (24)
#define FEATURE_ACMP_STS_CH_FLAG_MASK                   (0x00FF0000)
#define FEATURE_ACMP_STS_CH_FLAG_SHIFT                  (16)
#define FEATURE_ACMP_DEFAULT_CONT_PERIOD                (0x30)
#define FEATURE_ACMP_DEFAULT_CONT_POS                   (0x18)

#if FEATURE_ACMP_HAS_DAC
/* @brief Comparator fix DAC input to mux side */
#define FEATURE_ACMP_DAC_FIX_SELECTION                  (0U)
#define ACMP_DAC_RESOLUTION                             (255U)
#endif /* FEATURE_ACMP_HAS_DAC */


/* IPC module features */
/* @brief IPC support SIRC DIV 4 clock */
#define FEATURE_IPC_SUPPORT_SIRC_DIV4_CLK               (1)
/* @brief IPC support FAST BUS clock */
#define FEATURE_IPC_SUPPORT_FAST_BUS_CLK                (1)

/*! @brief Clock names. */
typedef enum {
    /* IPC clocks */
    DMA_CLK                  =  0U, /*!< DMA        clock source */
    TRACE_CLK                =  1U, /*!< TRACE      clock source */
    GPIO_CLK                 =  2U, /*!< GPIO       clock source */
    PCTRLA_CLK               =  3U, /*!< PCTRLA     clock source */
    PCTRLB_CLK               =  4U, /*!< PCTRLB     clock source */
    PCTRLC_CLK               =  5U, /*!< PCTRLC     clock source */
    PCTRLD_CLK               =  6U, /*!< PCTRLD     clock source */
    PCTRLE_CLK               =  7U, /*!< PCTRLE     clock source */
    LINFlexD0_CLK            =  8U, /*!< LINFlexD0  clock source */
    LINFlexD1_CLK            =  9U, /*!< LINFlexD1  clock source */
    LINFlexD2_CLK            = 10U, /*!< LINFlexD2  clock source */
    LINFlexD3_CLK            = 11U, /*!< LINFlexD3  clock source */
    LINFlexD4_CLK            = 12U, /*!< LINFlexD4  clock source */
    LINFlexD5_CLK            = 13U, /*!< LINFlexD5  clock source */
    LINFlexD6_CLK            = 14U, /*!< LINFlexD6  clock source */
    LINFlexD7_CLK            = 15U, /*!< LINFlexD7  clock source */
    LINFlexD8_CLK            = 16U, /*!< LINFlexD8  clock source */
    LINFlexD9_CLK            = 17U, /*!< LINFlexD9  clock source */
    I2C0_CLK                 = 18U, /*!< I2C0       clock source */
    I2C1_CLK                 = 19U, /*!< I2C1       clock source */
    I2C2_CLK                 = 20U, /*!< I2C2       clock source */
    I2C3_CLK                 = 21U, /*!< I2C3       clock source */
    I2C4_CLK                 = 22U, /*!< I2C4       clock source */
    SPI0_CLK                 = 23U, /*!< SPI0       clock source */
    SPI1_CLK                 = 24U, /*!< SPI1       clock source */
    SPI2_CLK                 = 25U, /*!< SPI2       clock source */
    SPI3_CLK                 = 26U, /*!< SPI3       clock source */
    SPI4_CLK                 = 27U, /*!< SPI4       clock source */
    SPI5_CLK                 = 28U, /*!< SPI5       clock source */
    SPI6_CLK                 = 29U, /*!< SPI6       clock source */
    SPI7_CLK                 = 30U, /*!< SPI7       clock source */
    FlexCAN0_CLK             = 31U, /*!< FlexCAN0   clock source */
    FlexCAN1_CLK             = 32U, /*!< FlexCAN1   clock source */
    FlexCAN2_CLK             = 33U, /*!< FlexCAN2   clock source */
    FlexCAN3_CLK             = 34U, /*!< FlexCAN3   clock source */
    FlexCAN4_CLK             = 35U, /*!< FlexCAN4   clock source */
    FlexCAN5_CLK             = 36U, /*!< FlexCAN5   clock source */
    FlexCAN6_CLK             = 37U, /*!< FlexCAN6   clock source */
    FlexCAN7_CLK             = 38U, /*!< FlexCAN7   clock source */
    ADC0_CLK                 = 39U, /*!< ADC0       clock source */
    ADC1_CLK                 = 40U, /*!< ADC1       clock source */
    ACMP0_CLK                = 41U, /*!< ACMP0      clock source */
    ACMP1_CLK                = 42U, /*!< ACMP1      clock source */
    PTU0_CLK                 = 43U, /*!< PTU0       clock source */
    PTU1_CLK                 = 44U, /*!< PTU1       clock source */
    TMU_CLK                  = 45U, /*!< TMU        clock source */
    eTMR0_CLK                = 46U, /*!< eTMR0      clock source */
    eTMR1_CLK                = 47U, /*!< eTMR1      clock source */
    eTMR2_CLK                = 48U, /*!< eTMR2      clock source */
    eTMR3_CLK                = 49U, /*!< eTMR3      clock source */
    eTMR4_CLK                = 50U, /*!< eTMR4      clock source */
    eTMR5_CLK                = 51U, /*!< eTMR5      clock source */
    MPWM0_CLK                = 52U, /*!< MPWM0      clock source */
    MPWM1_CLK                = 53U, /*!< MPWM1      clock source */
    MPWM2_CLK                = 54U, /*!< MPWM2      clock source */
    TMR0_CLK                 = 55U, /*!< TMR0       clock source */
    TMR1_CLK                 = 56U, /*!< TMR1       clock source */
    SENT0_CLK                = 57U, /*!< SENT0      clock source */
    SENT1_CLK                = 58U, /*!< SENT1      clock source */
    pTMR0_CLK                = 59U, /*!< pTMR0      clock source */
    pTMR1_CLK                = 60U, /*!< pTMR1      clock source */
    pTMR2_CLK                = 61U, /*!< pTMR2      clock source */
    lpTMR0_CLK               = 62U, /*!< lpTMR0     clock source */
    RTC_CLK                  = 63U, /*!< RTC        clock source */
    WKU_CLK                  = 64U, /*!< WKU        clock source */
    PCRC_CLK                  = 65U, /*!< PCRC        clock source */
    TRNG_CLK                 = 66U, /*!< TRNG       clock source */
    HCU_CLK                  = 67U, /*!< HCU        clock source */
    QSPI_CLK                 = 68U, /*!< QSPI       clock source */
    WDG_CLK                  = 69U, /*!< WDG        clock source */
    FMU_CLK                  = 70U, /*!< FMU        clock source */
    INTM_CLK                 = 71U, /*!< INTM       clock source */
    EMU_CLK                  = 72U, /*!< EMU        clock source */
    SAI0_CLK                 = 73U, /*!< SAI0       clock source */
    SAI1_CLK                 = 74U, /*!< SAI1       clock source */
    STU_CLK                  = 75U, /*!< STU        clock source */
    ENET_CLK                 = 76U, /*!< ENET       clock source */
    CIM_CLK                  = 77U, /*!< CIM        clock source */
    SCU_CLK                  = 78U, /*!< SCU        clock source */
    PCU_CLK                  = 79U, /*!< PCU        clock source */
    RCU_CLK                  = 80U, /*!< RCU        clock source */

    IPC_PERI_END_OF_CLOCK    = 81U, /*!< IPC_PERI_END_OF_CLOCK */
    IPC_SIRC_DIV4_CLK        = 82U, /*!< IPC_SIRC   clock source */
    IPC_FIRC_CLK             = 83U, /*!< IPC_FIRC   clock source */
    IPC_FXOSC_CLK            = 84U, /*!< IPC_FXOSC  clock source */
    IPC_SXOSC_CLK            = 85U, /*!< IPC_SXOSC  clock source */
    IPC_PLL_CLK              = 86U, /*!< IPC_PLL    clock source */

    IPC_END_OF_CLOCK         = 87U, /*!< IPC_END_OF_CLOCK */
    CORE_CLK                 = 88U, /*!< CORE       clock source */
    FAST_BUS_CLK             = 89U, /*!< FAST_BUS   clock source */
    SLOW_BUS_CLK             = 90U, /*!< SLOW_BUS   clock source */

    CLOCK_NAME_COUNT         = 91U, /*!< CLOCK_NAME_COUNT */
} clock_names_t;
/*! @brief IPC clock name mappings
 *  Mappings between clock names and peripheral clock control indexes.
 *  If there is no peripheral clock control index for a clock name,
 *  then the corresponding value is IPC_INVALID_INDEX.
 */
#define IPC_DMA_INDEX          (  8U)
#define IPC_TRACE_INDEX        ( 15U)
#define IPC_GPIO_INDEX         ( 17U)
#define IPC_PCTRLA_INDEX       ( 18U)
#define IPC_PCTRLB_INDEX       ( 19U)
#define IPC_PCTRLC_INDEX       ( 20U)
#define IPC_PCTRLD_INDEX       ( 21U)
#define IPC_PCTRLE_INDEX       ( 22U)
#define IPC_LINFlexD0_INDEX    ( 27U)
#define IPC_LINFlexD1_INDEX    ( 28U)
#define IPC_LINFlexD2_INDEX    ( 29U)
#define IPC_LINFlexD3_INDEX    ( 30U)
#define IPC_LINFlexD4_INDEX    ( 31U)
#define IPC_LINFlexD5_INDEX    ( 32U)
#define IPC_LINFlexD6_INDEX    ( 33U)
#define IPC_LINFlexD7_INDEX    ( 34U)
#define IPC_LINFlexD8_INDEX    ( 35U)
#define IPC_LINFlexD9_INDEX    ( 36U)
#define IPC_I2C0_INDEX         ( 37U)
#define IPC_I2C1_INDEX         ( 38U)
#define IPC_I2C2_INDEX         ( 39U)
#define IPC_I2C3_INDEX         ( 40U)
#define IPC_I2C4_INDEX         ( 41U)
#define IPC_SPI0_INDEX         ( 42U)
#define IPC_SPI1_INDEX         ( 43U)
#define IPC_SPI2_INDEX         ( 44U)
#define IPC_SPI3_INDEX         ( 45U)
#define IPC_SPI4_INDEX         ( 46U)
#define IPC_SPI5_INDEX         ( 47U)
#define IPC_SPI6_INDEX         ( 48U)
#define IPC_SPI7_INDEX         ( 49U)
#define IPC_FlexCAN0_INDEX     ( 52U)
#define IPC_FlexCAN1_INDEX     ( 53U)
#define IPC_FlexCAN2_INDEX     ( 54U)
#define IPC_FlexCAN3_INDEX     ( 55U)
#define IPC_FlexCAN4_INDEX     ( 56U)
#define IPC_FlexCAN5_INDEX     ( 57U)
#define IPC_FlexCAN6_INDEX     ( 58U)
#define IPC_FlexCAN7_INDEX     ( 59U)
#define IPC_ADC0_INDEX         ( 64U)
#define IPC_ADC1_INDEX         ( 65U)
#define IPC_ACMP0_INDEX        ( 67U)
#define IPC_ACMP1_INDEX        ( 68U)
#define IPC_PTU0_INDEX         ( 70U)
#define IPC_PTU1_INDEX         ( 71U)
#define IPC_TMU_INDEX          ( 72U)
#define IPC_eTMR0_INDEX        ( 73U)
#define IPC_eTMR1_INDEX        ( 74U)
#define IPC_eTMR2_INDEX        ( 75U)
#define IPC_eTMR3_INDEX        ( 76U)
#define IPC_eTMR4_INDEX        ( 77U)
#define IPC_eTMR5_INDEX        ( 78U)
#define IPC_MPWM0_INDEX        ( 81U)
#define IPC_MPWM1_INDEX        ( 82U)
#define IPC_MPWM2_INDEX        ( 83U)
#define IPC_TMR0_INDEX         ( 84U)
#define IPC_TMR1_INDEX         ( 85U)
#define IPC_SENT0_INDEX        ( 88U)
#define IPC_SENT1_INDEX        ( 89U)
#define IPC_pTMR0_INDEX        ( 90U)
#define IPC_pTMR1_INDEX        ( 91U)
#define IPC_pTMR2_INDEX        ( 92U)
#define IPC_lpTMR0_INDEX       ( 93U)
#define IPC_RTC_INDEX          ( 96U)
#define IPC_WKU_INDEX          ( 98U)
#define IPC_PCRC_INDEX         (101U)
#define IPC_TRNG_INDEX         (102U)
#define IPC_HCU_INDEX          (103U)
#define IPC_QSPI_INDEX         (104U)
#define IPC_WDG_INDEX          (106U)
#define IPC_FMU_INDEX          (112U)
#define IPC_INTM_INDEX         (113U)
#define IPC_EMU_INDEX          (114U)
#define IPC_SAI0_INDEX         (115U)
#define IPC_SAI1_INDEX         (116U)
#define IPC_STU_INDEX          (119U)
#define IPC_ENET_INDEX         (121U)
#define IPC_CIM_INDEX          (123U)
#define IPC_SCU_INDEX          (124U)
#define IPC_PCU_INDEX          (126U)
#define IPC_RCU_INDEX          (127U)
#define IPC_IPC_PERI_END_OF_CLOCK_INDEX   (128U)
#define IPC_IPC_SIRC_INDEX     (129U)
#define IPC_IPC_FIRC_INDEX     (130U)
#define IPC_IPC_FXOSC_INDEX    (131U)
#define IPC_IPC_SXOSC_INDEX    (132U)
#define IPC_IPC_PLL_INDEX      (133U)
#define IPC_IPC_END_OF_CLOCK_INDEX   (134U)
#define IPC_CORE_INDEX         (135U)
#define IPC_FAST_BUS_INDEX     (136U)
#define IPC_SLOW_BUS_INDEX     (137U)
#define IPC_CLOCK_NAME_COUNT_INDEX   (138U)

#define  IPC_CLOCK_NAME_MAPPINGS \
{                                \
    IPC_DMA_INDEX,            /*!< DMA        clock source    8u */ \
    IPC_TRACE_INDEX,          /*!< TRACE      clock source   15u */ \
    IPC_GPIO_INDEX,           /*!< GPIO       clock source   17u */ \
    IPC_PCTRLA_INDEX,         /*!< PCTRLA     clock source   18u */ \
    IPC_PCTRLB_INDEX,         /*!< PCTRLB     clock source   19u */ \
    IPC_PCTRLC_INDEX,         /*!< PCTRLC     clock source   20u */ \
    IPC_PCTRLD_INDEX,         /*!< PCTRLD     clock source   21u */ \
    IPC_PCTRLE_INDEX,         /*!< PCTRLE     clock source   22u */ \
    IPC_LINFlexD0_INDEX,      /*!< LINFlexD0  clock source   27u */ \
    IPC_LINFlexD1_INDEX,      /*!< LINFlexD1  clock source   28u */ \
    IPC_LINFlexD2_INDEX,      /*!< LINFlexD2  clock source   29u */ \
    IPC_LINFlexD3_INDEX,      /*!< LINFlexD3  clock source   30u */ \
    IPC_LINFlexD4_INDEX,      /*!< LINFlexD4  clock source   31u */ \
    IPC_LINFlexD5_INDEX,      /*!< LINFlexD5  clock source   32u */ \
    IPC_LINFlexD6_INDEX,      /*!< LINFlexD6  clock source   33u */ \
    IPC_LINFlexD7_INDEX,      /*!< LINFlexD7  clock source   34u */ \
    IPC_LINFlexD8_INDEX,      /*!< LINFlexD8  clock source   35u */ \
    IPC_LINFlexD9_INDEX,      /*!< LINFlexD9  clock source   36u */ \
    IPC_I2C0_INDEX,           /*!< I2C0       clock source   37u */ \
    IPC_I2C1_INDEX,           /*!< I2C1       clock source   38u */ \
    IPC_I2C2_INDEX,           /*!< I2C2       clock source   39u */ \
    IPC_I2C3_INDEX,           /*!< I2C3       clock source   40u */ \
    IPC_I2C4_INDEX,           /*!< I2C4       clock source   41u */ \
    IPC_SPI0_INDEX,           /*!< SPI0       clock source   42u */ \
    IPC_SPI1_INDEX,           /*!< SPI1       clock source   43u */ \
    IPC_SPI2_INDEX,           /*!< SPI2       clock source   44u */ \
    IPC_SPI3_INDEX,           /*!< SPI3       clock source   45u */ \
    IPC_SPI4_INDEX,           /*!< SPI4       clock source   46u */ \
    IPC_SPI5_INDEX,           /*!< SPI5       clock source   47u */ \
    IPC_SPI6_INDEX,           /*!< SPI6       clock source   48u */ \
    IPC_SPI7_INDEX,           /*!< SPI7       clock source   49u */ \
    IPC_FlexCAN0_INDEX,       /*!< FlexCAN0   clock source   52u */ \
    IPC_FlexCAN1_INDEX,       /*!< FlexCAN1   clock source   53u */ \
    IPC_FlexCAN2_INDEX,       /*!< FlexCAN2   clock source   54u */ \
    IPC_FlexCAN3_INDEX,       /*!< FlexCAN3   clock source   55u */ \
    IPC_FlexCAN4_INDEX,       /*!< FlexCAN4   clock source   56u */ \
    IPC_FlexCAN5_INDEX,       /*!< FlexCAN5   clock source   57u */ \
    IPC_FlexCAN6_INDEX,       /*!< FlexCAN6   clock source   58u */ \
    IPC_FlexCAN7_INDEX,       /*!< FlexCAN7   clock source   59u */ \
    IPC_ADC0_INDEX,           /*!< ADC0       clock source   64u */ \
    IPC_ADC1_INDEX,           /*!< ADC1       clock source   65u */ \
    IPC_ACMP0_INDEX,          /*!< ACMP0      clock source   67u */ \
    IPC_ACMP1_INDEX,          /*!< ACMP1      clock source   68u */ \
    IPC_PTU0_INDEX,           /*!< PTU0       clock source   70u */ \
    IPC_PTU1_INDEX,           /*!< PTU1       clock source   71u */ \
    IPC_TMU_INDEX,            /*!< TMU        clock source   72u */ \
    IPC_eTMR0_INDEX,          /*!< eTMR0      clock source   73u */ \
    IPC_eTMR1_INDEX,          /*!< eTMR1      clock source   74u */ \
    IPC_eTMR2_INDEX,          /*!< eTMR2      clock source   75u */ \
    IPC_eTMR3_INDEX,          /*!< eTMR3      clock source   76u */ \
    IPC_eTMR4_INDEX,          /*!< eTMR4      clock source   77u */ \
    IPC_eTMR5_INDEX,          /*!< eTMR5      clock source   78u */ \
    IPC_MPWM0_INDEX,          /*!< MPWM0      clock source   81u */ \
    IPC_MPWM1_INDEX,          /*!< MPWM1      clock source   82u */ \
    IPC_MPWM2_INDEX,          /*!< MPWM2      clock source   83u */ \
    IPC_TMR0_INDEX,           /*!< TMR0       clock source   84u */ \
    IPC_TMR1_INDEX,           /*!< TMR1       clock source   85u */ \
    IPC_SENT0_INDEX,          /*!< SENT0      clock source   88u */ \
    IPC_SENT1_INDEX,          /*!< SENT1      clock source   89u */ \
    IPC_pTMR0_INDEX,          /*!< pTMR0      clock source   90u */ \
    IPC_pTMR1_INDEX,          /*!< pTMR1      clock source   91u */ \
    IPC_pTMR2_INDEX,          /*!< pTMR2      clock source   92u */ \
    IPC_lpTMR0_INDEX,         /*!< lpTMR0     clock source   93u */ \
    IPC_RTC_INDEX,            /*!< RTC        clock source   96u */ \
    IPC_WKU_INDEX,            /*!< WKU        clock source   98u */ \
    IPC_PCRC_INDEX,            /*!< PCRC        clock source  101u */ \
    IPC_TRNG_INDEX,           /*!< TRNG       clock source  102u */ \
    IPC_HCU_INDEX,            /*!< HCU        clock source  103u */ \
    IPC_QSPI_INDEX,           /*!< QSPI       clock source  104u */ \
    IPC_WDG_INDEX,            /*!< WDG        clock source  106u */ \
    IPC_FMU_INDEX,            /*!< FMU        clock source  112u */ \
    IPC_INTM_INDEX,           /*!< INTM       clock source  113u */ \
    IPC_EMU_INDEX,            /*!< EMU        clock source  114u */ \
    IPC_SAI0_INDEX,           /*!< SAI0       clock source  115u */ \
    IPC_SAI1_INDEX,           /*!< SAI1       clock source  116u */ \
    IPC_STU_INDEX,            /*!< STU        clock source  119u */ \
    IPC_ENET_INDEX,           /*!< ENET       clock source  121u */ \
    IPC_CIM_INDEX,            /*!< CIM        clock source  123u */ \
    IPC_SCU_INDEX,            /*!< SCU        clock source  124u */ \
    IPC_PCU_INDEX,            /*!< PCU        clock source  126u */ \
    IPC_RCU_INDEX,            /*!< RCU        clock source  127u */ \
    IPC_IPC_PERI_END_OF_CLOCK_INDEX,     /*!< IPC_PERI_END_OF_CLOCK clock source  128u */ \
    IPC_IPC_SIRC_INDEX,       /*!< IPC_SIRC   clock source  129u */ \
    IPC_IPC_FIRC_INDEX,       /*!< IPC_FIRC   clock source  130u */ \
    IPC_IPC_FXOSC_INDEX,      /*!< IPC_FXOSC  clock source  131u */ \
    IPC_IPC_SXOSC_INDEX,      /*!< IPC_SXOSC  clock source  132u */ \
    IPC_IPC_PLL_INDEX,        /*!< IPC_PLL    clock source  133u */ \
    IPC_IPC_END_OF_CLOCK_INDEX,     /*!< IPC_END_OF_CLOCK clock source  134u */ \
    IPC_CORE_INDEX,           /*!< CORE       clock source  135u */ \
    IPC_FAST_BUS_INDEX,       /*!< FAST_BUS   clock source  136u */ \
    IPC_SLOW_BUS_INDEX,       /*!< SLOW_BUS   clock source  137u */ \
}

/* DMA module features */

/* @brief Number of DMA channels. */
#define FEATURE_DMA_CHANNELS (32U)
/* @brief Number of DMA virtual channels. */
#define FEATURE_DMA_VIRTUAL_CHANNELS (FEATURE_DMA_CHANNELS * DMA_INSTANCE_COUNT)
/* @brief Number of DMA interrupt lines. */
#define FEATURE_DMA_CHANNELS_INTERRUPT_LINES (32U)
/* @brief Number of DMA virtual interrupt lines. */
#define FEATURE_DMA_VIRTUAL_CHANNELS_INTERRUPT_LINES ((uint32_t)FEATURE_DMA_CHANNELS_INTERRUPT_LINES * (uint32_t)DMA_INSTANCE_COUNT)
/* @brief Number of DMA error interrupt lines. */
#define FEATURE_DMA_ERROR_INTERRUPT_LINES (1U)
/* @brief Number of DMA virtual error interrupt lines. */
#define FEATURE_DMA_VIRTUAL_ERROR_INTERRUPT_LINES ((uint32_t)FEATURE_DMA_ERROR_INTERRUPT_LINES * (uint32_t)DMA_INSTANCE_COUNT)
/* @brief DMA module has error interrupt. */
#define FEATURE_DMA_HAS_ERROR_IRQ
/* @brief DMA module separate interrupt lines for each channel */
#define FEATURE_DMA_SEPARATE_IRQ_LINES_PER_CHN
/* @brief Clock name for DMA */
#define FEATURE_DMA_CLOCK_NAMES {DMA_CLK}
/* @brief DMA channel width based on number of CTSs: 2^N, N=4,5,... */
#define FEATURE_DMA_CH_WIDTH (5U)
/* @brief DMA channel to instance */
#define FEATURE_DMA_VCH_TO_INSTANCE(x) 	((x) >> (uint32_t)FEATURE_DMA_CH_WIDTH)
/* @brief DMA virtual channel to channel */
#define FEATURE_DMA_VCH_TO_CH(x)        ((x) & ((uint32_t)FEATURE_DMA_CHANNELS - 1U))
/* @brief DMA supports bus bandwidth control. */
#define FEATURE_DMA_ENGINE_STALL
/* @brief DMA supports 8 bytes transfer size. */
#define FEATURE_DMA_TRANSFER_SIZE_8B    (1U)
/* @brief DMA supports 16 bytes transfer size. */
#define FEATURE_DMA_TRANSFER_SIZE_16B    (0U)
/* @brief DMA supports 32 bytes transfer size. */
#define FEATURE_DMA_TRANSFER_SIZE_32B    (0U)
/* @brief DMA supports 64 bytes transfer size. */
#define FEATURE_DMA_TRANSFER_SIZE_64B    (0U)

/* DMAMUX module features */

/* @brief DMAMUX peripheral is available in silicon. */
#define FEATURE_DMAMUX_AVAILABLE
/* @brief Number of DMA channels. */
#define FEATURE_DMAMUX_CHANNELS (32U)
/* @brief Conversion from request source to the actual DMAMUX channel */
#define FEATURE_DMAMUX_REQ_SRC_TO_CH(x) (x)
/* @brief Mapping between request source and DMAMUX instance */
#define FEATURE_DMAMUX_REQ_SRC_TO_INSTANCE(x) (0U)
/* @brief Conversion from DMA channel index to DMAMUX channel. */
#define FEATURE_DMAMUX_DMA_CH_TO_CH(x) (x)
/* @brief Conversion from DMAMUX channel DMAMUX register index. */
#define FEATURE_DMAMUX_CHN_REG_INDEX(x) (x)

/*!
 * @brief Structure for the DMA hardware request
 *
 * Defines the structure for the DMA hardware request collections. The user can configure the
 * hardware request into DMAMUX to trigger the DMA transfer accordingly. The index
 * of the hardware request varies according  to the to SoC.
 */
typedef enum {
    DMA_REQ_DISABLED         =   0U,     /*!< Disabled         dma request    0u */ \
    DMA_REQ_LINFlexD0_RX     =   2U,     /*!< LINFlexD0_RX     dma request    2u */ \
    DMA_REQ_LINFlexD0_TX     =   3U,     /*!< LINFlexD0_TX     dma request    3u */ \
    DMA_REQ_LINFlexD1_RX     =   4U,     /*!< LINFlexD1_RX     dma request    4u */ \
    DMA_REQ_LINFlexD1_TX     =   5U,     /*!< LINFlexD1_TX     dma request    5u */ \
    DMA_REQ_LINFlexD2_RX     =   6U,     /*!< LINFlexD2_RX     dma request    6u */ \
    DMA_REQ_LINFlexD2_TX     =   7U,     /*!< LINFlexD2_TX     dma request    7u */ \
    DMA_REQ_I2C1_RX          =   8U,     /*!< I2C1_RX          dma request    8u */ \
    DMA_REQ_I2C1_TX          =   9U,     /*!< I2C1_TX          dma request    9u */ \
    DMA_REQ_LINFlexD3_RX     =  10U,     /*!< LINFlexD3_RX     dma request   10u */ \
    DMA_REQ_LINFlexD3_TX     =  11U,     /*!< LINFlexD3_TX     dma request   11u */ \
    DMA_REQ_LINFlexD4_RX     =  12U,     /*!< LINFlexD4_RX     dma request   12u */ \
    DMA_REQ_LINFlexD4_TX     =  13U,     /*!< LINFlexD4_TX     dma request   13u */ \
    DMA_REQ_SPI0_RX          =  14U,     /*!< SPI0_RX          dma request   14u */ \
    DMA_REQ_SPI0_TX          =  15U,     /*!< SPI0_TX          dma request   15u */ \
    DMA_REQ_SPI1_RX          =  16U,     /*!< SPI1_RX          dma request   16u */ \
    DMA_REQ_SPI1_TX          =  17U,     /*!< SPI1_TX          dma request   17u */ \
    DMA_REQ_SPI2_RX          =  18U,     /*!< SPI2_RX          dma request   18u */ \
    DMA_REQ_SPI2_TX          =  19U,     /*!< SPI2_TX          dma request   19u */ \
    DMA_REQ_eTMR1_CH0        =  20U,     /*!< eTMR1_CH0        dma request   20u */ \
    DMA_REQ_eTMR1_CH1        =  21U,     /*!< eTMR1_CH1        dma request   21u */ \
    DMA_REQ_eTMR1_CH2        =  22U,     /*!< eTMR1_CH2        dma request   22u */ \
    DMA_REQ_eTMR1_CH3        =  23U,     /*!< eTMR1_CH3        dma request   23u */ \
    DMA_REQ_eTMR1_CH4        =  24U,     /*!< eTMR1_CH4        dma request   24u */ \
    DMA_REQ_eTMR1_CH5        =  25U,     /*!< eTMR1_CH5        dma request   25u */ \
    DMA_REQ_eTMR1_CH6        =  26U,     /*!< eTMR1_CH6        dma request   26u */ \
    DMA_REQ_eTMR1_CH7        =  27U,     /*!< eTMR1_CH7        dma request   27u */ \
    DMA_REQ_eTMR2_CH0        =  28U,     /*!< eTMR2_CH0        dma request   28u */ \
    DMA_REQ_eTMR2_CH1        =  29U,     /*!< eTMR2_CH1        dma request   29u */ \
    DMA_REQ_eTMR2_CH2        =  30U,     /*!< eTMR2_CH2        dma request   30u */ \
    DMA_REQ_eTMR2_CH3        =  31U,     /*!< eTMR2_CH3        dma request   31u */ \
    DMA_REQ_eTMR2_CH4        =  32U,     /*!< eTMR2_CH4        dma request   32u */ \
    DMA_REQ_eTMR2_CH5        =  33U,     /*!< eTMR2_CH5        dma request   33u */ \
    DMA_REQ_eTMR2_CH6        =  34U,     /*!< eTMR2_CH6        dma request   34u */ \
    DMA_REQ_eTMR2_CH7        =  35U,     /*!< eTMR2_CH7        dma request   35u */ \
    DMA_REQ_eTMR0            =  36U,     /*!< eTMR0            dma request   36u */ \
    DMA_REQ_eTMR3            =  37U,     /*!< eTMR3            dma request   37u */ \
    DMA_REQ_eTMR4            =  38U,     /*!< eTMR4            dma request   38u */ \
    DMA_REQ_eTMR5            =  39U,     /*!< eTMR5            dma request   39u */ \
    DMA_REQ_ADC0             =  42U,     /*!< ADC0             dma request   42u */ \
    DMA_REQ_ADC1             =  43U,     /*!< ADC1             dma request   43u */ \
    DMA_REQ_I2C0_RX          =  44U,     /*!< I2C0_RX          dma request   44u */ \
    DMA_REQ_I2C0_TX          =  45U,     /*!< I2C0_TX          dma request   45u */ \
    DMA_REQ_PTU0             =  46U,     /*!< PTU0             dma request   46u */ \
    DMA_REQ_PTU1             =  47U,     /*!< PTU1             dma request   47u */ \
    DMA_REQ_ACMP0            =  48U,     /*!< ACMP0            dma request   48u */ \
    DMA_REQ_GPIOA            =  49U,     /*!< GPIOA            dma request   49u */ \
    DMA_REQ_GPIOB            =  50U,     /*!< GPIOB            dma request   50u */ \
    DMA_REQ_GPIOC            =  51U,     /*!< GPIOC            dma request   51u */ \
    DMA_REQ_GPIOD            =  52U,     /*!< GPIOD            dma request   52u */ \
    DMA_REQ_GPIOE            =  53U,     /*!< GPIOE            dma request   53u */ \
    DMA_REQ_FLEXCAN0         =  54U,     /*!< FLEXCAN0         dma request   54u */ \
    DMA_REQ_FLEXCAN1         =  55U,     /*!< FLEXCAN1         dma request   55u */ \
    DMA_REQ_FLEXCAN2         =  56U,     /*!< FLEXCAN2         dma request   56u */ \
    DMA_REQ_FLEXCAN3         =  57U,     /*!< FLEXCAN3         dma request   57u */ \
    DMA_REQ_FLEXCAN4         =  58U,     /*!< FLEXCAN4         dma request   58u */ \
    DMA_REQ_lpTMR0           =  59U,     /*!< lpTMR0           dma request   59u */ \
    DMA_REQ_FLEXCAN5         =  60U,     /*!< FLEXCAN5         dma request   60u */ \
    DMA_REQ_SPI3_RX          =  64U,     /*!< SPI3_RX          dma request   64u */ \
    DMA_REQ_SPI3_TX          =  65U,     /*!< SPI3_TX          dma request   65u */ \
    DMA_REQ_SPI4_RX          =  66U,     /*!< SPI4_RX          dma request   66u */ \
    DMA_REQ_SPI4_TX          =  67U,     /*!< SPI4_TX          dma request   67u */ \
    DMA_REQ_SPI5_RX          =  68U,     /*!< SPI5_RX          dma request   68u */ \
    DMA_REQ_SPI5_TX          =  69U,     /*!< SPI5_TX          dma request   69u */ \
    DMA_REQ_I2C2_RX          =  70U,     /*!< I2C2_RX          dma request   70u */ \
    DMA_REQ_I2C2_TX          =  71U,     /*!< I2C2_TX          dma request   71u */ \
    DMA_REQ_LINFlexD5_TX     =  72U,     /*!< LINFlexD5_TX     dma request   72u */ \
    DMA_REQ_LINFlexD5_RX     =  73U,     /*!< LINFlexD5_RX     dma request   73u */ \
    DMA_REQ_HCU_IN           =  74U,     /*!< HCU_IN           dma request   74u */ \
    DMA_REQ_HCU_OUT          =  75U,     /*!< HCU_OUT          dma request   75u */ \
    DMA_REQ_LINFlexD6_TX     =  76U,     /*!< LINFlexD6_TX     dma request   76u */ \
    DMA_REQ_LINFlexD6_RX     =  77U,     /*!< LINFlexD6_RX     dma request   77u */ \
    DMA_REQ_LINFlexD7_TX     =  78U,     /*!< LINFlexD7_TX     dma request   78u */ \
    DMA_REQ_LINFlexD7_RX     =  79U,     /*!< LINFlexD7_RX     dma request   79u */ \
    DMA_REQ_LINFlexD8_TX     =  80U,     /*!< LINFlexD8_TX     dma request   80u */ \
    DMA_REQ_LINFlexD8_RX     =  81U,     /*!< LINFlexD8_RX     dma request   81u */ \
    DMA_REQ_LINFlexD9_TX     =  82U,     /*!< LINFlexD9_TX     dma request   82u */ \
    DMA_REQ_LINFlexD9_RX     =  83U,     /*!< LINFlexD9_RX     dma request   83u */ \
    DMA_REQ_I2C3_RX          =  84U,     /*!< I2C3_RX          dma request   84u */ \
    DMA_REQ_I2C3_TX          =  85U,     /*!< I2C3_TX          dma request   85u */ \
    DMA_REQ_SPI6_RX          =  86U,     /*!< SPI6_RX          dma request   86u */ \
    DMA_REQ_SPI6_TX          =  87U,     /*!< SPI6_TX          dma request   87u */ \
    DMA_REQ_SPI7_RX          =  88U,     /*!< SPI7_RX          dma request   88u */ \
    DMA_REQ_SPI7_TX          =  89U,     /*!< SPI7_TX          dma request   89u */ \
    DMA_REQ_I2C4_RX          =  90U,     /*!< I2C4_RX          dma request   90u */ \
    DMA_REQ_I2C4_TX          =  91U,     /*!< I2C4_TX          dma request   91u */ \
    DMA_REQ_ACMP1            =  94U,     /*!< ACMP1            dma request   94u */ \
    DMA_REQ_FLEXCAN6         =  96U,     /*!< FLEXCAN6         dma request   96u */ \
    DMA_REQ_FLEXCAN7         =  97U,     /*!< FLEXCAN7         dma request   97u */ \
    DMA_REQ_SENT0_FAST       = 102U,     /*!< SENT0_FAST       dma request  102u */ \
    DMA_REQ_SENT0_SLOW       = 103U,     /*!< SENT0_SLOW       dma request  103u */ \
    DMA_REQ_SENT1_FAST       = 104U,     /*!< SENT1_FAST       dma request  104u */ \
    DMA_REQ_SENT1_SLOW       = 105U,     /*!< SENT1_SLOW       dma request  105u */ \
    DMA_REQ_QSPI_RX          = 106U,     /*!< QSPI_RX          dma request  106u */ \
    DMA_REQ_QSPI_TX          = 107U,     /*!< QSPI_TX          dma request  107u */ \
    DMA_REQ_MPWM0            = 108U,     /*!< MPWM0            dma request  108u */ \
    DMA_REQ_MPWM1            = 109U,     /*!< MPWM1            dma request  109u */ \
    DMA_REQ_MPWM2            = 110U,     /*!< MPWM2            dma request  110u */ \
    DMA_REQ_EMAC_CH0         = 112U,     /*!< EMAC_CH0         dma request  112u */ \
    DMA_REQ_EMAC_CH1         = 113U,     /*!< EMAC_CH1         dma request  113u */ \
    DMA_REQ_EMAC_CH2         = 114U,     /*!< EMAC_CH2         dma request  114u */ \
    DMA_REQ_EMAC_CH3         = 115U,     /*!< EMAC_CH3         dma request  115u */ \
    DMA_REQ_SAI0_RX_CH0      = 116U,     /*!< SAI0_RX_CH0      dma request  116u */ \
    DMA_REQ_SAI0_RX_CH1      = 117U,     /*!< SAI0_RX_CH1      dma request  117u */ \
    DMA_REQ_SAI0_RX_CH2      = 118U,     /*!< SAI0_RX_CH2      dma request  118u */ \
    DMA_REQ_SAI0_RX_CH3      = 119U,     /*!< SAI0_RX_CH3      dma request  119u */ \
    DMA_REQ_SAI0_TX_CH0      = 120U,     /*!< SAI0_TX_CH0      dma request  120u */ \
    DMA_REQ_SAI0_TX_CH1      = 121U,     /*!< SAI0_TX_CH1      dma request  121u */ \
    DMA_REQ_SAI0_TX_CH2      = 122U,     /*!< SAI0_TX_CH2      dma request  122u */ \
    DMA_REQ_SAI0_TX_CH3      = 123U,     /*!< SAI0_TX_CH3      dma request  123u */ \
    DMA_REQ_SAI1_RX          = 124U,     /*!< SAI1_RX          dma request  124u */ \
    DMA_REQ_SAI1_TX          = 125U,     /*!< SAI1_TX          dma request  125u */ \
} dma_request_source_t;

/* TMU module features */
/*!
 * @brief Enumeration for trigger source module of TMU
 *
 * Describes all possible inputs (trigger sources) of the TMU IP
 * This enumeration depends on the supported instances in device
 */
enum tmu_trigger_source_e
{
    TMU_TRIG_SOURCE_DISABLED                       = 0U,
    TMU_TRIG_SOURCE_VDD                            = 1U,
    TMU_TRIG_SOURCE_TMU_IN0                        = 2U,
    TMU_TRIG_SOURCE_TMU_IN1                        = 3U,
    TMU_TRIG_SOURCE_TMU_IN2                        = 4U,
    TMU_TRIG_SOURCE_TMU_IN3                        = 5U,
    TMU_TRIG_SOURCE_TMU_IN4                        = 6U,
    TMU_TRIG_SOURCE_TMU_IN5                        = 7U,
    TMU_TRIG_SOURCE_TMU_IN6                        = 8U,
    TMU_TRIG_SOURCE_TMU_IN7                        = 9U,
    TMU_TRIG_SOURCE_TMU_IN8                        = 10U,
    TMU_TRIG_SOURCE_TMU_IN9                        = 11U,
    TMU_TRIG_SOURCE_TMU_IN10                       = 12U,
    TMU_TRIG_SOURCE_TMU_IN11                       = 13U,
    TMU_TRIG_SOURCE_ACMP0_OUT                      = 14U,
    TMU_TRIG_SOURCE_ACMP1_OUT                      = 15U,
    TMU_TRIG_SOURCE_pTMR0_CH0                      = 17U,
    TMU_TRIG_SOURCE_pTMR0_CH1                      = 18U,
    TMU_TRIG_SOURCE_pTMR0_CH2                      = 19U,
    TMU_TRIG_SOURCE_pTMR0_CH3                      = 20U,
    TMU_TRIG_SOURCE_lpTMR0                         = 21U,
    TMU_TRIG_SOURCE_eTMR0_INIT_TRIG                = 22U,
    TMU_TRIG_SOURCE_eTMR0_EXT_TRIG                 = 23U,
    TMU_TRIG_SOURCE_eTMR1_INIT_TRIG                = 24U,
    TMU_TRIG_SOURCE_eTMR1_EXT_TRIG                 = 25U,
    TMU_TRIG_SOURCE_eTMR2_INIT_TRIG                = 26U,
    TMU_TRIG_SOURCE_eTMR2_EXT_TRIG                 = 27U,
    TMU_TRIG_SOURCE_eTMR3_INIT_TRIG                = 28U,
    TMU_TRIG_SOURCE_eTMR3_EXT_TRIG                 = 29U,
    TMU_TRIG_SOURCE_ADC0_COCO                      = 30U,
    TMU_TRIG_SOURCE_ADC0_AWD                       = 31U,
    TMU_TRIG_SOURCE_ADC1_COCO                      = 32U,
    TMU_TRIG_SOURCE_ADC1_AWD                       = 33U,
    TMU_TRIG_SOURCE_PTU0_TRIG                      = 34U,
    TMU_TRIG_SOURCE_PTU0_PULSE_OUT                 = 36U,
    TMU_TRIG_SOURCE_PTU1_TRIG                      = 37U,
    TMU_TRIG_SOURCE_PTU1_PULSE_OUT                 = 39U,
    TMU_TRIG_SOURCE_RTC_ALARM                      = 43U,
    TMU_TRIG_SOURCE_RTC_SECOND                     = 44U,
    TMU_TRIG_SOURCE_I2C0_MASTER_TRIG               = 55U,
    TMU_TRIG_SOURCE_I2C0_SLAVE_TRIG                = 56U,
    TMU_TRIG_SOURCE_SPI0_FRAME                     = 59U,
    TMU_TRIG_SOURCE_SPI0_RX_DATA                   = 60U,
    TMU_TRIG_SOURCE_SPI1_FRAME                     = 61U,
    TMU_TRIG_SOURCE_SPI1_RX_DATA                   = 62U,
    TMU_TRIG_SOURCE_CIM_SW_TRIG                    = 63U,
    TMU_TRIG_SOURCE_I2C1_MASTER_TRIG               = 67U,
    TMU_TRIG_SOURCE_I2C1_SLAVE_TRIG                = 68U,
    TMU_TRIG_SOURCE_eTMR4_INIT_TRIG                = 69U,
    TMU_TRIG_SOURCE_eTMR4_EXT_TRIG                 = 70U,
    TMU_TRIG_SOURCE_eTMR5_INIT_TRIG                = 71U,
    TMU_TRIG_SOURCE_eTMR5_EXT_TRIG                 = 72U,
    TMU_TRIG_SOURCE_SPI2_FRAME                     = 77U,
    TMU_TRIG_SOURCE_SPI2_RX_DATA                   = 78U,
    TMU_TRIG_SOURCE_SPI3_FRAME                     = 79U,
    TMU_TRIG_SOURCE_SPI3_RX_DATA                   = 80U,
    TMU_TRIG_SOURCE_SPI4_FRAME                     = 81U,
    TMU_TRIG_SOURCE_SPI4_RX_DATA                   = 82U,
    TMU_TRIG_SOURCE_SPI5_FRAME                     = 83U,
    TMU_TRIG_SOURCE_SPI5_RX_DATA                   = 84U,
    TMU_TRIG_SOURCE_I2C2_MASTER_TRIG               = 85U,
    TMU_TRIG_SOURCE_I2C2_SLAVE_TRIG                = 86U,
    TMU_TRIG_SOURCE_pTMR1_CH0                      = 87U,
    TMU_TRIG_SOURCE_pTMR1_CH1                      = 88U,
    TMU_TRIG_SOURCE_pTMR1_CH2                      = 89U,
    TMU_TRIG_SOURCE_pTMR1_CH3                      = 90U,
    TMU_TRIG_SOURCE_pTMR2_CH0                      = 91U,
    TMU_TRIG_SOURCE_pTMR2_CH1                      = 92U,
    TMU_TRIG_SOURCE_pTMR2_CH2                      = 93U,
    TMU_TRIG_SOURCE_pTMR2_CH3                      = 94U,
    TMU_TRIG_SOURCE_I2C3_MASTER_TRIG               = 100U,
    TMU_TRIG_SOURCE_I2C3_SLAVE_TRIG                = 101U,
    TMU_TRIG_SOURCE_I2C4_MASTER_TRIG               = 102U,
    TMU_TRIG_SOURCE_I2C4_SLAVE_TRIG                = 103U,
    TMU_TRIG_SOURCE_SPI6_FRAME                     = 104U,
    TMU_TRIG_SOURCE_SPI6_RX_DATA                   = 105U,
    TMU_TRIG_SOURCE_SPI7_FRAME                     = 106U,
    TMU_TRIG_SOURCE_SPI7_RX_DATA                   = 107U,
    TMU_TRIG_SOURCE_CM7_TXEV                       = 110U,
    TMU_TRIG_SOURCE_FMU_ERR_DO0                    = 112U,
    TMU_TRIG_SOURCE_FMU_ERR_DO1                    = 113U,
    TMU_TRIG_SOURCE_MPWM0_TRIG_OUT0_3              = 116U,
    TMU_TRIG_SOURCE_MPWM0_TRIG_OUT4_7              = 117U,
    TMU_TRIG_SOURCE_MPWM0_TRIG_OUT8_11             = 118U,
    TMU_TRIG_SOURCE_MPWM0_TRIG_OUT12_15            = 119U,
    TMU_TRIG_SOURCE_MPWM1_TRIG_OUT0_3              = 120U,
    TMU_TRIG_SOURCE_MPWM1_TRIG_OUT4_7              = 121U,
    TMU_TRIG_SOURCE_MPWM1_TRIG_OUT8_11             = 122U,
    TMU_TRIG_SOURCE_MPWM1_TRIG_OUT12_15            = 123U,
    TMU_TRIG_SOURCE_MPWM2_TRIG_OUT0_3              = 124U,
    TMU_TRIG_SOURCE_MPWM2_TRIG_OUT4_7              = 125U,
    TMU_TRIG_SOURCE_MPWM2_TRIG_OUT8_11             = 126U,
    TMU_TRIG_SOURCE_MPWM2_TRIG_OUT12_15            = 127U,
};

/*!
 * @brief Enumeration for target module of TMU
 *
 * Describes all possible outputs (target modules) of the TMU IP
 * This enumeration depends on the supported instances in device
 */
enum tmu_target_module_e
{
    TMU_TARGET_MODULE_TMU_OUT0                       = 4U,
    TMU_TARGET_MODULE_TMU_OUT1                       = 5U,
    TMU_TARGET_MODULE_TMU_OUT2                       = 6U,
    TMU_TARGET_MODULE_TMU_OUT3                       = 7U,
    TMU_TARGET_MODULE_TMU_OUT4                       = 8U,
    TMU_TARGET_MODULE_TMU_OUT5                       = 9U,
    TMU_TARGET_MODULE_TMU_OUT6                       = 10U,
    TMU_TARGET_MODULE_TMU_OUT7                       = 11U,
    TMU_TARGET_MODULE_ADC0_EXT_TRIG0                 = 12U,
    TMU_TARGET_MODULE_ADC0_EXT_TRIG1                 = 13U,
    TMU_TARGET_MODULE_ADC0_EXT_TRIG2                 = 14U,
    TMU_TARGET_MODULE_ADC0_EXT_TRIG3                 = 15U,
    TMU_TARGET_MODULE_ADC1_EXT_TRIG0                 = 16U,
    TMU_TARGET_MODULE_ADC1_EXT_TRIG1                 = 17U,
    TMU_TARGET_MODULE_ADC1_EXT_TRIG2                 = 18U,
    TMU_TARGET_MODULE_ADC1_EXT_TRIG3                 = 19U,
    TMU_TARGET_MODULE_TMU_OUT8                       = 24U,
    TMU_TARGET_MODULE_TMU_OUT9                       = 25U,
    TMU_TARGET_MODULE_TMU_OUT10                      = 26U,
    TMU_TARGET_MODULE_TMU_OUT11                      = 27U,
    TMU_TARGET_MODULE_ACMP0_SAMPLE_INPUT             = 28U,
    TMU_TARGET_MODULE_ACMP1_SAMPLE_INPUT             = 29U,
    TMU_TARGET_MODULE_MPWM_TRIG0                     = 30U,
    TMU_TARGET_MODULE_MPWM_TRIG1                     = 31U,
    TMU_TARGET_MODULE_MPWM_TRIG2                     = 32U,
    TMU_TARGET_MODULE_MPWM_TRIG3                     = 33U,
    TMU_TARGET_MODULE_MPWM_TRIG4                     = 34U,
    TMU_TARGET_MODULE_MPWM_TRIG5                     = 35U,
    TMU_TARGET_MODULE_MPWM_TRIG6                     = 36U,
    TMU_TARGET_MODULE_MPWM_TRIG7                     = 37U,
    TMU_TARGET_MODULE_eTMR0_HWTRIG                   = 40U,
    TMU_TARGET_MODULE_eTMR0_FAULT0                   = 41U,
    TMU_TARGET_MODULE_eTMR0_FAULT1                   = 42U,
    TMU_TARGET_MODULE_eTMR0_FAULT2                   = 43U,
    TMU_TARGET_MODULE_eTMR1_HWTRIG                   = 44U,
    TMU_TARGET_MODULE_eTMR1_FAULT0                   = 45U,
    TMU_TARGET_MODULE_eTMR1_FAULT1                   = 46U,
    TMU_TARGET_MODULE_eTMR1_FAULT2                   = 47U,
    TMU_TARGET_MODULE_eTMR2_HWTRIG                   = 48U,
    TMU_TARGET_MODULE_eTMR2_FAULT0                   = 49U,
    TMU_TARGET_MODULE_eTMR2_FAULT1                   = 50U,
    TMU_TARGET_MODULE_eTMR2_FAULT2                   = 51U,
    TMU_TARGET_MODULE_eTMR3_HWTRIG                   = 52U,
    TMU_TARGET_MODULE_eTMR3_FAULT0                   = 53U,
    TMU_TARGET_MODULE_eTMR3_FAULT1                   = 54U,
    TMU_TARGET_MODULE_eTMR3_FAULT2                   = 55U,
    TMU_TARGET_MODULE_PTU0_TRIG_IN                   = 56U,
    TMU_TARGET_MODULE_PTU1_TRIG_IN                   = 60U,
    TMU_TARGET_MODULE_eTMR0_CH0_IN                   = 64U,
    TMU_TARGET_MODULE_eTMR0_CH1_IN                   = 65U,
    TMU_TARGET_MODULE_eTMR0_CH2_IN                   = 66U,
    TMU_TARGET_MODULE_eTMR0_CH3_IN                   = 67U,
    TMU_TARGET_MODULE_eTMR1_CH0_IN                   = 68U,
    TMU_TARGET_MODULE_eTMR1_CH1_IN                   = 69U,
    TMU_TARGET_MODULE_eTMR1_CH2_IN                   = 70U,
    TMU_TARGET_MODULE_eTMR1_CH3_IN                   = 71U,
    TMU_TARGET_MODULE_eTMR2_CH0_IN                   = 72U,
    TMU_TARGET_MODULE_eTMR2_CH1_IN                   = 73U,
    TMU_TARGET_MODULE_eTMR2_CH2_IN                   = 74U,
    TMU_TARGET_MODULE_eTMR2_CH3_IN                   = 75U,
    TMU_TARGET_MODULE_LINFLEXD0_TRIG                 = 76U,
    TMU_TARGET_MODULE_LINFLEXD1_TRIG                 = 77U,
    TMU_TARGET_MODULE_LINFLEXD2_TRIG                 = 78U,
    TMU_TARGET_MODULE_LINFLEXD3_TRIG                 = 79U,
    TMU_TARGET_MODULE_LINFLEXD4_TRIG                 = 80U,
    TMU_TARGET_MODULE_LINFLEXD5_TRIG                 = 81U,
    TMU_TARGET_MODULE_LINFLEXD6_TRIG                 = 82U,
    TMU_TARGET_MODULE_LINFLEXD7_TRIG                 = 83U,
    TMU_TARGET_MODULE_I2C0_TRIG                      = 84U,
    TMU_TARGET_MODULE_LINFLEXD8_TRIG                 = 85U,
    TMU_TARGET_MODULE_LINFLEXD9_TRIG                 = 86U,
    TMU_TARGET_MODULE_eTMR3_CH0_IN                   = 88U,
    TMU_TARGET_MODULE_eTMR3_CH1_IN                   = 89U,
    TMU_TARGET_MODULE_eTMR3_CH2_IN                   = 90U,
    TMU_TARGET_MODULE_eTMR3_CH3_IN                   = 91U,
    TMU_TARGET_MODULE_SPI0_TRIG                      = 92U,
    TMU_TARGET_MODULE_SPI1_TRIG                      = 96U,
    TMU_TARGET_MODULE_SPI2_TRIG                      = 97U,
    TMU_TARGET_MODULE_SPI3_TRIG                      = 98U,
    TMU_TARGET_MODULE_SPI4_TRIG                      = 99U,
    TMU_TARGET_MODULE_lpTMR0_ALT0                    = 100U,
    TMU_TARGET_MODULE_CM7_RXEV                       = 101U,
    TMU_TARGET_MODULE_SPI5_TRIG                      = 104U,
    TMU_TARGET_MODULE_SPI6_TRIG                      = 105U,
    TMU_TARGET_MODULE_SPI7_TRIG                      = 106U,
    TMU_TARGET_MODULE_I2C1_TRIG                      = 108U,
    TMU_TARGET_MODULE_I2C2_TRIG                      = 109U,
    TMU_TARGET_MODULE_I2C3_TRIG                      = 110U,
    TMU_TARGET_MODULE_I2C4_TRIG                      = 111U,
    TMU_TARGET_MODULE_eTMR4_HWTRIG                   = 112U,
    TMU_TARGET_MODULE_eTMR5_HWTRIG                   = 116U,
    TMU_TARGET_MODULE_MPWM_TRIG8                     = 119U,
    TMU_TARGET_MODULE_MPWM_TRIG9                     = 120U,
    TMU_TARGET_MODULE_MPWM_TRIG10                    = 121U,
    TMU_TARGET_MODULE_MPWM_TRIG11                    = 122U,
    TMU_TARGET_MODULE_MPWM_TRIG12                    = 123U,
    TMU_TARGET_MODULE_MPWM_TRIG13                    = 124U,
    TMU_TARGET_MODULE_MPWM_TRIG14                    = 125U,
    TMU_TARGET_MODULE_MPWM_TRIG15                    = 126U,
};

/* @brief Constant array storing the value of all TMU output(target module) identifiers */
#define FEATURE_TMU_TARGET_MODULE                     \
    {                                                 \
    TMU_TARGET_MODULE_TMU_OUT0,                       \
    TMU_TARGET_MODULE_TMU_OUT1,                       \
    TMU_TARGET_MODULE_TMU_OUT2,                       \
    TMU_TARGET_MODULE_TMU_OUT3,                       \
    TMU_TARGET_MODULE_TMU_OUT4,                       \
    TMU_TARGET_MODULE_TMU_OUT5,                       \
    TMU_TARGET_MODULE_TMU_OUT6,                       \
    TMU_TARGET_MODULE_TMU_OUT7,                       \
    TMU_TARGET_MODULE_ADC0_EXT_TRIG0,                 \
    TMU_TARGET_MODULE_ADC0_EXT_TRIG1,                 \
    TMU_TARGET_MODULE_ADC0_EXT_TRIG2,                 \
    TMU_TARGET_MODULE_ADC0_EXT_TRIG3,                 \
    TMU_TARGET_MODULE_ADC1_EXT_TRIG0,                 \
    TMU_TARGET_MODULE_ADC1_EXT_TRIG1,                 \
    TMU_TARGET_MODULE_ADC1_EXT_TRIG2,                 \
    TMU_TARGET_MODULE_ADC1_EXT_TRIG3,                 \
    TMU_TARGET_MODULE_TMU_OUT8,                       \
    TMU_TARGET_MODULE_TMU_OUT9,                       \
    TMU_TARGET_MODULE_TMU_OUT10,                      \
    TMU_TARGET_MODULE_TMU_OUT11,                      \
    TMU_TARGET_MODULE_ACMP0_SAMPLE_INPUT,             \
    TMU_TARGET_MODULE_ACMP1_SAMPLE_INPUT,             \
    TMU_TARGET_MODULE_MPWM_TRIG0,                     \
    TMU_TARGET_MODULE_MPWM_TRIG1,                     \
    TMU_TARGET_MODULE_MPWM_TRIG2,                     \
    TMU_TARGET_MODULE_MPWM_TRIG3,                     \
    TMU_TARGET_MODULE_MPWM_TRIG4,                     \
    TMU_TARGET_MODULE_MPWM_TRIG5,                     \
    TMU_TARGET_MODULE_MPWM_TRIG6,                     \
    TMU_TARGET_MODULE_MPWM_TRIG7,                     \
    TMU_TARGET_MODULE_eTMR0_HWTRIG,                   \
    TMU_TARGET_MODULE_eTMR0_FAULT0,                   \
    TMU_TARGET_MODULE_eTMR0_FAULT1,                   \
    TMU_TARGET_MODULE_eTMR0_FAULT2,                   \
    TMU_TARGET_MODULE_eTMR1_HWTRIG,                   \
    TMU_TARGET_MODULE_eTMR1_FAULT0,                   \
    TMU_TARGET_MODULE_eTMR1_FAULT1,                   \
    TMU_TARGET_MODULE_eTMR1_FAULT2,                   \
    TMU_TARGET_MODULE_eTMR2_HWTRIG,                   \
    TMU_TARGET_MODULE_eTMR2_FAULT0,                   \
    TMU_TARGET_MODULE_eTMR2_FAULT1,                   \
    TMU_TARGET_MODULE_eTMR2_FAULT2,                   \
    TMU_TARGET_MODULE_eTMR3_HWTRIG,                   \
    TMU_TARGET_MODULE_eTMR3_FAULT0,                   \
    TMU_TARGET_MODULE_eTMR3_FAULT1,                   \
    TMU_TARGET_MODULE_eTMR3_FAULT2,                   \
    TMU_TARGET_MODULE_PTU0_TRIG_IN,                   \
    TMU_TARGET_MODULE_PTU1_TRIG_IN,                   \
    TMU_TARGET_MODULE_eTMR0_CH0_IN,                   \
    TMU_TARGET_MODULE_eTMR0_CH1_IN,                   \
    TMU_TARGET_MODULE_eTMR0_CH2_IN,                   \
    TMU_TARGET_MODULE_eTMR0_CH3_IN,                   \
    TMU_TARGET_MODULE_eTMR1_CH0_IN,                   \
    TMU_TARGET_MODULE_eTMR1_CH1_IN,                   \
    TMU_TARGET_MODULE_eTMR1_CH2_IN,                   \
    TMU_TARGET_MODULE_eTMR1_CH3_IN,                   \
    TMU_TARGET_MODULE_eTMR2_CH0_IN,                   \
    TMU_TARGET_MODULE_eTMR2_CH1_IN,                   \
    TMU_TARGET_MODULE_eTMR2_CH2_IN,                   \
    TMU_TARGET_MODULE_eTMR2_CH3_IN,                   \
    TMU_TARGET_MODULE_LINFLEXD0_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD1_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD2_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD3_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD4_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD5_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD6_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD7_TRIG,                 \
    TMU_TARGET_MODULE_I2C0_TRIG,                      \
    TMU_TARGET_MODULE_LINFLEXD8_TRIG,                 \
    TMU_TARGET_MODULE_LINFLEXD9_TRIG,                 \
    TMU_TARGET_MODULE_eTMR3_CH0_IN,                   \
    TMU_TARGET_MODULE_eTMR3_CH1_IN,                   \
    TMU_TARGET_MODULE_eTMR3_CH2_IN,                   \
    TMU_TARGET_MODULE_eTMR3_CH3_IN,                   \
    TMU_TARGET_MODULE_SPI0_TRIG,                      \
    TMU_TARGET_MODULE_SPI1_TRIG,                      \
    TMU_TARGET_MODULE_SPI2_TRIG,                      \
    TMU_TARGET_MODULE_SPI3_TRIG,                      \
    TMU_TARGET_MODULE_SPI4_TRIG,                      \
    TMU_TARGET_MODULE_lpTMR0_ALT0,                    \
    TMU_TARGET_MODULE_CM7_RXEV,                       \
    TMU_TARGET_MODULE_SPI5_TRIG,                      \
    TMU_TARGET_MODULE_SPI6_TRIG,                      \
    TMU_TARGET_MODULE_SPI7_TRIG,                      \
    TMU_TARGET_MODULE_I2C1_TRIG,                      \
    TMU_TARGET_MODULE_I2C2_TRIG,                      \
    TMU_TARGET_MODULE_I2C3_TRIG,                      \
    TMU_TARGET_MODULE_I2C4_TRIG,                      \
    TMU_TARGET_MODULE_eTMR4_HWTRIG,                   \
    TMU_TARGET_MODULE_eTMR5_HWTRIG,                   \
    TMU_TARGET_MODULE_MPWM_TRIG8,                     \
    TMU_TARGET_MODULE_MPWM_TRIG9,                     \
    TMU_TARGET_MODULE_MPWM_TRIG10,                    \
    TMU_TARGET_MODULE_MPWM_TRIG11,                    \
    TMU_TARGET_MODULE_MPWM_TRIG12,                    \
    TMU_TARGET_MODULE_MPWM_TRIG13,                    \
    TMU_TARGET_MODULE_MPWM_TRIG14,                    \
    TMU_TARGET_MODULE_MPWM_TRIG15,                    \
    }

/* SPI module features */
/* @brief Initial value for state structure */
#define FEATURE_SPI_STATE_STRUCTURES_NULL {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL}
/* @brief Has transmitter/receiver DMA enable bits. */
#define FEATURE_SPI_HAS_DMA_ENABLE (1)
/* @brief Clock indexes for SPI clock */
#define FEATURE_SPI_CLOCKS_NAMES {SPI0_CLK, SPI1_CLK, SPI2_CLK, SPI3_CLK, SPI4_CLK, SPI5_CLK, SPI6_CLK, SPI7_CLK}
/* @brief SPI support extern device selection */
#define FEATURE_SPI_HAS_EXTERNAL_DEVICES_SELECTION  (8)

/* ADC module features */

/*! @brief ADC feature flag for extended number of SC1 and R registers,
 * generically named 'alias registers' */
#define FEATURE_ADC_HAS_EXTRA_NUM_REGS                    (0)

#define NUMBER_OF_ALT_CLOCKS  ADC_CLK_ALT_1
/*! @brief ADC feature flag for defining number of external ADC channels.
 * If each ADC instance has different number of external channels, then
 * this define is set with the maximum value. */
#define FEATURE_ADC_MAX_NUM_EXT_CHANS                     (32)
/*! @brief Support Injection Sample Mode */
#define FEATURE_ADC_SUPPORT_INJECTION_MODE                (0)
/*! @brief ADC WDG Channel count */
#define FEATURE_ADC_WDG_CHANNEL_COUNT                     (4)
#define ADC_CLOCKS                                        {ADC0_CLK, ADC1_CLK}

/*! @brief ADC default clock divider from RM */
#define ADC_DEFAULT_CLOCK_DIVIDER                         (0x02U)
/*! @brief ADC default Sample Time from RM */
#define ADC_DEFAULT_SAMPLE_TIME                           (0x0CU)
/*! @brief ADC default Start Time from RM */
#define ADC_DEFAULT_START_TIME                            (0x20U)
/*! @brief Max of adc clock frequency */
#define ADC_CLOCK_FREQ_MAX_RUNTIME                        (50000000u)
/*! @brief Min of adc clock frequency */
#define ADC_CLOCK_FREQ_MIN_RUNTIME                        (2000000u)

/* CAN module features */

/* @brief Frames available in Rx FIFO flag shift */
#define FEATURE_CAN_RXFIFO_FRAME_AVAILABLE  (5U)
/* @brief Rx FIFO warning flag shift */
#define FEATURE_CAN_RXFIFO_WARNING          (6U)
/* @brief Rx FIFO overflow flag shift */
#define FEATURE_CAN_RXFIFO_OVERFLOW         (7U)
/* @brief The list contains definitions of the FD feature support on all instances */
#define FEATURE_CAN_INSTANCES_HAS_FD        {true, true, true, true, true, true}
/* @brief Has Flexible Data Rate for CAN0 */
#define FEATURE_CAN0_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN1 */
#define FEATURE_CAN1_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN2 */
#define FEATURE_CAN2_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN3 */
#define FEATURE_CAN3_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN4 */
#define FEATURE_CAN4_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN5 */
#define FEATURE_CAN5_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN6 */
#define FEATURE_CAN6_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN7 */
#define FEATURE_CAN7_HAS_FD                 (1)

#define FEATURE_CAN_HAS_RAM_ECC             (1)

/* @brief Enhance fifo message buffer count */
#define FEATURE_CAN_ERFFELn_COUNT (128U)

/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN0 */
#define FEATURE_CAN0_MAX_MB_NUM             (128U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN1 */
#define FEATURE_CAN1_MAX_MB_NUM             (128U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN2 */
#define FEATURE_CAN2_MAX_MB_NUM             (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN3 */
#define FEATURE_CAN3_MAX_MB_NUM             (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN4 */
#define FEATURE_CAN4_MAX_MB_NUM             (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN5 */
#define FEATURE_CAN5_MAX_MB_NUM             (32U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN6 */
#define FEATURE_CAN6_MAX_MB_NUM             (32U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN7 */
#define FEATURE_CAN7_MAX_MB_NUM             (32U)

/* @brief Has PE clock source select (bit field CAN_CTRL1[CLKSRC]). */
#define FEATURE_CAN_HAS_PE_CLKSRC_SELECT    (1)
/* @brief Has DMA enable (bit field MCR[DMA]). */
#define FEATURE_CAN_HAS_DMA_ENABLE          (1)

/* @brief Maximum number of Message Buffers supported for payload size 8 for any of the CAN instances */
#define FEATURE_CAN_MAX_MB_NUM              (128U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for any of the CAN instances */
#define FEATURE_CAN_MAX_MB_NUM_ARRAY        { FEATURE_CAN0_MAX_MB_NUM, \
                                              FEATURE_CAN1_MAX_MB_NUM, \
                                              FEATURE_CAN2_MAX_MB_NUM, \
                                              FEATURE_CAN3_MAX_MB_NUM, \
                                              FEATURE_CAN4_MAX_MB_NUM, \
                                              FEATURE_CAN5_MAX_MB_NUM, \
                                              FEATURE_CAN6_MAX_MB_NUM, \
                                              FEATURE_CAN7_MAX_MB_NUM}
/* @brief Has Pretending Networking mode */
#define FEATURE_CAN_HAS_PRETENDED_NETWORKING    (0)
/* @brief Has Stuff Bit Count Enable Bit */
#define FEATURE_CAN_HAS_STFCNTEN_ENABLE         (0)
/* @brief Has ISO CAN FD Enable Bit */
#define FEATURE_CAN_HAS_ISOCANFDEN_ENABLE       (1)
/* @brief Has Message Buffer Data Size Region 1 */
#define FEATURE_CAN_HAS_MBDSR1                  (1)
/* @brief Has Message Buffer Data Size Region 2 */
#define FEATURE_CAN_HAS_MBDSR2                  (1)
/* @brief Has Message Buffer Data Size Region 3 */
#define FEATURE_CAN_HAS_MBDSR3                  (1)
/* @brief DMA hardware requests for all FlexCAN instances */
#define FEATURE_CAN_DMA_REQUESTS              { DMA_REQ_FLEXCAN0, \
                                                DMA_REQ_FLEXCAN1, \
                                                DMA_REQ_FLEXCAN2, \
                                                DMA_REQ_FLEXCAN3, \
                                                DMA_REQ_FLEXCAN4, \
                                                DMA_REQ_FLEXCAN6, \
                                                DMA_REQ_FLEXCAN7, \
                                                DMA_REQ_FLEXCAN8 }

/* @brief Maximum number of Message Buffers IRQs */
#define FEATURE_CAN_MB_IRQS_MAX_COUNT       (8U)
/* @brief Message Buffers IRQs */
#define FEATURE_CAN_MB_IRQS                 { CAN_ORed_0_15_MB_IRQS, CAN_ORed_16_31_MB_IRQS,  CAN_ORed_32_47_MB_IRQS,  CAN_ORed_48_63_MB_IRQS, CAN_ORed_64_79_MB_IRQS, CAN_ORed_80_95_MB_IRQS, CAN_ORed_96_111_MB_IRQS, CAN_ORed_112_127_MB_IRQS }
/* @brief Has Wake Up Irq channels (CAN_Wake_Up_IRQS_CH_COUNT > 0u) */
#define FEATURE_CAN_HAS_WAKE_UP_IRQ         (1)
/* @brief Has Self Wake Up mode */
#define FEATURE_CAN_HAS_SELF_WAKE_UP        (1)
/* @brief Has Flexible Data Rate */
#define FEATURE_CAN_HAS_FD                  (1)
/* @brief Clock name for the PE oscillator clock source */
#define FEATURE_CAN_PE_OSC_CLK_NAME         IPC_FXOSC_CLK
/* @bried FlexCAN has Detection And Correction of Memory Errors */
#define FEATURE_CAN_HAS_MEM_ERR_DET			(1)


#define FEATURE_CAN_RAM_COUNT               (512)
#define FEATURE_CAN_RXIMR_COUNT             (128)
/* @brief Has enhanced rx fifo feature for CAN0 */
#define FEATURE_CAN0_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN1 */
#define FEATURE_CAN1_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN2 */
#define FEATURE_CAN2_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN3 */
#define FEATURE_CAN3_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN4 */
#define FEATURE_CAN4_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN5 */
#define FEATURE_CAN5_HAS_ENHANCE_RX_FIFO (0)
/* @brief Has enhance rx fifo feture for CAN6 */
#define FEATURE_CAN6_HAS_ENHANCE_RX_FIFO (0)
/* @brief Has enhance rx fifo feture for CAN7 */
#define FEATURE_CAN7_HAS_ENHANCE_RX_FIFO (0)

/* @brief Has enhance rx fifo feture for all CAN */
#define FEATURE_CAN_HAS_ENHANCE_RX_FIFO  (1)

/* @brief The list contains definitions of the rx-fifo feature support on all instances */
#define FEATURE_CAN_INSTANCES_HAS_ENHANCE_RX_FIFO        { FEATURE_CAN0_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN1_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN2_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN3_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN4_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN5_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN6_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN7_HAS_ENHANCE_RX_FIFO }

#define FEATURE_CAN_ENHANCE_RX_FIFO_COUNT (32U)

/* pTMR module features */

/* @brief Clock names for pTMR */
#define pTMR_CLOCK_NAMES {SLOW_BUS_CLK,SLOW_BUS_CLK,SLOW_BUS_CLK}

/* @brief pTMR instance list */
typedef enum {
    pTMR0_INST = 0,
    pTMR1_INST = 1,
    pTMR2_INST = 2,
} ptmr_instance_t;

/* @brief pTMR channels list for each instance */
typedef enum {
    pTMR_CH0 = 0U,
    pTMR_CH1 = 1U,
    pTMR_CH2 = 2U,
    pTMR_CH3 = 3U,
} ptmr_channel_t;

/* TMR module features */
#define TMR_CLOCK_NAMES {SLOW_BUS_CLK}

/* lpTMR module features */

/* @brief lpTMR pulse counter input options */
#define FEATURE_LPTMR_HAS_INPUT_ALT1_SELECTION  (1U)
/* @brief Clock names for lpTMR. */
#define lpTMR_CLOCK_NAMES {lpTMR0_CLK }
/* @brief alt1 input for lpTMR. */
#define FEATURE_lpTMR_HAS_INPUT_ALT1_SELECTION  (1U)
/* @brief lpTMR ip clock selection */
#define FEATURE_lpTMR_HAS_IP_CLOCK_SELECTION    (1U)
/* @brief lpTMR ip clock selection */
#define FEATURE_lpTMR_HAS_CLOCK_SELECTION       (1U)
/* @brief lpTMR clock source support ipc */
#define FEATURE_lpTMR_CLKSRC_SUPPORT_IPC        (1U)
/* @brief lpTMR clock source support firc */
#define FEATURE_lpTMR_CLKSRC_SUPPORT_FIRC       (0U)
/* @brief lpTMR clock source support firc */
#define FEATURE_lpTMR_CLKSRC_SUPPORT_SIRC       (0U)
/* @brief lpTMR clock source support firc */
#define FEATURE_lpTMR_CLKSRC_SUPPORT_SIRC_DIV4  (1U)
/* @brief lpTMR clock source support sxosc */
#define FEATURE_lpTMR_CLKSRC_SUPPORT_SXOSC      (1U)
/* @brief lpTMR clock source support lpo */
#define FEATURE_lpTMR_CLKSRC_SUPPORT_LPO        (0U)

/* TMR module features */

/* @brief TMR has interrupt enable configuration */
#define FEATURE_TMR_HAS_INTERRUPT_ENABLE_CONFIG        (1U)

/* RTC module features */

/*!
 * @brief RTC clock select
 * Implements : rtc_clk_source_t_Class
 */
typedef enum
{
    RTC_CLK_SRC_OSC_32KHZ = 0x00U,   /*!< RTC Prescaler increments from IPC function clock */
    RTC_CLK_SRC_RTC_PIN = 0x01U,   /*!< RTC Prescaler increments from RTC pin */
    RTC_CLK_SRC_SIRC_DIV4 = 0x02U,   /*!< RTC Prescaler increments from SIRC clock */
    RTC_CLK_SRC_BUS = 0x03U, /*!< RTC Prescaler increments using bus clock */
} rtc_clk_source_t;
/* @brief RTC has separate second irq */
#define FEATURE_RTC_HAS_SEPARATE_SECOND_IRQ     (1U)
/* @brief RTC has 32.768 OSC clock support */
#define FEATURE_RTC_HAS_OSC                     (1U)

/* LINFlexD module features */

/* @brief LINFlexD has dma support */
#define LINFlexD_CLOCK_NAMES                     {LINFlexD0_CLK, \
                                                  LINFlexD1_CLK, \
                                                  LINFlexD2_CLK, \
                                                  LINFlexD3_CLK, \
                                                  LINFlexD4_CLK, \
                                                  LINFlexD5_CLK, \
                                                  LINFlexD6_CLK, \
                                                  LINFlexD7_CLK, \
                                                  LINFlexD8_CLK, \
                                                  LINFlexD9_CLK  }
#define FEATURE_LINFlexD_ORED_INT_LINES          (1)
#define FEATURE_LINFlexD_HAS_DMA_ENABLED         (1)
/* @brief Address of the least significant byte in a 32-bit register (depends on endianess) */
#define FEATURE_LINFlexD_LSB_ADDR(reg)  ((uint32_t)(&(reg)) + 3U)
/* @brief Address of the least significant word in a 32-bit register (depends on endianess) */
#define FEATURE_LINFlexD_LSW_ADDR(reg)  ((uint32_t)(&(reg)) + 2U)
/* @brief LINFlexD "instance-implemented filters" mapping */
#define FEATURE_LINFlexD_INST_HAS_IFCR {true, true}
#define FEATURE_LINFlexD_TIMEOUT_LOOPS           (0xFFU)
/* @brief LINFlexD no receive edge detection support */
#define FEATURE_LINFlexD_HAS_RED_IRQ             (1)
/* @brief LINFlexD has start detection support */
#define FEATURE_LINFlexD_HAS_STA_IRQ             (1)
/* @brief LINFlexD no different memory map */
#define FEATURE_LINFlexD_HAS_DIFFERENT_MEM_MAP   (0)
/* @brief LINFlexD no RX TX error INT lines */
#define FEATURE_LINFlexD_RX_TX_ERR_INT_LINES     (0)

/* SENT module features */

/* @brief SENT interrupt sources */
#define FEATURE_SENT_HAS_COMBINED_IRQ           (1)
#define FEATURE_SENT_IRQS                       {{SENT0_IRQn, SENT0_IRQn, SENT0_IRQn, SENT0_IRQn}, \
                                                 {SENT1_IRQn, SENT1_IRQn, SENT1_IRQn, SENT1_IRQn}}
/* @brief SENT clock sources */
#define FEATURE_SENT_CLOCK_NAMES                {SENT0_CLK, SENT1_CLK}
#define FEATURE_SENT_FAST_DMA_REQS              {DMA_REQ_SENT0_FAST, DMA_REQ_SENT1_FAST,}
#define FEATURE_SENT_SLOW_DMA_REQS              {DMA_REQ_SENT0_SLOW, DMA_REQ_SENT1_SLOW}

/* HCU module features */
#define FEATURE_HCU_AES_ENGINE                           (1U)
#define FEATURE_HCU_SM4_ENGINE                           (1U)
#define FEATURE_HCU_SHA_ENGINE                           (1U)
#define FEATURE_HCU_ONE_LOOP_DATA_SIZE                   (4U)
#define FEATURE_HCU_ONE_LOOP_INPUT_WATERMARK             (3U)
#define FEATURE_HCU_HAS_FIXED_DMA                        (1U)
#define FEATURE_HCU_HAS_SHA_AUTHORIZE                    (1U)
#define FEATURE_HCU_AES_CTR_ENGINE                       (1U)
#define FEATURE_HCU_AES_CCM_ENGINE                       (1U)

#define FEATURE_HCU_FIFO_SIZE_IN_WORDS                   (8U)

/* @brief EFM AES area base address */
#define FEATURE_EFM_AES_KEY_BASE_ADDR                    (0x10000800U)
/* @brief EFM AES key size (5 for 256bit, 4 for 128bit)*/
#define FEATURE_EFM_AES_KEY_SIZE                         (0x20U)
#define FEATURE_EFM_AES_KEY_COUNT_MAX                    (128U)
#define FEATURE_HCU_GET_KEY_ADDR(key_id)                 (FEATURE_EFM_AES_KEY_BASE_ADDR + ((key_id) << 5U))

/* WKU module features */
#define FEATURE_WKU_MAX_CHANNEL_COUNT                    (64U)
/* WKU module support pin isolation feature */
#define FEATURE_WKU_SUPPORT_PIN_ISOLATION                (1U)
/* WKU module support filter clock selection */
#define FEATURE_WKU_SUPPORT_FILTER_CLOCK_SEL             (1u)




/* OSIF module features */

#define FEATURE_OSIF_USE_SYSTICK                         (1)
#define FEATURE_OSIF_FREERTOS_ISR_CONTEXT_METHOD         (1) /* Cortex M device */


/* PTU module features */

/* @brief PTU has instance back to back mode between PTU0 CH0 and PTU1 CH0 pre-triggers */
#define FEATURE_PTU_HAS_INSTANCE_BACKTOBACK     (0)

/* @brief PTU has inter-channel back to back mode between PTUx CH0 and PTUx CH1 pre-triggers */
#define FEATURE_PTU_HAS_INTERCHANNEL_BACKTOBACK (0)


/* QSPI module features */

/* @brief Message Buffers IRQs */
#define FEATURE_QSPI_IRQS                 { QSPI_IRQn }

/*! @brief First address of the serial flash device on the AHB bus. */
#define FEATURE_QSPI_AMBA_BASE  0x68000000U
/*! @brief Last address of the serial flash device on the AHB bus. */
#define FEATURE_QSPI_AMBA_END   0x6FFFFFFFU

 /*! @brief Size of AHB buffer. */
#define FEATURE_QSPI_AHB_BUF_SIZE   256U
 /*! @brief Size of TX buffer. */
#define FEATURE_QSPI_TX_BUF_SIZE    128U

 /*! @brief List of DMA Tx requests */
#define FEATURE_QSPI_DMA_TX_REQ   {DMA_REQ_QSPI_TX}
 /*! @brief List of DMA Rx requests */
#define FEATURE_QSPI_DMA_RX_REQ   {DMA_REQ_QSPI_RX}

/* FMU module features */

/*! @brief Implements Key to lock for all registers */
#define FEATURE_FMU_LOCK_KEY               (uint16_t)(0x46B0)
/*! @brief Implements Key to unlock for all registers */
#define FEATURE_FMU_UNLOCK_KEY             (uint16_t)(0xA765)
/*! @brief Implements Key to set operation code */
#define FEATURE_FMU_OP_KEY                 (uint16_t)(0xFA65)
/*! @brief Implements Key to select functional clock */
#define FEATURE_FMU_FUNC_CLKSEL_KEY        (uint8_t)(0x5A)
/*! @brief Implements Fault Status Key */
#define FEATURE_FMU_FAULT_ST_KEY           (uint32_t)(0x52CB6701U)
/*! @brief Implements Key to select clock source of FMU fail to react detect function */
#define FEATURE_FMU_DETECT_CLKSEL_KEY      (uint8_t)(0xA5)
/*! @brief Implements Key to assert fault injection */
#define FEATURE_FMU_FAULT_INJECTION_KEY    (uint8_t)(0x69)
/*! @brief FMU module has fault FSM status feature(Record the number of FAULT channel for FSM) */
#define FEATURE_FMU_HAS_FSM_ST             (0U)

/* EMU module features */

/*! @brief EMU Support 64 bit data error injection */
#define FEATURE_EMU_SUPPORT_64BIT_INJECTION               (1U)
/*! @brief EMU Global Enable KEY */
#define FEATURE_EMU_GLOBAL_ENABLE_KEY                     (1U)

/* ENET module features */
/*! @brief ENET module retry times timeout counter */
#define FEATURE_ENET_TIMEOUT_RETRY_TIMES   (0x10000U)

#if defined(USER_DEVICE_FEATURES_OVERRIDE)
#include "user_device_features_override.h"
#endif


#endif /* YTM32B1MD0_FEATURES_H */

/*******************************************************************************
 * EOF
 ******************************************************************************/
