/*
* Copyright 2020-2024 Yuntu Microelectronics co.,ltd
* All rights reserved.
*
* YUNTU Confidential. This software is owned or controlled by YUNTU and may
* only be used strictly in accordance with the applicable license terms. By expressly
* accepting such terms or by downloading, installing, activating and/or otherwise
* using the software, you are agreeing that you have read, and that you agree to
* comply with and are bound by, such license terms. If you do not agree to be
* bound by the applicable license terms, then you may not retain, install,
* activate or otherwise use the software. The production use license in
* Section 2.3 is expressly granted for this software.
*/
/*!
* @file motor_parameters.h
*/
#ifndef MOTOR_PARAMETERS_H
#define MOTOR_PARAMETERS_H

/*******************************************************************************
 * @brief User defined parameters
 * User should update following parameters according to the motor and hardware
 ******************************************************************************/
/*!
 * @brief PWM souce clock frequency
 * @type float
 * @unit Hz
 */
#define PWM_CLK_FREQ              (120000000.0f)
/*!
 * @brief PWM output frequency
 * @type float
 * @unit Hz
 */
#define PWM_FREQ                  (16000.0f)
/*!
 * @brief PWM current sample window width
 * User should consider the deadtime, current noise and ADC sample time.
 * For one resistor current sample, SVPWM calcualtion function will inert a minimum sample window when current sample
 * window is less than given value, when there is enough sample window, software will set sample point at middle of
 * the window.
 * @note Set pwm window as small as possible according to the hardware and ADC sample time. Large window will cause
 * motor voltage vector distortion.
 * @type float
 * @unit Second
 */
#define PWM_WINDOW_SEC              (0)
/*!
 * @brief PWM deadtime
 * User should set the deadtime according to the Mosfet and hardware. Large deadtime will cause low efficiency and voltage
 * vector distortion.
 * @type float
 * @unit Second
 */
#define PWM_DEADTIME_SEC          (0.000001f)

/*******************************************************************************
 * @brief Auto calculated parameters
 ******************************************************************************/
#define PWM_PERIOD_SEC              (1.0 / PWM_FREQ)                           /* Float - Calculated PWM period  */
#define PWM_CLK_SEC                 (1.0 / PWM_CLK_FREQ)                       /* Float - Calculated PWM clock period */
#define PWM_PERIOD_TICK             (uint32_t)(PWM_CLK_FREQ / PWM_FREQ)        /* uint32_t - Calculated PWM peiod cycles */
#define PWM_HALF_PERIOD_TICK        (PWM_PERIOD_TICK >> 1)                     /* uint32_t - PWM half peiod cycles */
#define PWM_WINDOW                  (int32_t)(PWM_WINDOW_SEC/PWM_CLK_SEC)    /* uint32_t - PWM half peiod cycles */
#define PWM_DEAD_TIME_WIDTH_TICK    (uint32_t)(PWM_DEADTIME_SEC/PWM_CLK_SEC) /* uint32_t - PWM half peiod cycles */

/*******************************************************************************
 * @brief Hardware and motor parameters
 * User should update following parameters according to the motor and hardware
 ******************************************************************************/
/*!
 * @brief Peak bus voltage
 * YTM32Z1MD0x MCU support measure VSM volatage with internal ADC, fix convertion ratio is 12:1,
 * the maximum bus voltage is 4 * 12 = 48V.
 * @type float
 * @unit V
 */
#define PEAK_VOLATGE                (105)
/*!
 * @brief Peak bus current ADC can measure
 * YTM32Z1MD0x MCU support differential current sensor with configurable gain, user should calculate the peak current
 * according to the sensor gain(G) and sample resistor value(R). The differential current sensor has fix 0.5 * Vref
 * offset so user can get peak current by following formula:
 * Peak current = (0.5 * Vref / G) / R
 * @note The peak current should be less than the sensor maximum current.
 * @type float
 * @unit A
 */
#define PEAK_CURRENT                (10)


/*!
 * @brief Mosfet on resistance
 * @type float
 * @unit ohm
 */
#define MOS_RS                      (0.05)


/*!
 * @brief Motor pole pair number
 * Software use this parameter to calculate the motor mechanical angle.
 * @type float
 * @unit 1
 */
#define MOTOR_NP                    (4)

/*!
 * @brief Motor phase resistance
 * @note When using multimeter mease the motor wire resistance, the result should be divided by 2 to get phase resistance.
 * Suggestions: Use milliohmmeter to measure low resistance, measure U-V, V-W, U-W, and get the average value.
 * phase resistance = ((R1 + R2 + R3) / 3 ) / 2
 * @type float
 * @unit ohm
 */
#define MOTOR_RS                    (0.251)
/*!
 * @brief Motor phase inductance
 * Use LCR meter to measure the motor phase inductance. Measure U-V, V-W, U-W, and get the average value.
 * phase inductance = ((L1 + L2 + L3) / 3 ) / 2
 * @type float
 * @unit H
 */
#define MOTOR_LS                    (0.00045158)

/*!
 * @brief Motor back EMF constant
 * Get the motor back EMF constant from motor datasheet or motor manufacturer.
 * Or force motor to rotate at a constant speed, measure the motor terminal voltage and speed, then calculate the KFI.
 * Suppose we get back EMF waveform as sine wave, the peak to peak voltage is Vpp, the frequency is F, then the KFI can
 * be calculated as:
 * KFI = Vpp / (4 * pi * F * sqrt(3))
 * Vpp unit is V, F unit is Hz.
 * @type float
 */
 #define MOTOR_KFI_VPP               (28.6f) 
 #define MOTOR_KFI_FREQ              (385.0f) 
 #define MOTOR_KFI (MOTOR_KFI_VPP / (4 * 3.1415926f * MOTOR_KFI_FREQ * 1.732f))

/*!
 * @brief observer filter coefficient
 * This parameter will be adapted to Q15 format in software.
 * @type float
 */
#define KFILTER_ESDQ                (0.244)

/*!
 * @brief d-axis current PI controller parameters
 * @note The PI controller parameters should be tuned according to the motor and hardware.
 * PI controller support output saturation, the output will be limited to the range of OUTMAX and OUTMIN.
 * @type int32_t(Q15 format)
 */
#define D_CURRENT_KP                (int32_t)(4285.3f * 1.44f * 1.8f)    /* int32_t [+0, +32767] */
#define D_CURRENT_KI                (int32_t)(331.8f  * 1.44f * 0.8f)       /* int32_t [+0, +32767] */
#define D_CURRENT_OUTMAX            (int32_t)(31500)    /* int32_t [-32768, +32767] */
#define D_CURRENT_OUTMIN            (int32_t)(-31500)   /* int32_t [-32768, +32767] */

/*!
 * @brief q-axis current PI controller parameters
 * @note The PI controller parameters should be tuned according to the motor and hardware.
 * PI controller support output saturation, the output will be limited to the range of OUTMAX and OUTMIN.
 * @type int32_t(Q15 format)
 */
#define Q_CURRENT_KP                (int32_t)(4285.3f * 1.44f * 1.8f)    /* int32_t [+0, +32767] */
#define Q_CURRENT_KI                (int32_t)(331.8f  * 1.44f * 0.8f)       /* int32_t [+0, +32767] */
#define Q_CURRENT_OUTMAX            (int32_t)(31500)    /* int32_t [-32768, +32767] */
#define Q_CURRENT_OUTMIN            (int32_t)(-31500)   /* int32_t [-32768, +32767] */

/*!
 * @brief speed PI controller parameters
 * @note The PI controller parameters should be tuned according to the motor and hardware.
 * PI controller support output saturation, the output will be limited to the range of OUTMAX and OUTMIN.
 * @type int32_t(Q15 format)
 */
#define SPEED_CTRL_KP                (int32_t)(1000000)    /* int32_t [+0, +32767] */
#define SPEED_CTRL_KI                (int32_t)(12)       /* int32_t [+0, +32767] */
#define SPEED_CTRL_OUTMAX            (int32_t)(30000)    /* int32_t [-32768, +32767] */
#define SPEED_CTRL_OUTMIN            (int32_t)(-30000)   /* int32_t [-32768, +32767] */


/*******************************************************************************
 * @brief Motor control parameters
 ******************************************************************************/

/*!
 * @brief Motor boot speed in rps
 * Final boot speed in rps will be calculated as MOTOR_BOOT_SPEED_RPM / 60
 * @type uint32_t
 */
#define MOTOR_BOOT_SPEED            (1000/60)
/*!
 * @brief Motor boot current
 * @type int32_t
 * @unit A
 */
#define MOTOR_BOOT_CURRENT          (1.5)
/*!
 * @brief Motor open loop voltage
 * @type int32_t [-32768, +32767]
 * @unit 
 */
#define MOTOR_OPENLOOP_VOLTAGE      (8000)
/*!
 * @brief Speed loop count for lock motor to zero position
 * @type uint32_t
 */
#define MOTOR_BOOT_LOCK_TIME        (1)
/*!
 * @brief Speed loop count for motor boot acceleration time
 * @type uint32_t
 */
#define MOTOR_BOOT_ACC_TIME         (1000)

#define ADC_REF_VOLTAGE             (5000)
#define CONV_ADC_VOLTAGE_TO_CODE(v) ((v) * 4096 / ADC_REF_VOLTAGE)
#define VDDD_MAX_VOLTAGE            (CONV_ADC_VOLTAGE_TO_CODE(2000))
#define VDDD_MIN_VOLTAGE            (CONV_ADC_VOLTAGE_TO_CODE(1000))
/*!
 * @brief
 */
#define MOTOR_BOOT_MIN_VOLTAGE      (780)
#define MOTOR_BOOT_MAX_VOLTAGE      (1460)
#define MOTOR_RUN_MIN_VOLTAGE       (690)
#define MOTOR_RUN_MAX_VOLTAGE       (1550)
#define MOTOR_RUN_MIN_RPM           (100)
#define MOTOR_RUN_MAX_RPM           (5000)
#define MOTOR_RUN_CLOSE_LOOP_MODE   (MOTOR_SPEED_LOOP)
#define MOTOR_BOOT_SPEED_RPM        (1500)


#endif /* MOTOR_PARAMETERS_H */

/*******************************************************************************
* EOF
******************************************************************************/
