/*
 * Copyright 2020-2023 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
/*!
 * @file zdu_driver.h
 */

#ifndef ZDU_DRIVER_H
#define ZDU_DRIVER_H

#include <stdint.h>
#include <stddef.h>
#include <stdbool.h>
#include "status.h"
#include "device_registers.h"

/*!
 * @addtogroup zdu_driver
 * @brief zdu driver
 * @{
 */

/*******************************************************************************
 * Macro Definitions
 ******************************************************************************/

/*******************************************************************************
 * Enumeration & Struct
 ******************************************************************************/
/*!
 * @brief Output trigger source selection
 *
 * Implements : zdu_out_trig_src_t_Class
 */
typedef enum
{
    ZDU_OUT_TRIG              = 0U,   /*!< Output as trigger */
    ZDU_OUT_TRIG_POS_EDGE     = 1U,   /*!< Output positive edge as trigger */
    ZDU_OUT_TRIG_NEG_EDGE     = 2U,   /*!< Output negative edge as trigger */
    ZDU_OUT_TRIG_POS_NEG_EDGE = 3U,   /*!< Output positive or negative edge as trigger */
} zdu_out_trig_src_t;

/*!
 * @brief Edge selection
 *
 * Implements : zdu_edge_sel_t_Class
 */
typedef enum
{
    ZDU_NO_EDGE      = 0U,  /*!< No edge detection */
    ZDU_RISING_EDGE  = 1U,  /*!< Rising edge detection */
    ZDU_FALLING_EDGE = 2U,  /*!< Falling edge detection */
    ZDU_DUAL_EDGE    = 3U,  /*!< Dual edge detection, only valid in interrupt, not for DMA */
} zdu_int_edge_sel_t;

/*!
 * @brief ZDU mode
 *
 * Implements : zdu_mode_t_Class
 */
typedef enum
{
    ZDU_COMMON_MODE = 0U, /*!< Common mode */
    ZDU_WINDOW_MODE = 1U, /*!< Window mode */
} zdu_mode_t;

/*!
 * @brief Select phase for the positive port of the comparator
 *
 * Implements : zdu_phase_sel_t_Class
 */
typedef enum
{
    ZDU_PHASE_DISABLED = 0U,  /*!< Disabled */
    ZDU_PHASE_U        = 1U,  /*!< Phase U */
    ZDU_PHASE_V        = 2U,  /*!< Phase V */
    ZDU_PHASE_W        = 3U,  /*!< Phase W */
} zdu_phase_sel_t;

/*!
 * @brief ZDU module configuration struct
 *
 * Implements : zdu_config_t_Class
 */
typedef struct
{
    zdu_mode_t mode;                 /*!< ZDU mode */
    zdu_phase_sel_t phaseSel;        /*!< Phase selection */
    bool outTrigEn;                  /*!< Output trigger source selection */
    zdu_out_trig_src_t outTrigSrc;   /*!< Output trigger source selection */
    bool intEn;                      /*!< Module interrupt enable/disable */
    zdu_int_edge_sel_t edgeIntSel;   /*!< Edge selection for interrupt*/
    uint8_t fltPer;                  /*!< Filter period(0~0xF), set 0 to disable filter */
    uint8_t fltCnt;                  /*!< Filter count(0~0xFF), set 0 to disable filter */
    bool PosPortToPinEn;             /*!< Positive port to pin enable/disable */
    bool NegPortToPinEn;             /*!< Negative port to pin enable/disable */
} zdu_user_config_t;

/*******************************************************************************
 * API Declaration
 ******************************************************************************/
/*!
 * @brief Initialize ZDU
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return status_t
 */
status_t ZDU_DRV_Init(uint32_t instance, const zdu_user_config_t *userCfg);

/*!
 * @brief Deinitialize ZDU
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return status_t
 */
status_t ZDU_DRV_Deinit(uint32_t instance);

/*!
 * @brief Enable ZDU
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return void
 */
void ZDU_DRV_Enable(const uint32_t instance);

/*!
 * @brief Disable ZDU
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return void
 */
void ZDU_DRV_Disable(const uint32_t instance);

/*!
 * @brief Get negative edge state
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return bool
 */
bool ZDU_DRV_GetNegEdgeState(const uint32_t instance);

/*!
 * @brief Get positive edge state
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return bool
 */
bool ZDU_DRV_GetPosEdgeState(const uint32_t instance);

/*!
 * @brief Clear negative edge flag
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return void
 */
void ZDU_DRV_ClearNegEdgeFlag(const uint32_t instance);

/*!
 * @brief Clear positive edge flag
 *
 * @param[in] instance  The ZDU peripheral instance
 * @return void
 */
void ZDU_DRV_ClearPosEdgeFlag(const uint32_t instance);

/*! @}*/

#endif /* ZDU_DRIVER_H */
