/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef PWD_DRIVER_H
#define PWD_DRIVER_H

/*! @file pwd_driver.h */

#include <stddef.h>
#include "device_registers.h"
#include "status.h"

/*!
 * @defgroup pwd_driver PWD Driver
 * @ingroup pwd
 * @brief Power-mode and Reset Monitor Unit Peripheral Driver
 * @details This section describes the programming interface of the PWD driver.
 * @{
 */
/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Definitions
 ******************************************************************************/


/*!
 * @brief PWD capture edge selection
 * Implements : pwd_capture_edge_t_Class
 */
typedef enum
{
    PWD_CAPTURE_EDGE_FALLING = 0U, /*!< Capture on falling edge */
    PWD_CAPTURE_EDGE_RISING = 1U, /*!< Capture on rising edge */
} pwd_capture_edge_t;

/*!
 * @brief PWD capture polarity selection
 * Implements : pwd_capture_polarity_t_Class
 */
typedef enum
{
    PWD_CAPTURE_POLARITY_LOW = 0U, /*!< Capture on low level */
    PWD_CAPTURE_POLARITY_HIGH = 1U, /*!< Capture on high level */
} pwd_capture_polarity_t;

/*!
 * @brief PWD monitor channel configuration
 * Implements : pwd_channel_config_t_Class
 */
typedef struct {
    uint8_t channelId;                      /*!< pwd channel index */
    bool channelEnable;                     /*!< Channel enable */
    bool intEnable;                         /*!< Channel interrupt enable */
    bool intErrorEnable;                    /*!< Channel error interrupt enable */
    bool dmaEnable;                         /*!< DMA enable */
    uint8_t dmaWatermark;                   /*!< DMA watermark */
    bool overRunOverWrite;                  /*!< Overrun overwrite */
    pwd_capture_edge_t captureEdge;         /*!< Capture edge */
    pwd_capture_polarity_t capturePolarity; /*!< Capture polarity */
    bool speedWdgEnable;                    /*!< Speed watchdog enable */
    uint16_t speedWdgTimeoutValue;          /*!< Speed watchdog timeout Value */
    uint16_t speedWdgThresholdValue;        /*!< Speed watchdog threshold Value */
} pwd_channel_config_t;

/*!
 * @brief PWD user configuration structure
 * Implements : pwd_user_config_t_Class
 */
typedef struct {
    uint16_t decoderDiv;        /*!< Module prescaler prescaler */
    uint16_t pulseCapDiv;       /*!< Capture scaler */
    uint8_t filterCount;        /*!< Filter count */
    uint8_t filterPrescaler;    /*!< Filter prescaler */
    bool moduleEnable;          /*!< Module enable */
    bool oneWireMode;           /*!< One wire mode */
    uint8_t channelCount;       /*!< Monitor channel count */
    pwd_channel_config_t *channelConfigs; /*!< Monitor channel configuration */
} pwd_user_config_t;

/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name PWD DRIVER API
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initializes the PWD module
 *
 * This function initializes PWD driver based on user configuration input.
 * The user must make sure that the clock is enabled
 *
 * @param[in] instance The PWD instance number
 * @param[in] userConfigPtr Pointer to structure of initialization
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t PWD_DRV_Init(uint32_t instance, const pwd_user_config_t *userConfigPtr);

/*!
 * @brief Initializes the PWD channel
 *
 * This function initializes pwd channel based on user configuration input.
 *
 * @param[in] instance The PWD instance number
 * @param[in] channelConfig Pointer to structure of PWD channel configurations
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t PWD_DRV_ChannelInit(uint32_t instance, const pwd_channel_config_t *channelConfig);


/*!
 * @brief PWD get capture value
 * This function gets the capture value of the specified channel
 * @param[in] instance The PWD instance number
 * @param[in] channel The channel number
 * @return The capture value
 */

uint32_t PWD_DRV_GetCaptureValue(uint32_t instance, uint8_t channel);

/*!
 * @brief PWD get capture data valid status
 * This function gets the capture data valid status of the specified channel
 * when captured data matches watermark setting in FIFO, the data is valid and
 * application need to read the data out of FIFO
 * @param[in] instance The PWD instance number
 * @param[in] channel The channel number
 * @return The capture data valid status
 */

bool PWD_DRV_GetCaptureDataValidStatus(uint32_t instance, uint8_t channel);

/*!
 * @brief PWD get channel error status
 * This function gets the channel error status of the specified channel, if multiple errors
 * occur, the error status will be ORed.
 * @param[in] instance The PWD instance number
 * @param[in] channel The channel number
 * @return The channel error status
 */

uint32_t PWD_DRV_GetChannelErrorStatus(uint32_t instance, uint8_t channel);

/*!
 * @brief  PWD clear channel error status
 * This function clears the channel error status of the specified channel
 * @param[in] instance The PWD instance number
 * @param[in] channel The channel number
 * @param[in] errorStatus The error status to be cleared
 */

void PWD_DRV_ClearChannelErrorStatus(uint32_t instance, uint8_t channel, uint32_t errorStatus);

/*!
 * @brief Sets the default configuration
 *
 * This function sets the default configuration
 *
 * @param[in] instance The PWD instance number
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t PWD_DRV_Deinit(uint32_t instance);

/*! @} */

#if defined(__cplusplus)
}
#endif

/*! @} */

#endif /* PWD_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
