/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef PRMU_DRIVER_H
#define PRMU_DRIVER_H

/*! @file prmu_driver.h */

#include <stddef.h>
#include "device_registers.h"
#include "status.h"

/*!
 * @defgroup prmu_driver PRMU Driver
 * @ingroup prmu
 * @brief Power-mode and Reset Monitor Unit Peripheral Driver
 * @details This section describes the programming interface of the PRMU driver.
 * @{
 */
/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*!
 * @brief PRMU monitor threshold
 * Implements : prmu_monitor_threshold_t_Class
 */
typedef enum
{
    PRMU_MONITOR_THRESHOLD_6P25MS = 0U, /*!< Monitor threshold 6.25ms */
    PRMU_MONITOR_THRESHOLD_12P5MS = 1U, /*!< Monitor threshold 12.5ms */
    PRMU_MONITOR_THRESHOLD_25MS = 2U, /*!< Monitor threshold 25ms */
    PRMU_MONITOR_THRESHOLD_50MS = 3U, /*!< Monitor threshold 50ms */
} prmu_monitor_threshold_t;

/*!
 * @brief PRMU user configuration structure
 * Implements : prmu_user_config_t_Class
 */
typedef struct {
    bool lpResetMonitorEnable;         /*!< Low power reset monitor enable */
    bool normalResetMonitorEnable;     /*!< Normal reset monitor enable */
    bool specialResetMonitorEnable;     /*!< Special reset monitor enable */
    prmu_monitor_threshold_t monitorThreshold; /*!< Monitor threshold */
} prmu_user_config_t;

/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name PRMU DRIVER API
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initializes the PRMU module
 *
 * This function initializes PRMU driver based on user configuration input.
 * The user must make sure that the clock is enabled
 *
 * @param[in] instance The PRMU instance number
 * @param[in] userConfigPtr Pointer to structure of initialization
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t PRMU_DRV_Init(uint32_t instance, const prmu_user_config_t *userConfigPtr);

/*!
 * @brief Sets the default configuration
 *
 * This function sets the default configuration
 *
 * @param[out] userConfigPtr Pointer to structure of initialization
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t PRMU_DRV_GetDefaultConfig(prmu_user_config_t *userConfigPtr);

/*!
 * @brief Sets the default configuration
 *
 * This function sets the default configuration
 *
 * @param[in] instance The PRMU instance number
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t PRMU_DRV_Deinit(uint32_t instance);

/*! @} */

#if defined(__cplusplus)
}
#endif

/*! @} */

#endif /* PRMU_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
