/*
 * Copyright 2020-2023 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
/*!
 * @file predrv_driver.h
 */

#ifndef PREDRV_DRIVER_H
#define PREDRV_DRIVER_H

#include <stdint.h>
#include <stdbool.h>
#include "status.h"

/*!
 * @addtogroup predrv_driver
 * @brief predrv driver
 * @{
 */

/*******************************************************************************
 * Macro Definitions
 ******************************************************************************/

/*******************************************************************************
 * Enumeration & Struct
 ******************************************************************************/
/*!
 * @brief PREDRV drive priority when fault event occurs
 *
 * Implements : predrv_fault_drive_priority_t_Class
 */
typedef enum
{
    PREDRV_DRIVE_GND_VBUS_TRISTATE, /*!< To ground -> bus voltage -> tristate */
    PREDRV_DRIVE_GND_TRISTATE_VBUS, /*!< To ground -> tristate -> bus voltage */
    PREDRV_DRIVE_VBUS_GND_TRISTATE, /*!< To vbus -> ground -> tristate */
    PREDRV_DRIVE_VBUS_TRISTATE_GND, /*!< To vbus -> tristate -> ground */
    PREDRV_DRIVE_TRISTATE_VBUS_GND, /*!< To vbus -> tristate -> ground */
    PREDRV_DRIVE_TRISTATE_GND_VBUS, /*!< To vbus -> ground -> tristate */
} predrv_fault_drive_priority_t;

/*!
 * @brief PREDRV phase x drive state
 *
 * Implements : predrv_phase_drive_state_t_Class
 */
typedef enum
{
    PREDRV_PHASE_DRIVE_TO_TRISTATE, /*!< Drive to tristate */
    PREDRV_PHASE_OUTPUT_PWM,        /*!< Output PWM as source */
    PREDRV_PHASE_DRIVE_TO_GND,      /*!< Drive to ground */
    PREDRV_PHASE_DRIVE_TO_VBUS,     /*!< Drive to bus voltage */
} predrv_phase_drive_state_t;

/*!
 * @brief PREDRV fault drive state
 *
 * Implements : predrv_fault_drive_state_t_Class
 */
typedef enum
{
    PREDRV_FAULT_DRIVE_TO_GND,               /*!< Drive to ground */
    PREDRV_FAULT_DRIVE_TO_VBUS,              /*!< Drive to bus voltage */
    PREDRV_FAULT_DRIVE_TO_TRISTATE,          /*!< Drive to tristate */
    PREDRV_FAULT_PREFAULT_DRIVE_TO_TRISTATE, /*!< Pre-drive to tristate */
} predrv_fault_drive_state_t;

/*!
 * @brief PREDRV VDS switch with or without debounce
 *
 * Implements : predrv_vds_switch_t_Class
 */
typedef enum
{
    PREDRV_SWITCH_WITH_DEBOUNCE,    /*!< Switch with debounce */
    PREDRV_SWITCH_WITHOUT_DEBOUNCE, /*!< Switch without debounce */
} predrv_vds_switch_t;

/*!
 * @brief Fault Automatic Recovery Mode
 *
 * Implements : predrv_fault_auto_recovery_mode_t_Class
 */
typedef enum
{
    PREDRV_MANUAL_CLEAR_FAULT_FLAG_THEN_AUTO_RECOVER, /*!< After clearing STS[FxF] flag manually, the fault recovers automatically */
    PREDRV_AUTO_RECOVER_WITHOUT_CLEAR_FAULT_FLAG, /*!< The fault recovers immediately, not care STS[FxF] flag */
    PREDRV_AUTO_RECOVER_WITH_CLEAR_FAULT_FLAG, /*!< The fault recovers immediately and STS[FxF] flag clear automatically */
} predrv_fault_auto_recovery_mode_t;

/*!
 * @brief PREDRV VDS monitor configuration
 *
 * Implements : predrv_vds_config_t_Class
 */
typedef struct
{
    predrv_vds_switch_t vdsSwitch; /*!< VDS switch selection */
    uint8_t vdsStableTime;         /*!< VDS stable time when top bridge is on */
    bool phaseUMonEn;              /*!< Phase U monitoring enable or not */
    bool phaseVMonEn;              /*!< Phase V monitoring enable or not */
    bool phaseWMonEn;              /*!< Phase W monitoring enable or not */
#if FEATURE_PREDRV_PHASE_COUNT > 3
    bool phaseTMonEn;              /*!< Phase T monitoring enable or not */
#endif /* FEATURE_PREDRV_PHASE_COUNT > 3 */
} predrv_vds_config_t;

/*!
 * @brief PREDRV protection configuration
 *
 * Implements : predrv_protect_config_t_Class
 */
typedef struct
{
    bool overCurrentProtEn; /*!< Over current protection enable */
    bool overTempProtEn;    /*!< Over temperature protection enable */
    bool underVolProtEn;    /*!< VS under voltage protection enable */
    bool overVolProtEn;     /*!< VSM over voltage protection enable */
} predrv_protect_config_t;

/*!
 * @brief PREDRV phase x drive configuration structure
 *
 * Implements : predrv_phase_drive_config_t_Class
 */
typedef struct
{
    predrv_phase_drive_state_t phaseUDrvState; /*!< Phase U drive state */
    predrv_phase_drive_state_t phaseVDrvState; /*!< Phase V drive state */
    predrv_phase_drive_state_t phaseWDrvState; /*!< Phase W drive state */
#if FEATURE_PREDRV_PHASE_COUNT > 3
    predrv_phase_drive_state_t phaseTDrvState; /*!< Phase T drive state */
#endif /* FEATURE_PREDRV_PHASE_COUNT > 3 */
} predrv_phase_drive_config_t;

/*!
 * @brief PREDRV fault drive configuration
 *
 * Implements : predrv_fault_drive_config_t_Class
 */
typedef struct
{
    predrv_fault_drive_state_t overCurDrvState;     /*!< Over current drive state */
    predrv_fault_drive_state_t vdsVolAbnomDrvState; /*!< VDS voltage abnormality drive state */
    predrv_fault_drive_state_t overTempDrvState;    /*!< Over temperature drive state */
    predrv_fault_drive_state_t underVolDrvState;    /*!< Under voltage drive state */
    predrv_fault_drive_state_t overVolDrvState;     /*!< Over voltage drive state */
} predrv_fault_drive_config_t;

/*!
 * @brief PREDRV fault channel configuration
 *
 * Implements : predrv_fault_ch_config_t_Class
 */
typedef struct
{
    uint8_t faultChannelId;                   /*!< Fault channel ID */
    bool faultEn;                             /*!< Fault enable */
    bool faultPol;                            /*!< Fault input polarity */
    predrv_fault_drive_state_t faultDrvState; /*!< Fault drive state */
} predrv_fault_ch_config_t;

/*!
 * @brief PREDRV fault configuration structure
 *
 * Implements : predrv_fault_config_t_Class
 */
typedef struct
{
    uint8_t faultChannelNum;                     /*!< Fault channel number which need to initialize */
    predrv_fault_ch_config_t *faultChConfig;     /*!< Fault channel configuration */
    predrv_fault_drive_config_t *faultDrvConfig; /*!< PREDRV fault drive configuration */
    predrv_fault_drive_priority_t faultDrvPrior; /*!< Fault drive priority */
    uint8_t faultFilterSampleCount;              /*!< The fault filter sample count */
    uint8_t faultFilterSamplePeriod;             /*!< The fault filter sample period */
    bool faultInputStretch;                      /*!< Fault input stretch enable */
    bool faultCombineLogic; /*!< When there is no clock, fault event is generated by combinational logic */
    bool pwmRecovery;       /*!< PWM recovery or not after fault recovery */
    predrv_fault_auto_recovery_mode_t faultAutoRecoveryMode; /*!< Fault automatical recovery mode */
} predrv_fault_config_t;

/*!
 * @brief PREDRV dead-time configuration
 *
 * Implements : predrv_dt_config_t_Class
 */
typedef struct
{
    uint8_t topBrgDt; /*!< External top bridge dead-time */
    uint8_t botBrgDt; /*!< External bottom bridge dead-time */
} predrv_dt_config_t;

/*!
 * @brief PREDRV module configuration struct
 *
 * Implements : predrv_config_t_Class
 */
typedef struct
{
    predrv_phase_drive_config_t *phaseDrvConfig; /*!< PREDRV phase drive configuration */
    predrv_fault_config_t *faultConfig;          /*!< PREDRV fault configuration */
    predrv_vds_config_t *vdsConfig;              /*!< PREDRV VDS monitor enable configuration */
    predrv_protect_config_t *protConfig;         /*!< PREDRV protect configuration */
    predrv_dt_config_t *dtConfig;                /*!< PREDRV dead-time configuration */
} predrv_config_t;

/*******************************************************************************
 * API Declaration
 ******************************************************************************/
/*!
 * @brief Enable PREDRV module
 *
 * @param[in] instance  The PREDRV peripheral instance
 */
void PREDRV_DRV_Enable(uint32_t instance);

/*!
 * @brief Disable PREDRV module
 *
 * @param[in] instance  The PREDRV peripheral instance
 */
void PREDRV_DRV_Disable(uint32_t instance);

/*!
 * @brief Initialize PREDRV module
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @param[in] config  PREDRV module configuration pointer
 * @return status_t
 */
status_t PREDRV_DRV_Init(uint32_t instance, const predrv_config_t *config);

/*!
 * @brief Deinitialize PREDRV module
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @return status_t
 */
status_t PREDRV_DRV_Deinit(uint32_t instance);

/*!
 * @brief Configure phase x drive
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @param[in] *config  Phase x drive configuration pointer
 * @return status_t
 */
status_t PREDRV_DRV_ConfigPhaseDrive(uint32_t instance, const predrv_phase_drive_config_t *config);

/*!
 * @brief Configure Drain-Source Voltage and its monitor enable or not
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @param[in] config  Drain-Source Voltage configuration pointer
 * @return status_t
 */
status_t PREDRV_DRV_ConfigVds(uint32_t instance, const predrv_vds_config_t *config);

/*!
 * @brief Configure protection
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @param[in] config  Protection configuration pointer
 * @return status_t
 */
status_t PREDRV_DRV_ConfigProtect(uint32_t instance, const predrv_protect_config_t *config);

/*!
 * @brief Configure fault
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @param[in] config  Fault configuration pointer
 * @return status_t
 */
status_t PREDRV_DRV_ConfigFault(uint32_t instance, const predrv_fault_config_t *config);

/*!
 * @brief Configure dead-time
 *
 * @param[in] instance  The PREDRV peripheral instance
 * @param[in] config  Dead-time configuration pointer
 * @return status_t
 */
status_t PREDRV_DRV_ConfigDeadtime(uint32_t instance, const predrv_dt_config_t *config);

/*! @}*/

#endif /* PREDRV_DRIVER_H */
