/*
 * Copyright 2020-2024 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @file omu_driver.h
 */

#ifndef OMU_DRIVER_H
#define OMU_DRIVER_H

#include <stddef.h>
#include <stdbool.h>
#include "device_registers.h"
#include "status.h"


/*!
 * @ingroup omu_drv
 * @{
 */

/*******************************************************************************
 * Enumerations.
 ******************************************************************************/
typedef enum
{
    OMU_SOFT_LOCK = 1,
    OMU_HARD_LOCK = 2
} omu_lock_type_t;

typedef enum
{
#if (FEATURE_OMU_OVERLAY_REGION_NUM == 8)
    OMU_REGION0 = 0,
    OMU_REGION1 = 1,
    OMU_REGION2 = 2,
    OMU_REGION3 = 3,
    OMU_REGION4 = 4,
    OMU_REGION5 = 5,
    OMU_REGION6 = 6,
    OMU_REGION7 = 7,
#else
    OMU_REGION8 = 8,
    OMU_REGION9 = 9,
    OMU_REGION10 = 10,
    OMU_REGION11 = 11,
    OMU_REGION12 = 12,
    OMU_REGION13 = 13,
    OMU_REGION14 = 14,
    OMU_REGION15 = 15
#endif
} omu_overlay_region_t;

typedef enum
{
    OMU_OVERLAY_SIZE_1KB = 0,
    OMU_OVERLAY_SIZE_2KB = 1,
    OMU_OVERLAY_SIZE_4KB = 2,
    OMU_OVERLAY_SIZE_8KB = 3,
    OMU_OVERLAY_SIZE_16KB = 4,
    OMU_OVERLAY_SIZE_32KB = 5,
    OMU_OVERLAY_SIZE_64KB = 6,
    OMU_OVERLAY_SIZE_128KB = 7,
    OMU_OVERLAY_SIZE_256KB = 8,
    OMU_OVERLAY_SIZE_512KB = 9,
    OMU_OVERLAY_SIZE_1MB = 10,
    OMU_OVERLAY_SIZE_2MB = 11,
    OMU_OVERLAY_SIZE_4MB = 12,
    OMU_OVERLAY_SIZE_8MB = 13
} omu_overlay_size_t;

typedef struct
{
    uint32_t regionId;
    uint32_t srcAddr;
    uint32_t dstAddr;
    omu_overlay_size_t size;
} omu_overlay_region_config_t;

typedef struct
{
    uint32_t regionNum;
    omu_overlay_region_config_t *overlayRegionConfig;
    bool errorIntEn;
} omu_overlay_config_t;

/*******************************************************************************
* Definitions
******************************************************************************/




/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name OMU Driver
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initialize OMU overlay
 * 
 * @param[in] instance  OMU hw instance
 * @param[in] config    OMU overlay configuration
 */
void OMU_DRV_Init(uint32_t instance, omu_overlay_config_t *config);

/*!
 * @brief Deinitialize OMU overlay
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_Deinit(uint32_t instance);

/*!
 * @brief Unlock OMU
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_Unlock(uint32_t instance);

/*!
 * @brief Lock OMU
 * 
 * @param[in] instance  OMU hw instance
 * @param[in] type      OMU lock type
 */
void OMU_DRV_Lock(uint32_t instance, omu_lock_type_t type);

/*!
 * @brief Enable OMU
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_Enable(uint32_t instance);

/*!
 * @brief Disable OMU
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_Disable(uint32_t instance);

/*!
 * @brief Enable OMU overlay region
 * 
 * @param[in] instance  OMU hw instance
 * @param[in] region    OMU overlay region number
 */
void OMU_DRV_EnableOverlayRegion(uint32_t instance, omu_overlay_region_t region);

/*!
 * @brief Disable OMU overlay region
 * 
 * @param[in] instance  OMU hw instance
 * @param[in] region    OMU overlay region number
 */
void OMU_DRV_DisableOverlayRegion(uint32_t instance, omu_overlay_region_t region);

/*!
 * @brief Get OMU overlay region status
 * 
 * @param[in] instance  OMU hw instance
 * @param[in] region    OMU overlay region number
 */
bool OMU_DRV_GetOverlayRegionStatus(uint32_t instance, omu_overlay_region_t region);

/*!
 * @brief Get OMU error flag
 * 
 * @param[in] instance  OMU hw instance
 */
uint8_t OMU_DRV_GetErrorFlag(uint32_t instance);

/*!
 * @brief Clear OMU error flag
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_ClearErrorFlag(uint32_t instance);

/*!
 * @brief Enable OMU interrupt
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_EnableInterrupt(uint32_t instance);

/*!
 * @brief Disable OMU interrupt
 * 
 * @param[in] instance  OMU hw instance
 */
void OMU_DRV_DisableInterrupt(uint32_t instance);

/*! @}*/
#if defined(__cplusplus)
}
#endif

/*! @}*/

#endif /* OMU_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
