/*
 * Copyright 2020-2024 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @file mrc_driver.h
 */

#ifndef MRC_DRIVER_H
#define MRC_DRIVER_H

#include <stddef.h>
#include <stdbool.h>
#include "device_registers.h"
#include "status.h"


/*!
 * @ingroup mrc_drv
 * @{
 */

/*******************************************************************************
 * Enumerations.
 ******************************************************************************/
typedef struct
{
    uint32_t regionNum;
    uint32_t regionBaseMsk;
    uint32_t regionBasePos;
    uint32_t regionSizeMax;
    uint32_t masterNum;
} mrc_inst_param_t;

/*!
 * @brief MRC Lock Type
 */
typedef enum
{
    MRC_UNLOCK = 0,
    MRC_SOFT_LOCK = 1,
    MRC_HARD_LOCK = 2
} mrc_lock_t;

/*!
 * @brief MRC Error Access Type
 */
typedef enum
{
    MRC_READ_ACCESS = 0,
    MRC_WRITE_ACCESS = 1,
} mrc_access_t;

/*!
 * @brief MRC Error Access Attributes
 */
typedef enum
{
    MRC_ATTR_NONE_SECURE_USER = 0,
    MRC_ATTR_SECURE_USER = 1,
    MRC_ATTR_NONE_SECURE_PRIVILEGED = 2,
    MRC_ATTR_SECURE_PRIVILEGED = 3,
} mrc_attr_t;

/*!
 * @brief MRC Access Control Policy(ACP) Configuration
 * | ACP | Secure Privileged | Secure User | Non-Secure Privileged | Non-Secure User |
 * | --- | ----------------  | ----------- | ----------            | --------------- |
 * | 0   | ---               | ---         | ---                   | ---             |
 * | 1   | R                 | R           | ---                   | ---             |
 * | 2   | RW                | ---         | ---                   | ---             |
 * | 3   | RW                | RW          | ---                   | ---             |
 * | 4   | RW                | RW          | R                     | ---             |
 * | 5   | RW                | RW          | R                     | R               |
 * | 6   | RW                | RW          | RW                    | ---             |
 * | 7   | RW                | RW          | RW                    | RW              |
 * Example:
 * When ACP is 1, corresponding slave can be accessed by read(R) operation only in the
 * Secure Privileged mode and Secure User mode.
 */
typedef enum
{
    MRC_ACP0 = 0,
    MRC_ACP1 = 1,
    MRC_ACP2 = 2,
    MRC_ACP3 = 3,
    MRC_ACP4 = 4,
    MRC_ACP5 = 5,
    MRC_ACP6 = 6,
    MRC_ACP7 = 7,
} mrc_acp_t;

typedef struct
{
    soc_master_t master;           /*! Master ID */
    mrc_acp_t acp;                 /*! Access Control Policy */
} mrc_macp_t;

typedef struct
{
    uint32_t regionId;             /*! Region ID */
    uint32_t regionBase;           /*! Region Base */
    uint32_t regionSize;           /*! Region Size, 2^regionSize KB */
    mrc_lock_t lockType;           /*! Lock Type, soft or hard */
    uint32_t masterNum;            /*! Master Number */
    mrc_macp_t *macp;              /*! Master ACP Configuration */
} mrc_region_t;

typedef struct
{
    uint32_t regionNum;            /*! Region Number */
    mrc_region_t *regionConfig;    /*! Region Configuration */
    bool uncovAreaEn;              /*! Uncover Area if needs to use MRC to manage */
    mrc_acp_t uncovAreaAcp;        /*! Uncover Area Access Control Policy */
    bool errorIntEn;               /*! Error Interrupt Enable */
} mrc_config_t;
/*******************************************************************************
* Definitions
******************************************************************************/




/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name MRC Driver
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initialize MRC
 * 
 * @param[in] instance  MRC hw instance
 * @param[in] config    MRC master configuration
 */
void MRC_DRV_Init(mrc_inst_t instance, mrc_config_t *config);

/*!
 * @brief Deinitialize MRC
 * 
 * @param[in] instance  MRC hw instance
 */
void MRC_DRV_Deinit(mrc_inst_t instance);

/*!
 * @brief Unlock MRC
 * 
 * @param[in] instance  MRC hw instance
 */
void MRC_DRV_Unlock(mrc_inst_t instance);

/*!
 * @brief Lock MRC
 * 
 * @param[in] instance  MRC hw instance
 * @param[in] type      MRC lock type
 */
void MRC_DRV_Lock(mrc_inst_t instance, mrc_lock_t type);

/*!
 * @brief Enable MRC
 * 
 * @param[in] instance  MRC hw instance
 */
void MRC_DRV_Enable(mrc_inst_t instance);

/*!
 * @brief Disable MRC
 * 
 * @param[in] instance  MRC hw instance
 */
void MRC_DRV_Disable(mrc_inst_t instance);

/*!
 * @brief Get error region ID
 */
uint8_t MRC_DRV_GetErrorRegionId(mrc_inst_t instance);

/*!
 * @brief Indicate if uncover error occurred
 */
bool MRC_DRV_IsUncoverError(mrc_inst_t instance);

/*!
 * @brief Get error master ID
 */
soc_master_t MRC_DRV_GetErrorMasterId(mrc_inst_t instance);

/*!
 * @brief Get error access attribute
 */
mrc_attr_t MRC_DRV_GetAccessAttribute(mrc_inst_t instance);

/*!
 * @brief Get error access type
 */
mrc_access_t MRC_DRV_GetAccessType(mrc_inst_t instance);

/*!
 * @brief Get error flag
 */
bool MRC_DRV_GetErrorFlag(mrc_inst_t instance);

/*!
 * @brief Clear error flag
 */
void MRC_DRV_ClearErrorFlag(mrc_inst_t instance);

/*! @}*/
#if defined(__cplusplus)
}
#endif

/*! @}*/

#endif /* MRC_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
