/*
 * Copyright 2020-2024 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @file mac_driver.h
 */

#ifndef MAC_DRIVER_H
#define MAC_DRIVER_H

#include <stddef.h>
#include <stdbool.h>
#include "device_registers.h"
#include "status.h"


/*!
 * @ingroup mac_drv
 * @{
 */

/*******************************************************************************
 * Enumerations.
 ******************************************************************************/
typedef enum
{
    MAC_UNLOCK = 0,
    MAC_SOFT_LOCK = 1,
    MAC_HARD_LOCK = 2
} mac_lock_t;

typedef enum
{
    MAC_FORCE_SECURE = 0,
    MAC_FORCE_NON_SECURE = 1,
    MAC_FORCE_USER = 2,
    MAC_FORCE_PRIVILEGED = 3,
    MAC_FOLLOW_MASTER = 4
} mac_attr_t;

typedef struct
{
    soc_slave_t port;
    bool disable;
} mac_slave_config_t;

typedef struct
{
    soc_master_t slot;
    uint32_t slaveNum;
    mac_slave_config_t *slaveConfig;
    mac_attr_t secAttr;
    mac_attr_t privAttr;
    bool errorIntEn;
} mac_master_config_t;

typedef struct
{
    uint32_t masterNum;
    mac_master_config_t *masterConfig;
} mac_config_t;

/*******************************************************************************
* Definitions
******************************************************************************/




/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name MAC Driver
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initialize MAC
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] config    MAC master configuration
 */
void MAC_DRV_Init(uint32_t instance, mac_config_t *config);

/*!
 * @brief Deinitialize MAC
 * 
 * @param[in] instance  MAC hw instance
 */
void MAC_DRV_Deinit(uint32_t instance);

/*!
 * @brief Unlock MAC
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 */
void MAC_DRV_Unlock(uint32_t instance, soc_master_t slot);

/*!
 * @brief Lock MAC
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 * @param[in] type      MAC lock type
 */
void MAC_DRV_Lock(uint32_t instance, soc_master_t slot, mac_lock_t type);

/*!
 * @brief Enable MAC
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 */
void MAC_DRV_Enable(uint32_t instance, soc_master_t slot);

/*!
 * @brief Disable MAC
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 */
void MAC_DRV_Disable(uint32_t instance, soc_master_t slot);

/*!
 * @brief Set secure attribute for master
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 * @param[in] attribute secure attribute
 */
void MAC_DRV_SetSecureAttribute(uint32_t instance, soc_master_t slot, mac_attr_t attribute);

/*!
 * @brief Set privilege attribute for master
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 * @param[in] attribute privilege attribute
 */
void MAC_DRV_SetPrivilegeAttribute(uint32_t instance, soc_master_t slot, mac_attr_t attribute);

/*!
 * @brief Enable or disable slave port
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 * @param[in] slave     MAC slave port
 * @param[in] enable    true to enable, false to disable
 */
void MAC_DRV_SetSlavePort(uint32_t instance, soc_master_t slot, soc_slave_t port, bool enable);

/*!
 * @brief Get error slave port
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 * @return MAC error slave port
 */
soc_slave_t MAC_DRV_GetErrorSlavePort(uint32_t instance, soc_master_t slot);

/*!
 * @brief Get MAC error flag
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 * @return MAC error flag
 */
uint8_t MAC_DRV_GetErrorFlag(uint32_t instance, soc_master_t slot);

/*!
 * @brief Clear MAC error flag
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 */
void MAC_DRV_ClearErrorFlag(uint32_t instance, soc_master_t slot);

/*!
 * @brief Enable MAC interrupt
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 */
void MAC_DRV_EnableInterrupt(uint32_t instance, soc_master_t slot);

/*!
 * @brief Disable MAC interrupt
 * 
 * @param[in] instance  MAC hw instance
 * @param[in] slot      MAC master slot
 */
void MAC_DRV_DisableInterrupt(uint32_t instance, soc_master_t slot);

/*! @}*/
#if defined(__cplusplus)
}
#endif

/*! @}*/

#endif /* MAC_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
