/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @file intm_driver.h
 * @version 1.4.0
 */

#ifndef INTM_DRIVER_H
#define INTM_DRIVER_H

#include "intm_hw_access.h"

/*!
 * @defgroup intm_driver 
 * @ingroup intm
 * @details This section describes the programming interface of the Intm Peripheral Driver.
 * @{
 */


/*******************************************************************************
 * Definitions
 ******************************************************************************/

typedef struct
{
    uint32_t threshold;         /*!< Config the maximum number of INTM clock cycles allowed for the monitored interrupt request*/
    IRQn_Type monitorIrqn;        /*!< Select the monitored interrupt source. */
    intm_montor_type monitorChannel; /*!< Select the monitor channel. */
} intm_config_t;

/*******************************************************************************
 * Function Prototypes for INTM
 ******************************************************************************/
/*!
 * @name Intm driver APIs
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initialize a INTM instance for operation.
 *
 * This function first resets the source triggers of all INTM target modules to their default values,
 * then configures the INTM with all the user defined in-out mappings.
 * This example shows how to set up the intm_config_t parameters and how to call the
 * INTM_DRV_Init() function with the required parameters:
 *  @code
 *   intm_config_t intmConfig=
 *   {
 *        .threshold = 200,
 *        .montorIrqn = pTMR0_CH0_IRQn,
 *        .monitorChannel = 0,
 *   };
 *
 *   INTM_DRV_Init(instance, &intmConfig,1);
 *   @endcode
 *
 * @param[in] instance          The INTM instance number.
 * @param[in] configPtr         Pointer to the user configuration structure.
 */
void INTM_DRV_Init(const uint32_t instance, const intm_config_t * configPtr);

/*!
 * @brief Start intm function.
 *
 * @param[in] instance          The INTM instance number.
 * @param[in] monitorChannel         The monitor channel.
 */
void INTM_DRV_Start(const uint32_t instance, uint32_t monitorChannel);

/*!
 * @brief  Configures the intm port physical parameters to access a device.
 *
 * @param[in] instance          The INTM instance number.
 * @param[in] configPtr         Pointer to the user configuration structure.
 */
void INTM_DRV_SetConfig(const uint32_t instance, const intm_config_t *configPtr);

/*!
 * @brief Reset to default values the source triggers corresponding to all target 
 *
 * @param[in] instance          The INTM instance number.
 */
void INTM_DRV_DeInit(const uint32_t instance);

/*!
 * @brief Clear interrupt flag and register GINTACK for next interrupt.
 *
 * @param[in] instance          The INTM instance number.
 * @param[in] configPtr         Pointer to the user configuration structure.
 */
void INTM_DRV_Clear(const uint32_t instance, const intm_config_t *configPtr);

/*!
 * @brief Disable the INTM module.
 *
 * @param[in] instance          The INTM instance number.
 */
void INTM_DRV_DisableChannel(const uint32_t instance, const uint32_t monitorChannel);

/*!
 * @brief Disable the INTM module.
 *
 * @param[in] instance          The INTM instance number.
 */
void INTM_DRV_DisableTotal(const uint32_t instance);

#if defined(__cplusplus)
}
#endif

/*! @}*/ /* End of Intm driver APIs*/
/*! @}*/ /* End of addtogroup intm_driver */

#endif /* INTM_DRIVER_H */
/*******************************************************************************
* EOF
******************************************************************************/
