/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef HSU_DRIVER_H
#define HSU_DRIVER_H

/*! @file hsu_driver.h */

#include <stddef.h>
#include "device_registers.h"
#include "status.h"

/*!
 * @defgroup hsu_driver HSU Driver
 * @ingroup hsu
 * @brief Hardware Semaphore Unit Peripheral Driver
 * @details This section describes the programming interface of the HSU driver.
 * @{
 */
/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Definitions
 ******************************************************************************/


/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name HSU DRIVER API
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initializes the HSU module
 *
 * This function initializes HSU driver based on user configuration input.
 * The user must make sure that the clock is enabled
 *
 * @param[in] instance The HSU instance number
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t HSU_DRV_Init(uint32_t instance);

/*!
 * @brief Sets the default configuration
 *
 * This function sets the default configuration
 *
 * @param[in] instance The HSU instance number
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t HSU_DRV_Deinit(uint32_t instance);

/*!
 * @brief try to get HSU gate
 *
 * This function tries to get target HSU gate and returns new result.
 *
 * If target gate is free, it will be locked and success status is returned.
 * If target gate is not available, it will be unlocked and failed status is returned.
 *
 * @param[in] instance The HSU instance number
 * @param[in] gate target gate number
 * @return gate acquired status
 */
status_t HSU_DRV_TryLockGate(uint32_t instance, uint8_t gate);

/*!
 * @brief Wait HSU gate lock
 *
 * This function wait and lock target gate.
 *
 * If target gate is not available, this function will keep waiting until target gate is available.
 *
 * @param[in] instance The HSU instance number
 * @param[in] gate target gate number
 * @return gate acquired status
 */
status_t HSU_DRV_WaitLockGate(uint32_t instance, uint8_t gate);

/*!
 * @brief HSU Release lock gate
 *
 * This function will release target.
 *
 * If target gate is not locked, this function will still return success status.
 * If current thread is not owner of target gate, this function will return failed status.
 *
 * @param[in] instance The HSU instance number
 * @param[in] gate target gate number
 * @return gate relase status
 */
status_t HSU_DRV_ReleaseGate(uint32_t instance, uint8_t gate);

/*!
 * @brief HSU Force Reset Gate
 *
 * This function will force reset target gate even if target gate is locked.
 *
 * @param[in] instance The HSU instance number
 * @param[in] gate target gate number
 * @return gate owner
 */

status_t HSU_DRV_ForceResetGate(uint32_t instance, uint8_t gate);

/*! @} */

#if defined(__cplusplus)
}
#endif

/*! @} */

#endif /* HSU_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
