/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @file fmu_driver.h
 * @version 1.4.0
 */

#ifndef FMU_DRIVER_H
#define FMU_DRIVER_H

#include <stdbool.h>
#include <stddef.h>
#include "status.h"
#include "device_registers.h"

/*!
 * @defgroup fmu_driver Fault Manager Unit Driver
 * @ingroup fmu
 * @details This section describes the fmu interface of the Fault Manager Unit Driver.
 * @{
 */

/*!
 * @addtogroup fmu_driver
 * @{
 */

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* Time Out Value in milliseconds for running an operation */
#define FMU_TIME_DELAY           (10U)

/*!
 * @brief Implements Count of Fault Channels
 * Implements : FMU_FCH_COUNT_Class
 */
#define FMU_FCH_COUNT              (uint8_t)(FMU_FCH_CTRL_COUNT)

/*!
 * @brief Implements Clear All Faults Symbol
 * Implements : FMU_CLEAR_ALL_FAULTS_Class
 */
#define FMU_CLEAR_ALL_FAULTS       (uint8_t)(FMU_FCH_COUNT + 1U)

/*!
 * @brief Timer Types
 * Implements : fmu_xtmr_type_t_Class
 */
typedef enum
{
    FMU_XTMR_CFG    = 0U, /*!< Timer Set for CFG State */
    FMU_XTMR_ALARM  = 1U, /*!< Timer Set for Alarm State */
    FMU_XTMR_EOUT   = 2U, /*!< Timer Set for Fault ERR OUT */
    FMU_XTMR_FFTR   = 3U, /*!< Timer Set for Fault ERR OUT */
} fmu_xtmr_type_t;

/*!
 * @brief State of an executed Operation
 * Implements : fmu_op_status_t_Class
 */
typedef enum
{
    FMU_OP_IDLE       = 0U,    /*!< FMU Operation Idle */
    FMU_OP_INPROGRESS = 1U,    /*!< FMU Operation In progress */
    FMU_OP_ABORTED    = 2U,    /*!< FMU Operation Aborted */
    FMU_OP_SUCCESSFUL = 3U     /*!< FMU Operation Successful */
} fmu_op_status_t;

/*!
 * @brief Operations that can be executed by FMU
 * Implements : fmu_run_op_t_Class
 */
typedef enum
{
    FMU_RUN_OP0     = 0U,     /*!< No operation */
    FMU_RUN_OP1     = 1U,     /*!< Put FMU in CFG state */
    FMU_RUN_OP2     = 2U,     /*!< Put FMU in RUN state */
    FMU_RUN_OP3     = 3U,     /*!< Clear FAULT_FSM_ST and ALARM_FSM_ST register */
    FMU_RUN_OP4     = 4U,     /*!< Set OPS to IDLE */
} fmu_run_op_t;

/*!
 * @brief Functional clock source.
 * Implements fmu_clock_src_t_Class
 */
typedef enum
{
    FMU_FUNC_CLOCK_SRC_SIRC_DIV4    = 0U,    /*!< SIRC clock divided by 4              */
    FMU_FUNC_CLOCK_SRC_FIRC         = 1U,    /*!< FIRC Clock                           */
    FMU_FUNC_CLOCK_SRC_SXOSC        = 2U,    /*!< Slow External OSC                    */
    FMU_FUNC_CLOCK_SRC_FXOSC        = 3U,    /*!< Fast External OSC                    */
    FMU_FUNC_CLOCK_SRC_PLL          = 4U,    /*!< PLL                                  */
    FMU_FUNC_CLOCK_SRC_NONE         = 7U     /*!< Reserved, no clock source selected   */
} fmu_clock_src_t;

/*!
 * @brief Fault Output Control for EOUT Signals
 * Implements : fmu_eout_control_t_Class
 */
typedef enum
{
    FMU_EOUT_CTRL_BY_FSM      = 0U,    /*!< Controlled by the FSM */
    FMU_EOUT_CTRL_BY_SOFT     = 1U,    /*!< Controlled by the Soft, FOUT_FRC */
} fmu_eout_control_t;

/*!
 * @brief Fault Output Mode for EOUT Signals in CFG Mode
 * Implements : fmu_eout_mode_cfg_t_Class
 */
typedef enum
{
    FMU_EOUT_CFG_HI         = 0U,    /*!< ErrOut High Impedance in CFG State */
    FMU_EOUT_CFG_AS_RUN     = 1U,    /*!< ErrOut Output as RUN State in CFG State */
    FMU_EOUT_CFG_AS_FAULT   = 2U,    /*!< ErrOut Output as Fault State in CFG State */
} fmu_eout_mode_cfg_t;

/*!
 * @brief State of the FMU Module
 * Implements : fmu_status_t_Class
 */
typedef enum
{
    FMU_STATUS_IDLE          = 0U,    /*!< FMU IDLE          */
    FMU_STATUS_RUN           = 1U,    /*!< FMU RUN           */
    FMU_STATUS_CFG           = 2U,    /*!< FMU Configuration */
    FMU_STATUS_ALARM         = 3U,    /*!< FMU Alarm         */
    FMU_STATUS_FAULT         = 4U,    /*!< FMU Fault         */
} fmu_status_t;

/*!
 * @brief Type of FSM Transfer Status
 * Implements : fmu_transfer_type_t_Class
 */
typedef enum
{
    FMU_FSM_RUN_TO_ALARM      = 0U,    /*!< The state transition from the RUN state to the ALARM state    */
    FMU_FSM_ALARM_TO_FAULT    = 1U,    /*!< The state transition from the ALARM state to the FAULT state  */
    FMU_FSM_RUN_TO_FAULT      = 2U,    /*!< The state transition from the RUN state to the FAULT state    */
    FMU_FSM_FAULT_TO_ALARM    = 3U,    /*!< The state transition from the FAULT state to the ALARM state  */
} fmu_transfer_type_t;

/*!
 * @brief Type of Interrupt Source
 * Implements : fmu_irq_type_t_Class
 */
typedef enum
{
    FMU_IRQ_NO              = 0U,   /*!< FMU ISR No Interrupt */
    FMU_IRQ_ALARM           = 1U,   /*!< FMU ISR Alarm Interrupt */
    FMU_IRQ_NMI             = 2U,   /*!< FMU ISR NMI Interrupt */
    FMU_IRQ_CFG_TO          = 4U,   /*!< FMU ISR CFG State Timeout Interrupt */
    FMU_IRQ_ERRIN           = 8U,   /*!< FMU ISR ErrIn Interrupt */
} fmu_irq_type_t;

/*!
 * @brief Event of Interrupt Source Reported By State FMU IRQ Structure
 * Implements : fmu_isr_event_t_Class
 */
typedef enum
{
    FMU_ISR_EVENT_NO       = 0U,   /*!< ISR EVENT No Interrupt Triggered */
    FMU_ISR_EVENT_ALARM    = 1U,   /*!< ISR EVENT Alarm Interrupt Triggered */
    FMU_ISR_EVENT_NMI      = 2U,   /*!< ISR EVENT NMI Interrupt Triggered */
    FMU_ISR_EVENT_CFG_TO   = 4U,   /*!< ISR EVENT CFG State TimeOut Interrupt Triggered */
    FMU_ISR_EVENT_ERRIN    = 8U,   /*!< ISR EVENT ErrIn Interrupt Triggered */
} fmu_isr_event_t;

/*!
 * @brief State of Semaphore
 * Implements : fmu_semaphore_t_Class
 */
typedef enum
{
    FMU_SEMAPHORE_UNLOCK = 0U, /*!< State of Semaphore Unlocked */
    FMU_SEMAPHORE_LOCK   = 1U  /*!< State of Semaphore Locked */
} fmu_semaphore_t;

/*!
 * @brief Structure for State of ISR for FMU
 * Implements : fmu_isr_state_t_Class
 */
typedef struct
{
    volatile fmu_semaphore_t   semaphore;                           /*!< Status of Semaphore */
             fmu_isr_event_t   eventType;                           /*!< Source of interrupt trigger */
             uint8_t           chCount;                          /*!< Number of configured Fault Channels */
             void          ( * callback )( void * param );  /*!< Callback Global Isr Function */
             void            * callbackParam;                       /*!< Parameter for Callback Global Isr Function */
} fmu_isr_state_t;

/*!
 * @brief ERR OUT configuration
 * Implements : fmu_eout_config_t_Class
 */
typedef struct
{
    bool eout0Enable;                  /*!< False -> EOUT Signals are High Impedance (inactive),
                                        *   True  -> EOUT Signals driven by FMU (active)       */
    bool eout1Enable;                  /*!< False -> EOUT Signals are High Impedance (inactive),
                                        *   True  -> EOUT Signals driven by FMU (active)       */
    fmu_eout_control_t control;        /*!< FSM control State By FSM or Soft   */
    bool bistableMode;                 /*!< True -> Bi-stable mode, False -> other mode, Only support Bi-stable mode */
    bool eout0pol;                     /*!< Polarity Selection : False -> EOUT0 low,  True -> EOUT0 high */
    bool eout1pol;                     /*!< Polarity Selection : False -> EOUT1 low,  True -> EOUT1 high */
    fmu_eout_mode_cfg_t eoutModeCfg;   /*!< ErrOut Output mode when CFG State */
    uint32_t outInterval;              /*!< Bistable Minimum Fault-Output (EOUT) Faulty Interval */
} fmu_eout_config_t;

/*!
 * @brief ERR IN configuration
 * Implements : fmu_ein_config_t_Class
 */
typedef struct
{
    uint8_t portNum;                   /*!< Number of ERR IN port (0~3) */
    bool einEnable;                    /*!< Enable ERR IN input*/
    bool polarity;                     /*!< Polarity Selection : False -> high to generate fault,  
                                                                 True  -> low to generate fault */
    bool resetEnable;                  /*!< Enable ERR IN to generate reset */
    bool intEnable;                    /*!< Enable ERR IN to generate NMI interrupt */
} fmu_ein_config_t;

/*!
 * @brief Global Driver FMU Config
 * Implements : fmu_config_t_Class
 */
typedef struct
{
    bool                       faultDebugEn;                    /*!< Enable or disable fault input in debug mode */
    fmu_clock_src_t            funcClk;                         /*!< Select clock source of functional */
    bool                       cfgTimeoutIntEn;                 /*!< Enable or disable CFG state timeout interrupt */
    uint32_t                   cfgTimeout;                      /*!< CFG state timeout value, divisible by 32, not configed in cfg */
    uint32_t                   alarmTimeout;                    /*!< Alarm timeout value */
    uint8_t                    chCount;                         /*!< Number of fault channels configured */
    const fmu_eout_config_t  * eoutCfg;                         /*!< Pointer to Configuration behavior for ERR OUT */
    uint8_t                    einCount;                        /*!< Count of the ERR IN which to be configured */
    const fmu_ein_config_t   * einCfgArr;                       /*!< Pointer to Configuration behavior for ERR IN */
    fmu_clock_src_t            fftrClk;                         /*!< Select clock source of output supervision unit */
    uint32_t                   fftrTimeout;                     /*!< Timeout value of output supervision unit, more than 4096 and divisible by 4096 */
    void                   ( * callback )( void * param );      /*!< Global ISR Callback */
    void                     * callbackParam;                   /*!< Global ISR Callback Parameter*/
} fmu_config_t;


/*!
 * @brief Fault Channel Config
 * Implements : fmu_channel_config_t_Class
 */
typedef struct
{
    uint8_t             channelID;                         /*!< Fault Channel id*/
    bool                alarmEnable;                       /*!< Enable the FMU FSM enter ALARM state */
    bool                alarmIntEnable;                    /*!< Enable alarm to generate interrupt */
    bool                faultIntEnable;                    /*!< Enable fault to generate interrupt */
    bool                errOutEnable;                      /*!< Enable fault to generate ERR OUT */
    bool                resetEnable;                       /*!< Enable fault to generate reset */
    void            ( * alarmCallback )( void * param );   /*!< Applicable only for alarm interrupt else NULL*/
    void              * alarmCallbackParam;                /*!< Alarm Callback Parameter */
    void            ( * faultCallback )( void * param );   /*!< Applicable only for fault interrupt else NULL*/
    void              * faultCallbackParam;                /*!< Fault Callback Parameter */
} fmu_channel_config_t;

/*******************************************************************************
* API
*******************************************************************************/
/*!
 * @name fmu Driver API
 * @{
 */
#if defined (__cplusplus)
extern "C" {
#endif

/*!
 * @brief Clear Faults Status
 *
 * Need to clear the fault register fault_S by setting bit position corresponding with
 * fault faultIndex. Any value higher than last faultNo will clear all faults
 *
 * @param[in] instance          number of FMU peripheral to clear fault.
 * @param[in] channel           fault channel to clear or all faults mask.
 * @return  STATUS_SUCCESS in case of success clearing fault(s)
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_ClearFaults(const uint32_t instance,  uint8_t channel);


/*!
 * @brief Run Operation issued by user.
 *
 * In some case operations need to unlock the FMU to allow to execute it
 * An failsafe timeout protection for execution of operation is implemented
 *
 * @param[in] instance    number of FMU instance.
 * @param[in] operation   operation to run.
 * @return  STATUS_SUCCESS in case of success run operation
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_RunOperation(const uint32_t instance, const fmu_run_op_t operation);


/*!
 * @brief Set Event for triggered interrupt in FMU IRQ Status
 *
 * This function sets the event without taking care of the semaphore status
 * in case of lock will override the status and set the new one and
 * unlocks the semaphore after operation completed.
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @param[in] event      type of interrupt event to set.
 */
void FMU_DRV_IrqSetEvent(const uint32_t instance, const fmu_isr_event_t event);

/*!
 * @brief Get Event for triggered interrupt in FMU IRQ Status
 *
 * Checks the state of semaphore and returns the interrupt event.
 * An failsafe timeout protection for execution of operation is implemented
 * in case the semaphore is locked and prevents access to the FMU IRQ Status structure.
 *
 * @param[in]   instance   instance number of FMU peripheral.
 * @param[out]  state   pointer to the location where to save the event.
 */
void FMU_DRV_IrqGetEvent(const uint32_t instance, fmu_isr_event_t * const state);

/*!
 * @brief Clear the flag of the NMI Interrupt
 *
 * Will clear the flag of the NMI Interrupt.
 *
 * @param[in]   instance   instance number of FMU peripheral.
 */
void FMU_DRV_ClearNmiIntFlag(const uint32_t instance);

/*!
 * @brief Get the status of of the ERRIN
 *
 * Will retrun the status of of the ERRIN.
 *
 * @param[in]   instance   instance number of FMU peripheral.
 * @param[in]   port       port number of ERRIN.
 * @return  true     if ERRIN fault has occured
 *          false    if ERRIN fault not has occured
 */
bool FMU_DRV_GetErrInStatus(const uint32_t instance, uint8_t port);

/*!
 * @brief Clear the flag of the ERRIN
 *
 * Will clear the flag of the ERRIN
 *
 * @param[in]   instance   instance number of FMU peripheral.
 * @param[in]   port       port number of ERRIN.
 */
void FMU_DRV_ClearErrInFlag(const uint32_t instance, uint8_t port);

/*!
 * @brief Checks the installed Callbacks in case of alarm
 *
 * In case of interrupt from Alarm source checks the triggered source fault
 * and If configured callback function then call it.
 *
 * @param[in] instance   instance number of FMU peripheral.
 */
void FMU_DRV_IrqAlarmCallback(const uint32_t instance);

/*!
 * @brief Global Interrupt FMU Handler
 *
 * In case of interrupt checks the triggered source for FMU
 * and update FMU IRQ Status with corresponding event and if a global callBackFunction
 * configured call it.
  * If a fault NMI is enabled then it will trigger to a non-maskable interrupt(NMI).
 *
 * @param[in] instance   instance number of FMU peripheral.
 */
void FMU_DRV_IrqHandler(const uint32_t instance);

/*!
 * @brief Get FMU Status
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @return FMU_STATUS_RUN             if FMU is in RUN mode
 *         FMU_STATUS_CFG             if FMU is in CFG mode
 *         FMU_STATUS_ALARM           if FMU is in Alarm state
 *         FMU_STATUS_FAULT           if FMU is in Fault state
 */
fmu_status_t FMU_DRV_GetFsmState(const uint32_t instance);

/*!
 * @brief Get default value of the configuration for FMU module
 *
 * This function will get default one non-critical fault and one control configuration.
 * Non-critical fault ID default value is 2U, base on other derivatives maybe select fault injection(fake fault) suitable.
 *
 * @param[out] controlConfig pointer to configuration structure for FMU
 * @param[out] chConfig      pointer to configuration structure for fault channel
 */
void FMU_DRV_GetDefaultConfig(fmu_config_t * controlConfig, fmu_channel_config_t * chConfig);

/*!
 * @brief Initialize FMU Module and Faults
 *
 * Will clear all errors to permit enter in config mode and  run mode
 *
 * @param[in] instance      instance number of FMU peripheral.
 * @param[in] config        pointer to configuration structure for FMU
 * @param[in] chConfigArr   pointer to configuration structure for Faults
 * @return  STATUS_SUCCESS in case of success setting of FMU
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_Init(const uint32_t instance, const fmu_config_t *config, const fmu_channel_config_t * chConfigArr[]);

/*!
 * @brief Set FMU Module in config mode
 *
 * Will clear all errors to permit enter in configuration mode, and lock the configuration
 * as user set it.
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @return  STATUS_SUCCESS in case of success setting of FMU
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_SetCfgState(const uint32_t instance);

/*!
 * @brief Set FMU Module in run mode
 *
 * Will clear errors if FMU is in Alarm or Fault State.
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @return  STATUS_SUCCESS in case of success setting of FMU
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_SetRunState(const uint32_t instance);

/*!
 * @brief Set Time Out Interval that permits the FMU module to be set in configuration mode
 *
 * If is in configuration mode will force it in run mode and will clear all faults.
 * User need to put again in configuration mode if need it the FMU module
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @param[in] value      amount of time that FMU can be in configuration mode, is a multiple of 32
 * @return  STATUS_SUCCESS in case of success setting of FMU
 *          STATUS_FMU_ERROR_CONFIG_TIMEOUT in case of FMU fails to set timer value
 */
status_t FMU_DRV_SetCfgStateTimer(const uint32_t instance, const uint32_t value);

/*!
 * @brief Get the Recorded Channel of FSM transition
 *
 * Will return the recorded channel number of FSM transition by transfer type
 *
 * @param[in]  instance   instance number of FMU peripheral.
 * @param[in]  type       fmu fsm transfer type
 * @return  channel number of FSM transition
 */
uint8_t FMU_DRV_GetFsmFreezeCh(const uint32_t instance, const fmu_transfer_type_t type);

/*!
 * @brief Clear the Recorded Channel
 *
 * Will clear the recorded channel number.
 *
 * @param[in]  instance   instance number of FMU peripheral.
 */
void FMU_DRV_ClearAllFsmFreezeCh(const uint32_t instance);

/*!
 * @brief Update the configuration of fault channel
 *
 * This function will set the fault channel configuration that need to be done in FMU Config State.
 * In case FMU is in run state will try to put in Config state if no errors
 * detected and the put back FMU in run state, in case of Alarm & Fault State
 * the function will report error.
 *
 * @param[in]  instance   instance number of FMU peripheral.
 * @param[in]  chCfg      pointer to the configuration structure containing fault channle settings
 * @return  STATUS_SUCCESS in case of success setting fault channle configuration
 *          STATUS_FMU_ERROR_FAULT_DETECTED in case of detection of fault errors
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_SetFaultChConfig(const uint32_t instance, fmu_channel_config_t const * chCfg);

/*!
 * @brief Update the configuration of EOUT signals
 *
 * This function will set the EOUT configuration that need to be done in FMU Config State.
 * In case FMU is in run state will try to put in Config state if no errors
 * detected and the put back FMU in run state, in case of Alarm & Fault State
 * the function will report error.
 *
 * @param[in]  instance   instance number of FMU peripheral.
 * @param[in]  eoutCfg    pointer to the configuration structure containing EOUT settings
 * @return  STATUS_SUCCESS in case of success setting EOUT configuration
 *          STATUS_FMU_ERROR_FAULT_DETECTED in case of detection of fault errors
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_SetErrOutConfig(const uint32_t instance, const fmu_eout_config_t * const eoutCfg);

/*!
 * @brief Update the configuration of EIN signals
 *
 * This function will set the EIN configuration that need to be done in FMU Config State.
 * In case FMU is in run state will try to put in Config state if no errors
 * detected and the put back FMU in run state, in case of Alarm & Fault State
 * the function will report error.
 *
 * @param[in]  instance   instance number of FMU peripheral.
 * @param[in]  einCount   count of the configuration structure
 * @param[in]  einCfgArr  pointer to the configuration structure containing EIN settings
 * @return  STATUS_SUCCESS in case of success setting EIN configuration
 *          STATUS_FMU_ERROR_FAULT_DETECTED in case of detection of fault errors
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_SetErrInConfig(const uint32_t instance, const uint8_t einCount, const fmu_ein_config_t * const einCfgArr);

/*!
 * @brief Get the Timer Value
 *
 * This function will update and return if the correct state correspond for running timer
 * based on the timer type selected by user.
 * @param[in]  instance   instance number of FMU peripheral.
 * @param[in]  type       type of timer to read based on fmu_xtmr_type_t enum structure
 * @return    uint32_t value of timer
 */
uint32_t FMU_DRV_GetXtmrValue(const uint32_t instance, const fmu_xtmr_type_t type);

/*!
 * @brief Set Fault Injection for test
 *
 * This function will inject a Fault in the peripheral for propose of testing the
 * behavior of Faults configured to allow an emulation of triggered fault set.
 * @param[in]  instance   instance number of FMU peripheral.
 * @param[in]  channel    trigger injection fault channel
 */
void FMU_DRV_SetFaultInjection(const uint32_t instance, const uint8_t channel, uint16_t time_window);

/*!
 * @brief De-initialize FMU Module and Faults
 *
 * Will clear all errors to permit enter in config mode and  run mode
 * The Module Register Values will be the reset values as chip specific
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @return  STATUS_SUCCESS in case of success setting of FMU
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 *          STATUS_TIMEOUT in case of a timeout operation
 */
status_t FMU_DRV_Deinit(const uint32_t instance);

/*!
 * @brief Disable a Fault Channel
 *
 * Will Update and Check status of the Fault number to be disabled, and report if fault is
 * triggered or in case of a timeout operation.
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @param[in] faultIdx   fault to disable.
 * @return  STATUS_SUCCESS in case of success disable of FMU Fault
 *          STATUS_FMU_ERROR_FAULT_DETECTED in case of detection of fault triggered source
 *          STATUS_TIMEOUT in case of a timeout operation
 *          STATUS_FMU_ERROR_SET_RUN in case of FMU fails to return to run mode
 *          STATUS_FMU_ERROR_SET_CFG in case of FMU fails to enter in configuration mode
 */
status_t FMU_DRV_DisableFaultCh(const uint32_t instance, uint8_t channel);

/**
 * @brief Check whether any fault occurs in Fault Status Register.
 *
 * @param[in] instance   instance number of FMU peripheral.
 * @return true, if any fault exist.
 */
bool FMU_DRV_CheckFaults(const uint32_t instance);

#if (FMU_INSTANCE_COUNT > 0U)
/*!
 * @brief Wrapper for Global Interrupt FMU Handler
 *
 */
void FMU0_IRQHandler(void);

#endif


/*! @} */

#if defined (__cplusplus)
}

#endif

/*! @} */

#endif /* FMU_DRIVER_Hs */

/*******************************************************************************
 * EOF
 ******************************************************************************/
