/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef CCU_DRIVER_H
#define CCU_DRIVER_H

/*! @file ccu_driver.h */

#include <stddef.h>
#include "device_registers.h"
#include "status.h"

/*!
 * @defgroup ccu_driver CCU Driver
 * @ingroup ccu
 * @brief Core Communication Unit Peripheral Driver
 * @details This section describes the programming interface of the CCU driver.
 * @{
 */
/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/*!
 * @brief CCU channel configuration structure.
 * Implements : ccu_channel_config_t_Class
 */
typedef struct 
{
    uint8_t channelID;          /*!< Index of channel */
    bool txIntEn;               /*!< TX mailbox interrupt enable */
    bool rxIntEn;               /*!< RX mailbox interrupt enable */
    bool rxTrigIntEn;           /*!< Receive trigger interrupt enable */
} ccu_channel_config_t;

/*!
 * @brief CCU group configuration structure.
 * Implements : ccu_group_config_t_Class
 */
typedef struct
{
    uint8_t groupID;            /*!< Index of group */
    ccu_master_id_t masterID;   /*!< Define the master that can access this group */
    uint8_t triggerSize;        /*!< Define the accessible trigger number in this group. */
    uint8_t mailboxSize;        /*!< Define the accessible mailbox number in this group. */
} ccu_group_config_t;

/*!
 * @brief CCU user configuration structure.
 *
 * This structure is used to configure the CCU.
 *
 * Implements : ccu_user_config_t_Class
 */
typedef struct
{
    bool softLock;                      /*!< Soft lock for CCU group config */
    bool hardLock;                      /*!< Hard lock for CCU group config */
    uint8_t groupSize;                  /*!< The size for CCU group */
    uint8_t channelSize;                /*!< The size for CCU channel */
    ccu_group_config_t *ccuGroup;       /*!< Pointer to CCU group configuration */
    ccu_channel_config_t *ccuChannel;   /*!< Pointer to CCU channel configuration */
} ccu_user_config_t;


/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name CCU DRIVER API
 * @{
 */

status_t CCU_DRV_Init(uint8_t instance, const ccu_user_config_t *userCfg);

status_t CCU_DRV_SendData(uint8_t instance, uint8_t channel, uint32_t data);

status_t CCU_DRV_RecvData(uint8_t instance, uint8_t channel, uint32_t* data);

void CCU_DRV_GenerateTrigger(uint8_t instance, uint8_t channel);

void CCU_DRV_ClearTriggerEvent(uint8_t instance, uint8_t channel);

#if defined(__cplusplus)
extern "C" {
#endif

/*! @} */

#if defined(__cplusplus)
}
#endif

/*! @} */

#endif /* CCU_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
