/*
 * Copyright 2020-2022 Yuntu Microelectronics Co., Ltd.
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef BMIC_DRIVER_H
#define BMIC_DRIVER_H

/*! @file bmic_driver.h */

#include <stddef.h>
#include "device_registers.h"
#include "status.h"

/*!
 * @defgroup bmic_driver BMIC Driver
 * @ingroup bmic
 * @brief Power-mode and Reset Monitor Unit Peripheral Driver
 * @details This section describes the programming interface of the BMIC driver.
 * @{
 */
/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Definitions
 ******************************************************************************/


/*!
 * @brief BMIC monitor timeout configuration
 * Implements : bmic_monitor_timeout_t_Class
 */
typedef enum
{
    BMIC_MONITOR_TIMEOUT_DISABLE = 0U, /*!< Monitor timeout disable */
    BMIC_MONITOR_TIMEOUT_RESERVED = 1U, /*!< Monitor timeout reserved */
    BMIC_MONITOR_TIMEOUT_128_CLK = 2U, /*!< Monitor timeout 128 bus clock */
    BMIC_MONITOR_TIMEOUT_256_CLK = 3U, /*!< Monitor timeout 256 bus clock */
    BMIC_MONITOR_TIMEOUT_384_CLK = 4U, /*!< Monitor timeout 384 bus clock */
    BMIC_MONITOR_TIMEOUT_512_CLK = 5U, /*!< Monitor timeout 512 bus clock */
    BMIC_MONITOR_TIMEOUT_640_CLK = 6U, /*!< Monitor timeout 640 bus clock */
    BMIC_MONITOR_TIMEOUT_768_CLK = 7U, /*!< Monitor timeout 768 bus clock */
    BMIC_MONITOR_TIMEOUT_896_CLK = 8U, /*!< Monitor timeout 896 bus clock */
    BMIC_MONITOR_TIMEOUT_1024_CLK = 9U, /*!< Monitor timeout 1024 bus clock */
    BMIC_MONITOR_TIMEOUT_1152_CLK = 10U, /*!< Monitor timeout 1152 bus clock */
    BMIC_MONITOR_TIMEOUT_1280_CLK = 11U, /*!< Monitor timeout 1280 bus clock */
    BMIC_MONITOR_TIMEOUT_1408_CLK = 12U, /*!< Monitor timeout 1408 bus clock */
    BMIC_MONITOR_TIMEOUT_1536_CLK = 13U, /*!< Monitor timeout 1536 bus clock */
    BMIC_MONITOR_TIMEOUT_1664_CLK = 14U, /*!< Monitor timeout 1664 bus clock */
    BMIC_MONITOR_TIMEOUT_1792_CLK = 15U, /*!< Monitor timeout 1792 bus clock */
} bmic_monitor_timeout_t;

/*!
 * @brief BMIC monitor channel configuration
 * Implements : bmic_monitor_channel_t_Class
 */
typedef struct {
    uint8_t monitorChannel;                 /*!< Monitor channel */
    bool masterPortEnable;                  /* Master port check enable */
    bool slavePortEnable;                   /* Slave port check enable */
    bmic_monitor_timeout_t timeoutThresh;   /*!< Monitor ready timeout threshold */
} bmic_monitor_channel_t;

/*!
 * @brief BMIC monitor threshold
 * Implements : bmic_monitor_threshold_t_Class
 */
typedef enum
{
    BMIC_MONITOR_THRESHOLD_6P25MS = 0U, /*!< Monitor threshold 6.25ms */
    BMIC_MONITOR_THRESHOLD_12P5MS = 1U, /*!< Monitor threshold 12.5ms */
    BMIC_MONITOR_THRESHOLD_25MS = 2U, /*!< Monitor threshold 25ms */
    BMIC_MONITOR_THRESHOLD_50MS = 3U, /*!< Monitor threshold 50ms */
} bmic_monitor_threshold_t;

/*!
 * @brief BMIC error injection configuration
 * Implements : bmic_error_inject_selection_t_Class
 */
typedef enum
{
    BMIC_NO_ERROR_INJECTION = 0U, /*!< Disable error injection */
    BMIC_DATA_PHASE_ERROR_INJECTION = 1U << 0U, /*!< Data phase error injection */
    BMIC_ADDRESS_PHASE_ERROR_INJECTION = 1U << 1U, /*!< Address phase error injection */
    BMIC_MASTER_TIMEOUT_ERROR_INJECTION = 1 << 2U, /*!< Master timeout error injection */
}bmic_error_inject_selection_t;

/*!
 * @brief BMIC user configuration structure
 * Implements : bmic_user_config_t_Class
 */
typedef struct {
    uint8_t count;      /*!< Monitor channel count */
    bmic_monitor_channel_t *channelConfigs; /*!< Monitor channel configuration */
} bmic_user_config_t;

/*!
 * @brief BMIC error injection configuration
 * Implements : bmic_error_inject_t_Class
 */

typedef struct
{
    bmic_error_inject_selection_t errorInject; /*!< Error injection selection */
    uint8_t slavePort;                         /*!< Slave port number */
    uint8_t masterPort;                        /*!< Master port number */
    uint8_t syndrome;                          /*!< Syndrome */
} bmic_error_inject_t;

/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name BMIC DRIVER API
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Initializes the BMIC module
 *
 * This function initializes BMIC driver based on user configuration input.
 * The user must make sure that the clock is enabled
 *
 * @param[in] instance The BMIC instance number
 * @param[in] userConfigPtr Pointer to structure of initialization
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t BMIC_DRV_Init(uint32_t instance, const bmic_user_config_t *userConfigPtr);

/*!
 * @brief Sets the default configuration
 *
 * This function sets the default configuration
 *
 * @param[in] channel The BMIC channel number
 * @param[out] channelConfigPtr Pointer to structure of initialization
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t BMIC_DRV_GetDefaultChannelConfig(uint8_t channel, bmic_monitor_channel_t *channelConfigPtr);

/*!
 * @brief BMIC error injection configuration
 *
 * This function configures the BMIC error injection.
 *
 * @param[in] instance The BMIC instance number
 * @param[in] errorInjectPtr Pointer to structure of initialization
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */

status_t BMIC_DRV_ErrorInject(uint32_t instance, const bmic_error_inject_t *errorInjectPtr);

/*!
 * @brief Sets the default configuration
 *
 * This function sets the default configuration
 *
 * @param[in] instance The BMIC instance number
 * @return Execution status (success)
 * @retval STATUS_SUCCESS Operation was successful
 */
status_t BMIC_DRV_Deinit(uint32_t instance);

/*! @} */

#if defined(__cplusplus)
}
#endif

/*! @} */

#endif /* BMIC_DRIVER_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
