/**
* @file    CddTrg_Lld_Tmu.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : CddTrg_Lld_Tmu
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V1.0.0.HF001
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
/*
 *  PRQA S 2985 Rule 2.2:  The operation here is redundant as the result's Value is always that of the left-hand operand.
 *  PRQA S 2986 Rule 2.2:  This operation is redundant. The value of the result is always that of the right-hand operand.
 *  PRQA E 4934 Rule 10.8:  A composite expression of 'essentially unsigned' type (unsigned long) is being cast to a 
 *                           different type category, 'enum'.
 */
 
#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "CddTrg_Lld_Tmu.h"
#include "CddTrg_Lld_Tmu_Reg.h"
#include "SchM_CddTrg.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CDDTRG_LLD_TMU_VENDOR_ID_C                      (180)
#define CDDTRG_LLD_TMU_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CDDTRG_LLD_TMU_AR_RELEASE_MINOR_VERSION_C       (4)
#define CDDTRG_LLD_TMU_AR_RELEASE_REVISION_VERSION_C    (0)
#define CDDTRG_LLD_TMU_SW_MAJOR_VERSION_C               (1)
#define CDDTRG_LLD_TMU_SW_MINOR_VERSION_C               (0)
#define CDDTRG_LLD_TMU_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CDDTRG_LLD_TMU header file are of the same vendor */
#if (CDDTRG_LLD_TMU_VENDOR_ID_C != CDDTRG_LLD_TMU_VENDOR_ID)
#error "CddTrg_Lld_Tmu.c and CddTrg_Lld_Tmu.h have different vendor ids"
#endif

/* Check if source file and CDDTRG_LLD_TMU header file are of the same Autosar version */
#if (( CDDTRG_LLD_TMU_AR_RELEASE_MAJOR_VERSION_C != CDDTRG_LLD_TMU_AR_RELEASE_MAJOR_VERSION) || \
      ( CDDTRG_LLD_TMU_AR_RELEASE_MINOR_VERSION_C != CDDTRG_LLD_TMU_AR_RELEASE_MINOR_VERSION) || \
      ( CDDTRG_LLD_TMU_AR_RELEASE_REVISION_VERSION_C != CDDTRG_LLD_TMU_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CddTrg_Lld_Tmu.c and CddTrg_Lld_Tmu.h are different"
#endif

/* Check if source file and CDDTRG_LLD_TMU header file are of the same Software version */
#if (( CDDTRG_LLD_TMU_SW_MAJOR_VERSION_C != CDDTRG_LLD_TMU_SW_MAJOR_VERSION) || \
      ( CDDTRG_LLD_TMU_SW_MINOR_VERSION_C != CDDTRG_LLD_TMU_SW_MINOR_VERSION) || \
      ( CDDTRG_LLD_TMU_SW_PATCH_VERSION_C != CDDTRG_LLD_TMU_SW_PATCH_VERSION))
#error "Software Version Numbers of CddTrg_Lld_Tmu.c and CddTrg_Lld_Tmu.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
#define CDDTRG_TMU_REG_ADDR32_READ(address) (*(volatile uint32 *)(address))
#define CDDTRG_TMU_REG_ADDR32_WRITE(address, value) ((*(volatile uint32 *)(address)) = (value))
#define CDDTRG_TMU_REG_ADDR32_AEARWRITE(address, mask, value) \
    (CDDTRG_TMU_REG_ADDR32_WRITE((address),                   \
                              ((CDDTRG_TMU_REG_ADDR32_READ(address) & ((uint32) ~(mask))) | (value))))

#define CDDTRG_TMU_REG_ADDR32_SET_BITS(address, mask) ((*(volatile uint32 *)(address)) |= (mask))
#define CDDTRG_TMU_REG_ADDR32_CLEAR_BITS(address, mask) ((*(volatile uint32 *)(address)) &= (~(mask)))
#define CDDTRG_TMU_REG_ADDR32_INDEX_SET_BIT(address, bitindex) ((*(volatile uint32 *)(address)) |= ((uint32)1 << (bitindex)))
#define CDDTRG_TMU_REG_ADDR32_INDEX_CLEAR_BIT(address, bitindex) ((*(volatile uint32 *)(address)) &= (~((uint32)1 << (bitindex))))

#define CDDTRG_TMU_REG_ADDR32_GET_BITS(address, mask) ((*(volatile uint32 *)(address)) & (mask))
#define CDDTRG_TMU_REG_ADDR32_INDEX_GET_BIT(address, bitindex) (((*(volatile uint32 *)(address)) >> (bitindex)) & ((uint32)1U))

/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
#define CDDTRG_START_SEC_CODE
#include "CddTrg_MemMap.h"
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/**
 * MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that
 * of the left-hand operand.This line of code has been maintained for the sake of code clarity and
 * readability.
 */
/* PRQA S 2985, 2986 EOF */
/**
 * @brief set Tmu trigger source for target
 *
 * @param TiggerTarget  The target channel
 * @param TiggerSource  The source channel
 * @return void
 */
CDDTRG_FUNC LOCAL_INLINE void CddTrg_Lld_TmuSetTrigSrcForTgt(CddTrg_TmuTriggerTargetType TiggerTarget, CddTrg_TmuTriggerSourceType TiggerSource)
{
    uint8 TmuMuxRegOffset = 0U;
    uint8 TmuMuxRegIndex = 0U;

    TmuMuxRegIndex = (uint8)TiggerTarget >> 0x02U;
    TmuMuxRegOffset = (uint8)TiggerTarget & 0x3U;
    /* Set MUX register */
    CDDTRG_TMU_REG_ADDR32_AEARWRITE((TMU_BASE_ADDR32 + TMU_MUX_OFFSET32((uint32)TmuMuxRegIndex)),
                                    ((uint32)TMU_MUX_SEL0_MASK << (TmuMuxRegOffset << 3U)),
                                    (((uint32)TMU_MUX_SEL0(TiggerSource)) << (TmuMuxRegOffset << 3U)));
}
/**
* @brief   get Tmu trigger source for target
*
*/
CDDTRG_FUNC LOCAL_INLINE CddTrg_TmuTriggerSourceType CddTrg_Lld_TmuGetTrigSrcForTgt(CddTrg_TmuTriggerTargetType TiggerTarget)
{
    CddTrg_TmuTriggerSourceType RetVal ;
    uint32 Temp = 0U;
    uint8 TmuMuxRegOffset = 0U;
    uint8 TmuMuxRegIndex = 0U;

    TmuMuxRegIndex = (uint8)TiggerTarget >> 0x02U;
    TmuMuxRegOffset = (uint8)(((uint8)TiggerTarget & 0x3U) << 0x3U);

    Temp = CDDTRG_TMU_REG_ADDR32_READ(TMU_BASE_ADDR32 + TMU_MUX_OFFSET32((uint32)TmuMuxRegIndex));
    /**
    * MR12 Rule 10.8: The value of a composite expression shall not be cast to a different essential type category or a wider essential type.
    * This line is reduced by bit and operation and the number of effective data bits.
    */
    RetVal = (CddTrg_TmuTriggerSourceType)((uint8)((Temp >> TmuMuxRegOffset) & TMU_MUX_SEL0_MASK));/* PRQA S 4934*/

    return RetVal;
}
#if 0
/**
 * @brief   set lock bit for target
 *
 * @param TiggerTarget  the target channel
 * @param LockBitSet    TRUE: set lock bit,
 *                      FALSE: clear lock bit
 * @return void
 */
CDDTRG_FUNC LOCAL_INLINE void CddTrg_Lld_TmuSetLockReg(CddTrg_TmuTriggerTargetType TiggerTarget, boolean LockBitSet)
{
    uint8 TmuMuxRegIndex = 0U;
    TmuMuxRegIndex = (uint8)TiggerTarget >> 0x02U;

    if (TRUE == LockBitSet)
    {
        CDDTRG_TMU_REG_ADDR32_SET_BITS(TMU_BASE_ADDR32 + TMU_MUX_OFFSET32(TmuMuxRegIndex), TMU_MUX_LOCK_MASK);
    }
    else
    {
        CDDTRG_TMU_REG_ADDR32_CLEAR_BITS(TMU_BASE_ADDR32 + TMU_MUX_OFFSET32(TmuMuxRegIndex), TMU_MUX_LOCK_MASK);
    }
}
#endif
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
/************************************************************************
 *                       CddTrg_Lld_Tmu_Init
 ***********************************************************************/
CDDTRG_FUNC void CddTrg_Lld_Tmu_Init(CddTrg_Lld_Tmu_ConfigType const *TmuCfgPtr)
{
    uint8 LoopCnt;
    /*Disable all target and unlock all channel*/
    for (LoopCnt = 0; LoopCnt < (CDDTRG_TMU_LINKER_TOTAL_COUNT >> 2U); ++LoopCnt)
    {
        CDDTRG_TMU_REG_ADDR32_WRITE(TMU_BASE_ADDR32 + TMU_MUX_OFFSET32((uint32)LoopCnt), 0x00000000U);
    }

    /*Init all configured channels*/
    for (LoopCnt = 0; LoopCnt < TmuCfgPtr->LinkerCount; ++LoopCnt)
    {
        CddTrg_Lld_TmuSetTrigSrcForTgt(TmuCfgPtr->LinkerPtr[LoopCnt].TiggerTarget, TmuCfgPtr->LinkerPtr[LoopCnt].TiggerSource);
    }
}
/************************************************************************
 *                       CddTrg_Lld_Tmu_DeInit
 ***********************************************************************/
CDDTRG_FUNC void CddTrg_Lld_Tmu_DeInit(void)
{
    /*Disable all target and unlock all channel*/
    for (uint8 LoopCnt = 0; LoopCnt < (CDDTRG_TMU_LINKER_TOTAL_COUNT >> 2U); ++LoopCnt)
    {
        CDDTRG_TMU_REG_ADDR32_WRITE(TMU_BASE_ADDR32 + TMU_MUX_OFFSET32((uint32)LoopCnt), 0x00000000U);
    }
}
/************************************************************************
 *                       CddTrg_Lld_Tmu_SetTriggerSource
 ***********************************************************************/
CDDTRG_FUNC void CddTrg_Lld_Tmu_SetTriggerSource(CddTrg_TmuTriggerTargetType TiggerTarget, CddTrg_TmuTriggerSourceType TiggerSource)
{
    SchM_Enter_CddTrg_CDDTRG_EXCLUSIVE_AREA_03();
    CddTrg_Lld_TmuSetTrigSrcForTgt(TiggerTarget, TiggerSource);
    SchM_Exit_CddTrg_CDDTRG_EXCLUSIVE_AREA_03();
}

/************************************************************************
 *                       CddTrg_Lld_Tmu_GetTriggerSource
 ***********************************************************************/
CDDTRG_FUNC CddTrg_TmuTriggerSourceType CddTrg_Lld_Tmu_GetTriggerSource(CddTrg_TmuTriggerTargetType TiggerTarget)
{
    CddTrg_TmuTriggerSourceType RetVal = TMU_TRIG_SOURCE_DISABLED;
    RetVal = CddTrg_Lld_TmuGetTrigSrcForTgt(TiggerTarget);
    return RetVal;
}


#define CDDTRG_STOP_SEC_CODE
#include "CddTrg_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file CddTrg_Lld_Tmu.c */

