/**
 * @file    CddTrg.c
==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : TMU
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V1.0.0.HF001
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif
/*
 * PRQA S 1504 Rule 8.7: Functions and objects should not be defined with external linkage
 *                       if they are referenced in only one translation unit.
 *
 * PRQA S 2877 Dir 4.1: Run-time failures shall be minimized.
 */
/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "CddTrg.h"
#if defined (CDDTRG_TMU_ENABLE)
#include "CddTrg_Lld_Tmu.h"
#endif
#if defined (CDDTRG_PTU_ENABLE)
#include "CddTrg_Lld_Ptu.h"
#endif
#include "OsIf.h"
#include "Det.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CDDTRG_VENDOR_ID_C                      (180)
#define CDDTRG_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CDDTRG_AR_RELEASE_MINOR_VERSION_C       (4)
#define CDDTRG_AR_RELEASE_REVISION_VERSION_C    (0)
#define CDDTRG_SW_MAJOR_VERSION_C               (1)
#define CDDTRG_SW_MINOR_VERSION_C               (0)
#define CDDTRG_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CDDTRG header file are of the same vendor */
#if (CDDTRG_VENDOR_ID_C != CDDTRG_VENDOR_ID)
#error "CddTrg.c and CddTrg.h have different vendor ids"
#endif

/* Check if source file and CDDTRG header file are of the same Autosar version */
#if (( CDDTRG_AR_RELEASE_MAJOR_VERSION_C != CDDTRG_AR_RELEASE_MAJOR_VERSION) || \
      ( CDDTRG_AR_RELEASE_MINOR_VERSION_C != CDDTRG_AR_RELEASE_MINOR_VERSION) || \
      ( CDDTRG_AR_RELEASE_REVISION_VERSION_C != CDDTRG_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CddTrg.c and CddTrg.h are different"
#endif

/* Check if source file and CDDTRG header file are of the same Software version */
#if (( CDDTRG_SW_MAJOR_VERSION_C != CDDTRG_SW_MAJOR_VERSION) || \
      ( CDDTRG_SW_MINOR_VERSION_C != CDDTRG_SW_MINOR_VERSION) || \
      ( CDDTRG_SW_PATCH_VERSION_C != CDDTRG_SW_PATCH_VERSION))
#error "Software Version Numbers of CddTrg.c and CddTrg.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/


/*==================================================================================================
 *                                          VARIATES
==================================================================================================*/
#if (CDDTRG_FIXED_PB_CONFIG == STD_ON)
extern const CddTrg_ConfigType CddTrg_PreCompileConfig;
#endif
#define CDDTRG_START_SEC_VAR_INIT_32
#include "CddTrg_MemMap.h"
/*
 * MR12 RULE 8.7 VIOLATION: CddTrg_ConfigPtr is used in integation tests too, thus couldn't adhere to M3CM Rule-8.7
 */
CDDTRG_VAR CddTrg_ConfigType const *CddTrg_ConfigPtr = NULL_PTR; /*PRQA S 1504*/
#define CDDTRG_STOP_SEC_VAR_INIT_32
#include "CddTrg_MemMap.h"

#if defined (CDDTRG_PTU_ENABLE)
#define CDDTRG_START_SEC_VAR_INIT_8
#include "CddTrg_MemMap.h"
/*
 * MR12 RULE 8.7 VIOLATION: CddTrg_Ptu_RunningStatus is used in integation tests too, thus couldn't adhere to M3CM Rule-8.7
 */
CDDTRG_VAR CddTrg_Ptu_RunningStatusType CddTrg_Ptu_RunningStatus[CDDTRG_PTU_CONFIG_CHANNEL];/*PRQA S 1504*/
#define CDDTRG_STOP_SEC_VAR_INIT_8
#include "CddTrg_MemMap.h"
#endif

#define CDDTRG_START_SEC_CODE
#include "CddTrg_MemMap.h"
/*==================================================================================================
 *                                   LOCAL FUNCTION DECLARATION
==================================================================================================*/
/*==================================================================================================
                                        GLOBAL FUNCTIONS
==================================================================================================*/
/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_Init
* Description   : CddTrg module initialization.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_Init(CddTrg_ConfigType const *ConfigPtr)
{

    if (CddTrg_ConfigPtr != NULL_PTR)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_INIT, CDDTRG_E_ALREADY_INITIALIZED);
#endif /* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
#if (CDDTRG_FIXED_PB_CONFIG == STD_ON)
        if (NULL_PTR != ConfigPtr)
        {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                  CDDTRG_SID_INIT, CDDTRG_E_PARAM_POINTER);
#endif /* CDDTRG_DEV_ERROR_DETECT == STD_ON */
        }
#else
        if (NULL_PTR == ConfigPtr)
        {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                  CDDTRG_SID_INIT, CDDTRG_E_PARAM_POINTER);
#endif /* CDDTRG_DEV_ERROR_DETECT == STD_ON */
        }
#endif
        else
        {
            CddTrg_ConfigType const *TempPtr;
            Std_ReturnType TempRet = E_OK;
#if (CDDTRG_FIXED_PB_CONFIG == STD_ON)
            TempPtr = &CddTrg_PreCompileConfig;
#else
            TempPtr = ConfigPtr;
#endif
            /*Check if the TMU and PTU are configured, when enabled*/
#if defined (CDDTRG_TMU_ENABLE)
            if (NULL_PTR == TempPtr->TmuCfgPtr)
            {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
                (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                      CDDTRG_SID_INIT, CDDTRG_E_PARAM_POINTER);
#endif /* CDDTRG_DEV_ERROR_DETECT == STD_ON */
                TempRet = E_NOT_OK;
            }
#endif /* CDDTRG_TMU_ENABLE */
#if defined (CDDTRG_PTU_ENABLE)
            if (NULL_PTR == TempPtr->PtuCfgPtr)
            {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
                (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                      CDDTRG_SID_INIT, CDDTRG_E_PARAM_POINTER);
#endif /* CDDTRG_DEV_ERROR_DETECT == STD_ON */
                TempRet = E_NOT_OK;
            }
#endif /* CDDTRG_PTU_ENABLE */
            if (E_OK == TempRet)
            {
                CddTrg_ConfigPtr = TempPtr;

#if defined (CDDTRG_TMU_ENABLE)
#ifdef CDDTRG_USER_MODE_SUPPORT
                OsIf_Trusted_Call1param(CddTrg_Lld_Tmu_Init, CddTrg_ConfigPtr->TmuCfgPtr);
#else

                CddTrg_Lld_Tmu_Init(CddTrg_ConfigPtr->TmuCfgPtr);
#endif
#endif
#if defined (CDDTRG_PTU_ENABLE)
                /*Init PTU*/
#ifdef CDDTRG_USER_MODE_SUPPORT
                OsIf_Trusted_Call1param(CddTrg_Lld_Ptu_Init, CddTrg_ConfigPtr->PtuCfgPtr);
#else
                CddTrg_Lld_Ptu_Init(CddTrg_ConfigPtr->PtuCfgPtr);
#endif
/*
 * MR12 Dir 4.1 VIOLATION: The micro CDDTRG_PTU_CONFIG_CHANNEL is generated according to user configuration,It is not
 * a fixed value for different applications,so there need to be a for loop.
 */
                for (uint8 LoopCnt = 0; LoopCnt < CDDTRG_PTU_CONFIG_CHANNEL; ++LoopCnt) /*PRQA S 2877*/
                {
                    CddTrg_Ptu_RunningStatus[LoopCnt] = CDDTRG_PTU_RUNNING;
                }
#endif
            }
        }
    }
}
/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_DeInit
* Description   : CddTrg module deinitialization.
*
* END**************************************************************************/
#if (CDDTRG_DE_INIT_API == STD_ON)
CDDTRG_FUNC void CddTrg_DeInit(void)
{

    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_DEINIT, CDDTRG_E_UNINIT);
#endif
    }
    else
    {
#ifdef CDDTRG_USER_MODE_SUPPORT
#if defined (CDDTRG_TMU_ENABLE)
        OsIf_Trusted_Call(CddTrg_Lld_Tmu_DeInit);
#endif
#if defined (CDDTRG_PTU_ENABLE)
        OsIf_Trusted_Call(CddTrg_Lld_Ptu_DeInit);
#endif
#else
#if defined (CDDTRG_TMU_ENABLE)
        CddTrg_Lld_Tmu_DeInit();
#endif
#if defined (CDDTRG_PTU_ENABLE)
        CddTrg_Lld_Ptu_DeInit();
/*
 * MR12 Dir 4.1 VIOLATION: The micro CDDTRG_PTU_CONFIG_CHANNEL is generated according to user configuration,It is not
 * a fixed value for different applications,so there need to be a for loop.
 */
        for (uint8 LoopCnt = 0; LoopCnt < CDDTRG_PTU_CONFIG_CHANNEL; ++LoopCnt) /*PRQA S 2877*/
        {
            CddTrg_Ptu_RunningStatus[LoopCnt] = CDDTRG_PTU_IDLE;
        }
#endif
#endif
        CddTrg_ConfigPtr = NULL_PTR;
    }
}
#endif /* (CDDTRG_DE_INIT_API == STD_ON) */
/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_SetTmuTriggerSource
* Description   : This function is used to set the trigger source of TMU.
*
* END**************************************************************************/
#if (CDDTRG_SET_TMU_TRIGGER_SOURCE_API == STD_ON)
CDDTRG_FUNC void CddTrg_SetTmuTriggerSource(CddTrg_Lld_Tmu_LinkerIdType TmuChannelIndex, CddTrg_TmuTriggerSourceType TiggerSource)
{

    /* Check Init state */
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_TRIGGER_SOURCE, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if (NULL_PTR == CddTrg_ConfigPtr->TmuCfgPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_TRIGGER_SOURCE, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* Check paramater : TmuChannelID */
    else if (TmuChannelIndex >= CddTrg_ConfigPtr->TmuCfgPtr->LinkerCount)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_TRIGGER_SOURCE, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* Check linker changeable */
    else if (FALSE == CddTrg_ConfigPtr->TmuCfgPtr->LinkerPtr[TmuChannelIndex].SourceChangeAble)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_TRIGGER_SOURCE, CDDTRG_E_LINKER_ID_UNCHANGEABLE);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(CddTrg_Lld_Tmu_SetTriggerSource, CddTrg_ConfigPtr->TmuCfgPtr->LinkerPtr[TmuChannelIndex].TiggerTarget, TiggerSource);
#else
        CddTrg_Lld_Tmu_SetTriggerSource(CddTrg_ConfigPtr->TmuCfgPtr->LinkerPtr[TmuChannelIndex].TiggerTarget, TiggerSource);
#endif
    }
}
#endif /* (CDDTRG_SET_TMU_TRIGGER_SOURCE_API == STD_ON) */
/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_GetTmuTriggerSource
* Description   : This function is used to get the trigger source of TMU.
*
* END**************************************************************************/
#if (CDDTRG_GET_TMU_TRIGGER_SOURCE_API == STD_ON)
CDDTRG_FUNC CddTrg_TmuTriggerSourceType CddTrg_GetTmuTriggerSource(CddTrg_Lld_Tmu_LinkerIdType TmuChannelIndex)
{
    CddTrg_TmuTriggerSourceType TiggerSource = TMU_TRIG_SOURCE_DISABLED;
    /* Check Init state */
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_GET_TRIGGER_SOURCE, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if (NULL_PTR == CddTrg_ConfigPtr->TmuCfgPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_GET_TRIGGER_SOURCE, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* Check paramater : TmuChannelID */
    else if (TmuChannelIndex > CddTrg_ConfigPtr->TmuCfgPtr->LinkerCount)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_GET_TRIGGER_SOURCE, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
#ifdef CDDTRG_USER_MODE_SUPPORT
        TiggerSource = OsIf_Trusted_Call_Return1param(CddTrg_Lld_Tmu_GetTriggerSource, \
                       CddTrg_ConfigPtr->TmuCfgPtr->LinkerPtr[TmuChannelIndex].TiggerTarget);
#else
        TiggerSource = CddTrg_Lld_Tmu_GetTriggerSource(CddTrg_ConfigPtr->TmuCfgPtr->LinkerPtr[TmuChannelIndex].TiggerTarget);
#endif
    }
    return TiggerSource;
}
#endif /* (CDDTRG_GET_TMU_TRIGGER_SOURCE_API == STD_ON) */

#if (CDDTRG_ENABLE_PTU_INSTANCE_API == STD_ON)

/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_EnablePtuInstance
* Description   : This function enables the specified PTU instance.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_EnablePtuInstance(uint8 PtuIntance)
{
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_ENABLE_PTU_INSTANCE, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if ((NULL_PTR == CddTrg_ConfigPtr->PtuCfgPtr)||(PtuIntance >= CddTrg_ConfigPtr->PtuCfgPtr->NumInstance))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_ENABLE_PTU_INSTANCE, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] != CDDTRG_PTU_IDLE)
    {
        (void)Det_ReportRuntimeError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                     CDDTRG_SID_ENABLE_PTU_INSTANCE, CDDTRG_E_WRONG_STATE);
    }
    else
    {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(CddTrg_Lld_Ptu_InsEnable, CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId);
#else
        CddTrg_Lld_Ptu_InsEnable(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId);
#endif
        CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] = CDDTRG_PTU_RUNNING;
    }
}

/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_EnablePtuInstance
* Description   : This function disables the specified PTU instance.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_DisablePtuInstance(uint8 PtuIntance)
{
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_DISABLE_PTU_INSTANCE, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if ((NULL_PTR == CddTrg_ConfigPtr->PtuCfgPtr)||(PtuIntance >= CddTrg_ConfigPtr->PtuCfgPtr->NumInstance))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_DISABLE_PTU_INSTANCE, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] != CDDTRG_PTU_RUNNING)
    {
        (void)Det_ReportRuntimeError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                     CDDTRG_SID_DISABLE_PTU_INSTANCE, CDDTRG_E_WRONG_STATE);
    }
    else
    {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(CddTrg_Lld_Ptu_InsDisable, CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId);
#else
        CddTrg_Lld_Ptu_InsDisable(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId);
#endif
        CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] = CDDTRG_PTU_IDLE;
    }
}
#endif /* (CDDTRG_ENABLE_PTU_INSTANCE_API == STD_ON) */

#if (CDDTRG_SET_PTU_SOFT_TRIGGER_API == STD_ON)
/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_SetPtuSoftTrigger
* Description   : This function sets a soft trigger for the specified PTU instance.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_SetPtuSoftTrigger(uint8 PtuIntance)
{
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_SOFT_TRIGGER, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if ((NULL_PTR == CddTrg_ConfigPtr->PtuCfgPtr)||(PtuIntance >= CddTrg_ConfigPtr->PtuCfgPtr->NumInstance))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_SOFT_TRIGGER, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].TriggerSource != PTU_SOFTWARE_TRIGGER)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_SOFT_TRIGGER, CDDTRG_E_WRONG_TRIGG_SRC);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] != CDDTRG_PTU_RUNNING)
    {
        (void)Det_ReportRuntimeError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                     CDDTRG_SID_SET_PTU_SOFT_TRIGGER, CDDTRG_E_WRONG_STATE);
    }
    else
    {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(CddTrg_Lld_Ptu_SwTrigger, CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId);
#else
        CddTrg_Lld_Ptu_SwTrigger(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId);
#endif
    }
}
#endif /* (CDDTRG_SET_PTU_SOFT_TRIGGER_API == STD_ON) */

#if (CDDTRG_SET_PTU_TIMER_API == STD_ON)
/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_SetPtuModTimer
* Description   : This function sets the MOD value for the specified PTU instance.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_SetPtuModTimer(uint8 PtuIntance, uint16 ModValue)
{
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_MOD_TIMER, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if ((NULL_PTR == CddTrg_ConfigPtr->PtuCfgPtr)||(PtuIntance >= CddTrg_ConfigPtr->PtuCfgPtr->NumInstance))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_MOD_TIMER, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] != CDDTRG_PTU_RUNNING)
    {
        (void)Det_ReportRuntimeError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                     CDDTRG_SID_SET_PTU_MOD_TIMER, CDDTRG_E_WRONG_STATE);
    }
    else {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(CddTrg_Lld_Ptu_SetModVaue, CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId, ModValue);
#else
        CddTrg_Lld_Ptu_SetModVaue(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId, ModValue);
#endif
    }
}

/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_SetPtuIntcDelayTimer
* Description   : This function sets the interrupt delay value for the 
*                 specified PTU instance.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_SetPtuIntcDelayTimer(uint8 PtuIntance, uint16 IntcDelayValue)
{
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_INTC_DELAY_TIMER, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if ((NULL_PTR == CddTrg_ConfigPtr->PtuCfgPtr)||(PtuIntance >= CddTrg_ConfigPtr->PtuCfgPtr->NumInstance))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_INTC_DELAY_TIMER, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] != CDDTRG_PTU_RUNNING)
    {
        (void)Det_ReportRuntimeError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                     CDDTRG_SID_SET_PTU_INTC_DELAY_TIMER, CDDTRG_E_WRONG_STATE);
    }
    else {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(CddTrg_Lld_Ptu_SetIntcDelayValue, CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId, IntcDelayValue);
#else
        CddTrg_Lld_Ptu_SetIntcDelayValue(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId, IntcDelayValue);
#endif
    }
}

/*FUNCTION**********************************************************************
*
* Function Name : CddTrg_SetPtuChannelDelayTimer
* Description   : This function sets a channel delay value for the specified
*                 PTU instance.
*
* END**************************************************************************/
CDDTRG_FUNC void CddTrg_SetPtuChannelDelayTimer(uint8 PtuIntance, uint8 Channel, uint16 ChannelDelayValue)
{
    if (NULL_PTR == CddTrg_ConfigPtr)
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_CHANNEL_DELAY_TIMER, CDDTRG_E_UNINIT);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    /* check Tmu supported or not*/
    else if ((NULL_PTR == CddTrg_ConfigPtr->PtuCfgPtr)||(PtuIntance >= CddTrg_ConfigPtr->PtuCfgPtr->NumInstance))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_CHANNEL_DELAY_TIMER, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if(0U==(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].ChanConfigs->EnableMask&((uint8)(1U<<Channel))))
    {
#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                              CDDTRG_SID_SET_PTU_CHANNEL_DELAY_TIMER, CDDTRG_E_PARAM_CHANNEL_ID);
#endif/* CDDTRG_DEV_ERROR_DETECT == STD_ON */
    }
    else if (CddTrg_Ptu_RunningStatus[CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId] != CDDTRG_PTU_RUNNING)
    {
        (void)Det_ReportRuntimeError(CDDTRG_MODULE_ID, CDDTRG_INSTANCE_ID,
                                     CDDTRG_SID_SET_PTU_CHANNEL_DELAY_TIMER, CDDTRG_E_WRONG_STATE);
    }
    else {
#ifdef CDDTRG_USER_MODE_SUPPORT
        OsIf_Trusted_Call3params(CddTrg_Lld_Ptu_SetChnDealyValue, CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId, Channel, ChannelDelayValue);
#else
        CddTrg_Lld_Ptu_SetChnDealyValue(CddTrg_ConfigPtr->PtuCfgPtr->InstanceConfigs[PtuIntance].InstanceId, Channel, ChannelDelayValue);
#endif
    }

}
#endif /* (CDDTRG_SET_PTU_TIMER_API == STD_ON) */

#define CDDTRG_STOP_SEC_CODE
#include "CddTrg_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file CddTrg.c */

