/**
 * @file    CddTrg.h
 * @version
 *
 * @brief   AUTOSAR Port module interface
 * @details API implementation for CDDTRG driver
 *
 * @addtogroup CDDTRG_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : TMU&PTU
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V1.0.0.HF001
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifndef CDDTRG_H
#define CDDTRG_H

#ifdef __cplusplus
extern "C" {
#endif
/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "CddTrg_Types.h"
#include "CddTrg_Cfg.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CDDTRG_VENDOR_ID                      (180)
#define CDDTRG_AR_RELEASE_MAJOR_VERSION       (4)
#define CDDTRG_AR_RELEASE_MINOR_VERSION       (4)
#define CDDTRG_AR_RELEASE_REVISION_VERSION    (0)
#define CDDTRG_SW_MAJOR_VERSION               (1)
#define CDDTRG_SW_MINOR_VERSION               (0)
#define CDDTRG_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/**
* @file     CddTrg.h
*/
/* Check if source file and DMA configuration header file are of the same vendor */
#if (CDDTRG_VENDOR_ID != CDDTRG_CFG_VENDOR_ID)
#error "CddTrg.h and CddTrg_Cfg.h have different vendor ids"
#endif

/* Check if source file and DMA configuration header file are of the same Autosar version */
#if ((CDDTRG_AR_RELEASE_MAJOR_VERSION != CDDTRG_CFG_AR_RELEASE_MAJOR_VERSION) || \
     (CDDTRG_AR_RELEASE_MINOR_VERSION != CDDTRG_CFG_AR_RELEASE_MINOR_VERSION) || \
     (CDDTRG_AR_RELEASE_REVISION_VERSION != CDDTRG_CFG_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CddTrg.h and CddTrg_Cfg.h are different"
#endif

/* Check if source file and DMA configuration header file are of the same software version */
#if ((CDDTRG_SW_MAJOR_VERSION != CDDTRG_CFG_SW_MAJOR_VERSION) || \
     (CDDTRG_SW_MINOR_VERSION != CDDTRG_CFG_SW_MINOR_VERSION) || \
     (CDDTRG_SW_PATCH_VERSION != CDDTRG_CFG_SW_PATCH_VERSION))
#error "Software Version Numbers of CddTrg.h and CddTrg_Cfg.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/** @brief CDDTRG MODULE INSTANCE ID */
#define CDDTRG_INSTANCE_ID ((uint8)0U)

/** @brief CDDTRG MODULE ID */
#define CDDTRG_MODULE_ID ((uint16)260U)

#if (CDDTRG_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup        CDDTRG_DEV_ERROR
 * @brief           PWM development errors
 * @{
 */
/** @brief API CddTrg_Init service called with wrong parameter. */
#define CDDTRG_E_INIT_FAILED ((uint8)0x10)
/** @brief API service used without module initialization. */
#define CDDTRG_E_UNINIT ((uint8)0x11)
/** @brief API service used with an invalid channel Identifier. */
#define CDDTRG_E_PARAM_CHANNEL_ID ((uint8)0x12)
/** @brief Usage of unauthorized PWM service on PWM channel configured a fixed period. */
#define CDDTRG_E_LINKER_ID_UNCHANGEABLE ((uint8)0x13)
/** @brief API CddTrg_Init service called while the PWM driver has already been initialised. */
#define CDDTRG_E_ALREADY_INITIALIZED ((uint8)0x14)
/** @brief API service is called using an invalid pointer (e.g. the pointer should not be NULL).  */
#define CDDTRG_E_PARAM_POINTER ((uint8)0x15)
/** @brief API CddTrg_SetPtuSoftTrigger is called with a PTU channel which is trigger by hardware.  */
#define CDDTRG_E_WRONG_TRIGG_SRC ((uint8)0x16)
/** @brief API is called when the channel is not running in proper state. */
#define CDDTRG_E_WRONG_STATE ((uint8)0x17)

/** @} */
#endif

/**
 * @defgroup        CDD_TRG_SID
 * @brief           Service ID number for all CddTrg driver services
 */
/** @{ */
/** @brief API Service ID for CddTrg_Init */
#define CDDTRG_SID_INIT (0x00U)
/** @brief API Service ID for CddTrg_DeInit */
#define CDDTRG_SID_DEINIT (0x01U)
/** @brief API Service ID for CddTrg_SetTmuTriggerSource */
#define CDDTRG_SID_SET_TRIGGER_SOURCE (0x02U)
/** @brief API Service ID for CddTrg_GetTmuTriggerSource */
#define CDDTRG_SID_GET_TRIGGER_SOURCE   (0x03U)
/** @brief API Service ID for CddTrg_EnablePtuInstance */
#define CDDTRG_SID_ENABLE_PTU_INSTANCE (0x04U)
/** @brief API Service ID for CddTrg_DisablePtuInstance */
#define CDDTRG_SID_DISABLE_PTU_INSTANCE (0x05U)
/** @brief API Service ID for CddTrg_SetPtuSoftTrigger */
#define CDDTRG_SID_SET_PTU_SOFT_TRIGGER (0x06U)
/** @brief API Service ID for CddTrg_SetPtuModTimer */
#define CDDTRG_SID_SET_PTU_MOD_TIMER (0x07U)
/** @brief API Service ID for CddTrg_SetPtuIntcDelayTimer */
#define CDDTRG_SID_SET_PTU_INTC_DELAY_TIMER (0x08U)
/** @brief API Service ID for CddTrg_SetPtuChannelDelayTimer */
#define CDDTRG_SID_SET_PTU_CHANNEL_DELAY_TIMER (0x09U)
/** @brief API Service ID for CddTrg_GetVersionInfo */
#define CDDTRG_SID_GET_VERSION_INFO (0x0AU)
/** @} */

/*==================================================================================================
 *                                             ENUMS
==================================================================================================*/

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/

/*==================================================================================================
 *                                            VARIATES
==================================================================================================*/
extern CddTrg_ConfigType const *CddTrg_ConfigPtr;
/*==================================================================================================
                                        GLOBAL FUNCTIONS
==================================================================================================*/

/**
* @brief            Initializes the CDDTRG driver.
* @details          This function initializes the CDDTRG driver.
* @param[in]        ConfigPtr Pointer to the configuration set.
* @return           none
*/
void CddTrg_Init(CddTrg_ConfigType const *ConfigPtr);

#if (CDDTRG_DE_INIT_API == STD_ON)
/**
* @brief            De-initializes the CDDTRG driver.
* @details          This function de-initializes the CDDTRG driver.
* @param[in]        none
* @return           none
*/
void CddTrg_DeInit(void);
#endif /* (CDDTRG_DE_INIT_API == STD_ON) */

#if (CDDTRG_SET_TMU_TRIGGER_SOURCE_API == STD_ON)
/**
* @brief            Sets the trigger source for the specified TMU channel.
* @details          This function sets the trigger source for the specified TMU channel.
* @param[in]        TmuChannelIndex The index of the TMU channel.
* @param[in]        TiggerSource The trigger source to be set.
* @return           none
*/
void CddTrg_SetTmuTriggerSource(CddTrg_Lld_Tmu_LinkerIdType TmuChannelIndex, CddTrg_TmuTriggerSourceType TiggerSource);
#endif /* (CDDTRG_SET_TMU_TRIGGER_SOURCE_API == STD_ON) */

#if (CDDTRG_GET_TMU_TRIGGER_SOURCE_API == STD_ON)
/**
* @brief            Gets the trigger source for the specified TMU channel.
* @details          This function gets the trigger source for the specified TMU channel.
* @param[in]        TmuChannelIndex The index of the TMU channel.
* @return           CddTrg_TmuTriggerSourceType
*/
CddTrg_TmuTriggerSourceType CddTrg_GetTmuTriggerSource(CddTrg_Lld_Tmu_LinkerIdType TmuChannelIndex);
#endif /* (CDDTRG_GET_TMU_TRIGGER_SOURCE_API == STD_ON) */

#if (CDDTRG_ENABLE_PTU_INSTANCE_API == STD_ON)
/**
 *  @brief            Enables the specified PTU instance.
 *  @details          This function enables the specified PTU instance.
 *  @param[in]        PtuIntance The PTU instance number.
 *  @return           none
 */
void CddTrg_EnablePtuInstance(uint8 PtuIntance);

/**
 *  @brief            Disables the specified PTU instance.
 *  @details          This function disables the specified PTU instance.
 *  @param[in]        PtuIntance The PTU instance number.
 *  @return           none
 */
void CddTrg_DisablePtuInstance(uint8 PtuIntance);
#endif /* (CDDTRG_ENABLE_PTU_INSTANCE_API == STD_ON) */

#if (CDDTRG_SET_PTU_SOFT_TRIGGER_API == STD_ON)
/**
 *  @brief            Sets a soft trigger for the specified PTU instance.
 *  @details          This function sets a soft trigger for the specified PTU instance.
 *  @param[in]        PtuIntance The PTU instance number.
 *  @return           none
 */
void CddTrg_SetPtuSoftTrigger(uint8 PtuIntance);
#endif /* (CDDTRG_SET_PTU_SOFT_TRIGGER_API == STD_ON) */

#if (CDDTRG_SET_PTU_TIMER_API == STD_ON)
/**
 *  @brief            Sets the mod timer value for the specified PTU instance.
 *  @details          This function sets the mod timer value for the specified PTU instance.
 *  @param[in]        PtuIntance The PTU instance number.
 *  @param[in]        ModValue The mod timer value to be set.
 *  @return           none
 */
void CddTrg_SetPtuModTimer(uint8 PtuIntance, uint16 ModValue);

/**
 *  @brief            Sets the interrupt delay timer value for the specified PTU instance.
 *  @details          This function sets the interrupt delay timer value for the specified PTU instance.
 *  @param[in]        PtuIntance The PTU instance number.
 *  @param[in]        IntcDelayValue The interruptdelay timer value to be set.
 *  @return           none
 */
void CddTrg_SetPtuIntcDelayTimer(uint8 PtuIntance, uint16 IntcDelayValue);

/**
 *  @brief            Sets the channel delay timer value for the specified PTU instance.
 *  @details          This function sets the channel delay timer value for the specified PTU instance.
 *  @param[in]        PtuIntance The PTU instance number.
 *  @param[in]        Channel The channel number.
 *  @param[in]        ChannelDelayValue The channel delay timer value to be set.
 *  @return           none
 */
void CddTrg_SetPtuChannelDelayTimer(uint8 PtuIntance, uint8 Channel, uint16 ChannelDelayValue);
#endif /* (CDDTRG_SET_PTU_TIMER_API == STD_ON) */

#if (((CDDTRG_VERSION_INFO_API == STD_ON) && (CDDTRG_DEV_ERROR_DETECT == STD_ON)))
/**
 * @brief            Returns the version information of this module.
 * @details
 *
 * @param[in]        none
 * @param[out]       versioninfo Pointer to where to store version information of this module
 *
 * @return           none
 */
#define CddTrg_GetVersionInfo(versioninfo)                                       \
    {                                                                             \
        /* Check for DET: CDDTRG_E_PARAM_POINTER */                              \
        if ((versioninfo) == NULL_PTR)                                            \
        {                                                                         \
            /* Report CDDTRG_E_PARAM_POINTER DET if service called with          \
               NULL_PTR                                                           \
            */                                                                    \
            Det_ReportError(                                                      \
                (uint16)CDDTRG_MODULE_ID,                                        \
                CDDTRG_INSTANCE_ID,                                              \
                CDDTRG_SID_GET_VERSION_INFO,                                     \
                CDDTRG_E_PARAM_POINTER);                                         \
        }                                                                         \
        else                                                                      \
        {                                                                         \
            /* Vendor ID information */                                           \
            ((Std_VersionInfoType *)(versioninfo))->vendorID = CDDTRG_VENDOR_ID; \
            /* Adc module ID information */                                       \
            ((Std_VersionInfoType *)(versioninfo))->moduleID = CDDTRG_MODULE_ID; \
            /* Adc module Software major version information */                   \
            ((Std_VersionInfoType *)(versioninfo))->sw_major_version =            \
                (uint8)CDDTRG_SW_MAJOR_VERSION;                                      \
            /* Adc module Software minor version information */                   \
            ((Std_VersionInfoType *)(versioninfo))->sw_minor_version =            \
                (uint8)CDDTRG_SW_MINOR_VERSION;                                      \
            /* Adc module Software patch version information */                   \
            ((Std_VersionInfoType *)(versioninfo))->sw_patch_version =            \
                (uint8)CDDTRG_SW_PATCH_VERSION;                                      \
        }                                                                         \
    }
#elif (((CDDTRG_VERSION_INFO_API == STD_ON) && (CDDTRG_DEV_ERROR_DETECT == STD_OFF)))
#define CddTrg_GetVersionInfo(versioninfo)                                   \
    {                                                                         \
        /* Vendor ID information */                                           \
        ((Std_VersionInfoType *)(versioninfo))->vendorID = CDDTRG_VENDOR_ID; \
        /* Adc module ID information */                                       \
        ((Std_VersionInfoType *)(versioninfo))->moduleID = CDDTRG_MODULE_ID; \
        /* Adc module Software major version information */                   \
        ((Std_VersionInfoType *)(versioninfo))->sw_major_version =            \
            (uint8)CDDTRG_SW_MAJOR_VERSION;                                      \
        /* Adc module Software minor version information */                   \
        ((Std_VersionInfoType *)(versioninfo))->sw_minor_version =            \
            (uint8)CDDTRG_SW_MINOR_VERSION;                                      \
        /* Adc module Software patch version information */                   \
        ((Std_VersionInfoType *)(versioninfo))->sw_patch_version =            \
            (uint8)CDDTRG_SW_PATCH_VERSION;                                      \
    }
#endif /* CDDTRG_VERSION_INFO_API == STD_ON && CDDTRG_DEV_ERROR_DETECT == STD_ON */

#ifdef __cplusplus
}
#endif

#endif /* End of file CddTrg.h */

