/**
* @file    StartupTest.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : StartupTest
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V1.1.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "OsIf.h"
#include "pSIP_Regfile.h"
#include "StartupTest.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define STARTUPTEST_VENDOR_ID_C                    (180)
#define STARTUPTEST_AR_REL_MAJOR_VER_C              (4)
#define STARTUPTEST_AR_REL_MINOR_VER_C              (4)
#define STARTUPTEST_AR_REL_REVISION_VER_C           (0)
#define STARTUPTEST_SW_MAJOR_VER_C                  (1)
#define STARTUPTEST_SW_MINOR_VER_C                  (1)
#define STARTUPTEST_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and STARTUPTEST header file are of the same vendor */
#if (STARTUPTEST_VENDOR_ID_C != STARTUPTEST_VENDOR_ID)
#error "StartupTest.c and StartupTest.h have different vendor ids"
#endif

/* Check if source file and STARTUPTEST header file are of the same Autosar version */
#if (( STARTUPTEST_AR_REL_MAJOR_VER_C != STARTUPTEST_AR_REL_MAJOR_VER) || \
      ( STARTUPTEST_AR_REL_MINOR_VER_C != STARTUPTEST_AR_REL_MINOR_VER) || \
      ( STARTUPTEST_AR_REL_REVISION_VER_C != STARTUPTEST_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of StartupTest.c and StartupTest.h are different"
#endif

/* Check if source file and STARTUPTEST header file are of the same Software version */
#if (( STARTUPTEST_SW_MAJOR_VER_C != STARTUPTEST_SW_MAJOR_VER) || \
      ( STARTUPTEST_SW_MINOR_VER_C != STARTUPTEST_SW_MINOR_VER) || \
      ( STARTUPTEST_SW_PATCH_VER_C != STARTUPTEST_SW_PATCH_VER))
#error "Software Version Numbers of StartupTest.c and StartupTest.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/

#define FLSTST_SIGNATURE_ADDR    0x00000000U
#define FLSTST_NUMBER_OF_SIGNATURE      10U
#define FLSTST_BGND_BLOCK_NUM           5U
#define FLSTST_FGND_BLOCK_NUM           3U

/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/

static void IntTst_StartupTest_Run(void);
static void CorTst_StartupTest_Run(void);
static void FlsTst_StartupTest_Run(void);
static void RamTst_StartupTest_Run(void);
static void DmaTst_StartupTest_Run(void);
static void EfmTst_StartupTest_Run(void);
static void EccTst_StartupTest_Run(void);
static void PpuTst_StartupTest_Run(void);
static void WdgTst_StartupTest_Run(void);
static void CrcTst_StartupTest_Run(void);
static void StkTst_StartupTest_Run(void);

/**
 * @brief   Run the IntTst startup test.
 * @details This function starts the IntTst test and checks the result,
 *          retrieves the test details if needed
 * @param   void
 * @return  void
 */
static void IntTst_StartupTest_Run(void)
{
    /* the array for INTM monitor channel test details record */
    IntTst_ChannelTestType IntTst_ChannelTest[INTTST_CHANNEL_INJECTION_TST_COUNT];
    /* the IntTst module test details */
    IntTst_TestDetailsType Test_Details = {INTTST_CHANNEL_INJECTION_TST_COUNT, IntTst_ChannelTest};
    /* start the IntTst module test */
    IntTst_Start();
    /* get and check the IntTst module test result */
    if (INTTST_RESULT_OK != IntTst_GetResult())
    {
        /* if the test result is not OK, get the test details */
        IntTst_GetDetailsResult(&Test_Details);
        while (TRUE)
        {
            /* IntTst result: IntTst not pass */
        }
    }
    /* IntTst result: Pass and get the test details if needed*/
    IntTst_GetDetailsResult(&Test_Details);
}

/**
 * @brief   Run the CorTst startup test.
 * @details This function runs all the foreground tests of the CorTst module,
 *          checks the results, and handles any errors.
 * @param   void
 * @return  void
 */
static void CorTst_StartupTest_Run(void)
{
    /* Run foreground tests. */
    for (CorTst_TestIdFgndType TestId = 0; TestId < CORTST_FGND_TEST_NUMBER; TestId++)
    {
        if (E_OK != CorTst_Start(TestId))
        {
            /* CorTst test execute failed */
            while (1)
            {
                /* CorTst test failed */
            }
        }
        if (CORTST_E_OKAY != CorTst_GetFgndResult())
        {
            /* CorTst test failed */
            while (1)
            {
                /* CorTst test failed */
            }
        }
    }
    /* CorTst result: Pass */
}

/**
 * @brief   Run the FlsTst startup test.
 * @details This function runs the FlsTst foreground tests, checks the results,
 *          and handles any errors.
 * @param   void
 * @return  void
 */
static void FlsTst_StartupTest_Run(void)
{
    FlsTst_TestResultFgndType TempResultFgnd = FLSTST_NOT_TESTED;
    FlsTst_TestSignatureFgndType TempSignatureFgnd;
    FlsTst_BlockIdFgndType FgndTestBlockId;
    for (FgndTestBlockId = 0; FgndTestBlockId < FLSTST_FGND_BLOCK_NUM; FgndTestBlockId++)
    {
        FlsTst_StartFgnd(FgndTestBlockId);
        TempResultFgnd = FlsTst_GetTestResultFgnd();
        TempSignatureFgnd = FlsTst_GetTestSignatureFgnd();
        if ((TempSignatureFgnd.FgndTestBlockId != FgndTestBlockId) || \
                (TempResultFgnd != FLSTST_OK))
        {
            FlsTst_DemoErrorDetails = FlsTst_GetErrorDetails();
            while (1)
            {
                /* FlsTst test failed */
            }
        }
    }
}

/**
 * @brief   Run the RamTst startup test.
 * @details This function runs the full RAM test with different algorithms,
 *          checks the results, and handles any errors.
 * @param   void
 * @return  void
 */
static void RamTst_StartupTest_Run(void)
{
    /*Full Test RAM with default RAM test algorithm of RAMTST_CHECKERBOARD_TEST*/
    RamTst_RunFullTest();
    RamTst_SelectAlgParams(RamTstConf_RamTstAlgParamsId_2);
    /*Full Test RAM with RAMTST_MARCH_TEST algorithm*/
    RamTst_RunFullTest();
    RamTst_SelectAlgParams(RamTstConf_RamTstAlgParamsId_3);
    /*Full Test RAM with RAMTST_WALK_PATH_TEST algorithm*/
    RamTst_RunFullTest();;
    RamTst_SelectAlgParams(RamTstConf_RamTstAlgParamsId_4);
    /*Full Test RAM with RAMTST_GALPAT_TEST algorithm*/
    RamTst_RunFullTest();
}

/**
 * @brief   Run the DmaTst startup test.
 * @details This function starts the DmaTst module test, checks the result,
 *          and retrieves the test details if needed.
 * @param   void
 * @return  void
 */
static void DmaTst_StartupTest_Run(void)
{
    /*This array to store detailed results.The array is of size +1 of the configured channels because Runtime Test is enabled*/
    DmaTst_GetErrorType Error[DMATST_CHANNEL_ERROR_TST_COUNT + DMATST_ENABLE_RUNTIMETEST];
    /*Launching Test*/
    DmaTst_Start();
    /*When the test is complete, get the test results*/
    if (DMATST_RESULT_OK != DmaTst_GetResult())
    {
        /* DMA test failed */
        DmaTst_GetDetailsResult(Error);
        while (1)
        {
            /* DMA test failed */
        }
    }
    /* DMA test passed and get the test details if needed */
    DmaTst_GetDetailsResult(Error);
}

/**
 * @brief   Run the EfmTst startup test.
 * @details This function initializes the Fls module, starts the EfmTst test,
 *          checks the result, and retrieves the test details if needed.
 * @param   void
 * @return  void
 */
static void EfmTst_StartupTest_Run(void)
{
    EfmTst_TestDetailsType EfmTst_TestDetails;
    /* Initialize the Fls module with the configuration */
    Fls_Init(&Fls_Config);
    /* Run the EfmTst startup test */
    EfmTst_Start();
    /* Check the EfmTst result */
    if (EFMTST_OK != EfmTst_GetResult())
    {
        /* If the test result is not OK, get the test details */
        EfmTst_GetDetailsResult(&EfmTst_TestDetails);
        /* EfmTst test failed */
        while (1)
        {
            /* EfmTst test failed */
        }
    }
    /* EfmTst result: Pass and get the test details if needed */
    EfmTst_GetDetailsResult(&EfmTst_TestDetails);
}

/**
 * @brief   Run the EccTst startup test.
 * @details This function starts the EccTst module test, checks the result,
 *          and retrieves the test details if needed.
 * @param   void
 * @return  void
 */
static void EccTst_StartupTest_Run(void)
{
    EccTst_TestDetailsType EccTst_TestDetails;
    /* Start the EccTst module test */
    EccTst_Start();
    /* Get and check the EccTst module test result */
    if (ECCTST_OK != EccTst_GetResult())
    {
        /* If the test result is not OK, get the test details */
        EccTst_GetDetailsResult(&EccTst_TestDetails);
        /* EccTst test failed */
        while (1)
        {
            /* EccTst test failed */
        }
    }
    /* EccTst result: Pass and get the test details if needed */
    EccTst_GetDetailsResult(&EccTst_TestDetails);
}

/**
 * @brief   Run the PpuTst startup test.
 * @details This function starts the PpuTst module test, checks the result,
 *          and retrieves the test details if needed.
 * @param   void
 * @return  void
 */
static void PpuTst_StartupTest_Run(void)
{
    PpuTst_TestDetailType PpuTstGetDetails[PPUTST_GROUP_COUNT];
    PpuTst_Init(NULL_PTR); /* Initialize the PpuTst module */
    /* Start the PpuTst module test */
    PpuTst_Start();
    /* Get and check the PpuTst module test result */
    if (PPUTST_RESULT_OK != PpuTst_GetTestResult())
    {
        /* If the test result is not OK, get the test details */
        PpuTst_GetDetailsResult(PpuTstGetDetails);
        /* PpuTst test failed */
        while (1)
        {
            /* PpuTst test failed */
        }
    }
    /* PpuTst result: Pass and get the test details if needed */
    PpuTst_GetDetailsResult(PpuTstGetDetails);
    /*check details result*/
    for (uint8 GroupIndex = 0; GroupIndex < PPUTST_GROUP_COUNT; ++GroupIndex)
    {
        if ((PPUTST_PATH_PASS != PpuTstGetDetails[GroupIndex].SLockPathErrorCheckState) ||
                (PPUTST_PATH_PASS != PpuTstGetDetails[GroupIndex].HLockPathErrorCheckState))
        {
            /* Error Notification for PpuTst */
            /* Add your error handling code here */
            while (1)
            {
                /* Infinite loop to indicate an error */
            }
        }
    }
}

/**
 * @brief   Run the WdgTst startup test.
 * @details This function starts the WdgTst module test, checks the result,
 *          and retrieves the test details if needed.
 * @param   void
 * @return  void
 */
static void WdgTst_StartupTest_Run(void)
{
    WdgTst_TestDetailsType ResultDetail;
    /* Start the WdgTst */
    if (E_OK != WdgTst_Start())
    {
        /* If the WdgTst start failed, handle the error */
        while (1)
        {
            /* WdgTst start failed */
        }
    }
    /* Get the test result of WdgTst */
    if (WDGTST_RESULT_OK != WdgTst_GetResult())
    {
        /* If the test result is not OK, get the test details */
        WdgTst_GetDetailsResult(&ResultDetail);
        /* WdgTst test failed */
        while (1)
        {
            /* WdgTst test failed */
        }
    }
    /* WdgTst result: Pass and get the test details if needed */
    WdgTst_GetDetailsResult(&ResultDetail);
    /* Check the details result */
    if ((WDGTST_RESULT_OK != ResultDetail.EwdgTestResult) ||
            (WDGTST_RESULT_OK != ResultDetail.WdgSoftLockTestResult) ||
            (WDGTST_RESULT_OK != ResultDetail.WdgTimeoutInterruptTestResult))
    {
        /* Error Notification for WdgTst */
        /* Add your error handling code here */
        while (1)
        {
            /* Infinite loop to indicate an error */
        }
    }
//    Sys_GoToSupervisor();
    REGFILE->DR[0]++; /* mark the WdgTst result  as valid */
//    Sys_GoToUser();
}

static void CrcTst_StartupTest_Run(void)
{
    CrcTst_CrcFuncTestDetailsType FgndTestDetails;
    /*run atomic test case under front-ground mode*/
    CrcTst_StartFgnd();
    /* Get the foreground test result */
    if (CRCTST_E_OKAY != CrcTst_GetFgndResult())
    {
        /* If the test result is not OK, get the test details */
        CrcTst_GetFgndDetailsResult(&FgndTestDetails);
        /* CrcTst test failed */
        while (1)
        {
            /* Infinite loop to indicate an error */
        }
    }
    /* CrcTst result: Pass and get the test details if needed */
    CrcTst_GetFgndDetailsResult(&FgndTestDetails);
}

static void StkTst_StartupTest_Run(void)
{
    /* Run the stack test via invoking the StkTst_MainFunction */
    StkTst_MainFunction();
    /* Check the stack overflow result. */
    if (STKTST_RESULT_OK != StkTst_GetResult())
    {
        /* If the test result is not OK, handle the error */
        while (1)
        {
            /* Infinite loop to indicate an error */
        }
    }
}

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

/**
 * @brief   Run the ClkTst startup test.
 * @details This function starts the ClkTst test and checks the result,
 *          retrieves the test details if needed
 * @param   void
 * @return  void
 */
void ClkTst_StartupTest(void)
{
    /* arrary to store the test details */
    ClkTst_ChannelResultType ClkTst_TestDetails[CLKTST_CMU_CHANNEL_NUMS];
    /* start the ClkTst */
    ClkTst_Init(NULL_PTR);
    ClkTst_Start();
    /* ClkTst result: Pass and get the test details if needed*/
    ClkTst_GetDetailsResult(ClkTst_TestDetails);
}

/**
 * @brief   Initialize the SafLib modules for startup tests.
 * @details This function initializes all the SafLib modules that are required for startup tests.
 *          It sets up the modules with their respective configurations.
 * @param   void
 * @note    The PpuTst_Init() must not be called here, as it will enable the register protection
 *          of some system modules, such as the WDG and IPC modules.
 * @return  void
 */
void SafLib_StartupTest_Init(void)
{
    FlsTst_Init(&FlsTst_Config);
    EfmTst_Init(NULL_PTR);
    RamTst_Init(NULL_PTR);
    StkTst_Init(NULL_PTR);
    WdgTst_Init(NULL_PTR);
    IntTst_Init(NULL_PTR);
    DmaTst_Init(NULL_PTR);
    EccTst_Init(NULL_PTR);
    CorTst_Init(NULL_PTR);
    CrcTst_Init(NULL_PTR);
}

/**
 * @brief   Run the SafLib startup tests.
 * @details This function runs all the startup tests for the SafLib modules.
 *          It initializes the modules and checks their results.
 * @param   void
 * @return  void
 * @note    CorTst -> StkTst -> ... ->
 */
void SafLib_StartupTest_Run(void)
{
    CorTst_StartupTest_Run();
    StkTst_StartupTest_Run();
    IntTst_StartupTest_Run();
    CrcTst_StartupTest_Run();
    FlsTst_StartupTest_Run();
    RamTst_StartupTest_Run();
    DmaTst_StartupTest_Run();
    EccTst_StartupTest_Run();
    EfmTst_StartupTest_Run();
    PpuTst_StartupTest_Run();
    WdgTst_StartupTest_Run();
    /* Add other startup tests here if needed */
    while (1)
    {
        /* wait for WDG timeout reset to finish the startup test */
    }
}

/**
 * @brief   De-initialize the SafLib modules.
 * @details This function de-initializes the SafLib modules that were initialized in SafLib_StartupTest_Init.
 *          It is called to clean up resources after the tests are completed.
 * @param   void
 * @return  void
 */
void SafLib_StartupTest_DeInit(void)
{
    /* De-initialize the SafLib modules if needed */
    DmaTst_DeInit();
    FlsTst_DeInit();
    PpuTst_DeInit();
    RamTst_DeInit();
    IntTst_DeInit();
    DmaTst_DeInit();
    EccTst_DeInit();
    CorTst_DeInit();
}

#ifdef __cplusplus
}
#endif

/* End of file StartupTest.c */
