/**
* @file    RuntimeTest.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : RuntimeTest
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V1.1.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "RuntimeTest.h"
#include "Devassert.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define RUNTIMETEST_VENDOR_ID_C                     (180)
#define RUNTIMETEST_AR_REL_MAJOR_VER_C              (4)
#define RUNTIMETEST_AR_REL_MINOR_VER_C              (4)
#define RUNTIMETEST_AR_REL_REVISION_VER_C           (0)
#define RUNTIMETEST_SW_MAJOR_VER_C                  (1)
#define RUNTIMETEST_SW_MINOR_VER_C                  (1)
#define RUNTIMETEST_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and RUNTIMETEST header file are of the same vendor */
#if (RUNTIMETEST_VENDOR_ID_C != RUNTIMETEST_VENDOR_ID)
#error "RuntimeTest.c and RuntimeTest.h have different vendor ids"
#endif

/* Check if source file and RUNTIMETEST header file are of the same Autosar version */
#if (( RUNTIMETEST_AR_REL_MAJOR_VER_C != RUNTIMETEST_AR_REL_MAJOR_VER) || \
      ( RUNTIMETEST_AR_REL_MINOR_VER_C != RUNTIMETEST_AR_REL_MINOR_VER) || \
      ( RUNTIMETEST_AR_REL_REVISION_VER_C != RUNTIMETEST_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of RuntimeTest.c and RuntimeTest.h are different"
#endif

/* Check if source file and RUNTIMETEST header file are of the same Software version */
#if (( RUNTIMETEST_SW_MAJOR_VER_C != RUNTIMETEST_SW_MAJOR_VER) || \
      ( RUNTIMETEST_SW_MINOR_VER_C != RUNTIMETEST_SW_MINOR_VER) || \
      ( RUNTIMETEST_SW_PATCH_VER_C != RUNTIMETEST_SW_PATCH_VER))
#error "Software Version Numbers of RuntimeTest.c and RuntimeTest.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/

/* Variable to keep track of the background RAM test algorithm parameter set */
static  uint8 BackgroundParamId = 1U;

static CorTst_ErrOkType CorTst_RuntimeTest_Result = {0U, CORTST_E_NOT_TESTED};

uint8 CrcTst_BgndComplete = 0; /* it will be set when all the background test items complete */

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/

static void RamTst_RuntimeTest_Run(void);
static void FlsTst_RuntimeTest_Run(void);
static void CorTst_RuntimeTest_Run(void);
static void StkTst_RuntimeTest_Run(void);
static void CrcTst_RuntimeTest_Run(void);
static void DmaTst_RuntimeTest_Run(void);

void TestCRC_Complete(void)
{
    CrcTst_BgndComplete = 1;
}

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

/**
 * @brief    Initialize the SafLib modules for runtime tests.
 * @details  This function initializes the SafLib modules that are required for runtime tests.
 * @param    void
 * @return   void
 * @note     This function should be called before running any runtime tests.
 */
void RuntimeTest_Init(void)
{
    /* Initialize the SafLib modules which support the runtime test  */
    RamTst_Init(NULL_PTR);
    RamTst_Allow();
    CorTst_Init(NULL_PTR);
    FlsTst_Init(&FlsTst_Config);
    CrcTst_Init(NULL_PTR);
    CrcTst_BgndComplete = 0; /* clear the test complete flag */
#if (CRCTST_SIGINFLASH_SUPPORT == STD_OFF)
    /*Get CRC Signature*/
    (void) CrcTst_GetRegisterSignature();
#endif
    CrcTst_StartBgnd(); /* start the CrcTst background test */
    StkTst_Init(NULL_PTR);
    DmaTst_Init(NULL_PTR);
    /* Add any necessary initialization code here */
}

/**
 * @brief    Run the SafLib runtime tests.
 * @details  This function runs all the runtime tests of the SafLib modules.
 *           It call the SafLib modules runtime test mainfunction and checks their results.
 * @param    void
 * @return   void
 */
void RuntimeTest_Run(void)
{
    RamTst_RuntimeTest_Run();
    FlsTst_RuntimeTest_Run();
    CorTst_RuntimeTest_Run();
    StkTst_RuntimeTest_Run();
    CrcTst_RuntimeTest_Run();
    DmaTst_RuntimeTest_Run();
}

/*
* @brief    Run the background RAM test.
* @details  This function runs the background RAM test via invoking the RamTst_MainFunction.
* @param    void
* @return   void
*/
static void RamTst_RuntimeTest_Run(void)
{
    if (TRUE == RamTstBackGroundCompleted)
    {
        /* Background RAM test completed, reset the flag */
        RamTstBackGroundCompleted = FALSE;
        if (BackgroundParamId > 4)
        {
            BackgroundParamId = 1U; /* Reset to the first algorithm parameter set */
        }
        RamTst_Stop();
        /* Select the next RAM test algorithm parameter set */
        RamTst_SelectAlgParams(BackgroundParamId);
        /* Allow the RAM test to continue */
        RamTst_Allow();
        ++BackgroundParamId;
    }
    /* Call the RamTst_MainFunction to continue the background RAM test */
    RamTst_MainFunction();
}

static void FlsTst_RuntimeTest_Run(void)
{
    /*FlsTst bgnd test*/
    FlsTst_MainFunction();
    if (FlsTst_DemoAbortEn == TRUE)
    {
        FlsTst_Abort();
        FlsTst_DemoAbortEn = FALSE;
    }
    if (FlsTst_DemoSuspendEn == TRUE)
    {
        FlsTst_Suspend();
        FlsTst_DemoSuspendEn = FALSE;
    }
    if (FlsTst_DemoResumeEn == TRUE)
    {
        FlsTst_Resume();
        FlsTst_DemoResumeEn = FALSE;
    }
}

static void CorTst_RuntimeTest_Run(void)
{
    /* Run the core test via invoking the CorTst_MainFunction */
    CorTst_MainFunction();
    /* Get the core test result */
    CorTst_GetCurrentStatus(&CorTst_RuntimeTest_Result);
    /* Check the core test result */
    DevAssert(CORTST_E_NOT_OK != CorTst_RuntimeTest_Result.Result);
}

static void StkTst_RuntimeTest_Run(void)
{
    /* Run the stack test via invoking the StkTst_MainFunction */
    StkTst_MainFunction();
    /* Check the stack overflow result. */
    DevAssert(StkTst_GetResult() == STKTST_RESULT_OK);
}

static void CrcTst_RuntimeTest_Run(void)
{
    CrcTst_TestResultType CrcTstCase_Result;
    CrcTst_TestResultType BgndTestDetails[2];
    /*run atomic test case under background mode*/
    CrcTst_MainFunction();
    /* Get the foreground test result *//* check whether all the test items are completed and verify the test result */
    if (1 == CrcTst_BgndComplete)
    {
        /* Get the background test result */
        CrcTstCase_Result = CrcTst_GetBgndResult();
        /* Check the result. */
        DevAssert(CRCTST_E_OKAY == CrcTstCase_Result);
        /* Get the background test details */
        CrcTst_GetBgndDetailsResult(BgndTestDetails);
        CrcTst_BgndComplete = 0; /* clear the test complete flag */
    }
}

static void DmaTst_RuntimeTest_Run(void)
{
    /*This array to store detailed results.The array is of size +1 of the configured channels because Runtime Test is enabled*/
    DmaTst_GetErrorType Error[DMATST_CHANNEL_ERROR_TST_COUNT + DMATST_ENABLE_RUNTIMETEST];
    /* Launch the DMA runtime test */
    DmaTst_MainFunction();
    /* Get and check the DMA test result */
    DevAssert(DMATST_RESULT_OK == DmaTst_GetResult());
    /* Get the detailed results of the DMA test */
    DmaTst_GetDetailsResult(Error);
}


#ifdef __cplusplus
}
#endif

/* End of file RuntimeTest.c */
