/**
 * @file    WdgTst.c
 * @version V2.0.0
 *
 * @brief   Safety library WdgTst module interface
 * @details API implementation for WdgTst driver
 *
 * @addtogroup WDGTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 SafLib
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : None
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 2986 Rule 2.2: There shall be no dead code. This assignment is redundant. The value of this object is never used before being modified.
 */

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "WdgTst.h"
#include "WdgTst_Lld.h"

#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#include "Platform.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define WDGTST_VENDOR_ID_C                      (180)
#define WDGTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define WDGTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define WDGTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define WDGTST_SW_MAJOR_VERSION_C               (2)
#define WDGTST_SW_MINOR_VERSION_C               (0)
#define WDGTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if WdgTst.c and WdgTst.h are of the same vendor */
#if (WDGTST_VENDOR_ID_C != WDGTST_VENDOR_ID)
#error "WdgTst.c and WdgTst.h have different vendor ids"
#endif

/* Check if WdgTst.c and WdgTst.h are of the same Autosar version */
#if ((WDGTST_AR_RELEASE_MAJOR_VERSION_C != WDGTST_AR_RELEASE_MAJOR_VERSION) || \
     (WDGTST_AR_RELEASE_MINOR_VERSION_C != WDGTST_AR_RELEASE_MINOR_VERSION) || \
     (WDGTST_AR_RELEASE_REVISION_VERSION_C != WDGTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of WdgTst.c and WdgTst.h are different"
#endif

/* Check if WdgTst.c and WdgTst.h are of the same Software version */
#if ((WDGTST_SW_MAJOR_VERSION_C != WDGTST_SW_MAJOR_VERSION) || \
     (WDGTST_SW_MINOR_VERSION_C != WDGTST_SW_MINOR_VERSION) || \
     (WDGTST_SW_PATCH_VERSION_C != WDGTST_SW_PATCH_VERSION))
#error "Software Version Numbers of WdgTst.c and WdgTst.h are different"
#endif

/* Check if WdgTst.c and WdgTst_Lld.h are of the same vendor */
#if (WDGTST_VENDOR_ID_C != WDGTST_LLD_VENDOR_ID)
#error "WdgTst.c and WdgTst_Lld.h have different vendor ids"
#endif

/* Check if WdgTst.c and WdgTst_Lld.h are of the same Autosar version */
#if ((WDGTST_AR_RELEASE_MAJOR_VERSION_C != WDGTST_LLD_AR_RELEASE_MAJOR_VERSION) || \
     (WDGTST_AR_RELEASE_MINOR_VERSION_C != WDGTST_LLD_AR_RELEASE_MINOR_VERSION) || \
     (WDGTST_AR_RELEASE_REVISION_VERSION_C != WDGTST_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of WdgTst.c and WdgTst_Lld.h are different"
#endif

/* Check if WdgTst.c and WdgTst_Lld.h are of the same Software version */
#if ((WDGTST_SW_MAJOR_VERSION_C != WDGTST_LLD_SW_MAJOR_VERSION) || \
     (WDGTST_SW_MINOR_VERSION_C != WDGTST_LLD_SW_MINOR_VERSION) || \
     (WDGTST_SW_PATCH_VERSION_C != WDGTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of WdgTst.c and WdgTst_Lld.h are different"
#endif

/*==================================================================================================
 *                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                       GLOBAL VARIABLES
==================================================================================================*/
#define WDGTST_START_SEC_VAR_INIT_8
#include "WdgTst_MemMap.h"

/**
 * @brief     WdgTst driver state
 * @private
 */
WDGTST_VAR static volatile WdgTst_StateType WdgTst_State = WDGTST_UNINIT;

#define WDGTST_STOP_SEC_VAR_INIT_8
#include "WdgTst_MemMap.h"

#define WDGTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "WdgTst_MemMap.h"
/**
 * @brief   Used to store the interrupt service function of the wdg before executing the wdgtst test
 * @private
 */
WDGTST_VAR static Platform_IrqHandlerType WdgTst_BackupIrqHandler = NULL_PTR;
/**
 * @brief   Used to pass parameters when install irq handler
 * @private
 */
WDGTST_VAR static Platform_IrqHandlerType WdgTst_DummyIrqHandler = NULL_PTR;
#define WDGTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "WdgTst_MemMap.h"

/*==================================================================================================
 *                                          ISR FUNCTIONS
==================================================================================================*/
#define WDGTST_START_SEC_CODE
#include "WdgTst_MemMap.h"

/**
 * @brief       Watchdog Test Interrupt Service Routine (ISR)
 * @details     This function serves as the ISR for handling watchdog-related interrupts.
 *              It calls the appropriate low-level driver (LLD) handlers based on WDG and EWDG modules.
 *              The function is conditionally compiled based on the enable status of
 *              the respective watchdog modules.
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_189
 */
WDGTST_FUNC static void WdgTst_Isr(void)
{
#if (WDGTST_WDG_ENABLE == STD_ON)
    WdgTst_Lld_WdgIrqHandler(WDG);
#endif
#if (WDGTST_EWDG_ENABLE == STD_ON)
    WdgTst_Lld_EwdgIrqHandler(EWDG);
#endif
}

/*==================================================================================================
 *                                        GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       Service for WdgTst initialization.
 * @details     This function initializes the WdgTst driver, include state variables, structures, etc.
 *              It checks the current state of the module and validates the configuration pointer.
 *              If the module is already initialized, it reports an error using the Det module (if enabled).
 *              If the configuration pointer is invalid, it also reports an error.
 *              If all checks pass, it calls the low-level driver initialization
 *              function and sets the module state to initialized.
 * @param[in]   ConfigPtr Pointer to configuration set
 * @table       @service_id:       0x00        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_188
 */
WDGTST_FUNC void WdgTst_Init(const WdgTst_ConfigType *ConfigPtr)
{
    if (WDGTST_INIT == WdgTst_State)
    {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDGTST_MODULE_ID,
            (uint8)WDGTST_INDEX,
            WDGTST_SID_INIT,
            (uint8)WDGTST_E_ALREADY_INITIALIZED
        );
#endif
    }
    else
    {
        /* Validate the configuration parameter of the WdgTst_Init API. In case an
         * error is detected, the function will report an error to Det,
         * depending on the environment the driver is run in. */
        /* For pre-compile, check DET error if the received config pointer is not NULL_PTR */
        if (NULL_PTR != ConfigPtr)
        {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError
            (
                (uint16)WDGTST_MODULE_ID,
                (uint8)WDGTST_INDEX,
                (uint8)WDGTST_SID_INIT,
                (uint8)WDGTST_E_INIT_FAILED
            );
#endif
        }
        else
        {
            WdgTst_Lld_Init();
            WdgTst_State = WDGTST_INIT;
        }
    }
}

/**
 * @brief       Service for executing Watchdog Test.
 * @details     This function starts the test of WdgTst module. It checks the current state of the module, and if the module is
 *              uninitialized, it reports an error using the Det module (if enabled) and returns 'E_NOT_OK'.
 *              If the module is initialized, it proceeds to install the interrupt handler and starts
 *              the watchdog timers test(both EWDG and WDG, if enabled).\n
 *              The behavior of this function depends on the following compile-time switches:\n
 *              WDGTST_DEV_ERROR_DETECT: If enabled, errors are reported using the Det module.\n
 *              WDGTST_EWDG_ENABLE: If enabled, the external watchdog (EWDG) is tested.\n
 *              WDGTST_WDG_ENABLE: If enabled, the internal watchdog (WDG) is tested.\n
 *
 * @table       @service_id:       0x01        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @return      Std_ReturnType
 * @retval      E_OK The WdgTst module was Executed successfully
 * @retval      E_NOT_OK The WdgTst module is uninitialized or executed failed
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_195
 */
WDGTST_FUNC Std_ReturnType WdgTst_Start(void)
{
    Std_ReturnType Ret = E_OK;
    if (WDGTST_UNINIT == WdgTst_State)
    {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDGTST_MODULE_ID,
            (uint8)WDGTST_INDEX,
            WDGTST_SID_START,
            (uint8)WDGTST_E_UNINIT
        );
#endif
        Ret = E_NOT_OK;
    }
    else
    {
        /* Enable WDG interrupt */
        (void)Platform_SetIrq(WDG_IRQn, TRUE);
        /* Install the WdgTst interrupt handler */
        (void)Platform_InstallIrqHandler(WDG_IRQn, WdgTst_Isr, &WdgTst_BackupIrqHandler);
#if (WDGTST_WDG_ENABLE == STD_ON)
        WdgTst_Lld_StartInit();
#endif
#if (WDGTST_EWDG_ENABLE == STD_ON)
        Ret = WdgTst_Lld_EwdgStart(EWDG);
#endif
#if (WDGTST_WDG_ENABLE == STD_ON)
        /* MR12 RULE 2.2 VIOLATION: If EWDG is not tested, the 'Ret' value of the result is always that of the right-hand operand.
         * But if EWDG is tested, '|=' is valid.
         */
        Ret |= WdgTst_Lld_WdgStart(WDG); /*PRQA S 2986*/
#endif
        WdgTst_Lld_TestComplete();
        /* Restore the WdgTst interrupt handler */
        (void)Platform_InstallIrqHandler(WDG_IRQn, WdgTst_BackupIrqHandler, &WdgTst_DummyIrqHandler);
    }
    return Ret;
}

#if (WDGTST_GET_RESULT_API == STD_ON)
/**
 * @brief       Service returns the result of the last executed Watchdog Test.
 * @details     This function returns the current test result of the WdgTst module. If the module is
 *              uninitialized, it reports an error using the Det module (if enabled) and returns
 *              'WDGTST_RESULT_NOT_TESTED'. If the module is initialized, it retrieves the test result
 *              from the low-level driver (LLD) using 'WdgTst_Lld_GetResult()'.\n
 *              The behavior of this function depends on the following compile-time switches:\n
 *              WDGTST_DEV_ERROR_DETECT: If enabled, errors are reported using the Det module.
 *
 * @table       @service_id:       0x02        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @return      WdgTst_TestResultType
 * @retval      WDGTST_RESULT_OK Test passed
 * @retval      WDGTST_RESULT_NOT_OK Test failed
 * @retval      WDGTST_RESULT_NOT_TESTED Test not executed
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_186
 */
WDGTST_FUNC WdgTst_TestResultType WdgTst_GetResult(void)
{
    WdgTst_TestResultType Result = WDGTST_RESULT_NOT_TESTED;
    if (WDGTST_UNINIT == WdgTst_State)
    {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDGTST_MODULE_ID,
            (uint8)WDGTST_INDEX,
            WDGTST_SID_GET_RESULT,
            (uint8)WDGTST_E_UNINIT
        );
#endif
    }
    else
    {
        Result = WdgTst_Lld_GetResult();
    }
    return Result;
}

/**
 * @brief       Service returns the detail result of the Watchdog Test.
 * @details     This function retrieves detailed test results of the WdgTst module and stores them in the
 *              provided 'ResultDetail' structure. If the module is uninitialized, it reports an error
 *              using the Det module (if enabled). If the 'ResultDetail' pointer is NULL, it also reports
 *              an error. If both checks pass, it retrieves the detailed results from the low-level driver
 *              (LLD) using 'WdgTst_Lld_GetDetailsResult()'.\n
 *              The behavior of this function depends on the following compile-time switches:\n
 *              WDGTST_DEV_ERROR_DETECT: If enabled, errors are reported using the Det module.
 * @table       @service_id:       0x03        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @param[out]  WdgTst_TestDetailsType *ResultDetail
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_835
 */
WDGTST_FUNC void WdgTst_GetDetailsResult(WdgTst_TestDetailsType *ResultDetail)
{
    if (WDGTST_UNINIT == WdgTst_State)
    {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDGTST_MODULE_ID,
            (uint8)WDGTST_INDEX,
            WDGTST_SID_GET_DETAILS_RESULT,
            (uint8)WDGTST_E_UNINIT
        );
#endif
    }
    /* Validate the configuration parameter of the WdgTst_GetDetailsResult API. In case an
     * error is detected, the function will report an error to Det,
     * depending on the environment the driver is run in. */
    else if (NULL_PTR == ResultDetail)
    {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDGTST_MODULE_ID,
            (uint8)WDGTST_INDEX,
            (uint8)WDGTST_SID_GET_DETAILS_RESULT,
            (uint8)WDGTST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        WdgTst_Lld_GetDetailsResult(ResultDetail);
    }
}
#endif

#if (WDGTST_VERSION_INFO_API == STD_ON)
/**
 * @brief       Service returns the version information of this module.
 * @details     This function retrieves the version information of the WdgTst module and stores it in the
 *              provided 'VersionInfo' structure. If the 'VersionInfo' pointer is NULL, it reports an error
 *              using the Det module (if enabled). If the pointer is valid, it populates the structure with
 *              the module's vendor ID, module ID, and software version information (major, minor, and patch versions).
 *              The version information includes: Module Id, Vendor Id, Vendor specific version number.\n
 *              The behavior of this function depends on the following compile-time switches:\n
 *              WDGTST_DEV_ERROR_DETECT: If enabled, errors are reported using the Det module.
 * @param[out]  VersionInfo Pointer to a structure of type 'Std_VersionInfoType' where the
 *                         version information will be stored. This parameter must not be NULL.
 * @table       @service_id:       0x04        \n
 *              @is_reentrant:     true       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_187
 */
WDGTST_FUNC void WdgTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    /* Validate the pointer to the version info */
    if (NULL_PTR == VersionInfo)
    {
#if (WDGTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDGTST_MODULE_ID,
            (uint8)WDGTST_INDEX,
            (uint8)WDGTST_SID_GET_VERSION_INFO,
            (uint8)WDGTST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        VersionInfo->vendorID = (uint16)WDGTST_VENDOR_ID;
        VersionInfo->moduleID = (uint16)WDGTST_MODULE_ID;
        VersionInfo->sw_major_version = (uint8)WDGTST_SW_MAJOR_VERSION;
        VersionInfo->sw_minor_version = (uint8)WDGTST_SW_MINOR_VERSION;
        VersionInfo->sw_patch_version = (uint8)WDGTST_SW_PATCH_VERSION;
    }
}
#endif

#define WDGTST_STOP_SEC_CODE
#include "WdgTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

