/**
 * @file    WdgTst_Types.h
 * @version V2.0.0
 *
 * @brief   Safety library WdgTst module interface
 * @details API implementation for WdgTst driver
 *
 * @addtogroup WDGTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 SafLib
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : None
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifndef WDGTST_TYPES_H
#define WDGTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
 *=================================================================================================*/
#include "WdgTst_Cfg.h"

/*==================================================================================================
 *                                   FILE VERSION INFORMATION
 *=================================================================================================*/
#define WDGTST_TYPES_VENDOR_ID                      (180)
#define WDGTST_TYPES_AR_RELEASE_MAJOR_VERSION       (4)
#define WDGTST_TYPES_AR_RELEASE_MINOR_VERSION       (4)
#define WDGTST_TYPES_AR_RELEASE_REVISION_VERSION    (0)
#define WDGTST_TYPES_SW_MAJOR_VERSION               (2)
#define WDGTST_TYPES_SW_MINOR_VERSION               (0)
#define WDGTST_TYPES_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
 *=================================================================================================*/
/* Check if WdgTst_Types.h and WdgTst_Cfg.h are of the same vendor */
#if (WDGTST_TYPES_VENDOR_ID != WDGTST_VENDOR_ID_CFG)
#error "WdgTst_Types.h and WdgTst_Cfg.h have different vendor ids"
#endif

/* Check if WdgTst_Types.h and WdgTst_Cfg.h are of the same Autosar version */
#if ((WDGTST_TYPES_AR_RELEASE_MAJOR_VERSION != WDGTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
      (WDGTST_TYPES_AR_RELEASE_MINOR_VERSION != WDGTST_AR_RELEASE_MINOR_VERSION_CFG) || \
      (WDGTST_TYPES_AR_RELEASE_REVISION_VERSION != WDGTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of WdgTst_Types.h and WdgTst_Cfg.h are different"
#endif

/* Check if WdgTst_Types.h and WdgTst_Cfg.h are of the same software version */
#if ((WDGTST_TYPES_SW_MAJOR_VERSION != WDGTST_SW_MAJOR_VERSION_CFG) || \
      (WDGTST_TYPES_SW_MINOR_VERSION != WDGTST_SW_MINOR_VERSION_CFG) || \
      (WDGTST_TYPES_SW_PATCH_VERSION != WDGTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of WdgTst_Types.h and WdgTst_Cfg.h are different"
#endif
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
 *=================================================================================================*/

#if (WDGTST_WDG_ENABLE == STD_ON)
/**
 * @brief   WDG clock sources.
 * @sw_type sw_detail
 * @private
 */
typedef enum
{
#if defined(CPU_YTM32B1ME0)
    WDGTST_WDG_CLKSRC_SIRC  = 0x00U,   /**< CPU_YTM32B1ME0: SIRC clock */
    WDGTST_WDG_CLKSRC_SXOSC = 0x01U    /**< CPU_YTM32B1ME0: SXOSC clock */
#elif defined(CPU_YTM32B1MD1) || defined(CPU_YTM32B1HA0)
    WDGTST_WDG_CLKSRC_IPC  = 0x00U,    /**< CPU_YTM32B1MD1/CPU_YTM32B1HA0: IPC clock */
    WDGTST_WDG_CLKSRC_BUS  = 0x01U     /**< CPU_YTM32B1MD1/CPU_YTM32B1HA0: BUS clock */
#elif defined(CPU_YTM32B1MC0)
    WDGTST_WDG_CLKSRC_LPO  = 0x00U,    /**< CPU_YTM32B1MC0: LPO clock */
    WDGTST_WDG_CLKSRC_SIRC = 0x01U,    /**< CPU_YTM32B1MC0: SIRC clock */
    WDGTST_WDG_CLKSRC_IPC  = 0x02U     /**< CPU_YTM32B1MC0: IPC clock */
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
} WdgTst_WdgClkSrcType;
#endif


#if (WDGTST_EWDG_ENABLE == STD_ON)
/**
 * @brief   EWDG clock sources.
 * @sw_type sw_detail
 * @private
 */
typedef enum
{
#if defined(CPU_YTM32B1ME0)
    WDGTST_EWDG_CLKSRC_SIRC  = 0x00U,  /**< CPU_YTM32B1ME0: SIRC clock */
    WDGTST_EWDG_CLKSRC_SXOSC = 0x01U   /**< CPU_YTM32B1ME0: SXOSC clock */
#elif defined(CPU_YTM32B1MD1)
    WDGTST_EWDG_CLKSRC_SIRC  = 0x00U   /**< CPU_YTM32B1MD1: SIRC clock */
#elif defined(CPU_YTM32B1MC0)
    WDGTST_EWDG_CLKSRC_SIRC  = 0x00U,  /**< CPU_YTM32B1MC0: SIRC clock */
    WDGTST_EWDG_CLKSRC_LPO   = 0x01U   /**< CPU_YTM32B1MC0: LPO clock */
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
} WdgTst_EwdgClkSrcType;
#endif

/**
* @brief   The type specifies the WdgTst possible masks.
* @sw_type sw_detail
*/
typedef uint8 WdgTst_MaskType;
/*==================================================================================================
 *                                             TYPEDEF
 *=================================================================================================*/
/**
 * @brief   This enumerated type contain the possible states of the WdgTst driver.
 * @sw_type sw_arch
 */
typedef enum
{
    WDGTST_UNINIT   = 0x00U, /**< WdgTst driver is not initialized */
    WDGTST_INIT     = 0x01U  /**< WdgTst driver is initialized */
} WdgTst_StateType;

/**
 * @brief   This enumerated type contain the possible results of all tests.
 * @sw_type sw_arch
 */
typedef enum
{
    WDGTST_RESULT_NOT_TESTED    = 0x00U, /**< Test not executed */
    WDGTST_RESULT_OK            = 0x01U, /**< Test passed */
    WDGTST_RESULT_NOT_OK        = 0x02U  /**< Test failed */
} WdgTst_TestResultType;

/**
 * @brief   Function pointer type for CrcTst front ground test error notification.
 * @sw_type sw_arch
 */
typedef void (*WdgTst_ErrNotificationType)(uint16 ModuleId);

/**
 * @brief   Test details structure for the WdgTst driver
 * @sw_type sw_arch
 */
typedef struct
{
    WdgTst_TestResultType TestResult;    /**< The test result of WdgTst driver */
#if (WDGTST_EWDG_ENABLE == STD_ON)
    WdgTst_TestResultType EwdgTestResult;   /**< The test result of Ewdg test */
#endif
#if (WDGTST_WDG_ENABLE == STD_ON)
    WdgTst_TestResultType WdgSoftLockTestResult;    /**< The test result of Wdg soft lock test */
    WdgTst_TestResultType WdgTimeoutInterruptTestResult;   /**< The test result of Wdg timeout interrupt test */
#endif
} WdgTst_TestDetailsType;

/**
 * @brief   Config structure for the WdgTst driver
 * @sw_type sw_arch
 */
typedef struct
{
    WdgTst_ErrNotificationType ErrNotification;  /**< Pointer to the test error notification function */
} WdgTst_ConfigType;


#ifdef __cplusplus
}
#endif

#endif /* WDGTST_TYPES_H */

/** @} */
