/**
 * @file    WdgTst_Lld.h
 * @version V2.0.0
 *
 * @brief   Safety library WdgTst module interface
 * @details API implementation for WdgTst driver
 *
 * @addtogroup WDGTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 SafLib
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : None
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
 *=================================================================================================*/

#ifndef WDGTST_LLD_H
#define WDGTST_LLD_H

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
 *=================================================================================================*/
#include "WdgTst_Types.h"
#if (WDGTST_WDG_ENABLE == STD_ON)
#include "pSIP_Wdg.h"
#endif
#if (WDGTST_EWDG_ENABLE == STD_ON)
#include "pSIP_Ewdg.h"
#endif

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
 *=================================================================================================*/
#define WDGTST_LLD_VENDOR_ID                      (180)
#define WDGTST_LLD_MODULE_ID                      (524)
#define WDGTST_LLD_AR_RELEASE_MAJOR_VERSION       (4)
#define WDGTST_LLD_AR_RELEASE_MINOR_VERSION       (4)
#define WDGTST_LLD_AR_RELEASE_REVISION_VERSION    (0)
#define WDGTST_LLD_SW_MAJOR_VERSION               (2)
#define WDGTST_LLD_SW_MINOR_VERSION               (0)
#define WDGTST_LLD_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
 *=================================================================================================*/
/* Check if WdgTst_Lld.h and WdgTst_Types.h are of the same vendor */
#if (WDGTST_LLD_VENDOR_ID != WDGTST_TYPES_VENDOR_ID)
#error "WdgTst_Lld.h and WdgTst_Types.h have different vendor ids"
#endif

/* Check if WdgTst_Lld.h and WdgTst_Types.h are of the same Autosar version */
#if ((WDGTST_LLD_AR_RELEASE_MAJOR_VERSION != WDGTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
         (WDGTST_LLD_AR_RELEASE_MINOR_VERSION != WDGTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
         (WDGTST_LLD_AR_RELEASE_REVISION_VERSION != WDGTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of WdgTst_Lld.h and WdgTst_Types.h are different"
#endif

/* Check if WdgTst_Lld.h and WdgTst_Types.h are of the same software version */
#if ((WDGTST_LLD_SW_MAJOR_VERSION != WDGTST_TYPES_SW_MAJOR_VERSION) || \
         (WDGTST_LLD_SW_MINOR_VERSION != WDGTST_TYPES_SW_MINOR_VERSION) || \
         (WDGTST_LLD_SW_PATCH_VERSION != WDGTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of WdgTst_Lld.h and WdgTst_Types.h are different"
#endif

/*==================================================================================================
                                    DEFINES AND MACROS
==================================================================================================*/

#if defined(CPU_YTM32B1MC0)
#if (WDGTST_EWDG_ENABLE == STD_ON)
#define EWDG    EWDG0
#endif /* (WDGTST_EWDG_ENABLE == STD_ON) */
#endif /* defined(CPU_YTM32B1MC0) */

/**
 * @defgroup  InterruptTriggerMasks
 * @brief     WdgTst possible interrupt masks and execution state masks.
 * @trace     YTDT_WdgTst_180
 * @sw_type   sw_detail
 * @{
 */
/** No interrupt mask */
#define WDGTST_INT_MASK_NONE            (0x00U)
/** WDG interrupt mask */
#define WDGTST_INT_MASK_WDG             (0x01U)
/** EWDG interrupt mask */
#define WDGTST_INT_MASK_EWDG            (0x02U)
/** All watch dog interrupts mask */
#define WDGTST_INT_MASK_ALL             (0x03U)
/** WDGTST WDG start mask */
#define WDGTST_WDG_STARTED_MASK         (0x04U)
/** WDGTST EWDG start mask */
#define WDGTST_EWDG_STARTED_MASK        (0x08U)
/** WDGTST WDG and EWDG start mask */
#define WDGTST_STARTED_MASK             (0x0CU)
/** @} */


/**
 * @defgroup  WDGLockTestMasks
 * @brief     WdgTst WDG lock test masks.
 * @sw_type   sw_detail
 * @trace YTSDS_WdgTst_913
 * @{
 */
/** No lock test start mask */
#define WDGTST_LOCK_MASK_NONE               (0x00U)
/** Mask for WDG control register (CR) start lock test */
#define WDGTST_LOCK_CR_START_MASK           (0x01U)
/** Mask for WDG window value register (WVR) start lock test */
#define WDGTST_LOCK_WVR_START_MASK          (0x02U)
/** Mask for WDG timer overflow register (TOVR) start lock test */
#define WDGTST_LOCK_TOVR_START_MASK         (0x04U)
/** Mask for WDG control register (CR) bus fault trigger */
#define WDGTST_LOCK_CR_BUSFAULT_MASK        (0x10U)
/** Mask for WDG window value register (WVR) bus fault trigger */
#define WDGTST_LOCK_WVR_BUSFAULT_MASK       (0x20U)
/** Mask for WDG timer overflow (TOVR) bus fault trigger */
#define WDGTST_LOCK_TOVR_BUSFAULT_MASK      (0x40U)
/** @} */


/*==================================================================================================
 *                                    FUNCTION PROTOTYPES
 *=================================================================================================*/
#define WDGTST_START_SEC_CODE
#include "WdgTst_MemMap.h"

/**
 * @brief       Initializes the WdgTst module.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_455
 */
WDGTST_FUNC void WdgTst_Lld_Init(void);

#if (WDGTST_WDG_ENABLE == STD_ON)
/**
 * @brief       Preparation to start testing, install bus fault handler.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_692
 */
WDGTST_FUNC void WdgTst_Lld_StartInit(void);
#endif

/**
 * @brief       Update the state and restore the bus fault handler.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_WdgTst_914
 */
WDGTST_FUNC void WdgTst_Lld_TestComplete(void);

/**
 * @brief       Get test result.
 * @return      WdgTst_TestResultType
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_659
 */
WDGTST_FUNC WdgTst_TestResultType WdgTst_Lld_GetResult(void);

/**
 * @brief       Get details test result.
 * @param[out]  TestDetails
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_658
 */
WDGTST_FUNC void WdgTst_Lld_GetDetailsResult(WdgTst_TestDetailsType *TestDetails);

#if (WDGTST_WDG_ENABLE == STD_ON)
/**
 * @brief       Execute the tests for WDG
 * @return      Std_ReturnType
 * @retval      E_OK WDG test running properly
 * @retval      E_NOT_OK WdgTst running failed
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_457
 */
WDGTST_FUNC Std_ReturnType WdgTst_Lld_WdgStart(volatile WDG_Type *Base);

/**
 * @brief       WDG instance interrupt handler.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_456
 */
WDGTST_FUNC void WdgTst_Lld_WdgIrqHandler(volatile WDG_Type *Base);

#endif /* WDGTST_WDG_ENABLE */

#if (WDGTST_EWDG_ENABLE == STD_ON)
/**
 * @brief       Execute the tests for EWDG
 * @return      Std_ReturnType
 * @retval      E_OK EWDG test execute properly
 * @retval      E_NOT_OK EWDG has been enabled and cannot be tested
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_452
 */
WDGTST_FUNC Std_ReturnType WdgTst_Lld_EwdgStart(volatile EWDG_Type *Base);

/**
 * @brief       EWDG instance interrupt handler.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTDT_WdgTst_451
 */
WDGTST_FUNC void WdgTst_Lld_EwdgIrqHandler(volatile EWDG_Type *Base);

#endif /* WDGTST_EWDG_ENABLE */

#define WDGTST_STOP_SEC_CODE
#include "WdgTst_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* WDGTST_LLD_H */

/** @} */
