/**
* @file    StkTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : StkTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 3408 Rule-8.4: A compatible declaration shall be visible when an object or function
 *                   with external linkage is defined.
 *
 */

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "StkTst.h"
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#if (STKTST_DEM_EVENT_REPORT == STD_ON)
#include "Dem.h"
#endif

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define STKTST_VENDOR_ID_C                      (180)
#define STKTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define STKTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define STKTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define STKTST_SW_MAJOR_VERSION_C               (2)
#define STKTST_SW_MINOR_VERSION_C               (0)
#define STKTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if StkTst.c and StkTst.h are of the same vendor */
#if (STKTST_VENDOR_ID_C != STKTST_VENDOR_ID)
#error "StkTst.c and StkTst.h have different vendor ids"
#endif

/* Check if StkTst.c and StkTst.h are of the same Autosar version */
#if ((STKTST_AR_RELEASE_MAJOR_VERSION_C != STKTST_AR_RELEASE_MAJOR_VERSION) || \
     (STKTST_AR_RELEASE_MINOR_VERSION_C != STKTST_AR_RELEASE_MINOR_VERSION) || \
     (STKTST_AR_RELEASE_REVISION_VERSION_C != STKTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of StkTst.c and StkTst.h are different"
#endif

/* Check if StkTst.c and StkTst.h are of the same Software version */
#if ((STKTST_SW_MAJOR_VERSION_C != STKTST_SW_MAJOR_VERSION) || \
     (STKTST_SW_MINOR_VERSION_C != STKTST_SW_MINOR_VERSION) || \
     (STKTST_SW_PATCH_VERSION_C != STKTST_SW_PATCH_VERSION))
#error "Software Version Numbers of StkTst.c and StkTst.h are different"
#endif

/*==================================================================================================
 *                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/
#if (STKTST_PRECOMPILE_SUPPORT == STD_ON)
/**
 * @brief  StkTst driver configuration structure
 * @sw_type     sw_detail
 * @private
*/
extern const StkTst_ConfigType StkTst_Config;
#endif /* #if (STKTST_PRECOMPILE_SUPPORT == STD_ON) */

/*==================================================================================================
 *                                       GLOBAL VARIABLES
==================================================================================================*/
#define STKTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "StkTst_MemMap.h"
/* MR12 RULE 8.4 VIOLATION: The variable will used in integration test,
* thus, couldn't adhere to M3CM Rule-8.4 and Rule-8.7.
*/
/**
 * @brief  The pointer to the configuration structure of the StkTst.
 * @sw_type     sw_detail
 * @private
*/
STKTST_VAR const StkTst_ConfigType *StkTst_ConfigPtr = NULL_PTR; /*PRQA S 3408*/

#define STKTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "StkTst_MemMap.h"

#define STKTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "StkTst_MemMap.h"

/**
 * @brief   StkTst result variable
 * @sw_type     sw_detail
 * @private
 */
STKTST_VAR static StkTst_TestResultType StkTst_TestResult;

/* MR12 RULE 8.4 VIOLATION: The variable will used in integration test,
* thus, couldn't adhere to M3CM Rule-8.4 and Rule-8.7.
*/
/**
 * @brief  The StkTst module running state
 * @sw_type     sw_detail
 * @private
*/
STKTST_VAR StkTst_StateType StkTst_Status; /*PRQA S 3408*/

#define STKTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "StkTst_MemMap.h"

/*==================================================================================================
 *                                       LOCAL FUNCTIONS
==================================================================================================*/
#define STKTST_START_SEC_CODE
#include "StkTst_MemMap.h"

/**
 * @brief       Service for StkTst Pad Stack.
 * @details     This function pads the detection region with a fixed value.
 * @param[in]   PadValue Pad Value
 *
 * @table       @service_id:       --    \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false \n
                @memory_map:       .mcal_text \n
 *
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTDT_StkTst_644
 */
STKTST_FUNC static void StkTst_PadStack(StkTst_PadType PadValue)
{
    volatile StkTst_AddrType *Addr = (volatile StkTst_AddrType *)StkTst_ConfigPtr->BotAddr;
    for (StkTst_SizeType Index = 0; Index < StkTst_ConfigPtr->DetSize; Index += sizeof(StkTst_AddrType))
    {
        *Addr = PadValue;
        ++Addr;
    }
}

/**
 * @brief       Service for StkTst get signature.
 * @details     This function calculates the signature of the detection region.
 *
 * @table       @service_id:       --    \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false \n
                @memory_map:       .mcal_text \n
 *
 * @return      StkTst_SignType Signature
 * @sw_type     sw_detail
 *
 * @trace YTDT_StkTst_639
 */
STKTST_FUNC static StkTst_SignType StkTst_GetSign(void)
{
    StkTst_SignType RetSign = 0U;
    volatile const StkTst_AddrType *Addr = (volatile const StkTst_AddrType *)StkTst_ConfigPtr->BotAddr;
    for (StkTst_SizeType Index = 0; Index < StkTst_ConfigPtr->DetSize; Index += sizeof(StkTst_AddrType))
    {
        RetSign += *Addr;
        ++Addr;
    }
    return RetSign;
}

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief  Service for StkTst initialization.
 * @details The function shall initialize all stack Test global variables.
 * @param[in] ConfigPtr Pointer to the stack Test configuration data structure.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_StkTst_641
 */
STKTST_FUNC void StkTst_Init(const StkTst_ConfigType *ConfigPtr)
{
    if (NULL_PTR != StkTst_ConfigPtr)
    {
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(STKTST_MODULE_ID, STKTST_INSTANCE_ID, STKTST_SID_INIT, STKTST_E_ALREADY_INITIALIZED);
#endif /* (STKTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
#if (STKTST_PRECOMPILE_SUPPORT == STD_ON)
        /* For pre-compile, check DET error if the received config pointer is not NULL_PTR */
        if (NULL_PTR != ConfigPtr)
#else /* (STKTST_PRECOMPILE_SUPPORT == STD_OFF) */
        /* For post build, check DET error if the received config pointer is NULL_PTR */
        if (NULL_PTR == ConfigPtr)
#endif /* (STKTST_PRECOMPILE_SUPPORT == STD_ON) */
        {
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(STKTST_MODULE_ID, STKTST_INSTANCE_ID, STKTST_SID_INIT, STKTST_E_INIT_FAILED);
#endif /* (STKTST_DEV_ERROR_DETECT == STD_ON) */
        }
        else
        {
#if (STKTST_PRECOMPILE_SUPPORT == STD_ON)
            /* Save configuration pointer in global variable */
            StkTst_ConfigPtr = &StkTst_Config;
            (void)ConfigPtr;
#else
            StkTst_ConfigPtr = ConfigPtr;
#endif /* (STKTST_PRECOMPILE_SUPPORT == STD_ON) */
            StkTst_TestResult = STKTST_RESULT_NOT_TESTED;
            /* Pad the detection region with fixed value. */
            StkTst_PadStack(StkTst_ConfigPtr->PadValue);
            StkTst_Status = STKTST_INIT;
        }
    }
}

#if (STKTST_DEINIT_API == STD_ON)
/**
 * @brief  Service for StkTst De-initialization.
 * @details The function shall De-initialization all stack Test global variables.
 *
 * @table            @service_id:       0x02  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_StkTst_636
 */
STKTST_FUNC void StkTst_DeInit(void)
{
    if (STKTST_UNINIT == StkTst_Status)
    {
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(STKTST_MODULE_ID, STKTST_INSTANCE_ID, STKTST_SID_DEINIT, STKTST_E_UNINIT);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        /* Pad the detection region with zero. */
        StkTst_PadStack(0U);
        /* Set global configuration pointer back to NULL_PTR */
        StkTst_ConfigPtr = NULL_PTR;
        StkTst_TestResult = STKTST_RESULT_NOT_TESTED;
        StkTst_Status = STKTST_UNINIT;
    }
}
#endif /* STKTST_DEINIT_API == STD_ON */

/**
 * @brief  Cyclically called by scheduler to perform processing of Stack Test.
 * @details This function detect stack overflow and/or underflow in the system stack.
 *
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_StkTst_643
 */
STKTST_FUNC void StkTst_MainFunction(void)
{
    StkTst_SignType Sign;
    if (STKTST_UNINIT == StkTst_Status)
    {
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(STKTST_MODULE_ID, STKTST_INSTANCE_ID, STKTST_SID_MAIN_FUNCTION, STKTST_E_UNINIT);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        StkTst_Status = STKTST_TESTING;
        Sign = StkTst_GetSign();
        if (StkTst_ConfigPtr->OrgSign == Sign)
        {
            StkTst_TestResult = STKTST_RESULT_OK;
#if (STKTST_DEM_EVENT_REPORT == STD_ON)
            (void)Dem_SetEventStatus(STKTST_E_STACK_FAILURE_DEM_EVENT_ID, DEM_EVENT_STATUS_PREPASSED);
#endif /* STKTST_DEM_EVENT_REPORT == STD_ON */
        }
        else
        {
            StkTst_TestResult = STKTST_RESULT_NOT_OK;
#if (STKTST_NOTIFICATION_SUPPORTED == STD_ON)
            if (StkTst_ConfigPtr->OvfCbk != NULL_PTR)
            {
                StkTst_ConfigPtr->OvfCbk(STKTST_MODULE_ID);
            }
#endif /* STKTST_NOTIFICATION_SUPPORTED == STD_ON */
#if (STKTST_DEM_EVENT_REPORT == STD_ON)
            (void)Dem_SetEventStatus(STKTST_E_STACK_FAILURE_DEM_EVENT_ID, DEM_EVENT_STATUS_PREFAILED);
#endif /* STKTST_DEM_EVENT_REPORT == STD_ON */
        }
        StkTst_Status = STKTST_TESTED;
    }
}

#if (STKTST_GET_RESULT_API == STD_ON)
/**
 * @brief  Service to get result of the last executed Stack Test.
 * @details The function returns the result of the stack Test.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return StkTst_TestResultType
 * @retval STKTST_RESULT_NOT_TESTED: Test not executed.
 * @retval STKTST_RESULT_OK: Test passed.
 * @retval STKTST_RESULT_NOT_OK: Test failed.
 * @sw_type sw_detail
 *
 * @trace YTDT_StkTst_638
 */
STKTST_FUNC StkTst_TestResultType StkTst_GetResult(void)
{
    StkTst_TestResultType RetResult = STKTST_RESULT_NOT_TESTED;
    if (STKTST_UNINIT == StkTst_Status)
    {
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(STKTST_MODULE_ID, STKTST_INSTANCE_ID, STKTST_SID_GET_RESULT, STKTST_E_UNINIT);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        RetResult = StkTst_TestResult;
    }
    return RetResult;
}
#endif /* STKTST_GET_RESULT_API == STD_ON */

#if (STKTST_VERSION_INFO_API == STD_ON)
/**
 * @brief  Returns the version information of this module.
 * @param[out]  VersionInfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return  void
 * @sw_type sw_detail
 *
 * @trace YTDT_StkTst_640
 */
STKTST_FUNC void StkTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    if (NULL_PTR == VersionInfo)
    {
        /* Report STKTST_E_PARAM_POINTER DET if service called with NULL_PTR*/
#if (STKTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(STKTST_MODULE_ID, STKTST_INSTANCE_ID, STKTST_SID_GET_VERSION_INFO, STKTST_E_PARAM_POINTER);
#endif /* EFMTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        VersionInfo->vendorID = STKTST_VENDOR_ID;
        VersionInfo->moduleID = STKTST_MODULE_ID;
        VersionInfo->sw_major_version = STKTST_SW_MAJOR_VERSION;
        VersionInfo->sw_minor_version = STKTST_SW_MINOR_VERSION;
        VersionInfo->sw_patch_version = STKTST_SW_PATCH_VERSION;
    }
}
#endif /* STKTST_VERSION_INFO_API == STD_ON */

#define STKTST_STOP_SEC_CODE
#include "StkTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file StkTst.c */
