/*
* @file    StkTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : StkTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/


#ifndef STKTST_H
#define STKTST_H

#ifdef __cplusplus
extern "C"
{
#endif

/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "StkTst_Cfg.h"
#include "StkTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define STKTST_VENDOR_ID                      (180)
#define STKTST_AR_RELEASE_MAJOR_VERSION       (4)
#define STKTST_AR_RELEASE_MINOR_VERSION       (4)
#define STKTST_AR_RELEASE_REVISION_VERSION    (0)
#define STKTST_SW_MAJOR_VERSION               (2)
#define STKTST_SW_MINOR_VERSION               (0)
#define STKTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if StkTst.h and StkTst_Cfg.h are of the same vendor */
#if (STKTST_VENDOR_ID != STKTST_VENDOR_ID_CFG)
#error "StkTst.h and StkTst_Cfg.h have different vendor ids"
#endif

/* Check if StkTst.h and StkTst_Cfg.h are of the same Autosar version */
#if ((STKTST_AR_RELEASE_MAJOR_VERSION != STKTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (STKTST_AR_RELEASE_MINOR_VERSION != STKTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (STKTST_AR_RELEASE_REVISION_VERSION != STKTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of StkTst.h and StkTst_Cfg.h are different"
#endif

/* Check if StkTst.h and StkTst_Cfg.h are of the same software version */
#if ((STKTST_SW_MAJOR_VERSION != STKTST_SW_MAJOR_VERSION_CFG) || \
     (STKTST_SW_MINOR_VERSION != STKTST_SW_MINOR_VERSION_CFG) || \
     (STKTST_SW_PATCH_VERSION != STKTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of StkTst.h and StkTst_Cfg.h are different"
#endif

/* Check if StkTst.h and StkTst_Types.h are of the same vendor */
#if (STKTST_VENDOR_ID != STKTST_TYPES_VENDOR_ID)
#error "StkTst.h and StkTst_Types.h have different vendor ids"
#endif

/* Check if StkTst.h and StkTst_Types.h are of the same Autosar version */
#if ((STKTST_AR_RELEASE_MAJOR_VERSION != STKTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (STKTST_AR_RELEASE_MINOR_VERSION != STKTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (STKTST_AR_RELEASE_REVISION_VERSION != STKTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of StkTst.h and StkTst_Types.h are different"
#endif

/* Check if StkTst.h and StkTst_Types.h are of the same software version */
#if ((STKTST_SW_MAJOR_VERSION != STKTST_TYPES_SW_MAJOR_VERSION) || \
     (STKTST_SW_MINOR_VERSION != STKTST_TYPES_SW_MINOR_VERSION) || \
     (STKTST_SW_PATCH_VERSION != STKTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of StkTst.h and StkTst_Types.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

/**
 * @defgroup StackTst_ID_Information_Macro
 * @brief   Service ID number for all StkTst driver services
 * @sw_type sw_detail
 * @{
 *
 * @trace YTDT_StkTst_890
 */
#define STKTST_INSTANCE_ID             ((uint8)0U)   /**< StkTst instance ID */
#define STKTST_MODULE_ID               ((uint16)522U) /**< StkTst module ID */
#define STKTST_SID_INIT                 (0x01U)  /**< API Service ID for StkTst_Init */
#define STKTST_SID_DEINIT               (0x02U)  /**< API Service ID for StkTst_DeInit */
#define STKTST_SID_MAIN_FUNCTION        (0x03U)  /**< API Service ID for StkTst_MainFunction */
#define STKTST_SID_GET_RESULT           (0x04U)  /**< API Service ID for StkTst_GetResult */
#define STKTST_SID_GET_VERSION_INFO     (0x05U)  /**< API Service ID for StkTst_GetVersionInfo */
/** @} */

#if (STKTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup StackTst_Dev_Error_Macro
 * @brief   StackTst module supported development error detailed list.
 * @sw_type sw_detail
 * @{
 *
 * @trace YTDT_StkTst_480
 */
#define STKTST_E_ALREADY_INITIALIZED    (0x01U) /**< API StkTst_Init service called while the StkTst driver has already been initialised */
#define STKTST_E_INIT_FAILED            (0x02U) /**< API StkTst_Init service called with wrong parameter */
#define STKTST_E_UNINIT                 (0x03U) /**< API service used without module initialization */
#define STKTST_E_PARAM_POINTER          (0x04U) /**< StkTst API service is called using an invalid pointer (e.g. the pointer should not be NULL) */
/** @} */
#endif

/*==================================================================================================
 *                                  GLOBAL CONSTANT DECLARATIONS
==================================================================================================*/

/*==================================================================================================
 *                                    FUNCTION PROTOTYPES
==================================================================================================*/
#define STKTST_START_SEC_CODE
#include "StkTst_MemMap.h"

/**
 * @brief  Service for StkTst initialization.
 * @details The function shall initialize all stack Test global variables.
 * @param[in] ConfigPtr Pointer to the stack Test configuration data structure.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
STKTST_FUNC void StkTst_Init(const StkTst_ConfigType *ConfigPtr);

#if (STKTST_DEINIT_API == STD_ON)
/**
 * @brief  Service for StkTst De-initialization.
 * @details The function shall De-initialization all stack Test global variables.
 *
 * @table            @service_id:       0x02  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
STKTST_FUNC void StkTst_DeInit(void);
#endif /* STKTST_DEINIT_API == STD_ON */

/**
 * @brief  Cyclically called by scheduler to perform processing of Stack Test.
 * @details This function detect stack overflow and/or underflow in the system stack.
 *
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
STKTST_FUNC void StkTst_MainFunction(void);

#if (STKTST_GET_RESULT_API == STD_ON)
/**
 * @brief  Service to get result of the last executed Stack Test.
 * @details The function returns the result of the stack Test.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true \n
                     @autosar_api:      false \n
 *
 * @return StkTst_TestResultType
 * @retval STKTST_RESULT_NOT_TESTED: Test not executed.
 * @retval STKTST_RESULT_OK: Test passed.
 * @retval STKTST_RESULT_NOT_OK: Test failed.
 * @sw_type sw_arch
 *
 */
STKTST_FUNC StkTst_TestResultType StkTst_GetResult(void);
#endif /* STKTST_GET_RESULT_API == STD_ON */

#if (STKTST_VERSION_INFO_API == STD_ON)
/**
 * @brief  Returns the version information of this module.
 * @param[out]  versioninfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return  void
 * @sw_type sw_arch
 *
 */
STKTST_FUNC void StkTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* STKTST_VERSION_INFO_API == STD_ON */

#define STKTST_STOP_SEC_CODE
#include "StkTst_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* End of file StkTst.h */


