/**
* @file    RamTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : RamTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations List
 * PRQA S 3408 Rule 8.4: '%1s' has external linkage and is being defined without any previous declaration.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "RamTst.h"
#include "RamTst_Lld.h"
#include "OsIf.h"
#include "SchM_RamTst.h"
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* RAMTST_DEV_ERROR_DETECT == STD_ON */

#if ((RAMTST_ENABLE_DEM_RUNFL_ERROR_STATUS == STD_ON) || \
     (RAMTST_ENABLE_DEM_PART_ERROR_STATUS == STD_ON)  || \
     (RAMTST_ENABLE_DEM_MAIN_ERROR_STATUS == STD_ON))
#include "Dem.h"
#endif
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define RAMTST_VENDOR_ID_C                     (180)
#define RAMTST_AR_REL_MAJOR_VER_C              (4)
#define RAMTST_AR_REL_MINOR_VER_C              (4)
#define RAMTST_AR_REL_REVISION_VER_C           (0)
#define RAMTST_SW_MAJOR_VER_C                  (2)
#define RAMTST_SW_MINOR_VER_C                  (0)
#define RAMTST_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and RAMTST header file are of the same vendor */
#if (RAMTST_VENDOR_ID_C != RAMTST_VENDOR_ID)
#error "RamTst.c and RamTst.h have different vendor ids"
#endif

/* Check if source file and RAMTST header file are of the same Autosar version */
#if (( RAMTST_AR_REL_MAJOR_VER_C != RAMTST_AR_REL_MAJOR_VER) || \
      ( RAMTST_AR_REL_MINOR_VER_C != RAMTST_AR_REL_MINOR_VER) || \
      ( RAMTST_AR_REL_REVISION_VER_C != RAMTST_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of RamTst.c and RamTst.h are different"
#endif

/* Check if source file and RAMTST header file are of the same Software version */
#if (( RAMTST_SW_MAJOR_VER_C != RAMTST_SW_MAJOR_VER) || \
      ( RAMTST_SW_MINOR_VER_C != RAMTST_SW_MINOR_VER) || \
      ( RAMTST_SW_PATCH_VER_C != RAMTST_SW_PATCH_VER))
#error "Software Version Numbers of RamTst.c and RamTst.h are different"
#endif

/* Check if source file and RAMTST_LLD header file are of the same vendor */
#if (RAMTST_VENDOR_ID_C != RAMTST_LLD_VENDOR_ID)
#error "RamTst.c and RamTst_Lld.h have different vendor ids"
#endif

/* Check if source file and RAMTST_LLD header file are of the same Autosar version */
#if (( RAMTST_AR_REL_MAJOR_VER_C != RAMTST_LLD_AR_REL_MAJOR_VER) || \
      ( RAMTST_AR_REL_MINOR_VER_C != RAMTST_LLD_AR_REL_MINOR_VER) || \
      ( RAMTST_AR_REL_REVISION_VER_C != RAMTST_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of RamTst.c and RamTst_Lld.h are different"
#endif

/* Check if source file and RAMTST_LLD header file are of the same Software version */
#if (( RAMTST_SW_MAJOR_VER_C != RAMTST_LLD_SW_MAJOR_VER) || \
      ( RAMTST_SW_MINOR_VER_C != RAMTST_LLD_SW_MINOR_VER) || \
      ( RAMTST_SW_PATCH_VER_C != RAMTST_LLD_SW_PATCH_VER))
#error "Software Version Numbers of RamTst.c and RamTst_Lld.h are different"
#endif

/* Check if source file and SCHM_RamTst header file are of the same vendor */
#if (RAMTST_VENDOR_ID_C != SCHM_RAMTST_VENDOR_ID)
#error "RamTst.c and SchM_RamTst.h have different vendor ids"
#endif

/* Check if source file and SCHM_RamTst header file are of the same Autosar version */
#if (( RAMTST_AR_REL_MAJOR_VER_C != SCHM_RAMTST_AR_RELEASE_MAJOR_VERSION) || \
      ( RAMTST_AR_REL_MINOR_VER_C != SCHM_RAMTST_AR_RELEASE_MINOR_VERSION) || \
      ( RAMTST_AR_REL_REVISION_VER_C != SCHM_RAMTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of RamTst.c and SchM_RamTst.h are different"
#endif

/* Check if source file and SCHM_RamTst header file are of the same Software version */
#if (( RAMTST_SW_MAJOR_VER_C != SCHM_RAMTST_SW_MAJOR_VERSION) || \
      ( RAMTST_SW_MINOR_VER_C != SCHM_RAMTST_SW_MINOR_VERSION) || \
      ( RAMTST_SW_PATCH_VER_C != SCHM_RAMTST_SW_PATCH_VERSION))
#error "Software Version Numbers of RamTst.c and SchM_RamTst.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
#define RAMTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "RamTst_MemMap.h"
/* MR12 Rule 8.4: The variable of RamTstState is declared and used in RamTst.c and RamTst_Lld.c file, so it couldn't
 *                       adhere to M3CM Rule-8.4
 */
/**
 * @brief            Global variable to store the state during the RAM test process.
 * @private
 */
RAMTST_VAR RamTst_StateType RamTstState; /* PRQA S 3408 */
#define RAMTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "RamTst_MemMap.h"

extern const RamTst_ConfigType RamTst_Config;

/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
#define RAMTST_START_SEC_CODE
#include "RamTst_MemMap.h"

#if (RAMTST_ALLOW_API == STD_ON)
/**
 * @brief            Service for continuing the RAM Test after calling 'RamTst_Stop.
 * @details
 *
 * @table            @service_id:       0x03       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   false      \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_325
 */
RAMTST_FUNC void RamTst_Allow(void)
{
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_00();
    if (RAMTST_EXECUTION_STOPPED != RamTstState.ExecStatus)
    {
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_00();
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_ALLOW, RAMTST_E_STATUS_FAILURE);
#endif
    }
    else
    {
        RamTstState.ExecStatus = RAMTST_EXECUTION_RUNNING;
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_00();
    }
}
#endif

#if (RAMTST_CHANGE_NUM_OF_TESTED_CELLS_API == STD_ON)
/**
 * @brief            Service changes the current number of tested cells.
 * @details          This service can be used to change the number of tested cells per MainFunction cycle.
 *
 * @table            @service_id:       0x08       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        NewNumberOfTestedCells The new number of tested cells.
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_326
 */
RAMTST_FUNC void RamTst_ChangeNumberOfTestedCells(RamTst_NumberOfTestedCellsType NewNumberOfTestedCells)
{
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_01();
    if (RAMTST_EXECUTION_STOPPED != RamTstState.ExecStatus)
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_CHANGE_NUMBER_OF_TESTED_CELLS, RAMTST_E_STATUS_FAILURE);
#endif
    }
    else if ((NewNumberOfTestedCells > RamTstState.MaxNumTestedCell) || (NewNumberOfTestedCells < RamTstState.MinNumTestedCell))
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_CHANGE_NUMBER_OF_TESTED_CELLS, RAMTST_E_OUT_OF_RANGE);
#endif
    }
    else
    {
        RamTstState.NumTestedCell = NewNumberOfTestedCells;
    }
    SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_01();
}
#endif

/**
 * @brief            Service for RAM Test deinitialization.
 * @details
 *
 * @table            @service_id:       0x0c       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_327
 */
RAMTST_FUNC void RamTst_DeInit(void)
{
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
    if (RAMTST_EXECUTION_UNINIT == RamTstState.ExecStatus)
    {
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_DEINIT, RAMTST_E_UNINIT);
    }
    else
#endif
    {
        RamTst_Lld_DeInit();
    }
}

#if (RAMTST_GET_ALG_PARAMS_API == STD_ON)
/**
 * @brief            Service returns the ID of the current RAM Test algorithm parameter set.
 * @details
 *
 * @table            @service_id:       0x12       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           RamTst_AlgParamsIdType
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_335
 */
RAMTST_FUNC RamTst_AlgParamsIdType RamTst_GetAlgParams(void)
{
    RamTst_AlgParamsIdType RamTstAlgParamID = 0;
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
    if (RAMTST_EXECUTION_UNINIT == RamTstState.ExecStatus)
    {
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_GET_ALG_PARAMS, RAMTST_E_UNINIT);
    }
    else
#endif
    {
        RamTstAlgParamID = RamTstState.ActiveAlgParamID;
    }
    return RamTstAlgParamID;
}
#endif

#if (RAMTST_GET_EXECUTION_STATUS_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test execution status.
 * @details
 *
 * @table            @service_id:       0x04       \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           RamTst_ExecutionStatusType
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_336
 */
RAMTST_FUNC RamTst_ExecutionStatusType RamTst_GetExecutionStatus(void)
{
    return RamTstState.ExecStatus;
}
#endif

#if (RAMTST_GET_NUMBER_OF_TESTED_CELLS_API == STD_ON)
/**
 * @brief            Service returns the current number of tested cells per MainFunction cycle.
 * @details
 *
 * @table            @service_id:       0x09       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           RamTst_NumberOfTestedCellsType
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_337
 */
RAMTST_FUNC RamTst_NumberOfTestedCellsType RamTst_GetNumberOfTestedCells(void)
{
    RamTst_NumberOfTestedCellsType NumberOfTestedCells = 0U;
    if (RAMTST_EXECUTION_UNINIT == RamTstState.ExecStatus)
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_GET_NUMBER_OF_TESTED_CELLS, RAMTST_E_UNINIT);
#endif
    }
    else
    {
        NumberOfTestedCells = RamTstState.NumTestedCell;
    }
    return NumberOfTestedCells;
}
#endif

#if (RAMTST_GET_TEST_ALGORITHM_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test algorithm.
 * @details
 *
 * @table            @service_id:       0x07       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           RamTst_AlgorithmType
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_338
 */
RAMTST_FUNC RamTst_AlgorithmType RamTst_GetTestAlgorithm(void)
{
    RamTst_AlgorithmType Algorithm = RAMTST_ALGORITHM_UNDEFINED;
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
    if (RAMTST_EXECUTION_UNINIT == RamTstState.ExecStatus)
    {
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_GET_TEST_ALGORITHM, RAMTST_E_UNINIT);
    }
    else
#endif
    {
        Algorithm = RamTstState.RamTstAlgorithm;
    }
    return Algorithm;
}
#endif

#if (RAMTST_GET_TEST_RESULT_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test result.
 * @details
 *
 * @table            @service_id:       0x05       \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           RamTst_TestResultType
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_339
 */
RAMTST_FUNC RamTst_TestResultType RamTst_GetTestResult(void)
{
    RamTst_TestResultType Status = RAMTST_RESULT_NOT_TESTED;
    if (RAMTST_EXECUTION_UNINIT == RamTstState.ExecStatus)
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_GET_TEST_RESULT, RAMTST_E_UNINIT);
#endif
    }
    else
    {
        Status = RamTstState.RamTestResult;
    }
    return Status;
}
#endif

#if (RAMTST_GET_TEST_RESULT_PER_BLOCK_API == STD_ON)
/**
 * @brief            Service returns the current RAM Test result for the specified block.
 * @details
 *
 * @table            @service_id:       0x06       \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        BlockID Identifies the block
 * @return           RamTst_TestResultType
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_340
 */
RAMTST_FUNC RamTst_TestResultType RamTst_GetTestResultPerBlock(RamTst_NumberOfBlocksType BlockID)
{
    RamTst_TestResultType Status = RAMTST_RESULT_NOT_TESTED;
    if ((BlockID > RamTstState.NumBlocks) || (0U == BlockID))
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_GET_TEST_RESULT_PER_BLOCK, RAMTST_E_OUT_OF_RANGE);
#endif
    }
    else
    {
        SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_02();
        Status = RamTstState.RamTstBlockResultBuffer[BlockID - 1U];
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_02();
    }
    return Status;
}
#endif

/**
 * @brief            Service for RAM Test initialization.
 * @details
 *
 * @table            @service_id:       0x00       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        ConfigPtr Pointer to the selected configuration set.
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_341
 */
RAMTST_FUNC void RamTst_Init(const RamTst_ConfigType *ConfigPtr)
{
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR != ConfigPtr)
    {
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_INIT, RAMTST_E_PARAM_POINTER);
    }
    else if (RAMTST_EXECUTION_UNINIT != RamTstState.ExecStatus)
    {
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_INIT, RAMTST_E_STATUS_FAILURE);
    }
    else
#endif
    {
        RamTst_Lld_Init(&RamTst_Config);
    }
}

/**
 * @brief            Scheduled function for executing the RAM Test in the background.
 * @details
 *
 * @table            @service_id:       0x01       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_342
 */
RAMTST_FUNC void RamTst_MainFunction(void)
{
    uint32 CurrentBlockEndAddress;
    uint32 NumTestCell;
    Std_ReturnType BlockTestResult;
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_03();
    if (RAMTST_EXECUTION_RUNNING == RamTstState.ExecStatus)
    {
        /* Check wether the algorithm has been changed */
        if (RamTstState.MainStatePtr->MainActiveAlgParamID != RamTstState.ActiveAlgParamID)
        {
            /* update the state parameters of MainFunction used*/
            RamTstState.MainStatePtr->MainActiveAlgParamID = RamTstState.ActiveAlgParamID;
            RamTstState.ActiveAlgParamPtr = &RamTst_Config.RamTstAlgConfigPtr[RamTstState.ActiveAlgParamID - 1U];
            RamTstState.MainStatePtr->MainNumBlocks = RamTstState.ActiveAlgParamPtr->RamTstNumberOfBlocks;
            RamTstState.MainStatePtr->MainBlockTestedIndex = 0U;
            RamTstState.MainStatePtr->MainCellAddress = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[0U].RamTstStartAddress;
        }
        else
        {
            /* Get the next time cell address of MainFunction tested */
            RamTstState.CellAddress = RamTstState.MainStatePtr->MainCellAddress;
            CurrentBlockEndAddress =
                RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstEndAddress;
            /* If the current block test is finished*/
            if (RamTstState.CellAddress > CurrentBlockEndAddress)
            {
                if (RamTstState.RamTstBlockResultBuffer[RamTstState.MainStatePtr->MainBlockTestedIndex] != RAMTST_RESULT_NOT_OK)
                {
                    RamTstState.RamTstBlockResultBuffer[RamTstState.MainStatePtr->MainBlockTestedIndex] = RAMTST_RESULT_OK;
                }
                /* Test the the next block*/
                RamTstState.MainStatePtr->MainBlockTestedIndex += 1U;
                /* If all blocks are tested, testing start at beginning again*/
                if (RamTstState.MainStatePtr->MainBlockTestedIndex >= RamTstState.MainStatePtr->MainNumBlocks)
                {
                    RamTstState.MainStatePtr->MainBlockTestedIndex = 0U;
                    if (RAMTST_RESULT_NOT_OK != RamTstState.RamTestResult)
                    {
                        RamTstState.RamTestResult = RAMTST_RESULT_OK;
                        RamTst_TestCompletedNotification();
                    }
                }
                RamTstState.MainStatePtr->MainCellAddress =
                    RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstStartAddress;
            }
        }
        RamTstState.CellAddress = RamTstState.MainStatePtr->MainCellAddress;
        CurrentBlockEndAddress =
            RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstEndAddress;
        RamTstState.CurrentBlockFillPattern =
            RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstFillPattern;
        RamTstState.CurrentBlockPolicy =
            RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstPolicy;
        RamTstState.CurrentBlockRegion =
            RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstBlockRegion;
        NumTestCell = (CurrentBlockEndAddress - RamTstState.CellAddress + 1U) / 4U;
        /*For stack region it must be tested by a atom*/
        if (RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[RamTstState.MainStatePtr->MainBlockTestedIndex].RamTstBlockRegion != \
                BLOCK_REGION_IS_STACK)
        {
            /* The test cell for a cycle must be less than the backup region size*/
            if (NumTestCell > RamTstState.NumTestedCell)
            {
                NumTestCell = RamTstState.NumTestedCell;
            }
        }
        RamTstState.NumExecutedCell = NumTestCell;
        __disable_irq(); /*Disable Global Interrupt*/
#ifdef RAMTST_ENABLE_USER_MODE_SUPPORT
        BlockTestResult = (Std_ReturnType)OsIf_Trusted_Call_Return1param(RamTst_Lld_BlockTest, NumTestCell);
#else
        BlockTestResult = RamTst_Lld_BlockTest(NumTestCell);
#endif
        __enable_irq(); /*Enable Global Interrupt*/
        if (E_NOT_OK == BlockTestResult)
        {
            RamTstState.RamTstBlockResultBuffer[RamTstState.MainStatePtr->MainBlockTestedIndex] = RAMTST_RESULT_NOT_OK;
            RamTstState.RamTestResult = RAMTST_RESULT_NOT_OK;
            RamTstState.MainStatePtr->MainBlockTestedIndex += 1U;
#if (RAMTST_ENABLE_DEM_MAIN_ERROR_STATUS == STD_ON)
            (void)Dem_SetEventStatus(RAMTST_MAIN_RAM_FAILURE, DEM_EVENT_STATUS_FAILED); /*SWS_RamTst_01002*/
#endif
            RamTst_ErrorNotification();
        }
        RamTstState.CellAddress += NumTestCell * 4U;
        RamTstState.MainStatePtr->MainCellAddress = RamTstState.CellAddress;
    }
    SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_03();
}

#if (RAMTST_RESUME_API == STD_ON)
/**
 * @brief            Service for allowing to continue the background RAM Test at the point it was suspended.
 * @details
 *
 * @table            @service_id:       0x0e       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   false      \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_343
 */
RAMTST_FUNC void RamTst_Resume(void)
{
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_04();
    if (RAMTST_EXECUTION_SUSPENDED != RamTstState.ExecStatus)
    {
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_04();
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_RESUME, RAMTST_E_STATUS_FAILURE);
#endif /* RAMTST_DEV_ERROR_DETECT */
    }
    else
    {
        RamTstState.ExecStatus = RAMTST_EXECUTION_RUNNING;
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_04();
    }
}
#endif

#if (RAMTST_RUN_FULL_TEST_API == STD_ON)
/**
 * @brief            Service for executing the full RAM Test in the foreground.
 * @details
 *
 * @table            @service_id:       0x1004     \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_344
 */
RAMTST_FUNC void RamTst_RunFullTest(void)
{
    Std_ReturnType BlockTestResult = E_OK;
    uint32 CurrentBlockEndAddress = 0U;
    RamTst_NumberOfTestedCellsType NumTestCell = 0U;
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_05();
    if (RAMTST_EXECUTION_STOPPED != RamTstState.ExecStatus)
    {
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_05();
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_RUNFULL_TEST, RAMTST_E_STATUS_FAILURE);
#endif /* RAMTST_DEV_ERROR_DETECT */
    }
    else
    {
        RamTstState.ExecStatus = RAMTST_EXECUTION_RUNNING;
        RamTstState.RamTestResult = RAMTST_RESULT_UNDEFINED;
        for (uint16 BlockCount = 0; BlockCount < RamTstState.NumBlocks; ++BlockCount)
        {
            RamTstState.CellAddress = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockCount].RamTstStartAddress;
            RamTstState.CurrentBlockFillPattern = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockCount].RamTstFillPattern;
            RamTstState.CurrentBlockPolicy = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockCount].RamTstPolicy;
            RamTstState.CurrentBlockRegion = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockCount].RamTstBlockRegion;
            CurrentBlockEndAddress = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockCount].RamTstEndAddress;
            while (RamTstState.CellAddress < CurrentBlockEndAddress)
            {
                NumTestCell = (CurrentBlockEndAddress - RamTstState.CellAddress + 1U) / 4U;
                /* The test cell for a cycle must be less than the backup region size*/
                if (NumTestCell > RamTstState.MaxNumTestedCell)
                {
                    NumTestCell = RamTstState.MaxNumTestedCell;
                }
                RamTstState.NumExecutedCell = NumTestCell;
                __disable_irq(); /*Disable Global Interrupt*/
#ifdef RAMTST_ENABLE_USER_MODE_SUPPORT
                BlockTestResult = (Std_ReturnType)OsIf_Trusted_Call_Return1param(RamTst_Lld_BlockTest, NumTestCell);
#else
                BlockTestResult = RamTst_Lld_BlockTest(NumTestCell);
#endif
                __enable_irq(); /*Enable Global Interrupt*/
                if (E_NOT_OK == BlockTestResult)
                {
                    RamTstState.RamTstBlockResultBuffer[BlockCount] = RAMTST_RESULT_NOT_OK;
                    RamTstState.RamTestResult = RAMTST_RESULT_NOT_OK;
#if (RAMTST_ENABLE_DEM_RUNFL_ERROR_STATUS == STD_ON)
                    (void)Dem_SetEventStatus(RAMTST_RUNFL_RAM_FAILURE, DEM_EVENT_STATUS_FAILED); /*SWS_RamTst_00213*/
#endif
                    RamTst_ErrorNotification();
                    break;
                }
                RamTstState.CellAddress += NumTestCell * 4U;
            }
            if (RamTstState.RamTstBlockResultBuffer[BlockCount] != RAMTST_RESULT_NOT_OK)
            {
                RamTstState.RamTstBlockResultBuffer[BlockCount] = RAMTST_RESULT_OK;
            }
        }
        if (RAMTST_RESULT_UNDEFINED == RamTstState.RamTestResult)
        {
            RamTstState.RamTestResult = RAMTST_RESULT_OK;
        }
        RamTstState.ExecStatus = RAMTST_EXECUTION_STOPPED;
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_05();
    }
}
#endif

#if (RAMTST_RUN_PARTIAL_TEST_API == STD_ON)
/**
 * @brief            Service for testing one RAM block in the foreground.
 * @details
 *
 * @table            @service_id:       0x11       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        BlockId Identifies the single RAM block to be tested in the selected set of RamTstAlgParams.
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_345
 */
RAMTST_FUNC void RamTst_RunPartialTest(RamTst_NumberOfBlocksType BlockId)
{
    Std_ReturnType BlockTestResult = E_OK;
    RamTst_ExecutionStatusType OriginalExecState = RamTstState.ExecStatus;
    uint32 CurrentBlockEndAddress = 0U;
    RamTst_NumberOfTestedCellsType NumTestCell = 0U;
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_06();
    if ((RAMTST_EXECUTION_STOPPED != RamTstState.ExecStatus) && (RAMTST_EXECUTION_SUSPENDED != RamTstState.ExecStatus))
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_RUNPARTIAL_TEST, RAMTST_E_STATUS_FAILURE);
#endif
    }
    else if ((BlockId > RamTstState.NumBlocks) || (0U == BlockId))
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_RUNPARTIAL_TEST, RAMTST_E_OUT_OF_RANGE);
#endif
    }
    else
    {
        RamTstState.ExecStatus = RAMTST_EXECUTION_RUNNING;
        RamTstState.RamTstBlockResultBuffer[BlockId - 1U] = RAMTST_RESULT_UNDEFINED;
        RamTstState.CellAddress = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockId - 1U].RamTstStartAddress;
        RamTstState.CurrentBlockFillPattern = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockId - 1U].RamTstFillPattern;
        RamTstState.CurrentBlockPolicy = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockId - 1U].RamTstPolicy;
        RamTstState.CurrentBlockRegion = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockId - 1U].RamTstBlockRegion;
        CurrentBlockEndAddress = RamTstState.ActiveAlgParamPtr->RamTstBlockParamsPtr[BlockId - 1U].RamTstEndAddress;
        while (RamTstState.CellAddress < CurrentBlockEndAddress)
        {
            NumTestCell = (CurrentBlockEndAddress - RamTstState.CellAddress + 1U) / 4U;
            /* The test cell for a cycle must be less than the backup region size*/
            if (NumTestCell > RamTstState.MaxNumTestedCell)
            {
                NumTestCell = RamTstState.MaxNumTestedCell;
            }
            RamTstState.NumExecutedCell = NumTestCell;
            __disable_irq(); /*Disable Global Interrupt*/
#ifdef RAMTST_ENABLE_USER_MODE_SUPPORT
            BlockTestResult = (Std_ReturnType)OsIf_Trusted_Call_Return1param(RamTst_Lld_BlockTest, NumTestCell);
#else
            BlockTestResult = RamTst_Lld_BlockTest(NumTestCell);
#endif
            __enable_irq(); /*Enable Global Interrupt*/
            if (E_NOT_OK == BlockTestResult)
            {
                RamTstState.RamTstBlockResultBuffer[BlockId - 1U] = RAMTST_RESULT_NOT_OK;
#if (RAMTST_ENABLE_DEM_PART_ERROR_STATUS == STD_ON)
                (void)Dem_SetEventStatus(RAMTST_PART_RAM_FAILURE, DEM_EVENT_STATUS_FAILED); /*SWS_RamTst_01008*/
#endif
                break;
            }
            RamTstState.CellAddress += NumTestCell * 4U;
        }
        if (RamTstState.RamTstBlockResultBuffer[BlockId - 1U] != RAMTST_RESULT_NOT_OK)
        {
            RamTstState.RamTstBlockResultBuffer[BlockId - 1U] = RAMTST_RESULT_OK;
        }
    }
    RamTstState.ExecStatus = OriginalExecState;
    SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_06();
}
#endif

#if (RAMTST_SELECT_ALG_PARAMS_API == STD_ON)
/**
 * @brief            Service used to set the test algorithm and its parameter set.
 * @details
 *
 * @table            @service_id:       0x0b       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        NewAlgParamsId Identifies the parameter set to be used.
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_346
 */
RAMTST_FUNC void RamTst_SelectAlgParams(RamTst_AlgParamsIdType NewAlgParamsId)
{
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_07();
    if ((NewAlgParamsId > RamTstState.RamTstNumOfAlgParamSets) || (0U == NewAlgParamsId))
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_SELECT_ALG_PARAMS, RAMTST_E_OUT_OF_RANGE);
#endif
    }
    else if (RAMTST_EXECUTION_STOPPED != RamTstState.ExecStatus)
    {
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_SELECT_ALG_PARAMS, RAMTST_E_STATUS_FAILURE);
#endif
    }
    else
    {
        RamTstState.ActiveAlgParamID = NewAlgParamsId;
        RamTstState.MinNumTestedCell = RamTst_Config.RamTstMinNumberOfTestedCells;
        RamTstState.RamTstNumOfAlgParamSets = RamTst_Config.RamTstNumberOfAlgParamSets;
        RamTstState.ActiveAlgParamPtr = &RamTst_Config.RamTstAlgConfigPtr[RamTstState.ActiveAlgParamID - 1U];
        RamTstState.NumTestedCell = RamTstState.ActiveAlgParamPtr->RamTstNumberOfTestedCells;
        RamTstState.MaxNumTestedCell = RamTstState.ActiveAlgParamPtr->RamTstMaxNumberOfTestedCells;
        RamTstState.ExtNumTestedCell = RamTstState.ActiveAlgParamPtr->RamTstExtNumberOfTestedCells;
        RamTstState.NumBlocks = RamTstState.ActiveAlgParamPtr->RamTstNumberOfBlocks;
        RamTstState.RamTstAlgorithm = RamTstState.ActiveAlgParamPtr->RamTstAlgorithm;
        RamTstState.BackupStartAddress = RamTstState.ActiveAlgParamPtr->RamTstBackupStartAddress;
        for (RamTst_NumberOfBlocksType Block = 0; Block < (RamTst_NumberOfBlocksType)RAMTST_MAX_BLOCK; ++Block)
        {
            RamTstState.RamTstBlockResultBuffer[Block] = RAMTST_RESULT_NOT_TESTED;
        }
        RamTstState.RamTestResult = RAMTST_RESULT_NOT_TESTED;
    }
    SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_07();
}
#endif

#if (RAMTST_STOP_API == STD_ON)
/**
 * @brief            Service for stopping the RAM Test.
 * @details
 *
 * @table            @service_id:       0x02       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   false      \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_347
 */
RAMTST_FUNC void RamTst_Stop(void)
{
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_08();
    if ((RamTstState.ExecStatus == RAMTST_EXECUTION_UNINIT) || (RamTstState.ExecStatus == RAMTST_EXECUTION_STOPPED))
    {
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_08();
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_STOP, RAMTST_E_STATUS_FAILURE);
#endif
    }
    else
    {
        RamTstState.ExecStatus = RAMTST_EXECUTION_STOPPED;
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_08();
    }
}
#endif

#if (RAMTST_SUPSPEND_API == STD_ON)
/**
 * @brief            Service for suspending current operation of background RAM Test, until RESUME is called.
 * @details
 *
 * @table            @service_id:       0x0d       \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   false      \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_348
 */
RAMTST_FUNC void RamTst_Suspend(void)
{
    SchM_Enter_RamTst_RAMTST_EXCLUSIVE_AREA_09();
    if (RAMTST_EXECUTION_RUNNING != RamTstState.ExecStatus)
    {
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_09();
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_SUSPEND, RAMTST_E_STATUS_FAILURE);
#endif
    }
    else
    {
        RamTstState.ExecStatus = RAMTST_EXECUTION_SUSPENDED;
        SchM_Exit_RamTst_RAMTST_EXCLUSIVE_AREA_09();
    }
}
#endif

/**
 * @brief           The function RamTst_Error shall be called every time when a RAM failure has been detected
 *                  by the selected test algorithm in the background.
 * @details
 *
 * @table            @is_reentrant:     Don't care \n
 *                   @is_synchronous:   --         \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_328
 */
RAMTST_FUNC void RamTst_ErrorNotification(void)
{
#if (RAMTST_ERROR_NOTIFICATION == STD_ON)
    if (NULL_PTR != RamTst_Config.RamTstErrorNotification)
    {
        RamTst_Config.RamTstErrorNotification(RAMTST_MODULE_ID);
    }
    else
#endif
    {
        /*Do Nothing*/
    }
}

/**
 * @brief            The function RamTst_TestCompleted shall be called every time when all RAM blocks of the current test
 *                   configuration have been tested in the background.
 * @details
 *
 * @table            @is_reentrant:     Don't care \n
 *                   @is_synchronous:   --         \n
 *                   @autosar_api:      true       \n
 *                   @memory_map:       .mcal_text \n
 * @param[in]        void
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_349
 */
RAMTST_FUNC void RamTst_TestCompletedNotification(void)
{
#if (RAMTST_COMPLETED_NOTIFICATION == STD_ON)
    if (NULL_PTR != RamTst_Config.RamTstCompletedNotification)
    {
        RamTst_Config.RamTstCompletedNotification(RAMTST_MODULE_ID);
    }
    else
#endif
    {
        /*Do Nothing*/
    }
}

#if (RAMTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @details
 * @table            @service_id:       0x0a  \n
 *                   @is_reentrant:     true  \n
 *                   @is_synchronous:   true  \n
 *                   @autosar_api:      true  \n
 * @param[out]       Versioninfo Pointer to where to store version information of this module
 * @return           void
 * @sw_type          sw_detail
 *
 * @trace YTSDS_RamTst_743
 */
RAMTST_FUNC void RamTst_GetVersionInfo(Std_VersionInfoType *Versioninfo)
{
    /* Check for DET: RAMTST_E_PARAM_POINTER */
    if ((Versioninfo) == NULL_PTR)
    {
        /* Report RAMTST_E_PARAM_POINTER DET if service called with NULL_PTR
        */
#if (RAMTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(RAMTST_MODULE_ID, RAMTST_INSTANCE_ID, RAMTST_SID_GET_VERSION_INFO, RAMTST_E_PARAM_POINTER);
#endif /* RAMTST_DEV_ERROR_DETECT */
    }
    else
    {
        /* Vendor ID information */
        Versioninfo->vendorID = RAMTST_VENDOR_ID;
        /* RamTst module ID information */
        Versioninfo->moduleID = RAMTST_MODULE_ID;
        /* RamTst module Software major version information */
        Versioninfo->sw_major_version = (uint8)RAMTST_SW_MAJOR_VER;
        /* RamTst module Software minor version information */
        Versioninfo->sw_minor_version = (uint8)RAMTST_SW_MINOR_VER;
        /* RamTst module Software patch version information */
        Versioninfo->sw_patch_version = (uint8)RAMTST_SW_PATCH_VER;
    }
}
#endif /* RAMTST_VERSION_INFO_API */


#define RAMTST_STOP_SEC_CODE
#include "RamTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file RamTst.c */

