/*
* @file    RamTst_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : RamTst_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef RAMTST_TYPES_H
#define RAMTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "RamTst_Cfg.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define RAMTST_TYPES_VENDOR_ID                     (180)
#define RAMTST_TYPES_AR_REL_MAJOR_VER              (4)
#define RAMTST_TYPES_AR_REL_MINOR_VER              (4)
#define RAMTST_TYPES_AR_REL_REVISION_VER           (0)
#define RAMTST_TYPES_SW_MAJOR_VER                  (2)
#define RAMTST_TYPES_SW_MINOR_VER                  (0)
#define RAMTST_TYPES_SW_PATCH_VER                  (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and RAMTST configuration header file are of the same vendor */
#if (RAMTST_TYPES_VENDOR_ID != RAMTST_VENDOR_ID_CFG)
#error "RamTst_Types.h and RamTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and RAMTST configuration header file are of the same Autosar version */
#if ((RAMTST_TYPES_AR_REL_MAJOR_VER != RAMTST_AR_REL_MAJOR_VER_CFG) || \
     (RAMTST_TYPES_AR_REL_MINOR_VER != RAMTST_AR_REL_MINOR_VER_CFG) || \
     (RAMTST_TYPES_AR_REL_REVISION_VER != RAMTST_AR_REL_REVISION_VER_CFG))
#error "AutoSar Version Numbers of RamTst_Types.h and RamTst_Cfg.h are different"
#endif

/* Check if source file and RAMTST configuration header file are of the same software version */
#if ((RAMTST_TYPES_SW_MAJOR_VER != RAMTST_SW_MAJOR_VER_CFG) || \
     (RAMTST_TYPES_SW_MINOR_VER != RAMTST_SW_MINOR_VER_CFG) || \
     (RAMTST_TYPES_SW_PATCH_VER != RAMTST_SW_PATCH_VER_CFG))
#error "Software Version Numbers of RamTst_Types.h and RamTst_Cfg.h are different"
#endif

/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS
==================================================================================================*/
/**
 * @brief          The execution status of the RamTst module
 * @sw_type        sw_arch
 */
typedef enum
{
    RAMTST_EXECUTION_UNINIT    = 0x00U,  /**< The RAM Test is not initialized or not usable */
    RAMTST_EXECUTION_STOPPED   = 0x01U,  /**< The RAM Test is stopped and ready to be started in foreground or to be allowed in background */
    RAMTST_EXECUTION_RUNNING   = 0x02U,  /**< The RAM Test is currently running */
    RAMTST_EXECUTION_SUSPENDED = 0x03U,  /**< he background RAM Test is waiting to be resumed. */
} RamTst_ExecutionStatusType;

/**
 * @brief          The test Result of the RamTst module
 * @sw_type        sw_arch
 */
typedef enum
{
    RAMTST_RESULT_NOT_TESTED = 0x00U,  /**< The RAM Test is not executed */
    RAMTST_RESULT_OK         = 0x01U,  /**< The RAM Test has been tested with OK result */
    RAMTST_RESULT_NOT_OK     = 0x02U,  /**< The RAM Test has been tested with NOT-OK result */
    RAMTST_RESULT_UNDEFINED  = 0x03U,  /**< The RAM Test was stated, not all blocks have yet been tested and no block result is NOT-OK */
} RamTst_TestResultType;

/**
 * @brief          The algorithm type of the RamTst module
 * @sw_type        sw_arch
 */
typedef enum
{
    RAMTST_ALGORITHM_UNDEFINED = 0U,  /**< Undefined algorithm (uninitialized value) */
    RAMTST_CHECKERBOARD_TEST   = 1U,  /**< Checkerboard test algorithm */
    RAMTST_MARCH_TEST          = 2U,  /**< March test algorithm */
    RAMTST_WALK_PATH_TEST      = 3U,  /**< Walk path test algorithm */
    RAMTST_GALPAT_TEST         = 4U,  /**< Galpat test algorithm */
    RAMTST_TRANSP_GALPAT_TEST  = 5U,  /**< Transparent Galpat test algorithm */
    RAMTST_ABRAHAM_TEST        = 6U,  /**< Abraham test algorithm */
} RamTst_AlgorithmType;

/**
 * @brief          The policy regarding destruction or non-destruction of memory content
 * @sw_type        sw_arch
 */
typedef enum
{
    RAMTST_DESTRUCTIVE     = 0U,  /**< RAM test does not restore memory content */
    RAMTST_NON_DESTRUCTIVE = 1U,  /**< RAM test restores memory content */
} RamTst_BlockTestPolicyType;

/**
 * @brief          The region type of the RAM block to be tested
 * @sw_type        sw_arch
 */
typedef enum
{
    BLOCK_REGION_IS_NORMAL = 0U,  /**< RAM region is belongs to Normal block */
    BLOCK_REGION_IS_STACK  = 1U,  /**< RAM region is belongs to Stack block */
    BLOCK_REGION_IS_BACKUP = 2U,  /**< RAM region is belongs to Backup block */
} RamTst_BlockRegionType;

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/
/**
 * @brief          Data type used to identify a set of configuration parameters for a test algorithm
 * @sw_type        sw_arch
 */
typedef uint8 RamTst_AlgParamsIdType;

/**
 * @brief          Data type used to identify or count RAM blocks given in the test configuration parameters
 * @sw_type        sw_arch
 */
typedef uint16 RamTst_NumberOfBlocksType;

/**
 * @brief          Data type of number of tested RAM cells
 * @sw_type        sw_arch
 */
typedef uint32 RamTst_NumberOfTestedCellsType;

/**
 * @brief          Notification function type
 * @sw_type        sw_arch
 */
typedef void (*RamTst_NotificationType)(uint16 MdouleId);

/**
 * @brief          The Algorithm function type.
 * @sw_type        sw_arch
 */
typedef Std_ReturnType(*RamTst_AlgorithmFunctionType)(uint32 *const StartAddress, const uint32 Length);

/**
 * @brief          Block Configuration data structure
 * @sw_type        sw_arch
 */
typedef struct
{
    RamTst_NumberOfBlocksType RamTstBlockId;    /**< ID of the RAM block set*/
    RamTst_BlockRegionType RamTstBlockRegion;   /**< Region of the RAM block to be tested. */
    uint32 RamTstEndAddress;                    /**< End address of the RAM block to be tested. */
    uint32 RamTstStartAddress;                  /**< Start address of the RAM block to be tested. */
    uint32 RamTstFillPattern;                   /**< Pattern to be filled into each memory cell after destructive test of this block */
    RamTst_BlockTestPolicyType RamTstPolicy;    /**< Policy for testing the RAM block */
} RamTst_BlockConfigType;

/**
 * @brief          Algorithm Configuration data structure
 * @sw_type        sw_arch
 */
typedef struct
{
    RamTst_AlgParamsIdType RamTstAlgParamsId;           /**< ID of the RAM Test algorithm parameter set. */
    RamTst_AlgorithmType RamTstAlgorithm;               /**< The RAM Test algorithm. */
    uint32 RamTstExtNumberOfTestedCells;                /**< The absolute maximum value for the number of cells */
    uint32 RamTstMaxNumberOfTestedCells;                /**< The maximum value for the number of cells that can be tested
                                                            in one cycle of a background test.*/
    uint32 RamTstNumberOfTestedCells;                   /**< The number of RAM cells tested per MainFunction cycle*/
    RamTst_NumberOfBlocksType RamTstNumberOfBlocks;     /**< Number of RAM blocks configured using the container */
    const RamTst_BlockConfigType *RamTstBlockParamsPtr; /**< Pointer to the RAM block parameters. */
    uint32 RamTstBackupStartAddress;                    /**< Start address of the backup RAM */
    uint32 RamTstBackupEndAddress;                      /**< End address of the backup RAM */
} RamTst_AlgConfigType;

/**
 * @brief          Configuration data structure of the RamTst module
 * @sw_type        sw_arch
 */
typedef struct
{
    const RamTst_AlgConfigType *RamTstAlgConfigPtr;        /**< Pointer to the RAM Test algorithm configuration. */
    RamTst_AlgParamsIdType RamTstDefaultAlgParamsId;       /**< ID of the default RAM Test algorithm parameter set. */
    uint32 RamTstMinNumberOfTestedCells;                   /**< Minimum number of tested cells for one cycle of a background test. */
    RamTst_AlgParamsIdType RamTstNumberOfAlgParamSets;     /**< Number of configured parameter sets for the available test algorithms. */
    RamTst_NotificationType RamTstCompletedNotification;   /**< Pointer to the notification function for the completion of the RAM Test. */
    RamTst_NotificationType RamTstErrorNotification;       /**< Pointer to the notification function for the detection of a RAM failure. */
} RamTst_ConfigType;

/**
 * @brief          Extra State data structure for MainFunction test
 * @sw_type        sw_detail
 */
typedef struct
{
    RamTst_AlgParamsIdType MainActiveAlgParamID;  /**< Active algorithm parameter ID  for MainFunction*/
    RamTst_NumberOfBlocksType MainNumBlocks;      /**< Number of blocks to be tested */
    uint32 MainBlockTestedIndex;                  /**< Index of the block currently tested */
    uint32 MainCellAddress;                       /**< Address of the cell currently tested */
} RamTst_MainStateType;

/**
 * @brief          State data structure
 * @sw_type        sw_arch
 */
typedef struct
{
    RamTst_AlgParamsIdType ActiveAlgParamID;                         /**< Active algorithm parameter ID */
    const RamTst_AlgConfigType *ActiveAlgParamPtr;                   /**< Pointer to the active algorithm parameter */
    RamTst_AlgParamsIdType RamTstNumOfAlgParamSets;                  /**< Number of configured parameter sets for the available test algorithms */
    RamTst_AlgorithmType RamTstAlgorithm;                            /**< RAM Test algorithm */
    uint32 BackupStartAddress;                                       /**< Start address of the backup RAM */
    RamTst_NumberOfBlocksType NumBlocks;                             /**< Number of blocks to be tested */
    uint32 NumTestedCell;                                            /**< Number of cells tested or one cycle of a background test.*/
    uint32 NumExecutedCell;                                          /**< Number of cells executed */
    uint32 MaxNumTestedCell;                                         /**< Maximum number of cells tested */
    uint32 MinNumTestedCell;                                         /**< Minimum number of cells tested */
    uint32 ExtNumTestedCell;                                         /**< The absolute maximum value for the number of cells */
    uint32 CurrentBlockFillPattern;                                  /**< Pattern to be filled into each memory cell after destructive test of this
                                                                          block */
    RamTst_BlockTestPolicyType CurrentBlockPolicy;                   /**< Policy for testing the RAM block */
    RamTst_BlockRegionType CurrentBlockRegion;                       /**< Region of the RAM block to be tested */
    uint32 CellAddress;                                              /**< Address of the cell currently tested */
    RamTst_ExecutionStatusType ExecStatus;                           /**< Execution status of the RAM Test */
    RamTst_TestResultType RamTstBlockResultBuffer[RAMTST_MAX_BLOCK]; /**< Buffer for the results of each RAM block tests */
    RamTst_TestResultType RamTestResult;                             /**< Result of the overall RAM Test */
    RamTst_MainStateType *MainStatePtr;                              /**< Pointer to the MainFunction state */
} RamTst_StateType;

/**
 * @brief          The Algorithm test for specific block using for Lld layer.
 * @sw_type        sw_detail
 */
typedef struct
{
    uint32 CellAddress;                              /**< Address of the cell to be tested. */
    uint32 NumberCells;                              /**< Number of cells to be tested. */
    RamTst_BlockTestPolicyType CurrentBlockPolicy;   /**< Current block policy. */
    RamTst_BlockRegionType CurrentBlockRegion;       /**< Current block region. */
    uint32 FillData;                                 /**< Data to be written fot the tested cell if the block policy is RAMTST_DESTRUCTIVE. */
    uint32 BackupAddress;                            /**< Backup address (backup start address) */
} RamTst_AlgTstType;

/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/

#ifdef __cplusplus
}
#endif

#endif /* End of file RamTst_Types.h */

