/*
* @file    PpuTst_Lld.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : PpuTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef PPUTST_LLD_H
#define PPUTST_LLD_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "PpuTst_Types.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define PPUTST_LLD_VENDOR_ID                     (180)
#define PPUTST_LLD_AR_REL_MAJOR_VERSION          (4)
#define PPUTST_LLD_AR_REL_MINOR_VERSION          (4)
#define PPUTST_LLD_AR_REL_REVISION_VERSION       (0)
#define PPUTST_LLD_SW_MAJOR_VERSION              (2)
#define PPUTST_LLD_SW_MINOR_VERSION              (0)
#define PPUTST_LLD_SW_PATCH_VERSION              (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and PPUTST TYPES header file are of the same vendor */
#if (PPUTST_LLD_VENDOR_ID != PPUTST_TYPES_VENDOR_ID)
#error "PpuTst_Lld.h and PpuTst_Types.h have different vendor ids"
#endif

/* Check if source file and PPUTST TYPES header file are of the same Autosar version */
#if ((PPUTST_LLD_AR_REL_MAJOR_VERSION != PPUTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (PPUTST_LLD_AR_REL_MINOR_VERSION != PPUTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (PPUTST_LLD_AR_REL_REVISION_VERSION != PPUTST_TYPES_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of PpuTst_Lld.h and PpuTst_Types.h are different"
#endif

/* Check if source file and PPUTST TYPES header file are of the same Software version */
#if ((PPUTST_LLD_SW_MAJOR_VERSION != PPUTST_TYPES_SW_MAJOR_VERSION) || \
     (PPUTST_LLD_SW_MINOR_VERSION != PPUTST_TYPES_SW_MINOR_VERSION) || \
     (PPUTST_LLD_SW_PATCH_VERSION != PPUTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of PpuTst_Lld.h and PpuTst_Types.h are different"
#endif

/*==================================================================================================
 *                                         DEFINES AND MACROS
==================================================================================================*/
/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                                               ENUMS
==================================================================================================*/
/*==================================================================================================
 *                                         FUNCTION PROTOTYPES
==================================================================================================*/
#define PPUTST_START_SEC_CODE
#include "PpuTst_MemMap.h"

/**
 * @brief          This function initializes the soft lock and hard lock protections for the PPU group.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 */
PPUTST_FUNC void PpuTst_Lld_Init(const PpuTst_GroupSLockConfigType GroupConfig);

/**
 * @brief          This function unlocks the soft lock bit for a given PPU group configuration.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 */
PPUTST_FUNC void PpuTst_LLd_UnlockSoftLockBit(const PpuTst_GroupSLockConfigType GroupConfig);
/**
 * @brief          Updates the protected register for a given PPU group configuration.
 * @details        The function writes the register to inject the soft lock error.
 *                 Once LOCK_BIT is set, any write access to the corresponding normal register will cause transfer error.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 */
PPUTST_FUNC void PpuTst_Lld_UpdateProtectedRegister(const PpuTst_GroupSLockConfigType GroupConfig);

/**
 * @brief          Enable the hard lock bit for a given module group.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 */
PPUTST_FUNC void PpuTst_Lld_EnableHLockBit(const PpuTst_GroupSLockConfigType GroupConfig);

/**
 * @brief          Updates the soft lock bit for a given PPU group configuration.
 *
 * @table          @is_reentrant:     false  \n
 *                 @is_synchronous:   true   \n
 *                 @autosar_api:      false  \n
 *
 * @param[in]      GroupConfig The configuration of the PPU group.
 * @return         void
 * @sw_type        sw_detail
 */
PPUTST_FUNC void PpuTst_Lld_UpdateSLockBit(const PpuTst_GroupSLockConfigType GroupConfig);

#define PPUTST_STOP_SEC_CODE
#include "PpuTst_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* End of file PpuTst_Lld.h */
