/**
* @file    IntTst_Lld_Intm.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : IntTst_Lld_Intm
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "IntTst_Lld_Intm.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define INTTST_LLD_INTM_VENDOR_ID_C                     (180)
#define INTTST_LLD_INTM_AR_REL_MAJOR_VER_C              (4)
#define INTTST_LLD_INTM_AR_REL_MINOR_VER_C              (4)
#define INTTST_LLD_INTM_AR_REL_REVISION_VER_C           (0)
#define INTTST_LLD_INTM_SW_MAJOR_VER_C                  (2)
#define INTTST_LLD_INTM_SW_MINOR_VER_C                  (0)
#define INTTST_LLD_INTM_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and INTTST_LLD_INTM header file are of the same vendor */
#if (INTTST_LLD_INTM_VENDOR_ID_C != INTTST_LLD_INTM_VENDOR_ID)
#error "IntTst_Lld_Intm.c and IntTst_Lld_Intm.h have different vendor ids"
#endif

/* Check if source file and INTTST_LLD_INTM header file are of the same Autosar version */
#if (( INTTST_LLD_INTM_AR_REL_MAJOR_VER_C != INTTST_LLD_INTM_AR_REL_MAJOR_VER) || \
      ( INTTST_LLD_INTM_AR_REL_MINOR_VER_C != INTTST_LLD_INTM_AR_REL_MINOR_VER) || \
      ( INTTST_LLD_INTM_AR_REL_REVISION_VER_C != INTTST_LLD_INTM_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of IntTst_Lld_Intm.c and IntTst_Lld_Intm.h are different"
#endif

/* Check if source file and INTTST_LLD_INTM header file are of the same Software version */
#if (( INTTST_LLD_INTM_SW_MAJOR_VER_C != INTTST_LLD_INTM_SW_MAJOR_VER) || \
      ( INTTST_LLD_INTM_SW_MINOR_VER_C != INTTST_LLD_INTM_SW_MINOR_VER) || \
      ( INTTST_LLD_INTM_SW_PATCH_VER_C != INTTST_LLD_INTM_SW_PATCH_VER))
#error "Software Version Numbers of IntTst_Lld_Intm.c and IntTst_Lld_Intm.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

#define INTTST_START_SEC_CODE
#include "IntTst_MemMap.h"

/**
 * @brief      Initialize the INTM module
 * @param[in]  Base The INTM module register base address
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_685
 */
INTTST_FUNC void IntTst_LLd_Intm_Init(volatile INTM_Type *Base)
{
    /* enable the INTM module*/
    Base->GCTRL = INTM_GCTRL_GEN_MASK;
}

/**
 * @brief   Acknowledge the INTM interrupt by writing the IRQ number to the GINTACK register
 *
 * @param[in]  Base      The INTM module register base address
 * @param[in]  Channel   The INTM channel number
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:      .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_681
 */
INTTST_FUNC void IntTst_Lld_Intm_AckIrq(volatile INTM_Type *Base, uint8 Channel)
{
    /* clean the INTM channel timer by write zero */
    Base->MON[(uint8)Channel].TMR = 0U;
    /* write the IRQ number from SEL register config to ACK the INTM interrupt */
    Base->GINTACK = ((uint32)Base->MON[(uint8)Channel].SEL) & INTM_GINTACK_SRC_MASK;
    /* clean the ACK IRQ number for the next monitor capture */
    Base->GINTACK = ((uint32)0U) & INTM_GINTACK_SRC_MASK;
}

/**
 * @brief   Configure the INTM monitor channel
 *
 * @param[in]  Base         The INTM module register base address
 * @param[in]  Channel      The INTM channel number
 * @param[in]  IrqNumber   The INTM channel monitor peripheral IRQ interrupt number
 * @param[in]  Irq_Latency  The INTM channel monitor IRQ latency threshold value
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:      .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_682
 */
INTTST_FUNC void IntTst_Lld_Intm_ChannelConfig(volatile INTM_Type *Base, uint8 Channel, IRQn_Type IrqNumber, uint32 Irq_Latency)
{
    /* set the threshold value for the monitor channel */
    Base->MON[(uint8)Channel].THD = (Irq_Latency & INTM_MON_THD_THRESHOLD_MASK);
    /* select the IRQ number to be monitored and enable the monitor channel*/
    Base->MON[(uint8)Channel].SEL = INTM_MON_SEL_EN_MASK | ((uint32)IrqNumber & INTM_MON_SEL_IRQ_MASK);
}

/**
 * @brief   Reset the INTM monitor channel configuration
 *
 * @param[in]  Base         The INTM module register base address
 * @param[in]  Channel      The INTM channel number
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:      .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_686
 */
INTTST_FUNC void IntTst_Lld_Intm_ResetChannelConfig(volatile INTM_Type *Base, uint8 Channel)
{
    /* reset the threshold value for the monitor channel */
    Base->MON[(uint8)Channel].THD = INTM_MON_THD_THRESHOLD_MASK;
    /* reset the IRQ number to be monitored and disable the monitor channel*/
    Base->MON[(uint8)Channel].SEL = INTM_MON_SEL_EN(0) | INTM_MON_SEL_IRQ_MASK;
}

/**
 * @brief   Get the status of the INTM monitor channel
 *
 * @param[in]  Base         The INTM module register base address
 * @param[in]  Channel      The INTM channel number
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:      .mcal_text \n
 * @return boolean
 * @retval TRUE  The monitor channel is enabled and the status is overflow
 * @retval FALSE The monitor channel is disabled or the status is not overflow
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_684
 */
INTTST_FUNC boolean IntTst_Lld_Intm_GetChannelStatus(const volatile INTM_Type *Base, uint8 Channel)
{
    boolean Status = FALSE;
    /* get the status of the monitor channel and check it with the enable bit */
    if ((INTM_MON_STS_OF_MASK == (Base->MON[(uint8)Channel].STS & INTM_MON_STS_OF_MASK)) && \
            (INTM_MON_SEL_EN_MASK == (Base->MON[(uint8)Channel].SEL & INTM_MON_SEL_EN_MASK)))
    {
        Status = TRUE;
    }
    else
    {
        Status = FALSE;
    }
    return Status;
}

/**
 * @brief   Deinitialize the INTM module
 *
 * @param[in]  Base       The INTM module register base address
 * @table       @service_id:       --  \n
                @is_reentrant:     false  \n
                @is_synchronous:   true  \n
                @autosar_api:      false  \n
                @memory_map:      .mcal_text \n
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTSDS_IntTst_683
 */
INTTST_FUNC void IntTst_LLd_Intm_DeInit(volatile INTM_Type *Base)
{
    /* disable the INTM module*/
    Base->GCTRL = 0; /* clean the GEN bit */
}

#define INTTST_STOP_SEC_CODE
#include "IntTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file IntTst_Lld_Intm.c */
