/*
* @file    IntTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : IntTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef INTTST_H
#define INTTST_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "IntTst_Cfg.h"
#include "IntTst_Types.h"

#if (INTTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define INTTST_VENDOR_ID                      (180)
#define INTTST_AR_RELEASE_MAJOR_VERSION       (4)
#define INTTST_AR_RELEASE_MINOR_VERSION       (4)
#define INTTST_AR_RELEASE_REVISION_VERSION    (0)
#define INTTST_SW_MAJOR_VERSION               (2)
#define INTTST_SW_MINOR_VERSION               (0)
#define INTTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/

/* Check if header file and IntTst configuration header file are of the same vendor */
#if (INTTST_VENDOR_ID != INTTST_VENDOR_ID_CFG)
#error "IntTst.h and IntTst_Cfg.h have different vendor ids"
#endif

/* Check if header file and IntTst configuration header file are of the same Autosar version */
#if ((INTTST_AR_RELEASE_MAJOR_VERSION != INTTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (INTTST_AR_RELEASE_MINOR_VERSION != INTTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (INTTST_AR_RELEASE_REVISION_VERSION != INTTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of IntTst.h and IntTst_Cfg.h are different"
#endif

/* Check if header file and IntTst configuration header file are of the same software version */
#if ((INTTST_SW_MAJOR_VERSION != INTTST_SW_MAJOR_VERSION_CFG) || \
     (INTTST_SW_MINOR_VERSION != INTTST_SW_MINOR_VERSION_CFG) || \
     (INTTST_SW_PATCH_VERSION != INTTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of IntTst.h and IntTst_Cfg.h are different"
#endif

/* Check if header file and IntTst Types header file are of the same vendor */
#if (INTTST_VENDOR_ID != INTTST_TYPES_VENDOR_ID)
#error "IntTst.h and IntTst_Types.h have different vendor ids"
#endif

/* Check if header file and IntTst Types header file are of the same Autosar version */
#if ((INTTST_AR_RELEASE_MAJOR_VERSION != INTTST_AR_RELEASE_MAJOR_VERSION) || \
     (INTTST_AR_RELEASE_MINOR_VERSION != INTTST_AR_RELEASE_MINOR_VERSION) || \
     (INTTST_AR_RELEASE_REVISION_VERSION != INTTST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of IntTst.h and IntTst_Types.h are different"
#endif

/* Check if header file and IntTst Types header file are of the same software version */
#if ((INTTST_SW_MAJOR_VERSION != INTTST_TYPES_SW_MAJOR_VERSION) || \
     (INTTST_SW_MINOR_VERSION != INTTST_TYPES_SW_MINOR_VERSION) || \
     (INTTST_SW_PATCH_VERSION != INTTST_TYPES_SW_PATCH_VERSION))
#error "Software Version Numbers of IntTst.h and IntTst_Types.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

#define INTTST_INSTANCE_ID ((uint8)0U)       /** @brief INTTST MODULE INSTANCE ID */
#define INTTST_MODULE_ID   ((uint16)519U)    /** @brief INTTST MODULE ID */

#if (INTTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup        IntTst_Dev_Error
 * @brief           IntTst development error codes definition
 * @sw_type         sw_arch
 * @{
 */
#define INTTST_E_INVALID_STATE                 ((uint8)0x10)  /**< All API's called in wrong sequence shall return this error*/
#define INTTST_E_PARAM_POINTER                 ((uint8)0x11)  /**< Error in param pointer */
#define INTTST_E_UNINIT                        ((uint8)0x12)  /**< API called without module initialization */
/** @} */
#endif

/**
* @defgroup        IntTst_SID
* @brief           Service ID number for all IntTst driver services
* @sw_type         sw_arch
* @{
*/
#define INTTST_SID_INIT                        (0x01U)  /**< API Service ID for IntTst_Init */
#define INTTST_SID_DEINIT                      (0x02U)  /**< API Service ID for IntTst_DeInit */
#define INTTST_SID_START                       (0x03U)  /**< API Service ID for IntTst_Start */
#define INTTST_SID_GET_RESULT                  (0x04U)  /**< API Service ID for IntTst_GetResult */
#define INTTST_SID_GET_DETAILS_RESULT          (0x05U)  /**< API Service ID for IntTst_GetDetailsResult */
#define INTTST_SID_GET_VERSION_INFO            (0x06U)  /**< API Service ID for IntTst_GetVersionInfo */
/** @} */

#define INTTST_ERROR_CHECK_TIMEOUT             (0x1000U)  /**@brief INTM channel error check timeout*/

/* it should less than 12 CPU clock cycles to ensure the error injection is seccssfull */
#define INTTST_INTM_CHANNEL_LATENCY_COUNT      (2U)      /**@brief the INTM channel interrupt monitor latency count */

#define INTTST_PTMR_CHANNEL_TIMEOUT_VALUE      (500U)    /**@brief the pTMR channel timeout interrupt period, unit is slow-bus clock */

/*==================================================================================================
 *                                             ENUMS
==================================================================================================*/

/*==================================================================================================
 *                                          VARIATES
==================================================================================================*/

/*==================================================================================================
 *                                   EXTERNAL CONSTANTS
==================================================================================================*/

#if (INTTST_PRECOMPILE_SUPPORT == STD_ON)
/** @brief   IntTst driver configuration structure */
extern const IntTst_ConfigType IntTst_PreCompileConfig;
#else
/** @brief   IntTst driver configuration structure */
extern const IntTst_ConfigType IntTst_Config;
#endif

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/
/**
* @brief            Service for initialization of the IntTst driver.
* @details          This function initializes the run-time state structure to provide the INTM channel allocation,
*                   pTMR channel interrupt configuration as trigger sources, and track the state for channels.
*                   It also resets the INTM modules, initializes the module to user-defined settings and default settings.
*
* @table            @service_id:       0x01  \n
*                   @is_reentrant:     false \n
*                   @is_synchronous:   true  \n
*                   @autosar_api:      false \n
* @param[in]        IntTstConfig Pointer to the IntTst configuration structure.
* @return           void
* @sw_type          sw_arch
*
*/
void IntTst_Init(const IntTst_ConfigType *IntTstConfig);

/**
* @brief           Service for de-initialization of the IntTst driver.
* @details         This function resets the INTM module to reset state and disables the interrupt to the core.

* @table            @service_id:       0x02  \n
*                   @is_reentrant:     false \n
*                   @is_synchronous:   true  \n
*                   @autosar_api:      false \n
* @param[in]        void
* @return           void
* @sw_type          sw_arch
*
*/
void IntTst_DeInit(void);

/**
 * @brief          Service for the startup test function of the IntTst driver.
 * @details        This function is the startup test  function of the INTM test module. It performs various error checks
 *                 on the configured INTM monitor channels and updates the error status accordingly. It also
 *                 handles timeout scenarios and deinitialize the INTM monitor channel and pTMR channel after the test.
 *                 The overall result of the INTM test is updated based on the error status.
 *
 * @table          @service_id:       0x03  \n
 *                 @is_reentrant:     false \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 * @param[in]      void
 * @return         void
 * @sw_type        sw_arch
 *
 */
void IntTst_Start(void);

/**
 * @brief     Service to get the result of the IntTst execution.
 * @details   It returns the result of the IntTst execution with the IntTst module
 *            state check, and only if the IntTst module state is INTTST_TESTED or INTTST_TESTING,
 *            it will return the actual result of the IntTst execution.
 *
 * @table          @service_id:       0x04  \n
 *                 @is_reentrant:     false \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 *
 * @param     void
 * @return    IntTst_ResultType
 * @retval    INTTST_RESULT_NOT_TESTED - Test is not executed.
 * @retval    INTTST_RESULT_OK - All the test case of injection are passed.
 * @retval    INTTST_RESULT_NOT_OK - At least one test case of error injection is failed.
 * @sw_type   sw_arch
 */
IntTst_ResultType IntTst_GetResult(void);

/**
 * @brief          Service for get the test details of the IntTst execution
 * @details        This function is used to get the error details of the IntTst execution.
 *
 * @table          @service_id:       0x05  \n
 *                 @is_reentrant:     false \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 *
 * @param[inout]   IntTstDetails Pointer to where to store the test details.
 * @return         void
 * @sw_type        sw_arch
 */
void IntTst_GetDetailsResult(IntTst_TestDetailsType *IntTstDetails);

#if (INTTST_VERSION_INFO_API == STD_ON)
/**
 * @brief          Service for get the version information of this module
 * @details        This function is used to get the version information of this module.
 *
 * @table          @service_id:       0x06  \n
 *                 @is_reentrant:     true  \n
 *                 @is_synchronous:   true  \n
 *                 @autosar_api:      false \n
 * @param[out]     versioninfo Pointer to where to store version information of this module
 * @return         void
 * @sw_type        sw_arch
 */
void IntTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* INTTST_VERSION_INFO_API == STD_ON */

#ifdef __cplusplus
}
#endif

#endif /* End of file IntTst.h */

