/**
 * @file    FlsTst.c
 * @version
 *
 * @brief   AUTOSAR FlsTst module interface
 * @details API implementation for FLSTST driver
 *
 * @addtogroup FLSTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : FlsTst
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 0759 Rule-19.2: An object of union type has been defined.
 *
 * PRQA S 0306 Rule-11.4: A conversion should not be performed between a pointer to object
 *                        and an integer type.
 *
 * PRQA S 3415 Rule-13.5: The right hand operand of a logical && or || operator shall not contain
 *                         persistent side effects.
 *
 */
/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "FlsTst.h"
#if defined (FLSTST_E_FLSTST_FAILURE)
#include "Dem.h"
#endif /* #if defined (FLSTST_E_FLSTST_FAILURE) */
#include "SchM_FlsTst.h"
#include "Crc.h"
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* FLSTST_DEV_ERROR_DETECT == STD_ON */

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define FLSTST_VENDOR_ID_C                      (0xB4)
#define FLSTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define FLSTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define FLSTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define FLSTST_SW_MAJOR_VERSION_C               (2)
#define FLSTST_SW_MINOR_VERSION_C               (0)
#define FLSTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and FLSTST header file are of the same vendor */
#if (FLSTST_VENDOR_ID_C != FLSTST_VENDOR_ID)
#error "FlsTst.c and FlsTst.h have different vendor ids"
#endif

/* Check if source file and FLSTST header file are of the same Autosar version */
#if ((FLSTST_AR_RELEASE_MAJOR_VERSION_C != FLSTST_AR_RELEASE_MAJOR_VERSION) || \
     (FLSTST_AR_RELEASE_MINOR_VERSION_C != FLSTST_AR_RELEASE_MINOR_VERSION) || \
     (FLSTST_AR_RELEASE_REVISION_VERSION_C != FLSTST_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of FlsTst.c and FlsTst.h are different"
#endif

/* Check if source file and FLSTST header file are of the same Software version */
#if ((FLSTST_SW_MAJOR_VERSION_C != FLSTST_SW_MAJOR_VERSION) || \
     (FLSTST_SW_MINOR_VERSION_C != FLSTST_SW_MINOR_VERSION) || \
     (FLSTST_SW_PATCH_VERSION_C != FLSTST_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of FlsTst.c and FlsTst.h are different"
#endif

/* Check if source file and FLSTST Schm header file are of the same vendor */
#if (FLSTST_VENDOR_ID_C != SCHM_FLSTST_VENDOR_ID)
#error "FlsTst.c and SchM_FlsTst.h have different vendor ids"
#endif

/* Check if source file and FLSTST Schm header file are of the same Autosar version */
#if ((FLSTST_AR_RELEASE_MAJOR_VERSION_C != SCHM_FLSTST_AR_RELEASE_MAJOR_VERSION) || \
     (FLSTST_AR_RELEASE_MINOR_VERSION_C != SCHM_FLSTST_AR_RELEASE_MINOR_VERSION) || \
     (FLSTST_AR_RELEASE_REVISION_VERSION_C != SCHM_FLSTST_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of FlsTst.c and SchM_FlsTst.h are different"
#endif

/* Check if source file and FLSTST Schm header file are of the same Software version */
#if ((FLSTST_SW_MAJOR_VERSION_C != SCHM_FLSTST_SW_MAJOR_VERSION) || \
     (FLSTST_SW_MINOR_VERSION_C != SCHM_FLSTST_SW_MINOR_VERSION) || \
     (FLSTST_SW_PATCH_VERSION_C != SCHM_FLSTST_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of FlsTst.c and SchM_FlsTst.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#define FLSTST_START_SEC_VAR_CLEARED_32
#include "FlsTst_MemMap.h"
/* Variable to store the interval id for background test*/
FLSTST_VAR static uint32 FlsTst_BgndTestIntervalId;
#define FLSTST_STOP_SEC_VAR_CLEARED_32
#include "FlsTst_MemMap.h"

#define FLSTST_START_SEC_VAR_INIT_32
#include "FlsTst_MemMap.h"
/* Variable FlsTst_ConfigPtr to store data of FLSTST unit */
#if (FLSTST_PRECOMPILE_SUPPORT == STD_ON)
extern const FlsTst_ConfigType FlsTst_PreCompileConfig;
FLSTST_VAR static const FlsTst_ConfigType *FlsTst_ConfigPtr = NULL_PTR;
#else
FLSTST_VAR static const FlsTst_ConfigType *FlsTst_ConfigPtr = NULL_PTR;
#endif /* #if (FLSTST_PRECOMPILE_SUPPORT == STD_ON) */
#define FLSTST_STOP_SEC_VAR_INIT_32
#include "FlsTst_MemMap.h"

#define FLSTST_START_SEC_VAR_INIT_8
#include "FlsTst_MemMap.h"
/* variable  to store  FlsTst execution  state  */
FLSTST_VAR static volatile FlsTst_StateType FlsTst_CurrentState = FLSTST_UNINIT;
#define FLSTST_STOP_SEC_VAR_INIT_8
#include "FlsTst_MemMap.h"

#define FLSTST_START_SEC_VAR_CLEARED_8
#include "FlsTst_MemMap.h"
/* Variable to store the test execution status*/
FLSTST_VAR static uint8 FlsTst_TestComplete;
#define FLSTST_STOP_SEC_VAR_CLEARED_8
#include "FlsTst_MemMap.h"

#define FLSTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "FlsTst_MemMap.h"
/*  Variable to store all global variables as a structure */
FLSTST_VAR static FlsTst_BgndProcessDataType FlsTst_Var;
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
/* Variable to store the FlsTst test result  */
FLSTST_VAR static FlsTst_ErrorDetailsType FlsTst_ErrDetail;
#endif /* #if (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */
/* Variable to store all background blocks's test result for last interval */
FLSTST_VAR static FlsTst_BgndTestResultType FlsTst_LastBgndTestResult;
/* Variable to store all backgournd blokcs's test result for the runnning interval*/
FLSTST_VAR static FlsTst_BgndTestResultType FlsTst_CurBgndTestResult;
/* Variable to store the foreground test result  */
FLSTST_VAR static FlsTst_TestResultFgndType FlsTst_LastFgndResult;
/*Variable to store the signature result for last foreground test */
FLSTST_VAR static FlsTst_TestSignatureFgndType FlsTst_LastFgndSignature;
#define FLSTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "FlsTst_MemMap.h"

/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#define FLSTST_START_SEC_CODE
#include "FlsTst_MemMap.h"
/**
 * @brief          Service for reset the background test status variable.
 * @table            @service_id:       --  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_532
*/

FLSTST_FUNC LOCAL_INLINE void FlsTst_ResetBgndTestVar(void)
{
    FlsTst_Var.BgndConfigIndex = FLSTST_ZERO;
    FlsTst_Var.CalculatedCrc = FLSTST_CRC32_INITIAL_VALUE;
    FlsTst_Var.ReadAddress = FlsTst_ConfigPtr->BgndBlkConfig[0].TestBlockBaseAddress;
    FlsTst_Var.BgndCheckState = FLSTST_BGND_CHECK_INIT;
    FlsTst_CurBgndTestResult.SumResultFlag = FLSTST_RESULT_NOT_TESTED;
}

/**
 * @brief          Service for cache the result of last completed interval.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_519
*/
FLSTST_FUNC static void FlsTst_CacheBgndTestResult(void)
{
    uint32 LoopCnt = 0;
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
    if (FlsTst_CurBgndTestResult.SumResultFlag == FLSTST_RESULT_NOT_OK)
    {
        FlsTst_ErrDetail.BgndErrDetail.BgndErrIntervalId = FlsTst_CurBgndTestResult.TestIntervalId;
        FlsTst_ErrDetail.BgndErrDetail.BgndErrBlockCount = 0U;
    }
#endif /* #if (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */
    while (LoopCnt < FlsTst_ConfigPtr->MaxBgndBlkNo)
    {
        FlsTst_LastBgndTestResult.TestResultFlag[LoopCnt] = FlsTst_CurBgndTestResult.TestResultFlag[LoopCnt];
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
        FlsTst_LastBgndTestResult.TestSignature[LoopCnt].Crc32Data = FlsTst_CurBgndTestResult.TestSignature[LoopCnt].Crc32Data;
#endif /* #if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON) */
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
        if (FLSTST_RESULT_NOT_OK == FlsTst_CurBgndTestResult.TestResultFlag[LoopCnt])
        {
            FlsTst_ErrDetail.BgndErrDetail.BgndErrBlock[FlsTst_ErrDetail.BgndErrDetail.BgndErrBlockCount].Index = LoopCnt;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
            FlsTst_ErrDetail.BgndErrDetail.BgndErrBlock[FlsTst_ErrDetail.BgndErrDetail.BgndErrBlockCount].ErrSignature = \
                    FlsTst_CurBgndTestResult.TestSignature[LoopCnt].Crc32Data;
#endif /* #if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON) */
            ++FlsTst_ErrDetail.BgndErrDetail.BgndErrBlockCount;
        }
#endif /* #if (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */
        ++LoopCnt;
    }
    FlsTst_LastBgndTestResult.SumResultFlag = FlsTst_CurBgndTestResult.SumResultFlag;
    FlsTst_LastBgndTestResult.SumSignature = FlsTst_CurBgndTestResult.SumSignature;
    FlsTst_LastBgndTestResult.TestIntervalId = FlsTst_CurBgndTestResult.TestIntervalId;
}

/**
 * @brief           Service for calculate the crc for FlsTst.
 * @param Algorithm      Algorithm type
 * @param StartAddress   Start address of crc calculation
 * @param Lengh          Length of crc calculation
 * @param StoredCrc      Initial crc value,if FirstTimeFlag is FALSE,StoredCrc is the crc value of last calculation
 * @param FirstTimeFlag      TRUE:First time to calculate crc,FALSE:Not first time to calculate crc
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return uint32       Calculated crc value
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_520
 */
/* M3CM 4700 VIOLATION:This function  is for the calculation of CRC signature value. Several parameters need to be inputted,
 * including address, length, algorithm and initial value. Therefore, the STPAR cannot follow the standard.
*/
/*PRQA S 4700 ++*/
FLSTST_FUNC static uint32 FlsTst_CalculateCrc(FlsTst_TestAlgorithmType Algorithm, uint32 StartAddress, uint32 Lengh, \
        uint32 StoredCrc, boolean FirstTimeFlag) /*PRQA S 4700 --*/
{
    /* MR12 RULE 19.2 VIOLATION: The TempCrc need to be adapted to different algorithm results,
     * thus, couldn't adhere to M3CM Rule-19.2
     */
    FlsTst_CrcConvertType TempCrc; /*PRQA S 0759*/
    TempCrc.Crc32Data = FLSTST_CRC32_INITIAL_VALUE;
    /* Get the calculated CRC value from CRC unit*/
    if (FLSTST_16BIT_CRC == Algorithm)
    {
        /* MR12 RULE 11.4 VIOLATION: The parameter StartAddress is a configure and cacultle result in YTM Config tools,
         * thus, couldn't adhere to M3CM Rule-11.4
         */
        /* Select CRC16 and claculate CRC checksum*/
        TempCrc.Crc16Data = Crc_CalculateCRC16((const uint8 *)StartAddress, Lengh, (uint16)StoredCrc, FirstTimeFlag);/*PRQA S 0306*/
    }
    else if (FLSTST_32BIT_CRC == Algorithm)
    {
        /* MR12 RULE 11.4 VIOLATION: The parameter StartAddress is a configure and cacultle result in YTM Config tools,
        * thus, couldn't adhere to M3CM Rule-11.4
        */
        /* Select CRC32 and claculate CRC checksum*/
        TempCrc.Crc32Data = Crc_CalculateCRC32((const uint8 *)StartAddress, Lengh, StoredCrc, FirstTimeFlag);/*PRQA S 0306*/
    }
    else if (FLSTST_8BIT_CRC == Algorithm)
    {
        /* MR12 RULE 11.4 VIOLATION: The parameter StartAddress is a configure and cacultle result in YTM Config tools,
         * thus, couldn't adhere to M3CM Rule-11.4
         */
        /* Select CRC8 and claculate CRC checksum*/
        TempCrc.Crc8Data = Crc_CalculateCRC8((const uint8 *)StartAddress, Lengh, (uint8)StoredCrc, FirstTimeFlag);/*PRQA S 0306*/
    }
    else
    {
        /* No action required */
    }
    return (TempCrc.Crc32Data);
}
/**
 * @brief Read prestored signature value from specified address.
 *
 * @param SignatureAddress  Address of signature
 * @param Algorithm      Test Algorithm type
 * @table            @service_id:       --  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return FlsTst_CrcConvertType  Signature value
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_524
*/

FLSTST_FUNC static FlsTst_CrcConvertType FlsTst_GetSignatureFromFlash(uint32 SignatureAddress, FlsTst_TestAlgorithmType Algorithm)
{
    /* MR12 RULE 19.2 VIOLATION: The TespSignature need to be adapted to different algorithm results,
     * thus, couldn't adhere to M3CM Rule-19.2
     */
    FlsTst_CrcConvertType TespSignature;/*PRQA S 0759*/
    TespSignature.Crc32Data = FLSTST_CRC32_INITIAL_VALUE;
    /* Get the signature value from the flash */
    switch (Algorithm)
    {
        /* MR12 RULE 11.4 VIOLATION: The parameter SignatureAddress is a configure and cacultle result in YTM Config tools,
            * thus, couldn't adhere to M3CM Rule- 11.4
            */
        case FLSTST_8BIT_CRC:
            TespSignature.Crc8Data = (*((FlsTst_CrcConvertType *)SignatureAddress)).Crc8Data; /*PRQA S 0306*/
            break;
        case FLSTST_16BIT_CRC:
            TespSignature.Crc16Data = (*((FlsTst_CrcConvertType *)SignatureAddress)).Crc16Data; /*PRQA S 0306*/
            break;
        case FLSTST_32BIT_CRC:
            TespSignature.Crc32Data = (*((FlsTst_CrcConvertType *)SignatureAddress)).Crc32Data; /*PRQA S 0306*/
            break;
        default:
            /*Nothing to do*/
            break;
    }
    return TespSignature;
}
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
/**
 * @brief  Restore the error details data for FlsTst module.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 *
 * @trace YTSDS_FlsTst_533
 */
FLSTST_FUNC static void FlsTst_ResetDetailErrData(void)
{
    FlsTst_ErrDetail.BgndErrDetail.BgndErrIntervalId = 0U;
    FlsTst_ErrDetail.BgndErrDetail.BgndErrBlockCount = 0U;
    for (uint32 LoopCnt = 0; LoopCnt < FLSTST_MAX_NUMBER_OF_BGND_BLOCKS; ++LoopCnt)
    {
        FlsTst_ErrDetail.BgndErrDetail.BgndErrBlock[LoopCnt].Index = 0U;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
        FlsTst_ErrDetail.BgndErrDetail.BgndErrBlock[LoopCnt].ErrSignature = 0U;
#endif
    }
    FlsTst_ErrDetail.FgndErrDetail.FgndErrStatus = FALSE;
    FlsTst_ErrDetail.FgndErrDetail.FgndErrBlockIndex = 0U;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
    FlsTst_ErrDetail.FgndErrDetail.FgndErrSignature = 0U;
#endif
}
#endif
/**
 * @brief  Restore all global varible datas for FlsTst module
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_534
 */
FLSTST_FUNC static void FlsTst_ResetResultData(void)
{
    /* Reset the background test variable*/
    FlsTst_BgndTestIntervalId = 0U;
    FlsTst_LastBgndTestResult.TestIntervalId = 0U;
    FlsTst_CurBgndTestResult.TestIntervalId = 0U;
    for (uint32 LoopCnt = 0U; LoopCnt < FLSTST_MAX_NUMBER_OF_BGND_BLOCKS; ++LoopCnt)
    {
        FlsTst_LastBgndTestResult.TestResultFlag[LoopCnt] = FLSTST_RESULT_NOT_TESTED;
        FlsTst_CurBgndTestResult.TestResultFlag[LoopCnt] = FLSTST_RESULT_NOT_TESTED;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
        FlsTst_LastBgndTestResult.TestSignature[LoopCnt].Crc32Data = FLSTST_CRC32_INITIAL_VALUE;
        FlsTst_CurBgndTestResult.TestSignature[LoopCnt].Crc32Data = FLSTST_CRC32_INITIAL_VALUE;
#endif
    }
    FlsTst_LastBgndTestResult.SumResultFlag = FLSTST_RESULT_NOT_TESTED;
    FlsTst_CurBgndTestResult.SumResultFlag = FLSTST_RESULT_NOT_TESTED;
    FlsTst_LastBgndTestResult.SumSignature = FLSTST_CRC32_INITIAL_VALUE;
    FlsTst_CurBgndTestResult.SumSignature = FLSTST_CRC32_INITIAL_VALUE;
    FlsTst_LastFgndSignature.FgndTestBlockId = 0U;
    FlsTst_LastFgndSignature.FgndSignatureValue = 0U;
}
/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief  This function aborts the ongoing background test and set the state to FLSTST_ABORTED.
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_518
*/

FLSTST_FUNC void FlsTst_Abort(void)
{
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_ABORT, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        FlsTst_CurrentState = FLSTST_ABORTED;
    }
}

/**
 * @brief  The function de-initializes the Flash Test module and restore all process variable for FlsTst module.
 * @table            @service_id:       0x01  \n
                        @is_reentrant:     false \n
                        @is_synchronous:   true  \n
                        @autosar_api:      true  \n
                        @memory_map:       .mcal_text \n
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_521
 */
FLSTST_FUNC void FlsTst_DeInit(void)
{
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_DE_INIT, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Initialize the global variable to default values */
        FlsTst_LastFgndResult = FLSTST_NOT_TESTED;
        FlsTst_Var.BgndConfigIndex = 0U;
        FlsTst_Var.CalculatedCrc = 0U;
        /* set test block start address as 0 */
        FlsTst_Var.ReadAddress = 0U;
        FlsTst_Var.BgndCheckState = FLSTST_BGND_CHECK_INIT;
        FlsTst_ResetResultData();
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
        FlsTst_ResetDetailErrData();
#endif
        /* Set the driver status to idle */
        FlsTst_CurrentState = FLSTST_UNINIT;
        FlsTst_TestComplete = FALSE;
        FlsTst_ConfigPtr = NULL_PTR;
    }
}


#if (FLSTST_GET_CURRENT_STATE_API == STD_ON)
/**
 * @brief  The function returns the current state of the FlsTst driver.
 * @table            @service_id:       0x06  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 * @return FlsTst_StateType
 * @retval FLSTST_UNINIT: The Flash Test is not initialized or not usable.
 * @retval FLSTST_INIT: The Flash Test is initialized and ready to be started.
 * @retval FLSTST_RUNNING: The Flash Test is currently running.
 * @retval FLSTST_ABORTED: The Flash Test is aborted.
 * @retval FLSTST_SUSPENDED: The Flash Test is waiting to be resumed or is waiting to start forground mode test.
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_522
 */
FLSTST_FUNC FlsTst_StateType FlsTst_GetCurrentState(void)
{
    /* Local variable to store the driver status */
    FlsTst_StateType LenStatus;
    LenStatus = FlsTst_CurrentState;
    return LenStatus;
}
#endif/* #if (FLSTST_GET_CURRENT_STATE_API == STD_ON) */

#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
/**
 * @brief  The function returns the details error of the FlsTst blocks.
 * @table            @service_id:       0x0b  \n
                        @is_reentrant:     true \n
                        @is_synchronous:   true  \n
                        @autosar_api:      true  \n
                        @memory_map:       .mcal_text \n
 * @return FlsTst_ErrorDetailsType
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_523
 */
FLSTST_FUNC FlsTst_ErrorDetailsType FlsTst_GetErrorDetails(void)
{
    FlsTst_ErrorDetailsType RetTestResult;
    uint8 LoopCnt = 0;
    /* Initialize the return value */
    RetTestResult.FgndErrDetail.FgndErrStatus = FALSE;
    RetTestResult.FgndErrDetail.FgndErrBlockIndex = 0U;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
    RetTestResult.FgndErrDetail.FgndErrSignature = 0U;
#endif
    RetTestResult.BgndErrDetail.BgndErrIntervalId = 0U;
    RetTestResult.BgndErrDetail.BgndErrBlockCount = 0U;
    LoopCnt = 0U;
    while (LoopCnt < FLSTST_MAX_NUMBER_OF_BGND_BLOCKS)
    {
        RetTestResult.BgndErrDetail.BgndErrBlock[LoopCnt].Index = 0U;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
        RetTestResult.BgndErrDetail.BgndErrBlock[LoopCnt].ErrSignature = 0U;
#endif
        LoopCnt = LoopCnt + 1U;
    }
    /* Check if the module is initialized */
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_GET_ERROR_DETAILS, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Get the current FlsTst test result */
        RetTestResult = FlsTst_ErrDetail;
    }
    return RetTestResult;
}
#endif /* #if (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */

#if (FLSTST_GET_TEST_RESULT_BGND_API == STD_ON)
/**
 * @brief  The function returns the Background Flash Test result of last interval.
 * @table            @service_id:       0x07  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 * @return FlsTst_TestResultBgndType
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_525
*/
FLSTST_FUNC FlsTst_TestResultBgndType FlsTst_GetTestResultBgnd(void)
{
    FlsTst_TestResultBgndType LddTestResultBgnd;
    LddTestResultBgnd.BgndTestIntervalId = 0U;
    LddTestResultBgnd.BgndTestResult = FLSTST_RESULT_NOT_TESTED;
    /* Check if the module is initialized */
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_GET_TEST_RESULT_BGND, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Get the current test interval id and overall Bgnd test result  */
        LddTestResultBgnd.BgndTestIntervalId = FlsTst_LastBgndTestResult.TestIntervalId;
        LddTestResultBgnd.BgndTestResult = FlsTst_LastBgndTestResult.SumResultFlag;
    }
    return LddTestResultBgnd;
}
#endif /* #if (FLSTST_GET_TEST_RESULT_BGND_API == STD_ON) */

#if (FLSTST_GET_TEST_RESULT_FGND_API == STD_ON)
/**
 * @brief  The function returns the Flash Test result of last foreground test block.
 * @table            @service_id:       0x08  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 * @return FlsTst_TestResultFgndType
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_526
*/
FLSTST_FUNC FlsTst_TestResultFgndType FlsTst_GetTestResultFgnd(void)
{
    return FlsTst_LastFgndResult;
}
#endif /* #if (FLSTST_GET_TEST_RESULT_FGND_API == STD_ON) */

#if (FLSTST_GET_TEST_SIGNATURE_BGND_API == STD_ON)
/**
 * @brief            The function returns the signature and Test Interval Id of the last background test.
 *
 * @table            @service_id:       0x09  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           FlsTst_TestSignatureBgndType
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_527
 */
FLSTST_FUNC FlsTst_TestSignatureBgndType FlsTst_GetTestSignatureBgnd(void)
{
    /* Local variable to store the driver bgnd test result */
    FlsTst_TestSignatureBgndType LddTestResult;
    LddTestResult.BgndTestIntervalId = 0U;
    LddTestResult.BgndSignatureValue = 0U;
    /* Check if the module is initialized */
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_GET_TEST_SIGNATURE_BGND, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        if (FlsTst_LastBgndTestResult.SumResultFlag != FLSTST_RESULT_NOT_TESTED)
        {
            /* Get the last test interval id */
            LddTestResult.BgndTestIntervalId = FlsTst_LastBgndTestResult.TestIntervalId;
            LddTestResult.BgndSignatureValue = FlsTst_LastBgndTestResult.SumSignature;
        }
    }
    return LddTestResult;
}
#endif /* #if (FLSTST_GET_TEST_SIGNATURE_BGND_API == STD_ON) */

#if (FLSTST_GET_TEST_SIGNATURE_FGND_API == STD_ON)
/**
 * @brief            The function returns the signature of the last foreground test block.
 *
 * @table            @service_id:       0x0A  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           FlsTst_TestSignatureFgndType
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_528
 */
FLSTST_FUNC FlsTst_TestSignatureFgndType FlsTst_GetTestSignatureFgnd(void)
{
    /* Local variable to store the driver fgnd test result status */
    FlsTst_TestSignatureFgndType LulTestResult;
    LulTestResult.FgndTestBlockId = 0U;
    LulTestResult.FgndSignatureValue = 0U;
    SchM_Enter_FlsTst_FLSTST_EXCLUSIVE_AREA_01();
    if (FlsTst_LastFgndResult != FLSTST_NOT_TESTED)
    {
        LulTestResult.FgndTestBlockId = FlsTst_LastFgndSignature.FgndTestBlockId;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
        /* Load the signature of the last Fgnd check */
        LulTestResult.FgndSignatureValue = FlsTst_LastFgndSignature.FgndSignatureValue;
#else
        /* Load the signature of the last Fgnd check */
        LulTestResult.FgndSignatureValue = (uint32)FlsTst_LastFgndResult;
#endif
    }
    SchM_Exit_FlsTst_FLSTST_EXCLUSIVE_AREA_01();
    return LulTestResult;
}
#endif /* #if (FLSTST_GET_TEST_SIGNATURE_FGND_API == STD_ON) */

/**
 * @brief            The function initializes the Flash test driver and global variables,
 *                   then changes Flash test module state to FLSTST_INIT.
 *
 * @table            @service_id:       0x00  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @param[in]        ConfigPtr Pointer to configuration set
 * @return           void
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_530
 */
FLSTST_FUNC void FlsTst_Init(const FlsTst_ConfigType *ConfigPtr)
{
    Std_ReturnType LddRetVal = E_NOT_OK;
    /*Check the Flash Test driver has been initialized or not*/
    /*Trace SWS_FlsTst_00025*/
    if (FLSTST_UNINIT != FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report Error to DET */
        (void)Det_ReportError(FLSTST_MODULE_ID,
                              FLSTST_INSTANCE_ID, FLSTST_SID_INIT, FLSTST_E_ALREADY_INITIALIZED);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
#if (FLSTST_PRECOMPILE_SUPPORT == STD_ON)
        FlsTst_ConfigPtr = &FlsTst_PreCompileConfig;
        LddRetVal = E_OK;
#else
        /* Check if the Configuration pointer is NULL */
        if (NULL_PTR == ConfigPtr)
        {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
            /* Report Error to DET */
            (void)Det_ReportError(FLSTST_MODULE_ID,
                                  FLSTST_INSTANCE_ID, FLSTST_SID_INIT, FLSTST_E_PARAM_INVALID);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
            LddRetVal = E_NOT_OK;
        }
        else
        {
            LddRetVal = E_OK;
            FlsTst_ConfigPtr = ConfigPtr;
        }
#endif /* #if (FLSTST_PRECOMPILE_SUPPORT == STD_ON) */
        if (E_OK == LddRetVal)
        {
            /* set the bgnd test result as not tested */
            FlsTst_LastFgndResult = FLSTST_NOT_TESTED;
            FlsTst_TestComplete = FALSE;
            FlsTst_ResetResultData();
            /* No action required */
            FlsTst_Var.BgndCheckState = FLSTST_BGND_CHECK_INIT;
            FlsTst_CurrentState = FLSTST_INIT;
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
            FlsTst_ResetDetailErrData();
#endif
        }
    }
}
/**
 * @brief            The function tests the defined flash blocks in background mode, starting with
 *                    the first flash block in the FlsTstConfigParams.
 *
 * @table            @service_id:       0x0D  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   --    \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           void
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_531
 */
/* M3CM 4700 VIOLATION:This function governs the execution flow and result checking of the FlsTst background mode. It is of high complexity
 * and splitting it would reduce the readability of the code. Therefore, both STPTH, STMIF and STST3 cannot follow the standard.
*/
FLSTST_FUNC void FlsTst_MainFunction(void) /*PRQA S 4700 */
{
    /* Local pointer variable to hold the individual test block params */
    const FlsTst_BlockConfigType *LpFlsTstBlock;
    /* Variable to hold  number of byte to process in one call  */
    uint32 LulByteToProcess;
    /* Local union variable to access crc value */
    /* MR12 RULE 19.2 VIOLATION: The LddCrcConvert need to be adapted to different algorithm results,
     * thus, couldn't adhere to M3CM Rule-19.2
     */
    FlsTst_CrcConvertType LddCrcConvert; /*PRQA S 0759*/
    /* Variable to hold  loop count */
    uint32 LulLoopCount;
    /* Variable to hold  number of atomic cell count */
    uint32 LulAtomicCellCount;
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_MAIN_FUNCTION, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /*FLSTST State changed from FLSTST_INIT mode to RUNNING mode automatically*/
        if (FLSTST_INIT == FlsTst_CurrentState)
        {
            FlsTst_CurrentState = FLSTST_RUNNING;
            FlsTst_ResetBgndTestVar();
        }
        else
        {
            /* No action required */
        }
        if ((FLSTST_RUNNING == FlsTst_CurrentState) &&
                (FlsTst_ConfigPtr->MaxBgndBlkNo > FLSTST_ZERO))
        {
            if (TRUE == FlsTst_TestComplete)
            {
                /* Background test blocks are tested completely */
                if (FlsTst_BgndTestIntervalId >= FLSTST_TEST_INTERVAL_ID_END_VALUE)
                {
                    FlsTst_BgndTestIntervalId = FLSTST_ZERO;
                }
                else
                {
                    ++FlsTst_BgndTestIntervalId;
                }
                FlsTst_TestComplete = FALSE;
            }
            else
            {
                /* No action required */
            }
            FlsTst_CurBgndTestResult.TestIntervalId = FlsTst_BgndTestIntervalId;
            /* Load the test block parameters based on the index */
            LpFlsTstBlock = &(FlsTst_ConfigPtr->BgndBlkConfig[FlsTst_Var.BgndConfigIndex]);
            if (FLSTST_BGND_CHECK_INIT == FlsTst_Var.BgndCheckState)
            {
                /* Initialize the CRC value for new test */
                FlsTst_Var.BgndCheckState = FLSTST_BGND_CHECK_INPROGRESS;
            }
            else
            {
                /* No action required */
            }
            /* TRACE [R4,FlsTst139] */
            if (FLSTST_BGND_CHECK_INPROGRESS == FlsTst_Var.BgndCheckState)
            {
                /* Check block size is greater than no of cell to be tested in
                * one scheduled task (FlsTst_MainFunction() call), if yes load
                * maximum value configured for one scheduled task.
                */
                if (((LpFlsTstBlock->TestBlockBaseAddress +
                        LpFlsTstBlock->TestBlockSize) -
                        FlsTst_Var.ReadAddress)
                        > LpFlsTstBlock->NumberOfTestedCells)
                {
                    LulByteToProcess = LpFlsTstBlock->NumberOfTestedCells;
                }
                else
                {
                    LulByteToProcess = ((LpFlsTstBlock->TestBlockBaseAddress +
                                         LpFlsTstBlock->TestBlockSize) -
                                        FlsTst_Var.ReadAddress);
                }
                /* Initialize cell count to zero */
                LulLoopCount = 0;
                /*The First check of this block*/
                if (FlsTst_Var.ReadAddress == LpFlsTstBlock->TestBlockBaseAddress)
                {
                    if (LulByteToProcess >= FLSTST_NUMBER_OF_TESTED_CELLS_ATOMIC)
                    {
                        LulAtomicCellCount = FLSTST_NUMBER_OF_TESTED_CELLS_ATOMIC;
                    }
                    else
                    {
                        LulAtomicCellCount = LulByteToProcess;
                    }
                    SchM_Enter_FlsTst_FLSTST_EXCLUSIVE_AREA_02();
                    /* Calculate the CRC value for one cell atomic*/
                    FlsTst_Var.CalculatedCrc = FlsTst_CalculateCrc(LpFlsTstBlock->TestAlgorithm,
                                               FlsTst_Var.ReadAddress,
                                               LulAtomicCellCount,
                                               0x00, TRUE);
                    FlsTst_Var.ReadAddress += LulAtomicCellCount;
                    LulLoopCount = LulAtomicCellCount;
                    SchM_Exit_FlsTst_FLSTST_EXCLUSIVE_AREA_02();
                }
                /* calculate the CRC value for new cell */
                while ((FLSTST_RUNNING == FlsTst_CurrentState) && (LulLoopCount < LulByteToProcess))
                {
                    if ((LulByteToProcess - LulLoopCount) >= FLSTST_NUMBER_OF_TESTED_CELLS_ATOMIC)
                    {
                        LulAtomicCellCount = FLSTST_NUMBER_OF_TESTED_CELLS_ATOMIC;
                    }
                    else
                    {
                        LulAtomicCellCount = LulByteToProcess - LulLoopCount;
                    }
                    SchM_Enter_FlsTst_FLSTST_EXCLUSIVE_AREA_02();
                    /* Calculate the CRC value for one cell atomic*/
                    FlsTst_Var.CalculatedCrc = FlsTst_CalculateCrc(LpFlsTstBlock->TestAlgorithm,
                                               FlsTst_Var.ReadAddress,
                                               LulAtomicCellCount,
                                               FlsTst_Var.CalculatedCrc,
                                               FALSE);
                    FlsTst_Var.ReadAddress += LulAtomicCellCount;
                    LulLoopCount += LulAtomicCellCount;
                    SchM_Exit_FlsTst_FLSTST_EXCLUSIVE_AREA_02();
                }
                /*Check whether the block checked complete*/
                if (FlsTst_Var.ReadAddress >=
                        (LpFlsTstBlock->TestBlockBaseAddress +
                         LpFlsTstBlock->TestBlockSize))
                {
                    LddCrcConvert = FlsTst_GetSignatureFromFlash(LpFlsTstBlock->SignatureAddress,
                                    LpFlsTstBlock->TestAlgorithm);
                    /* Check the whether calculated crc matching with stored crc */
                    if (LddCrcConvert.Crc32Data == FlsTst_Var.CalculatedCrc)
                    {
                        /* TRACE [R4,FlsTst160] */
                        FlsTst_CurBgndTestResult.TestResultFlag[FlsTst_Var.BgndConfigIndex] = FLSTST_RESULT_OK;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
                        FlsTst_CurBgndTestResult.TestSignature[FlsTst_Var.BgndConfigIndex].Crc32Data = FlsTst_Var.CalculatedCrc;
#endif
                        /* Report error  and change overall bgnd result status to  ok */
#if defined(FLSTST_E_FLSTST_FAILURE)
                        (void)Dem_SetEventStatus(FLSTST_E_FLSTST_FAILURE, DEM_EVENT_STATUS_PASSED);
#endif
                    }
                    else
                    {
                        /* Report error  and change overall bgnd result status to not ok */
                        FlsTst_CurBgndTestResult.TestResultFlag[FlsTst_Var.BgndConfigIndex] = FLSTST_RESULT_NOT_OK;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
                        FlsTst_CurBgndTestResult.TestSignature[FlsTst_Var.BgndConfigIndex].Crc32Data = FlsTst_Var.CalculatedCrc;
#endif
                        FlsTst_CurBgndTestResult.SumResultFlag = FLSTST_RESULT_NOT_OK;
                        /* TRACE [R4,FlsTst013] */
#if defined(FLSTST_E_FLSTST_FAILURE)
                        (void)Dem_SetEventStatus(FLSTST_E_FLSTST_FAILURE, DEM_EVENT_STATUS_FAILED);
#endif
                    }
                    /* Increment the test block index */
                    ++FlsTst_Var.BgndConfigIndex;
                    /*Check this test interval completed or not*/
                    if (FlsTst_Var.BgndConfigIndex >= FlsTst_ConfigPtr->MaxBgndBlkNo)
                    {
                        /*Calculta the overall sigature result*/
                        if (FlsTst_CurBgndTestResult.SumResultFlag != FLSTST_RESULT_NOT_OK)
                        {
                            FlsTst_CurBgndTestResult.SumResultFlag = FLSTST_RESULT_OK;
                        }
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
                        /* MR12 RULE 11.4 VIOLATION: In order to guarantee the consent of the function interface,
                                 * thus, couldn't adhere to M3CM Rule-11.4
                                 */
                        LddCrcConvert.Crc32Data = FlsTst_CalculateCrc(FLSTST_32BIT_CRC,
                                                  (uint32)&FlsTst_CurBgndTestResult.TestSignature[0],/*PRQA S 0306*/
                                                  sizeof(FlsTst_CurBgndTestResult.TestSignature),
                                                  FLSTST_CRC32_INITIAL_VALUE,
                                                  TRUE);
                        FlsTst_CurBgndTestResult.SumSignature = LddCrcConvert.Crc32Data;
#else
                        if (FLSTST_RESULT_OK == FlsTst_CurBgndTestResult.SumResultFlag)
                        {
                            /* TRACE [R4,FlsTst161] */
                            FlsTst_CurBgndTestResult.SumSignature = FLSTST_RESULT_OK;
                        }
                        else
                        {
                            FlsTst_CurBgndTestResult.SumSignature = FLSTST_RESULT_NOT_OK;
                        }
#endif
                        SchM_Enter_FlsTst_FLSTST_EXCLUSIVE_AREA_02();
                        FlsTst_CacheBgndTestResult();
                        SchM_Exit_FlsTst_FLSTST_EXCLUSIVE_AREA_02();
#if (FLSTST_TEST_COMPLETED_NOTIFICATION_SUPPORTED == STD_ON)
                        if (NULL_PTR != FlsTst_ConfigPtr->BgndCompNotifyFuncPtr)
                        {
                            FlsTst_ConfigPtr->BgndCompNotifyFuncPtr();
                        }
                        else
                        {
                            /* No action required */
                        }
#endif/*(FLSTST_TEST_COMPLETED_NOTIFICATION_SUPPORTED == STD_ON)*/
                        /*TReset index and flstst state*/
                        FlsTst_Var.BgndConfigIndex = FLSTST_ZERO;
                        /* On completion of the one test cycle call-back notification will be invoked*/
                        FlsTst_TestComplete = TRUE;
                        FlsTst_CurrentState = FLSTST_INIT;
                    }
                    else
                    {
                        /* No action required */
                    }
                    /* Load the test block parameters based on the index */
                    FlsTst_Var.ReadAddress = FlsTst_ConfigPtr->BgndBlkConfig[FlsTst_Var.BgndConfigIndex].TestBlockBaseAddress;
                    FlsTst_Var.BgndCheckState = FLSTST_BGND_CHECK_INIT;
                }
                else
                {
                    /* No action required */
                }
            }
            else
            {
                /* No action required */
            }
        }
        else
        {
            /* No action required */
        }
    }
}

#if (FLSTST_SUSPEND_RESUME_API == STD_ON)
/*FUNCTION**********************************************************************
 *
 * Function Name : FlsTst_Resume
 * Description   : Service for continuing the Flash Test at the point
 *                  it was suspended.
 *
 *END**************************************************************************/
/**
 * @brief            Service for continuing the Flash Test at the point.The function change the execution state to
 *                   FLSTST_RUNNING when commanded to continue and the current execution state is FLSTST_SUSPENDED.
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           void
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_535
 */
FLSTST_FUNC void FlsTst_Resume(void)
{
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_RESUME, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else if (FLSTST_SUSPENDED != FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_RESUME, FLSTST_E_STATE_FAILURE);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        FlsTst_CurrentState = FLSTST_RUNNING;
    }
}
#endif  /* #if (FLSTST_SUSPEND_RESUME_API == STD_ON) */

/**
 * @brief            The function is only applicable for Foreground mode Flash Test operation.
 * @param[in]        FgndBlockId Number of the foreground test to be executed. This is dependent on configuration.
 *
 * @table            @service_id:       0x02  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           Std_ReturnType
 * @retval E_OK     Foreground Flash Test is started successfully
 * @retval E_NOT_OK Foreground Flash Test is not started
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_536
 */
FLSTST_FUNC Std_ReturnType FlsTst_StartFgnd(FlsTst_BlockIdFgndType FgndBlockId)
{
    /* Local variable to hold the return value of the function */
    Std_ReturnType LenReturnValue = E_OK;
    /* Local variable to hold stored crc */
    /* MR12 RULE 19.2 VIOLATION: The LulStoredCrc need to be adapted to different algorithm results,
     * thus, couldn't adhere to M3CM Rule-19.2
     */
    FlsTst_CrcConvertType LulStoredCrc; /*PRQA S 0759*/
    /* Local Pointer to FlsTstBlockConfigType configuration */
    const FlsTst_BlockConfigType *LpFlsTstBlock;
    /* Local union variable to access crc value */
    FlsTst_CrcConvertType LulCalculatedCrc; /*PRQA S 0759*/
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_START_FGND, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
        LenReturnValue = E_NOT_OK;
    }
    else if (FgndBlockId >= FlsTst_ConfigPtr->MaxFgndBlkNo)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that block id is out of range  */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_START_FGND, FLSTST_E_PARAM_INVALID);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
        LenReturnValue = E_NOT_OK;
    }
    else
    {
        SchM_Enter_FlsTst_FLSTST_EXCLUSIVE_AREA_00();
        /* Get the corresponding block parameters */
        LpFlsTstBlock = &(FlsTst_ConfigPtr->FgndBlkConfig[FgndBlockId]);
        /* Get the calculated CRC value from CRC unit and stored crc value.*/
        LulCalculatedCrc.Crc32Data = FlsTst_CalculateCrc(LpFlsTstBlock->TestAlgorithm, LpFlsTstBlock->TestBlockBaseAddress, \
                                     LpFlsTstBlock->TestBlockSize, 0x00000000U, TRUE);
        LulStoredCrc = FlsTst_GetSignatureFromFlash(LpFlsTstBlock->SignatureAddress, LpFlsTstBlock->TestAlgorithm);
        /* Check the calculated crc and stored crc*/
        if (LulCalculatedCrc.Crc32Data == LulStoredCrc.Crc32Data)
        {
            FlsTst_LastFgndResult = FLSTST_OK;
        }
        else
        {
            FlsTst_LastFgndResult = FLSTST_NOT_OK;
#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
            FlsTst_ErrDetail.FgndErrDetail.FgndErrBlockIndex = FgndBlockId;
            FlsTst_ErrDetail.FgndErrDetail.FgndErrStatus = TRUE;
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
            FlsTst_ErrDetail.FgndErrDetail.FgndErrSignature = LulCalculatedCrc.Crc32Data;
#endif /* #if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON) */
#endif /* #if (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */
#if (FLSTST_TEST_FAILED_NOTIFICATION == STD_ON)
            if (NULL_PTR != FlsTst_ConfigPtr->TestFailedNotifyFuncPtr)
            {
                FlsTst_ConfigPtr->TestFailedNotifyFuncPtr(FLSTST_MODULE_ID);
            }
            else
            {
                /* No action required */
            }
#endif /* #if (FLSTST_TEST_FAILED_NOTIFICATION == STD_ON) */
        }
        FlsTst_LastFgndSignature.FgndTestBlockId = FgndBlockId;
        FlsTst_LastFgndSignature.FgndSignatureValue = LulCalculatedCrc.Crc32Data;
        SchM_Exit_FlsTst_FLSTST_EXCLUSIVE_AREA_00();
    }
    return LenReturnValue;
}

#if (FLSTST_SUSPEND_RESUME_API == STD_ON)
/**
 * @brief            The function is only applicable for Background mode Flash Test operation.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           void
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_537
 */
FLSTST_FUNC void FlsTst_Suspend(void)
{
    if (FLSTST_UNINIT == FlsTst_CurrentState)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report error to DET that module is not initialized */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID,
                              FLSTST_SID_SUSPEND, FLSTST_E_UNINIT);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Suspend bgnd check only if it is in RUNNING or INIT state */
        /* MR12 RULE 13.5 VIOLATION: The FlsTst_CurrentState is a machine state,can only be in one stete at a time,
             * thus, couldn't adhere to M3CM Rule-13.5
             */
        if ((FLSTST_INIT == FlsTst_CurrentState) || (FLSTST_RUNNING == FlsTst_CurrentState)) /*PRQA S 3415*/
        {
            FlsTst_CurrentState = FLSTST_SUSPENDED;
        }
    }
}
#endif/* #if (FLSTST_SUSPEND_RESUME_API == STD_ON) */


#if (FLSTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Service returns the version information of this module.
 * @param[out]       VersionInfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x08  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      true  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           void
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_529
 */
FLSTST_FUNC void FlsTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    if (NULL_PTR == VersionInfo)
    {
#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
        /* Report FLSTST_E_PARAM_POINTER DET if service called with
           NULL_PTR
        */
        (void)Det_ReportError(FLSTST_MODULE_ID, FLSTST_INSTANCE_ID, FLSTST_SID_GET_VERSION_INFO, FLSTST_E_PARAM_POINTER);
#endif /* #if (FLSTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Vendor ID information */
        VersionInfo->vendorID = FLSTST_VENDOR_ID;
        /* FlsTst module ID information */
        VersionInfo->moduleID = FLSTST_MODULE_ID;
        /* FlsTst module Software major version information */
        VersionInfo->sw_major_version = (uint8)FLSTST_SW_MAJOR_VERSION;
        /* FlsTst module Software minor version information */
        VersionInfo->sw_minor_version = (uint8)FLSTST_SW_MINOR_VERSION;
        /* FlsTst module Software patch version information */
        VersionInfo->sw_patch_version = (uint8)FLSTST_SW_PATCH_VERSION;
    }
}
#endif /* #if (FLSTST_VERSION_INFO_API == STD_ON) */

#define FLSTST_STOP_SEC_CODE
#include "FlsTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

