/*
* @file    FlsTst_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : FlsTst_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef FLSTST_TYPES_H
#define FLSTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "FlsTst_Cfg.h"

/*
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 0750 Rule-19.2: An object of union type has been defined.
 *
*/

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define FLSTST_TYPES_VENDOR_ID                             (0xB4)
#define FLSTST_TYPES_AR_RELEASE_MAJOR_VERSION              (4)
#define FLSTST_TYPES_AR_RELEASE_MINOR_VERSION              (4)
#define FLSTST_TYPES_AR_RELEASE_REVISION_VERSION           (0)
#define FLSTST_TYPES_SW_MAJOR_VERSION                      (2)
#define FLSTST_TYPES_SW_MINOR_VERSION                      (0)
#define FLSTST_TYPES_SW_PATCH_VERSION                      (0)
/*==================================================================================================
                                        FILE VERSION CHECKS
==================================================================================================*/
/* Check if the file and FlsTst_Cfg header file are of the same vendor */
#if (FLSTST_TYPES_VENDOR_ID != FLSTST_VENDOR_ID_CFG)
#error "FlsTst_Types.h and FlsTst_Cfg.h have different vendor ids"
#endif

/* Check if the file and FlsTst_Cfg header file are of the same Autosar version */
#if ((FLSTST_TYPES_AR_RELEASE_MAJOR_VERSION != FLSTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (FLSTST_TYPES_AR_RELEASE_MINOR_VERSION != FLSTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (FLSTST_TYPES_AR_RELEASE_REVISION_VERSION != FLSTST_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of FlsTst_Types.h and FlsTst_Cfg.h are different"
#endif

/* Check if the file and FlsTst_Cfg header file are of the same Software version */
#if ((FLSTST_TYPES_SW_MAJOR_VERSION != FLSTST_SW_MAJOR_VERSION_CFG) || \
     (FLSTST_TYPES_SW_MINOR_VERSION != FLSTST_SW_MINOR_VERSION_CFG) || \
     (FLSTST_TYPES_SW_PATCH_VERSION != FLSTST_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of FlsTst_Types.h and FlsTst_Cfg.h are different"
#endif

/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/**
 * @defgroup FlsTst_ID_Information_Macro
 * @brief    The ID information list of FlsTst module .
 * @sw_type  sw_detail
 * @{
 *
 * @trace YTSDS_FlsTst_497
 */
/** @brief FLSTST MODULE INSTANCE ID */
#define FLSTST_INSTANCE_ID ((uint8)0U)

/** @brief FLSTST MODULE ID */
#define FLSTST_MODULE_ID ((uint16)104U)

/** @brief API Service ID for FlsTst_Init */
#define FLSTST_SID_INIT                                     (0x00U)
/** @brief API Service ID for FlsTst_DeInit */
#define FLSTST_SID_DE_INIT                                  (0x01U)
/** @brief API Service ID for FlsTst_StartFgnd */
#define FLSTST_SID_START_FGND                               (0x02U)
/** @brief API Service ID for FlsTst_Abort */
#define FLSTST_SID_ABORT                                    (0x03U)
/** @brief API Service ID for FlsTst_Suspend */
#define FLSTST_SID_SUSPEND                                  (0x04U)
/** @brief API Service ID for FlsTst_Resume */
#define FLSTST_SID_RESUME                                   (0x05U)
/** @brief API Service ID for FlsTst_GetCurrentState */
#define FLSTST_SID_GET_CURRENT_STATE                        (0x06U)
/** @brief API Service ID for FlsTst_GetTestResultBgnd */
#define FLSTST_SID_GET_TEST_RESULT_BGND                     (0x07U)
/** @brief API Service ID for FlsTst_GetVersionInfo */
#define FLSTST_SID_GET_VERSION_INFO                         (0x08U)
/** @brief API Service ID for FlsTst_GetTestSignatureBgnd */
#define FLSTST_SID_GET_TEST_SIGNATURE_BGND                  (0x09U)
/** @brief API Service ID for FlsTst_GetTestSignatureFgnd */
#define FLSTST_SID_GET_TEST_SIGNATURE_FGND                  (0x0AU)
/** @brief API Service ID for FlsTst_GetErrorDetails */
#define FLSTST_SID_GET_ERROR_DETAILS                        (0x0BU)
/** @brief API Service ID for FlsTst_MainFunction */
#define FLSTST_SID_MAIN_FUNCTION                            (0x0DU)
/** @brief API Service ID for FlsTst_GetTestResultFgnd */
#define FLSTST_SID_GET_TEST_RESULT_FGND                     (0x0FU)
/** @} */

#if (FLSTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup FlsTst_Dev_Error_Macro
 * @brief   FlsTst module supported development error detailed list.
 * @sw_type sw_detail
 * @{
 *
 * @trace YTSDS_FlsTst_492
 */
/** @brief Failure within Flash Test execution state */
#define FLSTST_E_STATE_FAILURE                           (0x01U)
/** @brief API parameter out of specified range */
#define FLSTST_E_PARAM_INVALID                           (0x02U)
/** @brief API service called without module initialization */
#define FLSTST_E_UNINIT                                  (0x03U)
/** @brief Flash Test module initialized repeatedly*/
#define FLSTST_E_ALREADY_INITIALIZED                     (0x04U)
/** @brief Flash Test configuration pointer error */
#define FLSTST_E_INIT_FAILED                             (0x05U)
/** @brief Pointer is a NULL pointer */
#define FLSTST_E_PARAM_POINTER                           (0x06U)
/** @} */
#endif

/** @brief FlsTst initialized*/
#define FLSTST_INITIALIZED                 (boolean)1U
/** @brief FlsTst uninitialized*/
#define FLSTST_UNINITIALIZED               (boolean)0U
/** @brief The initialized result for CRC16*/
#define FLSTST_CRC16_INITIAL_VALUE         (uint16)0xFFFFU
/** @brief The initialized result for CRC8*/
#define FLSTST_CRC8_INITIAL_VALUE          (uint8)0xFFU
/** @brief The initialized result for CRC32*/
#define FLSTST_CRC32_INITIAL_VALUE         (uint32)0x00000000U

/** @brief FlsTst zero*/
#define FLSTST_ZERO                        (uint8)0x00
/** @brief FlsTst one*/
#define FLSTST_ONE                         (uint8)0x01
/** @brief FlsTst two*/
#define FLSTST_TWO                         (uint8)0x02
/** @brief FlsTst three*/
#define FLSTST_THREE                       (uint8)0x03
/** @brief FlsTst four*/
#define FLSTST_FOUR                        (uint8)0x04
/** @brief FlsTst five*/
#define FLSTST_SIX                         (uint8)0x06
/** @brief FlsTst seven*/
#define FLSTST_EIGHT                       (uint8)0x08
/** @brief FlsTst nine*/
#define FLSTST_NINE                        (uint8)0x09
/** @brief FlsTst ten*/
#define FLSTST_TEN                         (uint8)0x0A

/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS
==================================================================================================*/

/**
 * @brief        This is a state value returned by the API service FlsTst_GetCurrentState().For the
 *                type FlsTst_StateType, the enumeration value FLSTST_UNINIT shall be the default
 *                value after a reset.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_36
 */
typedef enum
{
    FLSTST_UNINIT                            = 0x00, /**< The Flash Test is not initialized or not usable.*/
    FLSTST_INIT                              = 0x01, /**< The Flash Test is initialized and ready to be started.*/
    FLSTST_RUNNING                           = 0x02, /**< The Flash Test is currently running. */
    FLSTST_ABORTED                           = 0x03, /**< The Flash Test is aborted. */
    FLSTST_SUSPENDED                         = 0x04, /**< The Flash Test has beeb suspended and is waiting to be resumed.*/
} FlsTst_StateType;


/**
 * @brief       Type for test result status in foreground module,which is the return type of API service FlsTst_GetResultFgnd().
 *               The enumeration value FLSTST_NOT_TESTED shall be the default value after a reset.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_60
 */
typedef enum
{
    FLSTST_NOT_TESTED                        = 0x00, /**< There is no result available. */
    FLSTST_OK                                = 0x01, /**< The last foreground test result is OK. */
    FLSTST_NOT_OK                            = 0x02, /**< The last foreground test result is not OK */
} FlsTst_TestResultFgndType;


/**
 * @brief       Type for test result status in background flash test.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_68
 */
typedef enum
{
    FLSTST_RESULT_NOT_TESTED                 = 0x00, /**< There is no test result available. */
    FLSTST_RESULT_OK                         = 0x01, /**< The last Flash Test interval has been tested with OK result. */
    FLSTST_RESULT_NOT_OK                     = 0x02, /**< The last Flash Test interval has been tested with NOT-OK result. */
} FlsTst_TestResultType;

/**
 * @brief  Enumeration for background check state
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_489
 */
typedef enum
{
    FLSTST_BGND_CHECK_INIT                      = 0x00,/**< The test invterval not start*/
    FLSTST_BGND_CHECK_INPROGRESS                = 0x01,/**< The test invterval is in progress*/
    FLSTST_BGND_CHECK_COMPLETE                  = 0x02,/**< The test invterval is complete*/
} FlsTst_BgndStateType;


/**
 * @brief  This type specifies the identification (ID) for a Flash block to be tested in foreground mode, which is
 *         configured in the configuration structure.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_441
 */
typedef  uint32 FlsTst_BlockIdFgndType;

/**
 * @brief      Job end notification pointer type in backround mode. Point to test completed notification.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_455
 */
typedef void (*FlsTst_TestCompNotificationType)(void);

/**
 * @brief FlsTst test failed notification type.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_1062
*/
typedef void (*FlsTst_FailedNotificationType)(uint16 BlockId);



/**
 * @brief       Type for algorithm for foreground mode and background mode.
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_498
 */
typedef enum
{
    FLSTST_8BIT_CRC               = 0x00, /**< 8-bit CRC with SAEJ1850 calculation */
    FLSTST_16BIT_CRC              = 0x01, /**< 16-bit CRC calculation */
    FLSTST_32BIT_CRC              = 0x02, /**< 32-bit CRC calculation */
    FLSTST_CHECKSUM               = 0x03, /**< Checksum calculation */
    FLSTST_DUPLICATED_MEMORY      = 0x04, /**< Duplicated memory */
    FLSTST_ECC                    = 0x05, /**< ECC calculation */
} FlsTst_TestAlgorithmType;

/**
 * @brief       Type Declaration to store CRC values.
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_491
 */
/* MR12 RULE 19.2 VIOLATION: The number of signature bits varies among different algorithms. Therefore, in this case, a union type is used
 * to accommodate the storage of signatures calculated by different algorithms in the FlsTst module.
 */
/* PRQA S 0750 ++ */
typedef union
{

    uint32 Crc32Data;   /**< To get CRC32 value*/
    uint16 Crc16Data;   /**< To get CRC16 value*/
    uint8 Crc8Data;     /**< To get CRC8 value*/
    uint8 CrcData4[4];  /**< To store crc value */
} FlsTst_CrcConvertType; /* PRQA S 0750 -- */


/**
 * @brief       Type declaration for test memory block.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_440
 */
typedef struct
{
    uint32 TestBlockIndex;          /**< Index of the Flash block */
    uint32 TestBlockBaseAddress;    /**< Start Address of the Flash block */
    uint32 TestBlockSize;           /**< Flash test block size */
    uint32 NumberOfTestedCells;     /**< Number of cells to be tested in BGND mode during one scheduled task */
    uint32 SignatureAddress;        /**< Address of the signature reference value of the Flash test block  */
    FlsTst_TestAlgorithmType TestAlgorithm; /**< Test algorithm for foreground mode and background mode */

} FlsTst_BlockConfigType;

/**
 * @brief       This type of external data structure which contains the initialization data for the Flash Test.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_35
 */
typedef struct
{
    const FlsTst_BlockConfigType *BgndBlkConfig;  /**< Pointer to memory blocks definition to test in background mode */
    const FlsTst_BlockConfigType *FgndBlkConfig;  /**< Pointer to memory blocks definition to test in foreground mode */
    uint32 MaxBgndBlkNo; /**< Number of test blocks available for the background mode */
    uint32 MaxFgndBlkNo;  /**< Number of test blocks available for the foreground mode */
#if (FLSTST_TEST_COMPLETED_NOTIFICATION_SUPPORTED == STD_ON)
    FlsTst_TestCompNotificationType BgndCompNotifyFuncPtr;/**< Pointer to Flash Test completed notification in background mode */
#endif
#if (FLSTST_TEST_FAILED_NOTIFICATION == STD_ON)
    FlsTst_FailedNotificationType TestFailedNotifyFuncPtr;/**< Pointer to Flash Test failed notification*/
#endif
} FlsTst_ConfigType;

/**
 * @brief     Type Declaration to store values for TestResult in back ground.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_158
 */
typedef struct
{
    uint32 BgndTestIntervalId; /**< This variable will hold the last test interval id of the background test */
    FlsTst_TestResultType BgndTestResult; /**< This variable will hold the last background test result */
} FlsTst_TestResultBgndType;


/**
 * @brief     Type for test signature in background mode.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_67
 */
typedef struct
{
    uint32 BgndTestIntervalId; /**< This variable will hold the last test interval id of the background test */
    uint32 BgndSignatureValue; /**< This variable will hold the last background test signature value */
} FlsTst_TestSignatureBgndType;


/**
 * @brief       Type for test signature in foreground mode.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_62
 */
typedef struct
{
    FlsTst_BlockIdFgndType FgndTestBlockId; /**< This variable will hold the last test interval id of the foreground test */
    uint32 FgndSignatureValue;              /**< This variable will hold the last foreground test signature value */
} FlsTst_TestSignatureFgndType;

/**
 * @brief       Data Structure required for initializing the FlsTst test Block
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_496
 */
typedef struct
{
    uint32 ReadAddress; /**< Variable to store the test block start address */
    uint32 CalculatedCrc; /**< Variable to store the calculated CRC value */
    uint16 BgndConfigIndex; /**< Variable to store the index of the test block based on priority */
    FlsTst_BgndStateType BgndCheckState; /**< Variable to store the internal background check state */
} FlsTst_BgndProcessDataType;

/**
 * @brief Test result data structure for background test used in main function
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_490
 */
typedef struct
{
    uint32 TestIntervalId; /**< Variable to store the last test interval id of the background test */
    FlsTst_TestResultType TestResultFlag[FLSTST_MAX_NUMBER_OF_BGND_BLOCKS]; /**< Variable to store the last background test results for all blocks */
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
    FlsTst_CrcConvertType TestSignature[FLSTST_MAX_NUMBER_OF_BGND_BLOCKS]; /**< Variable to store the last background test signature values for all blocks */
#endif
    FlsTst_TestResultType SumResultFlag; /**< The sum result flag */
    uint32 SumSignature;                /**< The sum signature for all blocks */
} FlsTst_BgndTestResultType;

#if (FLSTST_GET_ERROR_DETAILS_API == STD_ON)
/**
 * @brief     Data Structure for background test error block details
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_494
 */
typedef struct
{
    uint32 Index;  /**< Index of the Flash block */
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
    uint32 ErrSignature; /**< Variable to store the last background test signature values for all blocks */
#endif
} FlsTst_ErrBlockType;


/**
 * @brief       Data Structure for background test error details
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_488
 */
typedef struct
{
    uint32 BgndErrIntervalId; /**<  This variable indicates the last interval id */
    uint32 BgndErrBlockCount; /**< This variable will hold the error block count of the background test in last interval */
    FlsTst_ErrBlockType BgndErrBlock[FLSTST_MAX_NUMBER_OF_BGND_BLOCKS]; /**< This variable will hold the error blocks index of the background test in last interval*/
} FlsTst_BgndErrDetailType;

/**
 * @brief       Data Structure for foreground test error details
 * @sw_type sw_detail
 *
 * @trace YTSDS_FlsTst_495
 */
typedef struct
{
    uint8 FgndErrStatus;  /**< This variable indicates  whether a foreground block test error has occurred */
    uint32 FgndErrBlockIndex; /**< This variable will hold the last error block id of the foreground test*/
#if (FLSTST_TEST_RESULT_SIGNATURE == STD_ON)
    uint32 FgndErrSignature; /**< This variable will hold the last error foreground test signature value */
#endif
} FlsTst_FgndErrDetailType;


/**
 * @brief    This type shall specify implementation specific error information monitored in the Flash test module.
 * @sw_type sw_arch
 *
 * @trace YTSAS_FlsTst_61
 */
typedef struct
{
    FlsTst_FgndErrDetailType FgndErrDetail; /**< This variable will hold the last foreground test error details */
    FlsTst_BgndErrDetailType BgndErrDetail; /**< This variable will hold the last background test error details */
} FlsTst_ErrorDetailsType;
#endif  /* (FLSTST_GET_ERROR_DETAILS_API == STD_ON) */

/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#ifdef __cplusplus
}
#endif

#endif /* End of file FlsTst_Types.h */
