/**
* @file    EfmTst_Lld.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : EfmTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 0306 Rule-11.4: A conversion should not be performed between a pointer to object
 *                        and an integer type.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "EfmTst_Lld.h"
#include "pSIP_Efm.h"
#include "Platform.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define EFMTST_LLD_VENDOR_ID_C                  (0xB4)
#define EFMTST_LLD_AR_RELEASE_MAJOR_VERSION_C              (4)
#define EFMTST_LLD_AR_RELEASE_MINOR_VERSION_C              (4)
#define EFMTST_LLD_AR_RELEASE_REVISION_VERSION_C           (0)
#define EFMTST_LLD_SW_MAJOR_VERSION_C                  (2)
#define EFMTST_LLD_SW_MINOR_VERSION_C                  (0)
#define EFMTST_LLD_SW_PATCH_VERSION_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EFMTST_LLD header file are of the same vendor */
#if (EFMTST_LLD_VENDOR_ID_C != EFMTST_LLD_VENDOR_ID)
#error "EfmTst_Lld.c and EfmTst_Lld.h have different vendor ids"
#endif

/* Check if source file and EFMTST_LLD header file are of the same Autosar version */
#if (( EFMTST_LLD_AR_RELEASE_MAJOR_VERSION_C != EFMTST_LLD_AR_RELEASE_MAJOR_VERSION) || \
      ( EFMTST_LLD_AR_RELEASE_MINOR_VERSION_C != EFMTST_LLD_AR_RELEASE_MINOR_VERSION) || \
      ( EFMTST_LLD_AR_RELEASE_REVISION_VERSION_C != EFMTST_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of EfmTst_Lld.c and EfmTst_Lld.h are different"
#endif

/* Check if source file and EFMTST_LLD header file are of the same Software version */
#if (( EFMTST_LLD_SW_MAJOR_VERSION_C != EFMTST_LLD_SW_MAJOR_VERSION) || \
      ( EFMTST_LLD_SW_MINOR_VERSION_C != EFMTST_LLD_SW_MINOR_VERSION) || \
      ( EFMTST_LLD_SW_PATCH_VERSION_C != EFMTST_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of EfmTst_Lld.c and EfmTst_Lld.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
extern const EfmTst_ConfigType *EfmTst_ConfigPtr;
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
#define EFMTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "EfmTst_MemMap.h"
#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
/**
 * @brief WRITE ENABLE test busfault handler pointer
 * @sw_type sw_detail
 * @private
 */
static Platform_IrqHandlerType EfmTst_SysExpPointer;
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON) */
#define EFMTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "EfmTst_MemMap.h"

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
#define EFMTST_START_SEC_CODE
#include "EfmTst_MemMap.h"
/**
 * @brief Set flash protect register
 * @param[in]  ChnIndex  The index of the protection register channel
 * @param[in]  ProtectReg  The value to be written into the protection register
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n

 *
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_540
*/
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_SetProtectReg(uint8 ChnIndex, uint32 ProtectReg)
{
    /* Set the protection register */
    EFM->ADDR_PROT[ChnIndex] = ProtectReg;
}

/**
 * @brief Service for getting flash hardware status.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n

 *
 * @return  boolean
 * @retval TRUE: The flash is in an idle state
 * @retval FALSE: The flash is in a non-idle state
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_887
*/
EFMTST_FUNC LOCAL_INLINE boolean EfmTst_Lld_CheckIdle(void)
{
    /* Check if the flash is in an idle state */
    return ((boolean)((EFM->STS & EFM_STS_IDLE_MASK) >> EFM_STS_IDLE_SHIFT));
}

#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
/**
 * @brief Service for checking the Write Enable Error flag in EFM status register.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n

 *
 * @return  boolean
 * @retval TRUE: Write Enable Error flag is set
 * @retval FALSE: Write Enable Error flag is not set
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_696
*/
EFMTST_FUNC static boolean EfmTst_Lld_CheckWeErrorFlag(void)
{
    boolean TempRet = FALSE;
    /* Check if the Write Enable Error flag is set */
    if (0U != (EFM->STS & EFM_STS_CI_WEERR_MASK))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}
#endif

#if (EFMTST_ENABLE_CMD_UNLOCK_TEST == STD_ON)
/**
 * @brief Service for checking the Command Error flag in EFM status register.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return boolean
 * @retval TRUE: Command Error flag is set
 * @retval FALSE: Command Error flag is not set
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_695
 */
EFMTST_FUNC static boolean EfmTst_Lld_CheckCmdErrorFlag(void)
{
    boolean TempRet = FALSE;
    /* Check if the Command Error flag is set */
    if (0U != (EFM->STS & EFM_STS_ACCERR_MASK))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}
#endif  /* #if (EFMTST_ENABLE_CMD_UNLOCK_TEST == STD_ON) */

/**
 * @brief Service for injecting an incorrect unlock command.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_888
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_ErrorUnlockCmd(void)
{
    /* Inject an incorrect unlock command */
    EFM->CMD_UNLOCK = EFMTST_ERROR_CMDUNLOCK_KEY;
}

/**
 * @brief Service for loading a command into the EFM command register.
 *
 * @param[in] Command The command to be loaded into the EFM command register.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_704
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_LoadCommand(uint8 Command)
{
    /* Load the command into the EFM command register */
    EFM->CMD = Command;
}

/**
 * @brief Service for clearing the status register of EFM.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_697
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_ClearRegSTS(void)
{
    /* Clear the status register */
    EFM->STS =  EFMTST_ERR_FLAGS_MASK | EFM_STS_DONE_MASK;
}

/**
 * @brief Service for executing a NOP command in EFM.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_700
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_EFMNopCommand(void)
{
    /* Execute a NOP command */
    EFM->CMD = 0U;
    EFM->STS = EFMTST_ERR_FLAGS_MASK;
}

#if (EFMTST_WEBIT_AVAILABLE == STD_ON)
/**
 * @brief Service for setting the Write Enable (WE) bit in EFM control register.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_705
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_SetWE(void)
{
    /* Set the Write Enable (WE) bit */
    EFM->CTRL |= EFM_CTRL_WE_MASK;
    __ASM("ISB");
    __ASM("DSB");
}
#endif

#if (EFMTST_WEBIT_AVAILABLE == STD_ON)
/**
 * @brief Service for clearing the Write Enable (WE) bit in EFM control register.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_698
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_ClearWE(void)
{
    __ASM("ISB");
    __ASM("DSB");
    /* Clear the Write Enable (WE) bit*/
    EFM->CTRL &= ~EFM_CTRL_WE_MASK;
}
#endif

#if ((EFMTST_WEBIT_AVAILABLE == STD_ON) && ((EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON) || (EFMTST_ENABLE_CMD_UNLOCK_TEST == STD_ON)))
/**
 * @brief Service for judging if the address is set within a timeout period.
 *
 * @param[in] Timeout The timeout period to wait for the address to be set.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return boolean
 * @retval TRUE: Address is set within the timeout period
 * @retval FALSE: Address is not set within the timeout period
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_703
 */
EFMTST_FUNC static boolean EfmTst_Lld_JudgeSetAddr(uint16 Timeout)
{
    uint16 TimeoutCnt = Timeout;
    boolean TempRet = FALSE;
    while (0U != TimeoutCnt)
    {
        if (0U != (EFM->STS & EFMTST_ADDR_SELECTED_FLAG_MASK))
        {
            TempRet = TRUE;
            break;
        }
        /*GCOVR_EXCL_START*/
        else
        {
            /*The integration test can't convered, this branch can be accessed only when the EFM hardware fails*/
            --TimeoutCnt;
        }
        /*GCOVR_EXCL_STOP*/
    }
    return TempRet;
}
#endif

#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
/**
 * @brief Service for handling BusFault exception interrupt.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_694
 */
EFMTST_FUNC LOCAL_INLINE void EfmTst_Lld_BusFaultExpIsr(void)
{
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear EMU error flags.
    * couldn't adhere to M3CM Rule-11.4
    */
    /*Clear the BusFault exception flag*/
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk | SCB_CFSR_PRECISERR_Msk; /*PRQA S 0306*/
}
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON) */

#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
/**
 * @brief Service for setting the BusFault exception handler.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_702
 */
EFMTST_FUNC static void EfmTst_Lld_Fls_SetExpHandler(void)
{
    /*************************************************************************************************/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear EMU error flags.
    * couldn't adhere to M3CM Rule-11.4
    */
    /*Enable usageFault exception*/
    SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk;/*PRQA S 0306*/
    SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk;/*PRQA S 0306*/
    (void)Platform_InstallIrqHandler(BusFault_IRQn, EfmTst_Lld_BusFaultExpIsr, &EfmTst_SysExpPointer);
}
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON) */

#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
/**
 * @brief Service for recovering the BusFault exception handler.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_701
 */
EFMTST_FUNC static void EfmTst_Lld_Fls_RecoveredExpHandler(void)
{
    Platform_IrqHandlerType TempBusFaultHandler;
    (void)Platform_InstallIrqHandler(BusFault_IRQn, EfmTst_SysExpPointer, &TempBusFaultHandler);
}
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON) */
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief This function will set flash protect register according to the configuration
 * @param[in]  SectorProtCfg  Pointer to the EFM test flash protection configuration data
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n

 *
 * @return     void
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_539
*/
EFMTST_FUNC void EfmTst_Lld_Init(EfmTst_SectorProtCfgType const *SectorProtCfg)
{
    /* Check if the flash is in an idle state */
    if (TRUE == EfmTst_Lld_CheckIdle())
    {
        EfmTst_Lld_ClearRegSTS();
        for (uint8 LoopCnt = 0; LoopCnt < EFMTST_INTERNAL_BLOCK_NUM; ++LoopCnt)
        {
            EfmTst_Lld_SetProtectReg(LoopCnt, ~(SectorProtCfg[LoopCnt]));
        }
    }
    else
    {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
        if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
        {
            EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
        }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
    }
}

#if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON)
/**
 * @brief Service for starting the Write Enable test.
 * @details The function initializes the test by clearing the status register, executing a NOP command, setting the exception handler,
 *          and writing a test value to a protected sector. It then checks if the address is set and verifies the Write Error flag.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @retval E_OK: The test of the Write Enable mechanism of EfmTst is OK.
 * @retval E_NOT_OK: The test of the Write Enable mechanism of EfmTst is NOT_OK.
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_706
 */
EFMTST_FUNC Std_ReturnType EfmTst_Lld_WriteEnable_Start(void)
{
    Std_ReturnType TempRet = E_NOT_OK;
    /* Check if the flash is in an idle state */
    if (TRUE == EfmTst_Lld_CheckIdle())
    {
        /* Clear the status register and execut a NOP command*/
        EfmTst_Lld_ClearRegSTS();
        EfmTst_Lld_EFMNopCommand();
        /* set the BusFault exception handler*/
        EfmTst_Lld_Fls_SetExpHandler();
        /* Perform operations on the flash without setting the WE bit. */
        *(volatile uint32 *)EfmTst_ConfigPtr->NoneProtectSectorAddr = 0x01234567U;
        /* Judge the test results*/
        if (FALSE == EfmTst_Lld_JudgeSetAddr(0x1000U))
        {
            if (TRUE == EfmTst_Lld_CheckWeErrorFlag())
            {
                TempRet = E_OK;
            }
            else
            {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
                if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
                {
                    EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
                }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
            }
        }
        else
        {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
            if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
            {
                EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
            }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
        }
        /* Recover the BusFault exception handler*/
        EfmTst_Lld_Fls_RecoveredExpHandler();
    }
    else
    {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
        if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
        {
            EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
        }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
    }
    return TempRet;
}
#endif /* #if (EFMTST_ENABLE_WRITE_ENABLE_TEST == STD_ON) */

#if (EFMTST_ENABLE_CMD_UNLOCK_TEST == STD_ON)
/**
 * @brief Service for starting the Command Unlock test.
 * @details The function initializes the test by clearing the status register, executing a NOP command, setting the Write Enable bit,
 *          and writing a test value to a protected sector. It then clears the Write Enable bit and checks if the address is set.
 *          If the address is not set, it loads a sector erase command and verifies the Command Error flag.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @retval E_OK: The test of the Command Unlock mechanism of EfmTst is OK.
 * @retval E_NOT_OK: The test of the Command Unlock mechanism of EfmTst is NOT_OK.
 * @sw_type sw_detail
 *
 * @trace YTDT_EfmTst_699
 */
EFMTST_FUNC Std_ReturnType EfmTst_Lld_CommandUnlock_Start(void)
{
    Std_ReturnType TempRet = E_NOT_OK;
    /* Check if the flash is in an idle state */
    if (TRUE == EfmTst_Lld_CheckIdle())
    {
        /* Clear the status register and execut a NOP command*/
        EfmTst_Lld_ClearRegSTS();
        EfmTst_Lld_EFMNopCommand();
        /* Set the WE bit before selecting the flash address*/
#if (EFMTST_WEBIT_AVAILABLE == STD_ON)
        EfmTst_Lld_SetWE();
#else
#if (PLATFORM_MPU_CFG == STD_ON)
        ARM_MPU_Disable();
#endif /* PLATFORM_MPU_CFG == STD_ON */
#endif /* EFMTST_WEBIT_AVAILABLE == STD_ON */
        *(volatile uint32 *)EfmTst_ConfigPtr->NoneProtectSectorAddr = 0x01234567U;
        /* Clear the WE bit after selecting the flash address*/
#if (EFMTST_WEBIT_AVAILABLE == STD_ON)
        EfmTst_Lld_ClearWE();
        /* Wait for the successful selection of the flash address*/
        if (FALSE == EfmTst_Lld_JudgeSetAddr(0x1000U))
        {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
            if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
            {
                EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
            }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
        }
        else
#else
#if (PLATFORM_MPU_CFG == STD_ON)
        ARM_MPU_Enable(MPU_CTRL_PRIVDEFENA_Msk);
#endif /* PLATFORM_MPU_CFG == STD_ON */
#endif /* EFMTST_WEBIT_AVAILABLE == STD_ON */
        {
            /* Load the incorrect command unlock key*/
            EfmTst_Lld_ErrorUnlockCmd();
            EfmTst_Lld_LoadCommand(EFMTST_SECTOR_ERASE_COMMAND_CODE);
            /* Judge the test results*/
            if (TRUE == EfmTst_Lld_CheckCmdErrorFlag())
            {
                TempRet = E_OK;
            }
            else
            {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
                if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
                {
                    EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
                }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
            }
        }
    }
    else
    {
#if (EFMTST_ERROR_NOTIFICATION==STD_ON)
        if (NULL_PTR != EfmTst_ConfigPtr->EfmTstErrorNotification)
        {
            EfmTst_ConfigPtr->EfmTstErrorNotification(EFMTST_MODULE_ID);
        }
#endif /* EFMTST_ERROR_NOTIFICATION == STD_ON */
    }
    return TempRet;
}
#endif /* #if (EFMTST_ENABLE_CMD_UNLOCK_TEST == STD_ON) */
#define EFMTST_STOP_SEC_CODE
#include "EfmTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file EfmTst_Lld.c */
