/*
* @file    EfmTst.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : EfmTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef EFMTST_H
#define EFMTST_H

#ifdef __cplusplus
extern "C" {
#endif


/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "EfmTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define EFMTST_VENDOR_ID                      (0xB4)
#define EFMTST_AR_RELEASE_MAJOR_VERSION       (4)
#define EFMTST_AR_RELEASE_MINOR_VERSION       (4)
#define EFMTST_AR_RELEASE_REVISION_VERSION    (0)
#define EFMTST_SW_MAJOR_VERSION               (2)
#define EFMTST_SW_MINOR_VERSION               (0)
#define EFMTST_SW_PATCH_VERSION               (0)


/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EFMTST TYPES header file are of the same vendor */
#if (EFMTST_VENDOR_ID != EFMTST_TYPES_VENDOR_ID)
#error "EfmTst.h and EfmTst_Types.h have different vendor ids"
#endif

/* Check if source file and EFMTST TYPES header file are of the same Autosar version */
#if ((EFMTST_AR_RELEASE_MAJOR_VERSION != EFMTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (EFMTST_AR_RELEASE_MINOR_VERSION != EFMTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (EFMTST_AR_RELEASE_REVISION_VERSION != EFMTST_TYPES_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers ofEfmTst.h and EfmTst_Types.h are different"
#endif

/* Check if source file and EFMTST TYPES header file are of the same software version */
#if ((EFMTST_SW_MAJOR_VERSION != EFMTST_TYPES_SW_MAJOR_VERSION) || \
     (EFMTST_SW_MINOR_VERSION != EFMTST_TYPES_SW_MINOR_VERSION) || \
     (EFMTST_SW_PATCH_VERSION != EFMTST_TYPES_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EfmTst.h and EfmTst_Types.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/**
* @brief           EfmTest error check timeout
* @sw_type         sw_detail
*/
#define EFMTST_ERROR_CHECK_TIMEOUT                    (0x1000U)

/**
 * @defgroup EfmTst_Dev_Error_Macro
 * @brief   EfmTst module supported development error detailed list.
 * @sw_type sw_detail
 * @{
 *
 * @trace YTDT_EfmTst_429
 */
#define EFMTST_E_ALREADY_INITIALIZED           ((uint8)0x01)  /**< All API's called in wrong sequence shall return this error*/
#define EFMTST_E_INIT_FAILED                   ((uint8)0x02)  /**< API EFMTST_Init service called with wrong parameter. */
#define EFMTST_E_PARAM_POINTER                 ((uint8)0x03)  /**< Error in param pointer */
#define EFMTST_E_UNINIT                        ((uint8)0x04)  /**< API called without module initialization */
#define EFMTST_E_INVALID_STATE                 ((uint8)0x05)  /**<  An API's called in an invalid state */
/** @} */

/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                            ENUMS
==================================================================================================*/

/*==================================================================================================
 *                                        TYPEDEF STRUCT
==================================================================================================*/

/*==================================================================================================
 *                                  GLOBAL CONSTANT DECLARATIONS
==================================================================================================*/

/*==================================================================================================
 *                                             FUNCTION
==================================================================================================*/
/**
 * @brief  Service for EFM Test initialization.
 * @details The function shall initialize all EFM Test relevant registers and global variables.
 * @param[in] ConfigPtr Pointer to the EFM Test configuration data structure.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
void EfmTst_Init(EfmTst_ConfigType const *ConfigPtr);

/**
 * @brief Service for EFM Test De-Initialization.
 * @details The function de-initializes all EFM Test relevant registers and global variables.
 *
 * @table            @service_id:       0x02    \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
void EfmTst_DeInit(void);

/**
 * @brief  Service for executing EFM Test.
 * @details The function executes all test items based on the configuration of the EfmTst module.
 *
 * @table            @service_id:       0x03  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
void EfmTst_Start(void);

/**
 * @brief  Service for getting EFM Test result.
 * @details The function returns the result of the EFM Test.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return EfmTst_TestResultType
 * @retval EFMTST_NOT_TESTED:The EFM Test is not completed.
 * @retval EFMTST_OK: The EFM Test test result is OK.
 * @retval EFMTST_NOT_OK: The EFM Test result is not OK.
 * @sw_type sw_arch
 *
 */
EfmTst_TestResultType EfmTst_GetResult(void);

/**
 * @brief Service for getting EFM Test result details.
 * @details The function returns the result details of the EFM Test driver.
 * @param[out] TestDetails  Pointer to the structure of EFMTst result detail data.
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return void
 * @sw_type sw_arch
 *
 */
void EfmTst_GetDetailsResult(EfmTst_TestDetailsType *TestDetails);

#if (EFMTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            Returns the version information of this module.
 * @param[out]       versioninfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x06  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
 *
 * @return           void
 * @sw_type          sw_arch
 *
 */
void EfmTst_GetVersionInfo(Std_VersionInfoType *VersionInfo);
#endif /* (EFMTST_VERSION_INFO_API == STD_ON) */

#ifdef __cplusplus
}
#endif

#endif  /* End of file EfmTst.h */


