/**
 * @file    EccTst.c
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : EccTst
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 0306 Rule-11.4: A conversion should not be performed between a pointer to object
 *                        and an integer type.
 *
 */
/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "EccTst_Lld_Fls.h"
#include "EccTst_Lld_Reg.h"
#include "Platform.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_LLD_FLS_VENDOR_ID_C                      (0xB4)
#define ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION_C       (4)
#define ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION_C       (4)
#define ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION_C    (0)
#define ECCTST_LLD_FLS_SW_MAJOR_VERSION_C               (2)
#define ECCTST_LLD_FLS_SW_MINOR_VERSION_C               (0)
#define ECCTST_LLD_FLS_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and EccTst_Lld_Fls header file are of the same vendor */
#if (ECCTST_LLD_FLS_VENDOR_ID_C != ECCTST_LLD_FLS_VENDOR_ID)
#error "EccTst_Lld_Fls.c and EccTst_Lld_Fls.h have different vendor ids"
#endif

/* Check if source file and EccTst_Lld_Fls header file are of the same Autosar version */
#if ((ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Fls.c and EccTst_Lld_Fls.h are different"
#endif

/* Check if source file and EccTst_Lld_Fls header file are of the same Software version */
#if ((ECCTST_LLD_FLS_SW_MAJOR_VERSION_C != ECCTST_LLD_FLS_SW_MAJOR_VERSION) || \
     (ECCTST_LLD_FLS_SW_MINOR_VERSION_C != ECCTST_LLD_FLS_SW_MINOR_VERSION) || \
     (ECCTST_LLD_FLS_SW_PATCH_VERSION_C != ECCTST_LLD_FLS_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EccTst_Lld_Fls.c and EccTst_Lld_Fls.h are different"
#endif

/* Check if source file and ECCTST register header file are of the same vendor */
#if (ECCTST_LLD_FLS_VENDOR_ID_C != ECCTST_LLD_REG_VENDOR_ID)
#error "EccTst_Lld_Fls.c and EccTst_Lld_Reg.h have different vendor ids"
#endif

/* Check if source file and ECCTST register header file are of the same Autosar version */
#if ((ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_REG_AR_REL_MAJOR_VER) || \
     (ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_REG_AR_REL_MINOR_VER) || \
     (ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_REG_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Fls.c and EccTst_Lld_Reg.h are different"
#endif

/* Check if source file and ECCTST register header file are of the same Software version */
#if ((ECCTST_LLD_FLS_SW_MAJOR_VERSION_C != ECCTST_LLD_REG_SW_MAJOR_VER) || \
     (ECCTST_LLD_FLS_SW_MINOR_VERSION_C != ECCTST_LLD_REG_SW_MINOR_VER) || \
     (ECCTST_LLD_FLS_SW_PATCH_VERSION_C != ECCTST_LLD_REG_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst_Lld_Fls.c and EccTst_Lld_Reg.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#define EFM_CTRL_FDBERR_MASK                (0x00000080U)
#define EFM_CTRL_FDBERR_SHIFT               (7U)
#define EFM_CTRL_FDBERR(x)                  (((uint32)(((uint32)(x)) << EFM_CTRL_FDBERR_SHIFT)) & EFM_CTRL_FDBERR_MASK)
#if defined (CPU_YTM32B1HA0)
#define ECCTST_FLS_TEST_DATA                (0x00000001U)
#define ECCTST_FLS_NO_ERR                   (0x00000000U)
#endif

#define FLSECC_TEST_INIT_SIGNATURE  0x5A5AU

#define FLSECC_FLSERRIRQ_SIGNATURE  0xE327U

#define FLSECC_BUSFAULT_SIGNATURE   0xD495U

#define FLSECC_SINGLEBIT_SIGNATURE  0xB97DU

#if defined (CPU_YTM32B1MC0)
#define FLSECC_DOUBLEBIT_SIGNATURE  0x6DE8U
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)
#define FLSECC_DOUBLEBIT_SIGNATURE  0xB97DU
#elif defined (CPU_YTM32B1HA0)
#define FLSECC_DOUBLEBIT_SIGNATURE  0x8ECFU
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/
/**
 * @brief Check ongoing FLASH ECC test result
 * @sw_type sw_detail
 *
 */
static EccTst_TestResultType EccTst_Lld_Fls_ResultCheck(FlsEcc_AddressType CheckAddr);

/**
 * @brief Check ongoing FLASH ECC test result
 * @sw_type sw_detail
 *
 */
static void EccTst_Lld_Fls_ClearUnRecoverr(void);

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/


/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/
/* MR12 RULE 8.4 VIOLATION: The variable will used in integration test,
* thus, couldn't adhere to M3CM Rule-8.4 and Rule-8.7.
*/
#define ECCTST_START_SEC_VAR_INIT_32
#include "EccTst_MemMap.h"
/**
 * @brief FLASH ECC test configuration data pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static FlsEcc_ConfigType const *FlsEcc_ConfigPtr = NULL_PTR;
#define ECCTST_STOP_SEC_VAR_INIT_32
#include "EccTst_MemMap.h"

#define ECCTST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "EccTst_MemMap.h"
/**
 * @brief FLASH ECC test status data
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static volatile FlsEcc_RunningStatusType FlsEcc_RunningData;
/**
 * @brief FLASH ECC test detail result
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static volatile FlsEcc_TestDetailsType FlsEcc_DetailResult;
/**
 * @brief FLASH ECC test busfault handler pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static Platform_IrqHandlerType FlsEcc_SysExpPointer;
/**
 * @brief FLASH ECC error handler pointer
 * @sw_type sw_detail
 * @private
 */
ECCTST_VAR static Platform_IrqHandlerType FlsEcc_EccIrqPointer;
#define ECCTST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "EccTst_MemMap.h"

/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#if defined(CPU_YTM32B1ME0)||defined (CPU_YTM32B1MD1)
#define ECCTST_START_SEC_RAMCODE
#include "EccTst_MemMap.h"
/**
 * @brief Activate the FLASH ECC injection feature.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_174
 * @private
 */
ECCTST_FUNC static void EccTst_Lld_Fls_EnInjectErr(void)
{
    /*disable Prefetch*/
    EFM->CTRL &= ~EFM_CTRL_PREFETCH_EN_MASK;
    /*enable injection*/
    EFM->CTRL |= EFM_CTRL_FDBERR_MASK;
}
/**
 * @brief Disable the FLASH ECC injection feature.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_108
 * @private
 */
ECCTST_FUNC static void EccTst_Lld_Fls_DisInjectErr(void)
{
    /*disable injection*/
    EFM->CTRL &= ~EFM_CTRL_FDBERR_MASK;
    /*enable Prefetch*/
    EFM->CTRL |= EFM_CTRL_PREFETCH_EN_MASK;
}
/**
 * @brief Execute the FLASH ECC injection test.
 * @param[in] TestAddr FLASH ECC injection test address.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @private
 */
ECCTST_FUNC static void EccTst_Lld_Fls_TestEnable(uint32 TestAddr)
{
    uint32 TempAddr = TestAddr;
    volatile uint32 TempData = 0;
    EccTst_Lld_Fls_EnInjectErr();
    /* MR12 RULE 11.4 VIOLATION: The parameter TestAddr is a configure and cacultle result in YTM Config tools,
    * thus, couldn't adhere to M3CM Rule-11.4
    */
    TempData = *((uint32 *)TempAddr);/*PRQA S 0306*/
    /* The purpose of the following code is simply to keep the previous line of code from being optimized by the compiler.*/
    TempData = TempData + 1U;
    if (TempData > 0x0FU)
    {
        __ASM("nop");
    }
    EccTst_Lld_Fls_DisInjectErr();
}
#define ECCTST_STOP_SEC_RAMCODE
#include "EccTst_MemMap.h"
#elif defined (CPU_YTM32B1MC0)
#define ECCTST_START_SEC_RAMCODE
#include "EccTst_MemMap.h"
/**
 * @brief Activate the FLASH ECC injection feature.
 * @param[in] TestMethod FLASH ECC injection error type.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_174
 * @private
 */
ECCTST_FUNC static void EccTst_Lld_Fls_EnInjectErr(FlsEcc_InjectErrorModeType TestMethod)
{
    /*disable Prefetch*/
    EFM->CTRL &= ~EFM_CTRL_PREFETCH_EN_MASK;
    /*enable injection*/
    if (FLSECC_BLOCK_ONEBIT_TEST == TestMethod)
    {
        EFM->ERR_INJECT_CTRL = EFM_ERR_INJECT_CTRL_ECCERR_INJ_EN(1);
    }
    else
    {
        EFM->ERR_INJECT_CTRL = EFM_ERR_INJECT_CTRL_ECCERR_INJ_EN(3);
    }
}
/**
 * @brief Disable the FLASH ECC injection feature.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_108
 * @private
 */
ECCTST_FUNC static void EccTst_Lld_Fls_DisInjectErr(void)
{
    EFM->ERR_INJECT_CTRL = EFM_ERR_INJECT_CTRL_ECCERR_INJ_EN(0);
    /*enable Prefetch*/
    EFM->CTRL |= EFM_CTRL_PREFETCH_EN_MASK;
}
/**
 * @brief Execute the FLASH ECC injection test.
 * @param[in] TestAddr FLASH ECC injection test address.
 * @param[in] TestMethod FLASH ECC injection error type.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_177
 * @private
 */

ECCTST_FUNC static void EccTst_Lld_Fls_TestEnable(uint32 InjectAddr, FlsEcc_InjectErrorModeType TestMethod)
{
    volatile uint32 TempData = 0U;
    /*enable the error injection*/
    EccTst_Lld_Fls_EnInjectErr(TestMethod);
    /*Trigger fls ECC error*/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read the InjectAddr to trigger ECC error,
    * and couldn't adhere to M3CM Rule-11.4.
    */
    TempData = *((uint32 *)InjectAddr); /*PRQA S 0306*/
    /* The purpose of the following code is simply to keep the previous line of code from being optimized by the compiler.*/
    TempData += 1U;
    if (TempData != 1U)
    {
        __ASM("nop");
    }
    EccTst_Lld_Fls_DisInjectErr();
}

#define ECCTST_STOP_SEC_RAMCODE
#include "EccTst_MemMap.h"
#elif defined (CPU_YTM32B1HA0)

#define ECCTST_START_SEC_CODE
#include "EccTst_MemMap.h"
/**
 * @brief Set injection test data.
 * @param[in] InjectData0 FLASH ECC injection test data.
 * @param[in] InkectData1 FLASH ECC injection test data.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_176
 */
ECCTST_FUNC LOCAL_INLINE void EccTst_Lld_Fls_InjectData(uint32 InjectData0, uint32 InkectData1)
{
    EFM->ECCERR_INJECT_MSK_0 = InjectData0;
    EFM->ECCERR_INJECT_MSK_1 = InkectData1;
}
/**
 * @brief Set injection test address.
 * @param[in] InjectAddr FLASH ECC injection test address.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_175
 */
ECCTST_FUNC LOCAL_INLINE void EccTst_Lld_Fls_InjectAddr(uint32 InjectAddr)
{
    EFM->ECCERR_INJECT_ADDR = InjectAddr;
}

/**
 * @brief Activate the FLASH ECC injection feature.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_174
 */
ECCTST_FUNC LOCAL_INLINE void EccTst_Lld_Fls_EnInjectErr(void)
{
    /*enable injection*/
    EFM->CTRL |= EFM_CTRL_ECCERR_INJ_EN_MASK;
    if (0x00000000U == (EFM->CTRL & EFM_CTRL_ECCERR_INJ_EN_MASK))
    {
        __ASM("nop");
    }
}
/**
 * @brief Disable the FLASH ECC injection feature.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_108
 */
ECCTST_FUNC static void EccTst_Lld_Fls_DisInjectErr(void)
{
    /*disable injection*/
    EFM->CTRL &= ~EFM_CTRL_ECCERR_INJ_EN_MASK;
    if ((EFM->CTRL & EFM_CTRL_ECCERR_INJ_EN_MASK) != 0x00000000U)
    {
        __ASM("nop");
    }
}
/**
 * @brief Execute the FLASH ECC injection test.
 * @param[in] TestAddr FLASH ECC injection test address.
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_177
 */
ECCTST_FUNC static void EccTst_Lld_Fls_TestEnable(uint32 InjectAddr, uint32 InjectData0, uint32 InkectData1)
{
    volatile uint32 TempData = 0;
    /*enable the error injection,the one bit ECC error will happen when read data in test address*/
    EccTst_Lld_Fls_EnInjectErr();
    /*Inject error data and address*/
    EccTst_Lld_Fls_InjectData(InjectData0, InkectData1);
    EccTst_Lld_Fls_InjectAddr(InjectAddr);
    __ASM("ISB");
    __ASM("DSB");
#if (PLATFORM_ENABLE_CACHE == STD_ON)
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to clear the DCACHE for ECC error injection,
    * and couldn't adhere to M3CM Rule-11.4.
    */
    SCB_InvalidateDCache_by_Addr((uint32 *)InjectAddr, 64); /*PRQA S 0306*/
#endif
    /*Trigger fls ECC error*/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read the InjectAddr to trigger ECC error,
    * and couldn't adhere to M3CM Rule-11.4.
    */
    TempData = *((uint32 *)InjectAddr); /*PRQA S 0306*/
    /*The purpose of the following code is simply to keep the previous line of code from being optimized by the compiler.*/
    TempData += 1U;
    if (TempData != 1U)
    {
        __ASM("nop");
    }
    EccTst_Lld_Fls_DisInjectErr();
}

#define ECCTST_STOP_SEC_CODE
#include "EccTst_MemMap.h"

#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif

#if defined (CPU_YTM32B1HA0)
#define ECCTST_START_SEC_CODE
#else
#define ECCTST_START_SEC_RAMCODE
#endif
#include "EccTst_MemMap.h"
/**
 * @brief This function is the ISR of system exception when FLASH ECC test running.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .ramcode \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_377
 */
ECCTST_FUNC static void EccTst_Lld_FlsEccExpIsr(void)
{
    uint32 TempData = 0;
    /* Disable error injection */
    EccTst_Lld_Fls_DisInjectErr();
    /*The result check for HA0 in two bit test and interrupt eable*/
#if defined (CPU_YTM32B1HA0)
    TempData = FlsEcc_RunningData.RunningBlockIndex;
    if (FLSECC_BLOCK_TWOBIT_TEST == FlsEcc_RunningData.BlockStatus)
    {
        FlsEcc_DetailResult.TwoBitTestResult[TempData] = \
                EccTst_Lld_Fls_ResultCheck(FlsEcc_ConfigPtr->TestBlockPtr[TempData].TwoBitTestAddr);
        EccTst_Lld_Fls_ClearUnRecoverr();
        FlsEcc_RunningData.BlockStatus = FLSECC_BLOCK_IDLE;
    }
#endif
    TempData = FlsEcc_RunningData.RunningSignature;
    FlsEcc_RunningData.RunningSignature = (TempData & 0xFFFF0000U) + \
                                          ((TempData ^ ((uint32)FLSECC_BUSFAULT_SIGNATURE)) & 0x0000FFFFU);
    /*Clear the BusFault exception flag*/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear BusFault error flags cuased by Fls ECC error injection,
    *  couldn't adhere to M3CM Rule-11.4
    */
    SCB->CFSR = SCB_CFSR_BUSFAULTSR_Msk | SCB_CFSR_PRECISERR_Msk; /*PRQA S 0306*/
}


/**
 * @brief Read FLASH one bit ECC error flag
 *
 * @table            @service_id:       --  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 *
 * @return boolean
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_146
*/
ECCTST_FUNC static boolean EccTst_Lld_Fls_GetRecEccErrFlag(void)
{
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)|| defined (CPU_YTM32B1MC0)
    return ((boolean)((EFM->STS & EFM_STS_RECOVERR_MASK) >> EFM_STS_RECOVERR_SHIFT));
#elif defined (CPU_YTM32B1HA0)
    return ((boolean)((EFM->STS & EFM_STS_CI_RECOVERR_MASK) >> EFM_STS_CI_RECOVERR_SHIFT));
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}
/**
 * @brief Read FLASH two bit ECC error flag
 *
 * @table            @service_id:       --  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 *
 * @return boolean
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_147
*/
ECCTST_FUNC static boolean EccTst_Lld_Fls_GetUnRecEccErrFlag(void)
{
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)|| defined (CPU_YTM32B1MC0)
    return ((boolean)((EFM->STS & EFM_STS_UNRECOVERR_MASK) >> EFM_STS_UNRECOVERR_SHIFT));
#elif defined (CPU_YTM32B1HA0)
    return ((boolean)((EFM->STS & EFM_STS_CI_UNRECOVERR_MASK) >> EFM_STS_CI_UNRECOVERR_SHIFT));
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}

/**
 * @brief Read FLASH one bit ECC error interrupt enable status
 *
 * @table            @service_id:       --  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 *
 * @return boolean
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_447
*/
ECCTST_FUNC static boolean EccTst_Lld_Fls_GetRecIntcEnable(void)
{
    return ((boolean)((EFM->CTRL & EFM_CTRL_RECOVERRIE_MASK) >> EFM_CTRL_RECOVERRIE_SHIFT));
}

/**
 * @brief Read FLASH two bit ECC error interrupt enable status
 *
 * @table            @service_id:       --  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 *
 * @return boolean
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_448
*/
ECCTST_FUNC static boolean EccTst_Lld_Fls_GetUnRecIntcEnable(void)
{
    return ((boolean)((EFM->CTRL & EFM_CTRL_UNRECOVERRIE_MASK) >> EFM_CTRL_UNRECOVERRIE_SHIFT));
}

#if defined (CPU_YTM32B1HA0)
#define ECCTST_STOP_SEC_CODE
#else
#define ECCTST_STOP_SEC_RAMCODE
#endif
#include "EccTst_MemMap.h"


#define ECCTST_START_SEC_CODE
#include "EccTst_MemMap.h"
/**
 * @brief Enable or disable the UNRECOVERRIE interrupt
 * @param[in] Enable UNRECOVERRIE interrupt enable or disable
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_149
 */
ECCTST_FUNC static void EccTst_Lld_Fls_SetUnrecErrIE(boolean Enable)
{
    if (TRUE == Enable)
    {
        EFM->CTRL |= EFM_CTRL_UNRECOVERRIE_MASK;
    }
    else
    {
        EFM->CTRL &= ~EFM_CTRL_UNRECOVERRIE_MASK;
    }
}

/**
 * @brief Enable or disable the RECOVERRIE interrupt
 * @param[in] Enable RECOVERRIE interrupt enable or disable
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_148
 */
ECCTST_FUNC static void EccTst_Lld_Fls_SetRecErrIE(boolean Enable)
{
    if (TRUE == Enable)
    {
        EFM->CTRL |= EFM_CTRL_RECOVERRIE_MASK;
    }
    else
    {
        EFM->CTRL &= ~EFM_CTRL_RECOVERRIE_MASK;
    }
}
/**
 * @brief Read FLASH ECC error address
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return uint32
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_145
 */
ECCTST_FUNC LOCAL_INLINE uint32 EccTst_Lld_Fls_GetEccErrAddr(void)
{
#if defined (CPU_YTM32B1HA0)
    return (EFM->ECC_ERR_ADDR);
#elif defined (CPU_YTM32B1MC0)
    return (EFM->ECCERR_ADDR);
#elif defined (CPU_YTM32B1MD1)||defined (CPU_YTM32B1ME0)
    return (EFM->ERR_ADDR);
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}

/**
 * @brief Read FLASH ECC error address
 * @param[in] EccIntcEnable FLASH ECC error interrupt enable or disable
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_144
 */
ECCTST_FUNC  static void EccTst_Lld_FLs_EccErrIntcInit(boolean EccIntcEnable)
{
    if (TRUE == EccIntcEnable)
    {
        /*Enable the FLASH ECC error interrupt*/
        EccTst_Lld_Fls_SetUnrecErrIE(TRUE);
        EccTst_Lld_Fls_SetRecErrIE(TRUE);
    }
    else
    {
        /*Disable the FLASH ECC error interrupt*/
        EccTst_Lld_Fls_SetUnrecErrIE(FALSE);
        EccTst_Lld_Fls_SetRecErrIE(FALSE);
    }
}


/**
 * @brief Initialize the data of the FLASH ECC test
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_134
 */
ECCTST_FUNC static void EccTst_Lld_Fls_DataInit(void)
{
    uint8 BlockCount = 0;
    while (BlockCount < ECCTST_FLASH_ECC_MAX_BLOCK_NUM)
    {
        FlsEcc_DetailResult.OneBitTestResult[BlockCount] = ECCTST_NOT_TEST;
        FlsEcc_DetailResult.TwoBitTestResult[BlockCount] = ECCTST_NOT_TEST;
        ++BlockCount;
    }
    FlsEcc_RunningData.TestStatus = ECCTST_UNINIT;
    FlsEcc_RunningData.BlockStatus = FLSECC_BLOCK_IDLE;
    FlsEcc_RunningData.RunningBlockIndex = 0;
    FlsEcc_RunningData.SumaryResult = ECCTST_NOT_TEST;
}

/**
 * @brief Check ongoing FLASH ECC test result
 * @param[in] CheckAddr  The ongoing FLASH ECC test address
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return EccTst_TestResultType
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_150
 */
ECCTST_FUNC static EccTst_TestResultType EccTst_Lld_Fls_ResultCheck(FlsEcc_AddressType CheckAddr)
{
    EccTst_TestResultType Ret = ECCTST_NOT_TEST;
    boolean IsError = FALSE;
    uint32 ErrorAddr = 0;
    if (FLSECC_BLOCK_ONEBIT_TEST ==  FlsEcc_RunningData.BlockStatus)
    {
        IsError = EccTst_Lld_Fls_GetRecEccErrFlag();
    }
    else
    {
        IsError = EccTst_Lld_Fls_GetUnRecEccErrFlag();
    }
    ErrorAddr = EccTst_Lld_Fls_GetEccErrAddr();
    if ((TRUE == IsError) && (ErrorAddr == CheckAddr))
    {
        Ret = ECCTST_OK;
    }
    else
    {
        Ret = ECCTST_NOT_OK;
    }
    return Ret;
}

/**
 * @brief Clear FLASH two bit ECC error flag
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_132
 */
ECCTST_FUNC static void EccTst_Lld_Fls_ClearRecoverr(void)
{
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)|| defined (CPU_YTM32B1MC0)
    EFM->STS = EFM_STS_RECOVERR_MASK;
#elif defined (CPU_YTM32B1HA0)
    EFM->STS = EFM_STS_CI_RECOVERR_MASK;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}

/**
 * @brief Clear FLASH two bit ECC error flag
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_133
 */
ECCTST_FUNC static void EccTst_Lld_Fls_ClearUnRecoverr(void)
{
#if defined (CPU_YTM32B1ME0) || defined (CPU_YTM32B1MD1)|| defined (CPU_YTM32B1MC0)
    EFM->STS = EFM_STS_UNRECOVERR_MASK;
#elif defined (CPU_YTM32B1HA0)
    EFM->STS = EFM_STS_CI_UNRECOVERR_MASK;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}
/**
 * @brief Inject one bit ECC error for a configured test block
 * @param[in] BlockIndex Test block index
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_135
 */
ECCTST_FUNC static void EccTst_Lld_Fls_OneBitInject(uint8 BlockIndex)
{
#if defined (CPU_YTM32B1ME0)|| defined (CPU_YTM32B1MD1)
    (void)BlockIndex;
#elif defined (CPU_YTM32B1HA0)
    EccTst_Lld_Fls_TestEnable(FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].OneBitTestAddr, \
                              ECCTST_FLS_TEST_DATA, ECCTST_FLS_NO_ERR);
#elif defined (CPU_YTM32B1MC0)
    EccTst_Lld_Fls_TestEnable(FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].OneBitTestAddr, \
                              FLSECC_BLOCK_ONEBIT_TEST);
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}

/**
 * @brief Inject two bit ECC error for a configured test block
 * @param[in] BlockIndex Test block index
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_139
 */
ECCTST_FUNC static void EccTst_Lld_Fls_TwoBitInject(uint8 BlockIndex)
{
#if defined (CPU_YTM32B1HA0)
    EccTst_Lld_Fls_TestEnable(FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].TwoBitTestAddr, \
                              ECCTST_FLS_TEST_DATA, ECCTST_FLS_TEST_DATA);
#elif defined (CPU_YTM32B1ME0)|| defined (CPU_YTM32B1MD1)
    EccTst_Lld_Fls_TestEnable(FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].TwoBitTestAddr);
#elif defined (CPU_YTM32B1MC0)
    EccTst_Lld_Fls_TestEnable(FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].TwoBitTestAddr, \
                              FLSECC_BLOCK_TWOBIT_TEST);
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}

/**
 * @brief Check the signature of FLS ECC test
 * @param[in]  Index: The index of the ongoing test block
 * @param[in]  RunningMode: The running mode of the ongoing test block
 * @param[in]  TempSignature: The signature of the ongoing test block
 * @table           @service_id:       --  \n
                    @is_reentrant:     true  \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
 * @return Std_ReturnType
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_369
*/
ECCTST_FUNC static Std_ReturnType EccTst_Lld_Fls_SignatureCheck(uint8 Index, FlsEcc_InjectErrorModeType RunningMode, uint32 TempSignature)
{
    Std_ReturnType RetValue = E_NOT_OK;
    uint32 TempInex = (((uint32)Index) << 16U) + \
                      (((uint32)RunningMode) << 24U);
    if (FLSECC_BLOCK_ONEBIT_TEST == RunningMode)
    {
        TempInex += FLSECC_SINGLEBIT_SIGNATURE;
    }
    else
    {
        TempInex += FLSECC_DOUBLEBIT_SIGNATURE;
    }
    /*The FLASH ECC test excute EFM irq*/
    if (TempInex == TempSignature)
    {
        RetValue = E_OK;
    }
    return RetValue;
}

/**
 * @brief Perform FLASH ECC testing, assess the outcomes, and document the results.
 * @param[in] BlockIndex Test block index
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_138
 */
ECCTST_FUNC static void EccTst_Lld_Fls_TestBlock(uint8 BlockIndex)
{
    /*step1: test one bit ECC error*/
    if (TRUE == FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].OneBitTestEn)
    {
        FlsEcc_RunningData.BlockStatus = FLSECC_BLOCK_ONEBIT_TEST;
        FlsEcc_RunningData.RunningSignature = (((uint32)BlockIndex) << 16U) + \
                                              (((uint32)FLSECC_BLOCK_ONEBIT_TEST) << 24U) + \
                                              FLSECC_TEST_INIT_SIGNATURE;
        EccTst_Lld_Fls_OneBitInject(BlockIndex);
        /*Aggregate test results*/
        if (EccTst_Lld_Fls_SignatureCheck(BlockIndex, FLSECC_BLOCK_ONEBIT_TEST, FlsEcc_RunningData.RunningSignature) != E_OK)
        {
            FlsEcc_DetailResult.OneBitTestResult[(BlockIndex % ECCTST_FLASH_ECC_MAX_BLOCK_NUM)] = ECCTST_NOT_OK; /*NOSONAR*/
        }
        if (FlsEcc_DetailResult.OneBitTestResult[BlockIndex] != ECCTST_OK) /*NOSONAR*/
        {
            FlsEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (FlsEcc_ConfigPtr->FlsEccTestFailedNotification != NULL_PTR)
            {
                FlsEcc_ConfigPtr->FlsEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
    }
    /*step2: test two bit ECC error*/
    if (TRUE == FlsEcc_ConfigPtr->TestBlockPtr[BlockIndex].TwoBitTestEn)
    {
        FlsEcc_RunningData.BlockStatus = FLSECC_BLOCK_TWOBIT_TEST;
        FlsEcc_RunningData.RunningSignature = (((uint32)BlockIndex) << 16U) + \
                                              (((uint32)FLSECC_BLOCK_TWOBIT_TEST) << 24U) + \
                                              FLSECC_TEST_INIT_SIGNATURE;
        EccTst_Lld_Fls_TwoBitInject(BlockIndex);
        /*Aggregate test results*/
        if (EccTst_Lld_Fls_SignatureCheck(BlockIndex, FLSECC_BLOCK_TWOBIT_TEST, FlsEcc_RunningData.RunningSignature) != E_OK)
        {
            FlsEcc_DetailResult.TwoBitTestResult[(BlockIndex % ECCTST_FLASH_ECC_MAX_BLOCK_NUM)] = ECCTST_NOT_OK;
        }
        if (FlsEcc_DetailResult.TwoBitTestResult[BlockIndex] != ECCTST_OK)/*NOSONAR*/
        {
            FlsEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
            if (FlsEcc_ConfigPtr->FlsEccTestFailedNotification != NULL_PTR)
            {
                FlsEcc_ConfigPtr->FlsEccTestFailedNotification(ECCTST_MODULE_ID);
            }
        }
    }
}

/**
 * @brief Install busfault exception handler in FLASH ECC test .
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_137
 */
ECCTST_FUNC static void EccTst_Lld_Fls_SetExpHandler(void)
{
    /*Enable usageFault exception*/
    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to configure SCB to clear EMU error flags.
    * couldn't adhere to M3CM Rule-11.4
    */
    SCB->SHCSR |= SCB_SHCSR_BUSFAULTENA_Msk;/*PRQA S 0306*/
    SCB->CCR |= SCB_CCR_BFHFNMIGN_Msk;/*PRQA S 0306*/
    (void)Platform_InstallIrqHandler(BusFault_IRQn, EccTst_Lld_FlsEccExpIsr, &FlsEcc_SysExpPointer);
}
/**
 * @brief Recover busfault exception handler in FLASH ECC test.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_136
 */
ECCTST_FUNC static void EccTst_Lld_Fls_RecoveredExpHandler(void)
{
    Platform_IrqHandlerType TempBusFaultHandler;
    (void)Platform_InstallIrqHandler(BusFault_IRQn, FlsEcc_SysExpPointer, &TempBusFaultHandler);
}


/**
 * @brief Disable the FLASH ECC error inject.
 *
 * @table            @service_id:       --      \n
                     @is_reentrant:     true    \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false   \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_109
*/
ECCTST_FUNC static void EccTst_Lld_Fls_EccErrIrqProcess(void)
{
    uint32 TempData;
    TempData = FlsEcc_RunningData.RunningBlockIndex;
    if (FLSECC_BLOCK_ONEBIT_TEST == FlsEcc_RunningData.BlockStatus)
    {
        FlsEcc_DetailResult.OneBitTestResult[TempData] = \
                EccTst_Lld_Fls_ResultCheck(FlsEcc_ConfigPtr->TestBlockPtr[TempData].OneBitTestAddr);
        EccTst_Lld_Fls_ClearRecoverr();
        /*Now the two bit error must be test ,so there is no nessary to judge two bit test enable flag.*/
    }
    else if (FLSECC_BLOCK_TWOBIT_TEST == FlsEcc_RunningData.BlockStatus)
    {
        FlsEcc_DetailResult.TwoBitTestResult[TempData] = \
                EccTst_Lld_Fls_ResultCheck(FlsEcc_ConfigPtr->TestBlockPtr[TempData].TwoBitTestAddr);
        EccTst_Lld_Fls_ClearUnRecoverr();
        FlsEcc_RunningData.BlockStatus = FLSECC_BLOCK_IDLE;
    }
    else
    {
        EccTst_Lld_Fls_ClearRecoverr();
        EccTst_Lld_Fls_ClearUnRecoverr();
    }
    TempData = FlsEcc_RunningData.RunningSignature;
    FlsEcc_RunningData.RunningSignature = (TempData & 0xFFFF0000U) + \
                                          ((TempData ^ ((uint32)FLSECC_FLSERRIRQ_SIGNATURE)) & 0x0000FFFFU);
    __ASM("ISB");
    __ASM("DSB");
}
#define ECCTST_STOP_SEC_CODE
#include "EccTst_MemMap.h"

#if defined (CPU_YTM32B1HA0)
#define ECCTST_START_SEC_CODE
#else
#define ECCTST_START_SEC_RAMCODE
#endif
#include "EccTst_MemMap.h"
/**
 * @brief The FLASH ECC error interrupt ISR.
 *
 * @table            @service_id:       --      \n
                     @is_reentrant:     true    \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false   \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_151
*/
ECCTST_FUNC static void EccTst_Lld_Fls_EfmEccIsr(void)
{
    boolean RecEccErrorFlag = FALSE;
    boolean UnRecEccErrorFlag = FALSE;
    boolean RecEccIntcEnable = FALSE;
    boolean UnRecEccIntcEnable = FALSE;
    /*The FLASH EccTst module shall double check the IRQ interrupt initialization in the ISR.*/
    RecEccErrorFlag = EccTst_Lld_Fls_GetRecEccErrFlag();
    UnRecEccErrorFlag = EccTst_Lld_Fls_GetUnRecEccErrFlag();
    RecEccIntcEnable = EccTst_Lld_Fls_GetRecIntcEnable();
    UnRecEccIntcEnable = EccTst_Lld_Fls_GetUnRecIntcEnable();
    if (((TRUE == RecEccErrorFlag) && (TRUE == RecEccIntcEnable)) || \
            ((TRUE == UnRecEccErrorFlag) && (TRUE == UnRecEccIntcEnable)))
    {
        EccTst_Lld_Fls_DisInjectErr();
        EccTst_Lld_Fls_EccErrIrqProcess();
    }
    else
    {
        /*Nothing to do*/
    }
}
#if defined (CPU_YTM32B1HA0)
#define ECCTST_STOP_SEC_CODE
#else
#define ECCTST_STOP_SEC_RAMCODE
#endif
#include "EccTst_MemMap.h"
/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define ECCTST_START_SEC_CODE
#include "EccTst_MemMap.h"
/**
 * @brief Service for FLASH ECC test initialization.
 * @details The function shall initialize all FLASH ECC Test relevant registers and global variables.
 * @param[in] FlsEccTstConfigPtr  FLASH ECC test configuration data pointer
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: FLASH ECC test initialization success
 * @retval E_NOT_OK: FLASH ECC test initialization failed
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_112
 */
ECCTST_FUNC Std_ReturnType EccTst_Lld_Fls_Init(FlsEcc_ConfigType const *FlsEccTstConfigPtr)
{
    Std_ReturnType Ret = E_OK;
    if (NULL_PTR == FlsEccTstConfigPtr)
    {
        /*return 'E_NOT_OK' if the FlsEccTstConfigPtr is NULL pointer*/
        Ret = E_NOT_OK;
    }
    else
    {
        EccTst_Lld_Fls_DataInit();
        /*Enable the FLASH ECC error interrupt*/
        (void)Platform_SetIrq(EFM_Ecc_IRQn, TRUE);
        EccTst_Lld_FLs_EccErrIntcInit(TRUE);
        FlsEcc_ConfigPtr = FlsEccTstConfigPtr;
        FlsEcc_RunningData.TestStatus = ECCTST_INIT;
    }
    return Ret;
}


/**
 * @brief Service for FLASH ECC test executing.
 * @details The function executes all FLASH ECC test items based on the configuration of the EccTst module.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: FLASH ECC test execution success
 * @retval E_NOT_OK: FLASH ECC test execution failed
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_113
 */
/* M3CM 4700 VIOLATION:This function governs the execution flow of the FLASH ECC test. It needs to make judgments on the test contents and
 * call the corresponding test functions accordingly. Therefore, the STCAL cannot follow the standard.
*/
ECCTST_FUNC Std_ReturnType EccTst_Lld_Fls_Start(void) /*PRQA S 4700*/
{
    Std_ReturnType Ret = E_OK;
    uint8 TempPriority;
    boolean RecEccErrorFlag = FALSE;
    boolean UnRecEccErrorFlag = FALSE;
    if (ECCTST_UNINIT == FlsEcc_RunningData.TestStatus)
    {
        /*return 'E_NOT_OK' if the EccTst module is not initialized*/
        Ret = E_NOT_OK;
    }
    else
    {
        if (ECCTST_TESTING == FlsEcc_RunningData.TestStatus)
        {
            Ret = E_NOT_OK;
        }
        else
        {
            RecEccErrorFlag = EccTst_Lld_Fls_GetRecEccErrFlag();
            UnRecEccErrorFlag = EccTst_Lld_Fls_GetUnRecEccErrFlag();
            MCAL_QUA_FAULT_INJECTION(ECCTST_LLD_FLS_ECCERR_FLAG);
            /*[YTSWS_EccTst_300]The EccTst module shall support pre-condition check before the commencement of testing.*/
            if ((TRUE == RecEccErrorFlag) || (TRUE == UnRecEccErrorFlag))
            {
                /*Clear the EFM ECC error flag*/
                EccTst_Lld_Fls_ClearRecoverr();
                EccTst_Lld_Fls_ClearUnRecoverr();
                RecEccErrorFlag = EccTst_Lld_Fls_GetRecEccErrFlag();
                UnRecEccErrorFlag = EccTst_Lld_Fls_GetUnRecEccErrFlag();
                MCAL_QUA_FAULT_INJECTION(ECCTST_LLD_FLS_ECCERR_FLAG);
                /*Check the EFM ECC error flag has been cleared or not*/
                if ((TRUE == RecEccErrorFlag) || (TRUE == UnRecEccErrorFlag))
                {
                    /*Set the EFM ECC test result is NOT_OK*/
                    uint8 BlockCount = 0;
                    while (BlockCount < ECCTST_FLASH_ECC_MAX_BLOCK_NUM)
                    {
                        FlsEcc_DetailResult.OneBitTestResult[BlockCount] = ECCTST_NOT_OK;
                        FlsEcc_DetailResult.TwoBitTestResult[BlockCount] = ECCTST_NOT_OK;
                        ++BlockCount;
                    }
                    FlsEcc_RunningData.SumaryResult = ECCTST_NOT_OK;
                    /*Called the test failed notification*/
                    if (FlsEcc_ConfigPtr->FlsEccTestFailedNotification != NULL_PTR)
                    {
                        FlsEcc_ConfigPtr->FlsEccTestFailedNotification(ECCTST_MODULE_ID);
                    }
                    /*changed the test status*/
                    FlsEcc_RunningData.TestStatus = ECCTST_TESTED;
                }
                else
                {
                    FlsEcc_RunningData.TestStatus = ECCTST_TESTING;
                }
            }
            else
            {
                FlsEcc_RunningData.TestStatus = ECCTST_TESTING;
            }
            if (ECCTST_TESTING == FlsEcc_RunningData.TestStatus)
            {
                /*Overloading Bus fault handlers*/
                EccTst_Lld_Fls_SetExpHandler();
                /*Enable EFM ECC error interrupt in NVIC*/
                (void)Platform_SetIrq(EFM_Ecc_IRQn, TRUE);
                EccTst_Lld_FLs_EccErrIntcInit(TRUE);
                /*Overloading EFM ECC irq handlers*/
                (void)Platform_InstallIrqHandler(EFM_Ecc_IRQn, EccTst_Lld_Fls_EfmEccIsr, &FlsEcc_EccIrqPointer);
                (void)Platform_GetIrqPriority(EFM_Ecc_IRQn, &TempPriority);
                (void)Platform_SetIrqPriority(EFM_Ecc_IRQn, 0u);
                /*running the test in order of block index*/
                for (uint8 BlockCount = 0; BlockCount < FlsEcc_ConfigPtr->TestBlockNum; ++BlockCount)
                {
                    FlsEcc_RunningData.RunningBlockIndex = BlockCount;
                    EccTst_Lld_Fls_TestBlock(FlsEcc_RunningData.RunningBlockIndex);
                }
                /*After all test blocks have been executed, if the test results remain as "NOT_TEST,"
                it indicates that the test was successful, and the result should be updated to "TEST_OK."*/
                if (ECCTST_NOT_TEST == FlsEcc_RunningData.SumaryResult)
                {
                    FlsEcc_RunningData.SumaryResult = ECCTST_OK;
                }
                /*Restores the bus fault handler function*/
                EccTst_Lld_Fls_RecoveredExpHandler();
                /*Restores the EFM ECC irq handler function*/
                Platform_IrqHandlerType TempEccIrqPointer;
                (void)Platform_SetIrqPriority(EFM_Ecc_IRQn, TempPriority);
                (void)Platform_InstallIrqHandler(EFM_Ecc_IRQn, FlsEcc_EccIrqPointer, &TempEccIrqPointer);
                /*changed the test status*/
                FlsEcc_RunningData.TestStatus = ECCTST_TESTED;
            }
        }
    }
    return Ret;
}
/**
 * @brief Service for getting FLASH ECC test result.
 * @details The function returns the result of the FLASH ECC test.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return EccTst_TestResultType
 * @retval ECCTST_NOT_TEST: FLASH ECC test is not running
 * @retval ECCTST_OK: FLASH ECC test result is OK
 * @retval ECCTST_NOT_OK: FLASH ECC test result is not OK
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_111
 */
ECCTST_FUNC EccTst_TestResultType EccTst_Lld_Fls_GetTestResult(void)
{
    return FlsEcc_RunningData.SumaryResult;
}

/**
 * @brief Service for getting FLASH ECC test error detail.
 * @details The function returns the error details of the FLASH ECC Test driver.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @param[out] ErrorDetailsPtr   FLASH ECC test error details pointer
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_110
 */
ECCTST_FUNC void EccTst_Lld_Fls_GetErrorDetail(FlsEcc_TestDetailsType *ErrorDetailsPtr)
{
    if (NULL_PTR == ErrorDetailsPtr)
    {
        /*Nothing to do */
    }
    else
    {
        *ErrorDetailsPtr = FlsEcc_DetailResult;
    }
}

/**
 * @brief Service for FLASH ECC Test De-Initialization.
 * @details The function de-initializes all FLASH ECC Test relevant registers and global variables.
 *
 * @table            @service_id:       --      \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 * @trace YTSDS_EccTst_107
 */
ECCTST_FUNC void EccTst_Lld_Fls_DeInit(void)
{
    /*Disable the FLASH ECC error interrupt*/
    EccTst_Lld_FLs_EccErrIntcInit(FALSE);
    FlsEcc_ConfigPtr = NULL_PTR;
    FlsEcc_RunningData.TestStatus = ECCTST_UNINIT;
    EccTst_Lld_Fls_DataInit();
}
#define ECCTST_STOP_SEC_CODE
#include "EccTst_MemMap.h"


#ifdef __cplusplus
}
#endif

/** @} */

