/**
 * @file    EccTst.c
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : EccTst
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "EccTst.h"
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
#include "EccTst_Lld_Fls.h"
#endif
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
#include "EccTst_Lld_Ram.h"
#endif
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
#include "EccTst_Lld_Can.h"
#endif
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_VENDOR_ID_C                      (0xB4)
#define ECCTST_AR_RELEASE_MAJOR_VERSION_C       (4)
#define ECCTST_AR_RELEASE_MINOR_VERSION_C       (4)
#define ECCTST_AR_RELEASE_REVISION_VERSION_C    (0)
#define ECCTST_SW_MAJOR_VERSION_C               (2)
#define ECCTST_SW_MINOR_VERSION_C               (0)
#define ECCTST_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ECCTST header file are of the same vendor */
#if (ECCTST_VENDOR_ID_C != ECCTST_VENDOR_ID)
#error "EccTst.c and EccTst.h have different vendor ids"
#endif

/* Check if source file and ECCTST header file are of the same Autosar version */
#if ((ECCTST_AR_RELEASE_MAJOR_VERSION_C != ECCTST_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_AR_RELEASE_MINOR_VERSION_C != ECCTST_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_AR_RELEASE_REVISION_VERSION_C != ECCTST_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EccTst.c and EccTst.h are different"
#endif

/* Check if source file and ECCTST header file are of the same Software version */
#if ((ECCTST_SW_MAJOR_VERSION_C != ECCTST_SW_MAJOR_VERSION) || \
     (ECCTST_SW_MINOR_VERSION_C != ECCTST_SW_MINOR_VERSION) || \
     (ECCTST_SW_PATCH_VERSION_C != ECCTST_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EccTst.c and EccTst.h are different"
#endif

#if (ECCTST_ENABLE_CAN_ECC_TEST == STD_ON)
/* Check if source file and EccTst_Lld_Can header file are of the same vendor */
#if (ECCTST_VENDOR_ID_C != ECCTST_LLD_CAN_VENDOR_ID)
#error "EccTst.c and EccTst_Lld_Can.h have different vendor ids"
#endif

/* Check if source file and EccTst_Lld_Can header file are of the same Autosar version */
#if ((ECCTST_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EccTst.c and EccTst_Lld_Can.h are different"
#endif

/* Check if source file and EccTst_Lld_Can header file are of the same Software version */
#if ((ECCTST_SW_MAJOR_VERSION_C != ECCTST_LLD_CAN_SW_MAJOR_VERSION) || \
     (ECCTST_SW_MINOR_VERSION_C != ECCTST_LLD_CAN_SW_MINOR_VERSION) || \
     (ECCTST_SW_PATCH_VERSION_C != ECCTST_LLD_CAN_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EccTst.c and EccTst_Lld_Can.h are different"
#endif
#endif /*(ECCTST_ENABLE_CAN_ECC_TEST == STD_ON)*/

#if (ECCTST_ENABLE_FLASH_ECC_TEST == STD_ON)
/* Check if source file and EccTst_Lld_Fls header file are of the same vendor */
#if (ECCTST_VENDOR_ID_C != ECCTST_LLD_FLS_VENDOR_ID)
#error "EccTst.c and EccTst_Lld_Fls.h have different vendor ids"
#endif
/* Check if source file and EccTst_Lld_Fls header file are of the same Autosar version */
#if ((ECCTST_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EccTst.c and EccTst_Lld_Fls.h are different"
#endif
/* Check if source file and EccTst_Lld_Fls header file are of the same Software version */
#if ((ECCTST_SW_MAJOR_VERSION_C != ECCTST_LLD_FLS_SW_MAJOR_VERSION) || \
     (ECCTST_SW_MINOR_VERSION_C != ECCTST_LLD_FLS_SW_MINOR_VERSION) || \
     (ECCTST_SW_PATCH_VERSION_C != ECCTST_LLD_FLS_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EccTst.c and EccTst_Lld_Fls.h are different"
#endif
#endif /*(ECCTST_ENABLE_FLASH_ECC_TEST == STD_ON)*/

#if (ECCTST_ENABLE_RAM_ECC_TEST == STD_ON)
/* Check if source file and EccTst_Lld_Ram header file are of the same vendor */
#if (ECCTST_VENDOR_ID_C != ECCTST_LLD_RAM_VENDOR_ID)
#error "EccTst.c and EccTst_Lld_Ram.h have different vendor ids"
#endif
/* Check if source file and EccTst lld header file are of the same Autosar version */
#if ((ECCTST_AR_RELEASE_MAJOR_VERSION_C != ECCTST_LLD_RAM_AR_RELEASE_MAJOR_VERSION) || \
     (ECCTST_AR_RELEASE_MINOR_VERSION_C != ECCTST_LLD_RAM_AR_RELEASE_MINOR_VERSION) || \
     (ECCTST_AR_RELEASE_REVISION_VERSION_C != ECCTST_LLD_RAM_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of EccTst.c and EccTst_Lld_Ram.h are different"
#endif
/* Check if source file and EccTst lld header file are of the same Software version */
#if ((ECCTST_SW_MAJOR_VERSION_C != ECCTST_LLD_RAM_SW_MAJOR_VERSION) || \
     (ECCTST_SW_MINOR_VERSION_C != ECCTST_LLD_RAM_SW_MINOR_VERSION) || \
     (ECCTST_SW_PATCH_VERSION_C != ECCTST_LLD_RAM_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of EccTst.c and EccTst_Lld_Ram.h are different"
#endif
#endif /*(ECCTST_ENABLE_RAM_ECC_TEST == STD_ON)*/

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#if (ECCTST_PRECOMPILE_SUPPORT == STD_ON)
/**
 * @brief   EccTst driver pre-compiled configuration data structure
 * @private
*/
extern const EccTst_ConfigType EccTst_PreCompileConfig;
#endif /* #if (ECCTST_PRECOMPILE_SUPPORT == STD_ON) */
#define ECCTST_START_SEC_VAR_INIT_UNSPECIFIED
#include "EccTst_MemMap.h"
/**
 * @brief   EccTst driver configuration data structure
 * @private
*/
ECCTST_VAR static const EccTst_ConfigType *EccTst_ConfigPtr = NULL_PTR;

#define ECCTST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "EccTst_MemMap.h"

#define ECCTST_START_SEC_VAR_INIT_8
#include "EccTst_MemMap.h"
/**
 * @brief   The EccTst module running state
 * @private
*/
ECCTST_VAR static EccTst_StateType EccTst_Status = ECCTST_UNINIT;
#define ECCTST_STOP_SEC_VAR_INIT_8
#include "EccTst_MemMap.h"
/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/


/*==================================================================================================
 *                                       SafMgr FUNCTIONS
==================================================================================================*/
#define ECCTST_START_SEC_CODE
#include "EccTst_MemMap.h"
/**
 * @brief  The function is responsible for the initialization of the EccTst module, which encompasses
 *         configuring all registers and global variables pertinent to ECC testing.
 * @param[in] ConfigPtr Pointer to the ECC Test configuration data structure.
 *
 * @table            @service_id:       0x01  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_118
 */
ECCTST_FUNC void EccTst_Init(EccTst_ConfigType const *ConfigPtr)
{
    /* Local variable to hold the return value of the function */
    Std_ReturnType RetVal = (Std_ReturnType)E_OK;
    if (ECCTST_UNINIT != EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID,
                              ECCTST_SID_INIT, ECCTST_E_ALREADY_INITIALIZED);
#endif /* (ECCTST_DEV_ERROR_DETECT == STD_ON) */
        RetVal = (Std_ReturnType)E_NOT_OK;
    }
    if ((Std_ReturnType)E_OK == RetVal)
    {
#if (ECCTST_PRECOMPILE_SUPPORT == STD_ON)
        if (NULL_PTR != ConfigPtr)
        {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
            /* Invalid pointer */
            (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID,
                                  ECCTST_SID_INIT, ECCTST_E_INIT_FAILED);
#endif /* (ECCTST_DEV_ERROR_DETECT == STD_ON) */
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
        else
        {
            EccTst_ConfigPtr = &EccTst_PreCompileConfig;
        }
#else /* (ECCTST_PRECOMPILE_SUPPORT == STD_OFF) */
        if (NULL_PTR == ConfigPtr)
        {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
            /* Invalid pointer */
            (void)Det_ReportError((uint16)ECCTST_MODULE_ID, ECCTST_INSTANCE_ID,
                                  ECCTST_SID_INIT, ECCTST_E_INIT_FAILED);
#endif /* (ECCTST_DEV_ERROR_DETECT == STD_ON) */
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
        else
        {
            EccTst_ConfigPtr = ConfigPtr;
        }
#endif /* (ECCTST_DEV_ERROR_DETECT == STD_ON) */
    }
    if ((Std_ReturnType)E_OK == RetVal)
    {
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        /* Initialize FLASH ECC test */
        RetVal = (Std_ReturnType)OsIf_Trusted_Call_Return1param(EccTst_Lld_Fls_Init, EccTst_ConfigPtr->FlsEccConfig);
#else
        /* Initialize FLASH ECC test */
        RetVal = EccTst_Lld_Fls_Init(EccTst_ConfigPtr->FlsEccConfig);
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        /* Initialize RAM ECC test */
        RetVal |= (Std_ReturnType)OsIf_Trusted_Call_Return1param(EccTst_Lld_Ram_Init, EccTst_ConfigPtr->RamEccConfig);
#else
        /* Initialize RAM ECC test */
        RetVal |= EccTst_Lld_Ram_Init(EccTst_ConfigPtr->RamEccConfig);
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        /* Initialize CAN ECC test */
        RetVal |= (Std_ReturnType)OsIf_Trusted_Call_Return1param(EccTst_Lld_Can_Init, EccTst_ConfigPtr->CanEccConfig);
#else
        /* Initialize CAN ECC test */
        RetVal |= EccTst_Lld_Can_Init(EccTst_ConfigPtr->CanEccConfig);
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON) */
        if (E_OK == RetVal)
        {
            EccTst_Status = ECCTST_INIT;
        }
        else
        {
            EccTst_ConfigPtr = NULL_PTR;
            EccTst_Status = ECCTST_UNINIT;
        }
    }
}

/**
* @brief The API for the EccTst injection test executes all test items according
*        to the configuration of the EccTst module.
*
* @table            @service_id:       0x03  \n
                    @is_reentrant:     false \n
                    @is_synchronous:   true  \n
                    @autosar_api:      false  \n
                    @memory_map:       .mcal_text \n
*
* @return None
* @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_119
*/
ECCTST_FUNC void EccTst_Start(void)
{
    /* Local variable to hold the return value of the function */
    Std_ReturnType RetValue = E_OK;
    if (ECCTST_UNINIT == EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_START, ECCTST_E_UNINIT);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (ECCTST_TESTING == EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_START, ECCTST_E_STATE_FAILURE);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        EccTst_Status = ECCTST_TESTING;
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        /* Execute FLASH ECC test */
        RetValue = (Std_ReturnType)OsIf_Trusted_Call_Return(EccTst_Lld_Fls_Start);
#else
        /* Execute FLASH ECC test */
        RetValue = EccTst_Lld_Fls_Start();
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        /* Execute RAM ECC test */
        RetValue |= (Std_ReturnType)OsIf_Trusted_Call_Return(EccTst_Lld_Ram_Start);
#else
        /* Execute RAM ECC test */
        RetValue |= EccTst_Lld_Ram_Start();
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        /* Execute CAN ECC test */
        RetValue |= (Std_ReturnType)OsIf_Trusted_Call_Return(EccTst_Lld_Can_Start);
#else
        /* Execute CAN ECC test */
        RetValue |= EccTst_Lld_Can_Start();
#endif /* ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON) */
        if (E_OK == RetValue)
        {
            EccTst_Status = ECCTST_TESTED;
        }
        else
        {
            EccTst_Status = ECCTST_INIT;
        }
    }
}

/**
 * @brief  The API for users to read the results of the EccTst injection test. After the test is completed,
 *         users can retrieve the overall test results of the EccTst module through this API.
 *
 * @table            @service_id:       0x04  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return EccTst_TestResultType
 * @retval ECCTST_NOT_TEST:The ECC Test is not completed.
 * @retval ECCTST_OK: The ECC Test test result is OK.
 * @retval ECCTST_NOT_OK: The ECC Test result is not OK.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_116
 */
ECCTST_FUNC EccTst_TestResultType EccTst_GetResult(void)
{
    /* Local variable to hold the return value of the function */
    EccTst_TestResultType RetValue = ECCTST_NOT_TEST;
    if (ECCTST_UNINIT == EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_GET_RESULT, ECCTST_E_UNINIT);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (ECCTST_INIT == EccTst_Status)
    {
        /*Nothing to do*/
    }
    else
    {
        uint8 ErrObj = 0U;
        /* Get FLASH ECC test result  is not tested or not*/
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
        EccTst_TestResultType FlsEccTempRst = EccTst_Lld_Fls_GetTestResult();
        if (ECCTST_OK == FlsEccTempRst)
        {
            RetValue = ECCTST_OK;
        }
        else if (ECCTST_NOT_OK == FlsEccTempRst)
        {
            RetValue = ECCTST_NOT_OK;
        }
        else
        {
            ErrObj |= (uint8)ECCTST_FLASH_ECC;
        }
#endif /* #if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON) */
        /* Get RAM ECC test result  is not tested or not*/
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
        EccTst_TestResultType RamEccTempRst = EccTst_Lld_Ram_GetInjectRes();
        if (ECCTST_NOT_OK == RamEccTempRst)
        {
            RetValue = ECCTST_NOT_OK;
        }
        else if (ECCTST_OK == RamEccTempRst)
        {
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_OFF)
            RetValue = ECCTST_OK;
#else
            /* Do nothing */
#endif
        }
        else
        {
            ErrObj |= (uint8)ECCTST_RAM_ECC;
        }
#endif
        /* Get CAN ECC test result  is not tested or not*/
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
        EccTst_TestResultType CanEccTempTst = EccTst_Lld_Can_GetTestResult();
        if (ECCTST_NOT_OK == CanEccTempTst)
        {
            RetValue = ECCTST_NOT_OK;
        }
        else if (ECCTST_OK == CanEccTempTst)
        {
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_OFF)&&(ECCTST_ENABLE_RAM_ECC_TEST==STD_OFF)
            RetValue = ECCTST_OK;
#else
            /* Do nothing */
#endif
        }
        else
        {
            ErrObj |= (uint8)ECCTST_CAN_ECC;
        }
#endif /* #if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON) */
        /*mux judge the result is not tested or not*/
        if (0U != ErrObj)
        {
            RetValue = ECCTST_NOT_TEST;
        }
    }
    return RetValue;
}

/**
 * @brief API for users to read the detailed results of the EccTst injection test. After the test is completed,
 *    users can retrieve the detailed results of each test object within the EccTst module through this API.
 * @param[out] EccErrorDetailPtr  Pointer to the structure of EccTst result detail data.
 *
 * @table            @service_id:       0x05  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_115
 */
ECCTST_FUNC void EccTst_GetDetailsResult(EccTst_TestDetailsType *EccErrorDetailPtr)
{
    if (ECCTST_UNINIT == EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_GET_ERROR_DETAIL, ECCTST_E_UNINIT);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (NULL_PTR == EccErrorDetailPtr)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_GET_ERROR_DETAIL, ECCTST_E_PARAM_POINTER);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        /* Get FLASH ECC test error detail information */
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
        EccTst_Lld_Fls_GetErrorDetail(&EccErrorDetailPtr->FlsEccResultDetails);
#endif /* #if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
        EccTst_Lld_Ram_InjectGetErr(&EccErrorDetailPtr->RamEccResultDetails);
#endif /* #if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
        EccTst_Lld_Can_GetErrorDetail(&EccErrorDetailPtr->CanEccResultDetails);
#endif /* #if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON) */
    }
}

/**
 * @brief The deinitialization API for the EccTst module. Through this API, users can restore the registers,
 *        global variables used by the EccTst module to their reset state.
 *
 * @table            @service_id:       0x02    \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_114
 */
ECCTST_FUNC void EccTst_DeInit(void)
{
    /* Local variable to hold the return value of the function */
    if (ECCTST_UNINIT == EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_DEINIT, ECCTST_E_UNINIT);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else if (ECCTST_TESTING == EccTst_Status)
    {
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        /* Report to DET */
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_DEINIT, ECCTST_E_STATE_FAILURE);
#endif /* ECCTST_DEV_ERROR_DETECT == STD_ON */
    }
    else
    {
        /* De-init the FLASH ECC test */
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(EccTst_Lld_Fls_DeInit);
#else
        EccTst_Lld_Fls_DeInit();
#endif /* #ifdef ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(EccTst_Lld_Ram_DeInit);
#else
        EccTst_Lld_Ram_DeInit();
#endif /* #ifdef ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON) */
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
#ifdef ECCTST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(EccTst_Lld_Can_DeInit);
#else
        EccTst_Lld_Can_DeInit();
#endif /* #ifdef ECCTST_ENABLE_USER_MODE_SUPPORT */
#endif /* #if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON) */
        EccTst_ConfigPtr = NULL_PTR;
        EccTst_Status = ECCTST_UNINIT;
    }
}

#if (ECCTST_VERSION_INFO_API == STD_ON)
/**
 * @brief            The API for reading the software version number of the EccTst module.
 * @param[out]       VersionInfo Pointer to where to store version information of this module
 *
 * @table            @service_id:       0x06  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return           void
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_117
 */
ECCTST_FUNC void EccTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    if (NULL_PTR == VersionInfo)
    {
        /* Report ECCTST_E_PARAM_POINTER DET if service called with
           NULL_PTR
        */
#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(ECCTST_MODULE_ID, ECCTST_INSTANCE_ID, ECCTST_SID_GET_VERSION_INFO, ECCTST_E_PARAM_POINTER);
#endif /* #if (ECCTST_DEV_ERROR_DETECT == STD_ON) */
    }
    else
    {
        /* Vendor ID information */
        VersionInfo->vendorID = ECCTST_VENDOR_ID;
        /* EccTst module ID information */
        VersionInfo->moduleID = ECCTST_MODULE_ID;
        /* EccTst module Software major version information */
        VersionInfo->sw_major_version = (uint8)ECCTST_SW_MAJOR_VERSION;
        /* EccTst module Software minor version information */
        VersionInfo->sw_minor_version = (uint8)ECCTST_SW_MINOR_VERSION;
        /* EccTst module Software patch version information */
        VersionInfo->sw_patch_version = (uint8)ECCTST_SW_PATCH_VERSION;
    }
}
#endif /* #if ((ECCTST_VERSION_INFO_API == STD_ON) */

#define ECCTST_STOP_SEC_CODE
#include "EccTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

