/**
 * @file    EccTst_Types.h
 * @version
 *
 * @brief   AUTOSAR EccTst module interface
 * @details API implementation for ECCTST driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */

/*
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : EccTst_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef ECCTST_TYPES_H
#define ECCTST_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif
/*
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 1039 Rule-1.2: Language extensions should not be used.
 *
 */
/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "EccTst_Cfg.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_TYPES_VENDOR_ID                     (180)
#define ECCTST_TYPES_AR_REL_MAJOR_VER              (4)
#define ECCTST_TYPES_AR_REL_MINOR_VER              (4)
#define ECCTST_TYPES_AR_REL_REVISION_VER           (0)
#define ECCTST_TYPES_SW_MAJOR_VER                  (2)
#define ECCTST_TYPES_SW_MINOR_VER                  (0)
#define ECCTST_TYPES_SW_PATCH_VER                  (0)
/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ECCTST configuration header file are of the same vendor */
#if (ECCTST_TYPES_VENDOR_ID != ECCTST_VENDOR_ID_CFG)
#error "EccTst_Types.h and EccTst_Cfg.h have different vendor ids"
#endif

/* Check if source file and ECCTST configuration header file are of the same Autosar version */
#if ((ECCTST_TYPES_AR_REL_MAJOR_VER != ECCTST_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (ECCTST_TYPES_AR_REL_MINOR_VER != ECCTST_AR_RELEASE_MINOR_VERSION_CFG) || \
     (ECCTST_TYPES_AR_REL_REVISION_VER != ECCTST_AR_RELEASE_REVISION_VERSION_CFG) \
    )
#error "AutoSar Version Numbers of EccTst_Types.h and EccTst_Cfg.h are different"
#endif

/* Check if source file and ECCTST configuration header file are of the same Software version */
#if ((ECCTST_TYPES_SW_MAJOR_VER != ECCTST_SW_MAJOR_VERSION_CFG) || \
     (ECCTST_TYPES_SW_MINOR_VER != ECCTST_SW_MINOR_VERSION_CFG) || \
     (ECCTST_TYPES_SW_PATCH_VER != ECCTST_SW_PATCH_VERSION_CFG) \
    )
#error "Software Version Numbers of EccTst_Types.h and EccTst_Cfg.h are different"
#endif
/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/**
 * @defgroup EccTst_ID_Info_Macro
 * @brief   The ID information list of EccTst module .
 * @sw_type sw_detail
 * @{
 *
 * @trace YTSDS_EccTst_247
 */
#define ECCTST_INSTANCE_ID                     ((uint8)0U) /**< ECCTST MODULE INSTANCE ID */
#define ECCTST_MODULE_ID                       ((uint16)517U)/**< ECCTST MODULE ID */
#define ECCTST_SID_INIT                        (0x01U)  /**< API Service ID for EccTst_Init */
#define ECCTST_SID_DEINIT                      (0x02U)  /**< API Service ID for EccTst_DeInit */
#define ECCTST_SID_START                       (0x03U)  /**< API Service ID for EccTst_Start */
#define ECCTST_SID_GET_RESULT                  (0x04U)  /**< API Service ID for EccTst_GetResult */
#define ECCTST_SID_GET_ERROR_DETAIL            (0x05U)  /**< API Service ID for EccTst_GetErrorDetail */
#define ECCTST_SID_GET_VERSION_INFO            (0x06U)  /**< API Service ID for EccTst_GetVersionInfo */
/** @} */

#if (ECCTST_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup EccTst_Dev_Error_Macro
 * @brief   EccTst module supported development error detailed list.
 * @sw_type sw_detail
 * @{
 *
 * @trace YTSDS_EccTst_246
 */
#define ECCTST_E_ALREADY_INITIALIZED           ((uint8)0x01)  /**< ECC Test module is already initialized*/
#define ECCTST_E_INIT_FAILED                   ((uint8)0x02)  /**< API EccTst_Init service called with wrong parameter. */
#define ECCTST_E_PARAM_POINTER                 ((uint8)0x03)  /**< Error in param pointer */
#define ECCTST_E_UNINIT                        ((uint8)0x04)  /**< API called without module initialization */
#define ECCTST_E_STATE_FAILURE                 ((uint8)0x05)  /**< Failure within ECC Test execution state */
/** @} */
#endif

/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                TYPEDEF
==================================================================================================*/

/**
 * @brief Type of ECC test result status,which is the return type of API service EccTst_GetTestResult().
 * @details The enumeration value ECCTST_NOT_TEST shall be the default value after a reset.
 * @sw_type sw_arch
 *
*/
typedef enum
{
    ECCTST_NOT_TEST                        = 0x00U, /**< The ECC test is not running,no result to be read */
    ECCTST_OK                                = 0x01U, /**< The ECC test result is OK */
    ECCTST_NOT_OK                            = 0x02U, /**< The ECC test result is not OK */
} EccTst_TestResultType;

/**
 * @brief ECC test failed notification type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_478
*/
typedef void (*EccTst_FailedNotificationType)(uint16 MoudleId);

/**
 * @brief  Type of ECC test execution state.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_688
*/
typedef enum
{
    ECCTST_UNINIT                            = 0x00, /**< The ECC test is uninitialized   */
    ECCTST_INIT                              = 0x01, /**< The ECC test is initialized   */
    ECCTST_TESTING                           = 0x02, /**< The ECC test is running   */
    ECCTST_TESTED                            = 0x03, /**< The ECC test is completed   */
} EccTst_StateType;

#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)

/**
 * @brief type of FLASH ECC test block testing status.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_475
 */
typedef enum
{
    FLSECC_BLOCK_IDLE                         = 0x00, /**< The test block is not tested */
    FLSECC_BLOCK_ONEBIT_TEST                  = 0x01, /**< The test block is running one bit test */
    FLSECC_BLOCK_TWOBIT_TEST                  = 0x02, /**< The test block is running two bit test */
} FlsEcc_InjectErrorModeType;

/**
 * @brief type of FLASH ECC test address
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_92
 */
typedef uint32 FlsEcc_AddressType;

/**
 * @brief FLASH ECC test block configuration type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_93
 */
typedef struct
{
    uint8 BlockIndex;     /*!< FLASH ECC test block index */
    boolean OneBitTestEn;   /*!< One bit test enable */
    FlsEcc_AddressType OneBitTestAddr;    /*!< FLASH ECC one bit test address */
    boolean TwoBitTestEn;   /*!< Two bit test enable */
    FlsEcc_AddressType TwoBitTestAddr;    /*!< FLASH ECC two bit test address */
    uint16 TestDataSize; /*!< FLASH ECC test data size */
} FlsEcc_BlockConfigType;

/**
 * @brief FLASH ECC test configuration type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_95
 */
typedef struct
{
    EccTst_FailedNotificationType FlsEccTestFailedNotification; /*!< FLASH ECC test failed notification for APP*/
    uint8 TestBlockNum;  /*!< FLASH ECC test blocks number */
    FlsEcc_BlockConfigType const *TestBlockPtr;  /*!< FLASH ECC test blocks configuration pointer */
} FlsEcc_ConfigType;

/**
 * @brief Types of the FLASH ECC test execution.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_99
*/
typedef struct
{
    EccTst_StateType TestStatus;    /*!< FLASH ECC test module state */
    FlsEcc_InjectErrorModeType BlockStatus; /*!< FLASH ECC test running block status */
    uint8 RunningBlockIndex;    /*!< FLASH ECC test running block index */
    uint32 RunningSignature;    /*!< FLASH ECC test running signature */
    EccTst_TestResultType SumaryResult;    /*!< FLASH ECC test summary result */
} FlsEcc_RunningStatusType;

/**
 * @brief  Type for detailed error information of the FLS ECC test.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_98
 */
typedef struct
{
    EccTst_TestResultType OneBitTestResult[ECCTST_FLASH_ECC_MAX_BLOCK_NUM];    /*!< FLASH ECC one bit test result */
    EccTst_TestResultType TwoBitTestResult[ECCTST_FLASH_ECC_MAX_BLOCK_NUM];    /*!< FLASH ECC two bit test result */
} FlsEcc_TestDetailsType;
#endif /* (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON) */

#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
/**
 * @brief Type of RAM ECC test mode
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_103
*/
typedef enum
{
    RAMECC_NONE_TEST                         = 0x00, /**< No RAM ECC test is performed */
    RAMECC_DATA_CORRECTABLE_TEST             = 0x01, /**< Data single correctable error test */
    RAMECC_DATA_NONE_CORRECTABLE_TEST        = 0x02, /**< Data none correctable error test */
    RAMECC_CHKBIT_TEST                       = 0x03, /**< Check bit error test */
} RamEcc_InjectErrorModeType;

/**
 * @brief    RAM ECC test channel configuration type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_101
*/
typedef struct
{
    RamEcc_RegionNameType RamErrorInjectionChannelId; /**< RAM ECC test channel ID */
    uint32 RamErrorInjectionAddr; /**< RAM ECC test injection address */
    uint32 RamErrInjectionDataBit0L; /**< RAM ECC test injection data bit 0 */
    uint32 RamErrInjectionDataBit1L; /**< RAM ECC test injection data bit 1 */
#if defined (CPU_YTM32B1HA0)
    uint32 RamErrInjectionDataBit0H; /**< RAM ECC test injection data bit 0 */
    uint32 RamErrInjectionDataBit1H; /**< RAM ECC test injection data bit 1 */
#endif
    uint32 RamErrInjectionChkBit; /**< RAM ECC test injection check bit */
} RamEcc_ChannelConfigType;

/**
 * @brief RAM ECC test channel detail result type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_479
*/
typedef struct
{
    RamEcc_RegionNameType ChannelId; /**< RAM ECC test channel ID */
    EccTst_TestResultType TestResult; /**< The test result of testing channel*/
} RamEcc_ChannelResultType;

/**
 * @brief Type for detailed error information of the RAM ECC test.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_105
 */
typedef struct
{
    /* MR12 RULE 2.2 VIOLATION: The length of this array is only 1 in YTM32B1MC0 due to the the limitation of EMU resources.
     * This line of code has been maintained to ensure compatibility with other MCUs.
     */
    RamEcc_ChannelResultType TestDetailResult[ECCTST_RAM_ECC_MAX_REGION_NUM]; /**< RAM ECC test detail results */ /*PRQA S 1039*/
} RamEcc_TestDetailsType;

/**
 * @brief RAM ECC test configuration type
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_102
*/
typedef struct
{
    uint8 RamErrorInjectionChannelCount;  /**< RAM ECC test channel count */
    EccTst_FailedNotificationType RamEccTestFailedNotification;  /**< RAM ECC error test failed notification */
    RamEcc_ChannelConfigType const *RamErrorInjectionChannel;   /**< RAM ECC test channel configuration */
} RamEcc_ConfigType;

/**
 * @brief Types of the RAM ECC test execution.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_106
*/
typedef struct
{
    uint8 RunningIndex;                 /*!< RAM ECC test running channel index */
    RamEcc_InjectErrorModeType RunningMode;   /*!< RAM ECC test running mode */
    uint32 RunningSignature;            /*!< RAM ECC test running signature */
    EccTst_TestResultType SumaryResult;   /*!< RAM ECC test summary result */
} RamEcc_RunningStatusType;
#endif /* (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON) */

#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
/**
 * @brief Type of CAN ECC test mode
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_87
*/
typedef enum
{
    CANECC_NONE_TEST                        = 0x00,/*!< No test is running   */
    CANECC_DATA_CORRENTABLE_TEST            = 0x01,/*!< Data Correntable test is running   */
    CANECC_DATA_NONE_CORRENTABLE_TEST       = 0x02,/*!< Data None Correntable test is running   */
    CANECC_CHKBIT_TEST                      = 0x03,/*!< Checksum bit test is running   */
} CanEcc_InjectErrorModeType;

/**
 * @brief Types of the CAN ECC test execution.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_86
*/
typedef struct
{
    uint8 RunningChn;                     /*!< The id of running channel */
    uint8 RunningIndex;                   /*!< The index of CanEcc config table */
    CanEcc_InjectErrorModeType TestMode;  /*!< Test mode */
    uint32 RunningSignature;              /*!< Running signature */
    EccTst_TestResultType SumaryResult;   /*!< CAN ECC Summary result */
} CanEcc_RunningStatusType;


/**
 * @brief CAN ECC injection test input type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_82
 */
typedef struct
{
    uint8  InitData;    /*!< The initial data of the injection test */
    uint32 TestAddr;    /*!< The test address of the injection test */
    uint32 InjAddr;      /*!< The injection address of the injection test */
    uint32 InjData;      /*!< The injection data of the injection test */
} CanEcc_InjctionInputType;

/**
 * @brief CAN ECC injection test output type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_83
 */
typedef struct
{
    uint8  DataResult;    /*!< The error type of ECC injection test */
    uint32 ErrorAddr;    /*!< The error address of ECC injection test */
    uint32 ErrorData;    /*!< The error data of ECC injection test */
    uint32 ErrorSyndrome;  /*!< The error mask of ECC injection test */
    uint32 ErrorReg;    /*!< The error status of ECC injection test */
} CanEcc_InjctionOutputType;

/**
 * @brief CAN ECC configuration type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_88
 */
typedef struct
{
    boolean InjectionTestEnable;        /*!< CAN ECC Injection test enable */
    uint8   TestInitialData;            /*!< CAN ECC test initial data */
    uint16  TestAddress;                /*!< CAN ECC test address accessed by MCU*/
    uint16  InjectAddress;              /*!< CAN ECC Inject test address */
    uint8   InjectionCorrentaleData;    /*!< CAN ECC Correntale Injection test data */
    uint8   InjectionNonCorrentaleData; /*!< CAN ECC Non-Correntale Injection test data */
    uint8   InjectionChecksum;          /*!< CAN ECC Injection test checksum */
} CanEcc_CanBlockType;

/**
 * @brief CAN ECC instacne configuration type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_89
 */
typedef struct
{
    uint8 EccTstCanHwId;           /*!< CAN ECC test hardware id */
    uint8 CanMbNumber;             /*!< CAN mail box number for the test instance  */
    boolean CanFDEnable;           /*!< CAN FD enable for the test instance*/
    boolean CanEnhancedFifoEnable;  /*!< CAN enhanced fifo enable for the test instance*/
    boolean CanHrTimeStampValid;    /*!< CAN hrtimestamp ram valid for the test instance*/
    const CanEcc_CanBlockType *EccTstCanBlockMessageBuffersPtr;        /*!< CAN ECC MSB test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockRxIndividualMasksPtr;     /*!< CAN ECC Rx individual mask test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockRxFilterIdhitsPtr;        /*!< CAN ECC Rx filter test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockRxFilterGlobalMasksPtr;   /*!< CAN ECC Rx global mask test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockTxSmbPtr;                 /*!< CAN ECC Tx Smb test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockRxSmbsPtr;                /*!< CAN ECC Rx Smb test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockRxSmbTimeStampsPtr;       /*!< CAN ECC Rx Smb time stamp test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockHrTimeStampPtr;           /*!< CAN ECC Hr Time stamp test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockEnhancedRxFifoPtr;        /*!< CAN ECC Enhance Rx fifo test block pointer */
    const CanEcc_CanBlockType *EccTstCanBlockEnhancedRxFifoFiltersPtr; /*!< CAN ECC Enhance Rx fifo filter test block pointer */
} CanEcc_CanInstType;


/**
 * @brief CAN ECC configuration type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_80
 */
typedef struct
{
    uint8 EccTstCanInstNum;                            /*!< CAN ECC test instance Count */
    EccTst_FailedNotificationType   CanEccTestFailedNotification;  /**< ECC error test failed notification */
    const CanEcc_CanInstType *EccTstCanInstPtr;        /*!< CAN ECC test instance pointer */
} CanEcc_ConfigType;

/**
 * @brief CAN ECC test channel detail error type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_81
 */
typedef struct
{
    uint8 CanInstanceId;    /*!< CAN instance id */
    EccTst_TestResultType CanEccDetailResult;    /*!< CAN ECC test result of this instance */
} CanEcc_ErrorDetailType;

/**
 * @brief CAN ECC test detail type.
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_85
 */
typedef struct
{
    CanEcc_ErrorDetailType CanEccResult[ECCTST_CAN_ECC_CHANNEL_NUM]; /*!< CAN ECC test result of all instances */
} CanEcc_TestDetailsType;

#endif /* (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON) */



/**
  * @brief  EccTst object enum
 * @sw_type sw_detail
 *
 * @trace YTSDS_EccTst_91
*/
typedef enum
{
    ECCTST_NO_TEST           = 0x00U,   /**< NO test object*/
    ECCTST_CAN_ECC           = 0x01U,   /**< CAN ECC */
    ECCTST_RAM_ECC           = 0x02U,   /**< RAM ECC */
    ECCTST_FLASH_ECC         = 0x04U,   /**< FLASH ECC*/
    ECCTST_ALL_ECC           = 0x07U,   /**< ALL OBJECT*/
} EccTst_TestObjectIdType;

/**
 * @brief This type of external data structure which contains the initialization data for the ECC Test.
 * @sw_type sw_arch
 *
 */
typedef struct
{
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
    FlsEcc_ConfigType const *FlsEccConfig;   /**< FLASH ECC test configuration pointer */
#endif
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
    RamEcc_ConfigType const *RamEccConfig;   /**< RAM ECC test configuration pointer */
#endif
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
    const CanEcc_ConfigType *CanEccConfig;   /**< CAN ECC test configuration pointer */
#endif
} EccTst_ConfigType;

/**
 * @brief This type shall specify implementation specific error information monitored in the ECC test module.
 * @sw_type sw_arch
 *
 */
typedef struct
{
#if (ECCTST_ENABLE_FLASH_ECC_TEST==STD_ON)
    FlsEcc_TestDetailsType  FlsEccResultDetails;   /**< FLASH ECC test error detail pointer */
#endif
#if (ECCTST_ENABLE_RAM_ECC_TEST==STD_ON)
    RamEcc_TestDetailsType RamEccResultDetails;    /**< RAM ECC test error detail pointer */
#endif
#if (ECCTST_ENABLE_CAN_ECC_TEST==STD_ON)
    CanEcc_TestDetailsType CanEccResultDetails;    /**< CAN ECC test error detail pointer */
#endif
} EccTst_TestDetailsType;

/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#ifdef __cplusplus
}
#endif

#endif /* End of file EccTst_Types.h */

/** @} */
