/**
 * @file    EccTst_Lld_Fls.h
 * @version
 *
 * @brief   AUTOSAR EccTst_Lld_Fls module interface
 * @details API implementation for ECCTST_LLD_FLS driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Adc
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef ECCTST_LLD_FLS_H
#define ECCTST_LLD_FLS_H

#ifdef __cplusplus
extern "C" {
#endif


/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "EccTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_LLD_FLS_VENDOR_ID                      (0xB4)
#define ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION       (4)
#define ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION       (4)
#define ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION    (0)
#define ECCTST_LLD_FLS_SW_MAJOR_VERSION               (2)
#define ECCTST_LLD_FLS_SW_MINOR_VERSION               (0)
#define ECCTST_LLD_FLS_SW_PATCH_VERSION               (0)


/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ECCTST types header file are of the same vendor */
#if (ECCTST_LLD_FLS_VENDOR_ID != ECCTST_TYPES_VENDOR_ID)
#error "EccTst_Lld_Fls.h and EccTst_Types.h have different vendor ids"
#endif

/* Check if source file and ECCTST types header file are of the same Autosar version */
#if ((ECCTST_LLD_FLS_AR_RELEASE_MAJOR_VERSION != ECCTST_TYPES_AR_REL_MAJOR_VER) || \
     (ECCTST_LLD_FLS_AR_RELEASE_MINOR_VERSION != ECCTST_TYPES_AR_REL_MINOR_VER) || \
     (ECCTST_LLD_FLS_AR_RELEASE_REVISION_VERSION != ECCTST_TYPES_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Fls.h and EccTst_Types.h are different"
#endif

/* Check if source file and ECCTST types header file are of the same software version */
#if ((ECCTST_LLD_FLS_SW_MAJOR_VERSION != ECCTST_TYPES_SW_MAJOR_VER) || \
     (ECCTST_LLD_FLS_SW_MINOR_VERSION != ECCTST_TYPES_SW_MINOR_VER) || \
     (ECCTST_LLD_FLS_SW_PATCH_VERSION != ECCTST_TYPES_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst_Lld_Fls.h and EccTst_Types.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/

/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                             FUNCTIONS
==================================================================================================*/
/**
 * @brief Service for FLASH ECC test initialization.
 * @details The function shall initialize all FLASH ECC Test relevant registers and global variables.
 * @param[in] FlsEccTstConfigPtr  FLASH ECC test configuration data pointer
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: FLASH ECC test initialization success
 * @retval E_NOT_OK: FLASH ECC test initialization failed
 * @sw_type sw_detail
 */
Std_ReturnType EccTst_Lld_Fls_Init(FlsEcc_ConfigType const *FlsEccTstConfigPtr);

/**
 * @brief Service for FLASH ECC test executing.
 * @details The function executes all FLASH ECC test items based on the configuration of the EccTst module.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return Std_ReturnType
 * @retval E_OK: FLASH ECC test execution success
 * @retval E_NOT_OK: FLASH ECC test execution failed
 * @sw_type sw_detail
 */
Std_ReturnType EccTst_Lld_Fls_Start(void);

/**
 * @brief Service for getting FLASH ECC test result.
 * @details The function returns the result of the FLASH ECC test.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return EccTst_TestResultType
 * @retval FLSECC_NOT_TESTED: FLASH ECC test is not running
 * @retval FLSECC_OK: FLASH ECC test result is OK
 * @retval FLSECC_NOT_OK: FLASH ECC test result is not OK
 * @sw_type sw_detail
 */
EccTst_TestResultType EccTst_Lld_Fls_GetTestResult(void);

/**
 * @brief Service for getting FLASH ECC test error detail.
 * @details The function returns the error details of the FLASH ECC Test driver.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     true  \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @param[out] ErrorDetailsPtr   FLASH ECC test error details pointer
 * @return None
 * @sw_type sw_detail
 */
void EccTst_Lld_Fls_GetErrorDetail(FlsEcc_TestDetailsType *ErrorDetailsPtr);

/**
 * @brief Service for FLASH ECC Test De-Initialization.
 * @details The function de-initializes all FLASH ECC Test relevant registers and global variables.
 *
 * @table            @service_id:       --      \n
                     @is_reentrant:     false   \n
                     @is_synchronous:   true    \n
                     @autosar_api:      false    \n
                     @memory_map:       .mcal_text \n
 *
 * @return None
 * @sw_type sw_detail
 */
void EccTst_Lld_Fls_DeInit(void);

#ifdef __cplusplus
}
#endif

#endif /* ECCTST_H */

/** @} */

