/**
 * @file    EccTst_Lld_Can.h
 * @version
 *
 * @brief   AUTOSAR EccTst_Lld_Can module interface
 * @details API implementation for ECCTST_LLD_CAN driver
 *
 * @addtogroup ECCTST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : CAN
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef ECCTST_LLD_CAN_H
#define ECCTST_LLD_CAN_H

#ifdef __cplusplus
extern "C" {
#endif


/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "EccTst_Types.h"


/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ECCTST_LLD_CAN_VENDOR_ID                      (0xB4)
#define ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION       (4)
#define ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION       (4)
#define ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION    (0)
#define ECCTST_LLD_CAN_SW_MAJOR_VERSION               (2)
#define ECCTST_LLD_CAN_SW_MINOR_VERSION               (0)
#define ECCTST_LLD_CAN_SW_PATCH_VERSION               (0)



/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ECCTST types header file are of the same vendor */
#if (ECCTST_LLD_CAN_VENDOR_ID != ECCTST_TYPES_VENDOR_ID)
#error "EccTst_Lld_Can.h and EccTst_Types.h have different vendor ids"
#endif

/* Check if source file and ECCTST types header file are of the same Autosar version */
#if ((ECCTST_LLD_CAN_AR_RELEASE_MAJOR_VERSION != ECCTST_TYPES_AR_REL_MAJOR_VER) || \
     (ECCTST_LLD_CAN_AR_RELEASE_MINOR_VERSION != ECCTST_TYPES_AR_REL_MINOR_VER) || \
     (ECCTST_LLD_CAN_AR_RELEASE_REVISION_VERSION != ECCTST_TYPES_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of EccTst_Lld_Can.h and EccTst_Types.h are different"
#endif

/* Check if source file and ECCTST types header file are of the same software version */
#if ((ECCTST_LLD_CAN_SW_MAJOR_VERSION != ECCTST_TYPES_SW_MAJOR_VER) || \
     (ECCTST_LLD_CAN_SW_MINOR_VERSION != ECCTST_TYPES_SW_MINOR_VER) || \
     (ECCTST_LLD_CAN_SW_PATCH_VERSION != ECCTST_TYPES_SW_PATCH_VER) \
    )
#error "Software Version Numbers of EccTst_Lld_Can.h and EccTst_Types.h are different"
#endif
/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/


/*==================================================================================================
 *                                         EXTERNAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                             FUNCTIONS
==================================================================================================*/


/**
 * @brief      Service for CAN ECC test initialization.
 * @details The function shall initialize all CAN ECC Test relevant registers and global variables.
 * @param[in]  CanEccTstConfigPtr  Pointer to CAN ECC test configuration.
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     Std_ReturnType
 * @retval E_OK: CAN ECC test all instance init success
 * @retval E_NOT_OK: CAN ECC test at least one instance init failed
 * @sw_type sw_detail
 */
Std_ReturnType EccTst_Lld_Can_Init(CanEcc_ConfigType const *CanEccTstConfigPtr);

/**
 * @brief      Main function for CAN ECC test.
 * @details    This function serves as the main entry point for the CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     Std_ReturnType
 * @sw_type sw_detail
 */
Std_ReturnType EccTst_Lld_Can_Start(void);

/**
 * @brief      Get the result of the CAN ECC test.
 * @details    This function retrieves the result status of the CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     EccTst_TestResultType
 * @retval CANECC_NOT_TESTED: CAN ECC test has not been executed
 * @retval CANECC_OK: CAN ECC test result is OK
 * @retval CANECC_NOT_OK: CAN ECC test result is NOT OK
 * @sw_type sw_detail
 */
EccTst_TestResultType EccTst_Lld_Can_GetTestResult(void);

/**
 * @brief      Get detailed error information for the CAN ECC test.
 * @details    This function retrieves detailed error information for the CAN ECC test.
 * @param[out] InjectErrDetail  Pointer to the CAN ECC test error details
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     void
 * @sw_type sw_detail
 *
 */
void EccTst_Lld_Can_GetErrorDetail(CanEcc_TestDetailsType *InjectErrDetail);

/**
 * @brief      De-initialize the CAN ECC test.
 * @details    This function de-initializes all instances of the CAN ECC test.
 * @param[in]  void
 *
 * @table            @service_id:       --  \n
                     @is_reentrant:     false \n
                     @is_synchronous:   true  \n
                     @autosar_api:      false  \n
                     @memory_map:       .mcal_text \n
 *
 * @return     void
 * @sw_type sw_detail
 */
void EccTst_Lld_Can_DeInit(void);


#ifdef __cplusplus
}
#endif

#endif /* ECCTST_H */

/** @} */

