/**
* @file    DmaTst_Lld.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : DmaTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "pSIP_Dma.h"
#include "DmaTst_Lld.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define DMATST_LLD_VENDOR_ID_C                    (180)
#define DMATST_LLD_AR_REL_MAJOR_VER_C              (4)
#define DMATST_LLD_AR_REL_MINOR_VER_C              (4)
#define DMATST_LLD_AR_REL_REVISION_VER_C           (0)
#define DMATST_LLD_SW_MAJOR_VER_C                  (2)
#define DMATST_LLD_SW_MINOR_VER_C                  (0)
#define DMATST_LLD_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and DMATST_LLD header file are of the same vendor */
#if (DMATST_LLD_VENDOR_ID_C != DMATST_LLD_VENDOR_ID)
#error "DmaTst_Lld.c and DmaTst_Lld.h have different vendor ids"
#endif

/* Check if source file and DMATST_LLD header file are of the same Autosar version */
#if (( DMATST_LLD_AR_REL_MAJOR_VER_C != DMATST_LLD_AR_REL_MAJOR_VER) || \
      ( DMATST_LLD_AR_REL_MINOR_VER_C != DMATST_LLD_AR_REL_MINOR_VER) || \
      ( DMATST_LLD_AR_REL_REVISION_VER_C != DMATST_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of DmaTst_Lld.c and DmaTst_Lld.h are different"
#endif

/* Check if source file and DMATST_LLD header file are of the same Software version */
#if (( DMATST_LLD_SW_MAJOR_VER_C != DMATST_LLD_SW_MAJOR_VER) || \
      ( DMATST_LLD_SW_MINOR_VER_C != DMATST_LLD_SW_MINOR_VER) || \
      ( DMATST_LLD_SW_PATCH_VER_C != DMATST_LLD_SW_PATCH_VER))
#error "Software Version Numbers of DmaTst_Lld.c and DmaTst_Lld.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
#define DMATST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "DmaTst_MemMap.h"
DMATST_VAR static DmaTst_CallbackType DmaTst_ChannelErrorCallbackFunction;
#define DMATST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "DmaTst_MemMap.h"

/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
#ifndef DOC
#define DMATST_START_SEC_CONST_UNSPECIFIED
#include "DmaTst_MemMap.h"
/*! @brief Array of Base addresses for DMA instances. */
DMATST_CONST static volatile DMA_Type *const Dma_Base[DMA_INSTANCE_COUNT] = {DMA0};
#define DMATST_STOP_SEC_CONST_UNSPECIFIED
#include "DmaTst_MemMap.h"
#endif
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
#define DMATST_START_SEC_CODE
#include "DmaTst_MemMap.h"

/**
 * @brief            Clears channel source multiplexer configuration
 * @details          Resets DMA channel's hardware request source selection
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base address
 * @param[in]        Channel Channel number to configure
 * @return           void
 *
 * @trace            YTSDS_DmaTst_564
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_MuxClearChannelSource(volatile DMA_Type *Base, DmaTst_ChannelType Channel)
{
    Base->CHMUX[Channel] = 0U;
}

/**
 * @brief            Resets channel transfer descriptor registers
 * @details          Clears all configuration fields in Channel Transfer Submodule (CTS)
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base address
 * @param[in]        ChannelIndex Channel number to reset
 * @return           void
 *
 * @trace            YTSDS_DmaTst_554
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_CTSClearReg(volatile DMA_Type *Base, DmaTst_ChannelType ChannelIndex)
{
    Base->CTS[ChannelIndex].BCNT.BCNT = 0U;
    Base->CTS[ChannelIndex].SADDR = 0U;
    Base->CTS[ChannelIndex].SOFF = 0;
    Base->CTS[ChannelIndex].TCR = 0U;
    Base->CTS[ChannelIndex].STO = 0U;
    Base->CTS[ChannelIndex].DADDR = 0U;
    Base->CTS[ChannelIndex].DOFF = 0U;
    Base->CTS[ChannelIndex].TCNT.LKDIS = 0U;
    Base->CTS[ChannelIndex].DTO_RLD.DTO = 0U;
    Base->CTS[ChannelIndex].CSR = 0U;
    Base->CTS[ChannelIndex].TCNTRV = 0U;
}

/**
 * @brief            Controls DMA channel request enable state
 * @details          Safely enables/disables channel requests with DMA engine pause/resume
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @param[in]        Enable TRUE=enable requests, FALSE=disable
 * @return           void
 *
 * @trace            YTSDS_DmaTst_570
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_SetDmaRequestCmd(DmaTst_ChannelType Channel, boolean Enable)
{
    uint8 DmaInstance;
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    uint8 DmaActiveTimeout = 50U;
    /* Get DMA instance from virtual Channel */
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    DmaRegBase = Dma_Base[DmaInstance];
    /* Hardware clear of ReqEn causes inconsistent data writeback. It is necessary to add a DMA HALT,
     * then wait for several clock cycles before performing read and write operations.*/
    /* Halt DMA before changing the REQEN register*/
    DmaRegBase->CTRL |= 1UL << DMA_CTRL_PAUSE_SHIFT;
    /* Wait DMA active done*/
    while ((DmaRegBase->CTRL & DMA_CTRL_ACTIVE_MASK) != 0U)
    {
        --DmaActiveTimeout;
        if (0U == DmaActiveTimeout)
        {
            break;
        }
    }
    if (TRUE == Enable)
    {
        DmaRegBase->REQEN |= (0x01UL << Channel);
    }
    else
    {
        DmaRegBase->REQEN &= ~(0x01UL << Channel);
    }
    /* Resume DMA*/
    DmaRegBase->CTRL &= ~(1UL << DMA_CTRL_PAUSE_SHIFT);
}

/**
 * @brief            Performs full DMA instance initialization cleanup
 * @details          Resets CTRL register, disables all channel requests, clears CTS registers and hardware triggers
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base pointer
 * @return           void
 *
 * @trace            YTSDS_DmaTst_563
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_InstanceClear(volatile DMA_Type *Base)
{
    /* Clear the bit of CR register */
    uint32 RegValTemp;
    RegValTemp = Base->CTRL;
    RegValTemp &= ~DMA_CTRL_CLM_MASK;
    RegValTemp &= ~DMA_CTRL_CX_MASK;
    RegValTemp &= ~DMA_CTRL_ECX_MASK;
    RegValTemp &= ~DMA_CTRL_DBGDIS_MASK;
    RegValTemp &= ~DMA_CTRL_LOEN_MASK;
    Base->CTRL = RegValTemp;
    Base->CHEIE = 0x0;
    for (uint8 Channel = 0; Channel < DMATST_CHANNEL_HW_COUNT; ++Channel)
    {
        /* Disables the DMA Channel request*/
        DmaTst_Lld_SetDmaRequestCmd(Channel, FALSE);
        /* Clears all registers of DMA->CTS */
        DmaTst_Lld_CTSClearReg(Base, Channel);
        /* Clear all DMA hardware request*/
        DmaTst_Lld_MuxClearChannelSource(Base, Channel);
    }
}

/**
 * @brief            Configures DMA debug operation mode
 * @details          Sets/clears DBGDIS bit in CTRL register to control debug halt behavior
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base pointer
 * @param[in]        Debug TRUE=disable debug halt, FALSE=enable debug halt
 * @return           void
 *
 * @trace            YTSDS_DmaTst_568
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_SetDebugMode(volatile DMA_Type *Base, boolean Debug)
{
    uint32 RegValTemp;
    RegValTemp = Base->CTRL;
    RegValTemp &= ~DMA_CTRL_DBGDIS_MASK;
    RegValTemp |= DMA_CTRL_DBGDIS(Debug);
    Base->CTRL = RegValTemp;
}

/**
 * @brief            Configures continuous hardware trigger source
 * @details          Sets CHMUX to always-on trigger (value 62) for software-driven transfers
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Logical channel identifier
 * @return           void
 *
 * @trace            YTSDS_DmaTst_567
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_SetChannelSourceAlways(DmaTst_ChannelType Channel)
{
    uint8 DmaInstance;
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    /* Get DMA instance from virtual Channel */
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    DmaRegBase = Dma_Base[DmaInstance];
    /* Set request and the trigger to always*/
    DmaRegBase->CHMUX[Channel] = 62U;
}

/**
 * @brief            Enables error interrupts for specified channel
 * @details          Sets corresponding bit in CHEIE register to enable error detection
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @return           void
 *
 * @trace            YTSDS_DmaTst_555
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_EnableChannelInterrupt(DmaTst_ChannelType Channel)
{
    uint8 DmaInstance;
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    /* Get DMA instance from virtual Channel */
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    DmaRegBase = Dma_Base[DmaInstance];
    /*Enable Error interrupt*/
    DmaRegBase->CHEIE |= (uint32)0x01U << Channel;
}

/**
 * @brief            Enables transfer loop offset feature
 * @details          Sets LOEN bit in CTRL register to enable cyclic transfers
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @return           void
 *
 * @trace            YTSDS_DmaTst_557
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_EnableTransferLoopOffset(DmaTst_ChannelType Channel)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    /* Get DMA instance from virtual Channel */
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    uint32 RegValTemp;
    RegValTemp = Base->CTRL;
    RegValTemp &= ~DMA_CTRL_LOEN_MASK;
    RegValTemp |= DMA_CTRL_LOEN(TRUE);
    Base->CTRL = RegValTemp;
}

/**
 * @brief            Clears transfer completion status flags
 * @details          Writes to DONE register to reset channel's completion indicator
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base pointer
 * @param[in]        Channel Target channel number
 * @return           void
 *
 * @trace            YTSDS_DmaTst_551
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_ClearDoneStatusFlag(volatile DMA_Type *Base, DmaTst_ChannelType Channel)
{
    Base->DONE = (uint32)0x01U << Channel;
}

/**
 * @brief            Clears interrupt pending flags
 * @details          Resets CHTLDIF and CHTLHDIF registers for specified channel
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base pointer
 * @param[in]        Channel Target channel number
 * @return           void
 *
 * @trace            YTSDS_DmaTst_553
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_ClearIntStatusFlag(volatile DMA_Type *Base, DmaTst_ChannelType Channel)
{
    Base->CHTLDIF = (uint32)0x01U << Channel;
    Base->CHTLHDIF = (uint32)0x01U << Channel;
}

/**
 * @brief            Retrieves error interrupt pending status
 * @details          Returns current value of CHEIF register
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base pointer
 * @return           uint32 Bitmask of pending error interrupts
 *
 * @trace            YTSDS_DmaTst_559
 */
DMATST_FUNC LOCAL_INLINE uint32 DmaTst_Lld_GetErrorIntStatusFlag(const volatile DMA_Type *Base)
{
    return Base->CHEIF;
}

/**
 * @brief            Clears error interrupt flags
 * @details          Writes to CHEIF register to reset error status
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Base DMA register base pointer
 * @param[in]        Channel Target channel number
 * @return           void
 *
 * @trace            YTSDS_DmaTst_552
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_ClearErrorIntStatusFlag(volatile DMA_Type *Base, DmaTst_ChannelType Channel)
{
    Base->CHEIF = (uint32)0x01U << Channel;
}

/**
 * @brief            Configures basic transfer parameters in CTS registers
 * @details          Sets SADDR/DADDR, transfer attributes, offsets and initial CSR configuration
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @param[in]        SrcAddr 32-bit aligned source address
 * @param[in]        DestAddr 32-bit aligned destination address
 * @return           void
 *
 * @trace            YTSDS_DmaTst_566
 */
DMATST_FUNC LOCAL_INLINE void DmaTst_Lld_SetBasicConfigToReg(DmaTst_ChannelType Channel, uint32 SrcAddr, uint32 DestAddr)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    Base->CTS[Channel].CSR = (uint16)DMA_CTS_CSR_BWC(DMA_SET_CYCLES);
    Base->CTS[Channel].SADDR = SrcAddr;
    Base->CTS[Channel].DADDR = DestAddr;
    Base->CTS[Channel].TCR = (uint16)(DMA_CTS_TCR_SMOD(DMA_SET_MODULO) | DMA_CTS_TCR_SSIZE(DMA_SET_TRANSFER_SIZE));
    Base->CTS[Channel].TCR |= (uint16)(DMA_CTS_TCR_DMOD(DMA_SET_MODULO) | DMA_CTS_TCR_DSIZE(DMA_SET_TRANSFER_SIZE));
    Base->CTS[Channel].SOFF = DMA_SET_OFFSET;
    Base->CTS[Channel].DOFF = DMA_SET_OFFSET;
    Base->CTS[Channel].STO = 0U;
    Base->CHEIE |= (uint32)0x01U << Channel;
    Base->CTS[Channel].CSR |= (uint16)DMA_CTS_CSR_TDINT(FALSE);
    Base->CTS[Channel].CSR |= (uint16)DMA_CTS_CSR_RLDEN(FALSE);
    Base->CTS[Channel].DTO_RLD.DTO = 0U;
    Base->CTS[Channel].TCNTRV |= (uint16)DMA_CTS_TCNT_LKDIS_TCNT(1U);
    Base->CTS[Channel].TCNT.LKDIS |= (uint16)DMA_CTS_TCNT_LKDIS_TCNT(1U);
    Base->CTS[Channel].BCNT.LODIS = (DMA_SET_BCNT & DMA_CTS_BCNT_LODIS_BCNT_MASK);
    Base->CTS[Channel].CSR |= (uint16)DMA_CTS_CSR_DREQ(TRUE);
}

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            Configures DMA channel transfer parameters
 * @details          Sets up source/destination addresses and transfer attributes with alignment constraints
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Virtual channel identifier
 * @param[in]        SrcAddr Physical source address (32-bit aligned)
 * @param[in]        DestAddr Physical destination address (32-bit aligned)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_565
 */
DMATST_FUNC void DmaTst_Lld_SetBasicChannelTransfer(DmaTst_ChannelType Channel, uint32 SrcAddr, uint32 DestAddr)
{
    DmaTst_Lld_SetChannelSourceAlways(Channel);
    DmaTst_Lld_EnableTransferLoopOffset(Channel);
    DmaTst_Lld_SetBasicConfigToReg(Channel, SrcAddr, DestAddr);
}

/**
 * @brief            Registers error interrupt callback handler
 * @details          Stores callback function pointer in global state with exclusive area protection
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Callback Function pointer for error events
 * @return           void
 *
 * @trace            YTSDS_DmaTst_562
 */
DMATST_FUNC void DmaTst_Lld_InstallErrorCallback(DmaTst_CallbackType Callback)
{
    DmaTst_ChannelErrorCallbackFunction = Callback;
}

/**
 * @brief            Initializes DMA channel hardware configuration
 * @details          Performs instance reset, enables debug mode and continuous trigger
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Virtual channel identifier
 * @return           Std_ReturnType Initialization status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_549
 */
DMATST_FUNC Std_ReturnType DmaTst_Lld_ChannelInit(const DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_NOT_OK;
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    uint8 DmaInstance = 0U;
    DmaRegBase = Dma_Base[DmaInstance];
    DmaTst_Lld_InstanceClear(DmaRegBase);
    /* Set configuration of 'Halt on error'*/
    DmaTst_Lld_SetDebugMode(DmaRegBase, TRUE);
    DmaTst_Lld_EnableChannelInterrupt(Channel);
    ReStatus = E_OK;
    return ReStatus;
}

/**
 * @brief            Activates DMA channel transfer
 * @details          Sets START bit in Channel's CSR register through protected access
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @return           void
 *
 * @trace            YTSDS_DmaTst_0
 */
DMATST_FUNC void DmaTst_Lld_TriggerChannelStart(DmaTst_ChannelType Channel)
{
    uint8 DmaInstance;
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    /* Get DMA instance from virtual Channel */
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    DmaRegBase = Dma_Base[DmaInstance];
    DmaRegBase->CTS[Channel].CSR |= DMA_CTS_CSR_START_MASK;
}

/**
 * @brief            Aborts active DMA transfers
 * @details          Sets ECX bit in CTRL register to cancel all ongoing transfers
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_574
 */
DMATST_FUNC void DmaTst_Lld_TransferCancel(void)
{
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    DmaRegBase = Dma_Base[0];
    DmaRegBase->CTRL |= DMA_CTRL_ECX_MASK;
}

#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
/**
 * @brief            Enables scatter-gather descriptor reload
 * @details          Configures RLDEN flag and next CTS address for chained transfers
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @param[in]        NextCtsAddr Physical address of next CTS descriptor (32-byte aligned)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_556
 */
DMATST_FUNC void DmaTst_Lld_EnableRamReload(DmaTst_ChannelType Channel, uint32 NextCtsAddr)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    Base->CTS[Channel].CSR |= (uint16)DMA_CTS_CSR_RLDEN(TRUE);
    Base->CTS[Channel].DTO_RLD.RLD = NextCtsAddr;
}
#endif
/**
 * @brief            Resets DMA hardware to default state
 * @details          Performs software reset via IPC register with protection
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_548
 */
DMATST_FUNC void DmaTst_Lld_ChannelDeInit(void)
{
    /*Asserted a software reset*/
    DMA_REST_VIA_IPC->CTRL |= (uint32)DMA_CTRL_SWRESET_MASK;
    /*De-assert the software reset*/
    DMA_REST_VIA_IPC->CTRL &= (uint32)(~DMA_CTRL_SWRESET_MASK);
}

/**
 * @brief            Configures destination address offset
 * @details          Sets DOFF register for post-transfer address adjustment
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @param[in]        Offset Signed 16-bit offset value (-32768~32767)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_569
 */
DMATST_FUNC void DmaTst_Lld_SetDestOffset(DmaTst_ChannelType Channel, sint16 Offset)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    Base->CTS[Channel].DOFF = (uint16)Offset;
}

/**
 * @brief            Enables DMA channel linking for chained transfers
 * @details          Configures the CHCTRL.LINKEN bit and sets the LINKCH field to specify
 *                   the next channel in the transfer chain. This enables automatic
 *                   triggering of subsequent channels upon current transfer completion.
 * @table            @service_id:       --          \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @param[in]        ActiveChannel     Currently active DMA channel index (0-7)
 * @param[in]        NextChannel       Next DMA channel index to trigger (0-7)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_839
 */
DMATST_FUNC void DmaTst_Lld_EnableChannelLink(uint32 Channel, uint32 NextChannel)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    Base->CHMUX[NextChannel] = 62U;
    Base->CTS[Channel].CSR |= (uint16)DMA_CTS_CSR_TLKEN_MASK;
    Base->CTS[Channel].CSR |= (uint16)DMA_CTS_CSR_TLKCH(NextChannel);
}

/**
 * @brief            Configures source address offset
 * @details          Sets SOFF register for post-transfer address adjustment
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @param[in]        Offset Signed 16-bit offset value (-32768~32767)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_572
 */
DMATST_FUNC void DmaTst_Lld_SetSrcOffset(DmaTst_ChannelType Channel, sint16 Offset)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    Base->CTS[Channel].SOFF = (uint16)Offset;
}

/**
 * @brief            Sets transfer data quantity
 * @details          Configures BCNT register with bytes-per-loop-cycle value
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Target channel number
 * @param[in]        Nbytes Transfer bytes per loop cycle (0-4095)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_573
 */
DMATST_FUNC void DmaTst_Lld_SetTransferLoopbytes(DmaTst_ChannelType Channel, uint32 Nbytes)
{
    uint8 DmaInstance;
    volatile DMA_Type *Base = NULL_PTR;
    DmaInstance = (uint8)FEATURE_DMA_VCH_TO_INSTANCE(Channel);
    Base = Dma_Base[DmaInstance];
    Base->CTS[Channel].BCNT.LODIS = (Nbytes & DMA_CTS_BCNT_LODIS_BCNT_MASK);
}

/**
 * @brief            Retrieves DMA error status register value
 * @details          Returns contents of ERS register containing error type flags
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           uint32 Bitmask of active error status flags (DMATST_ERS_*_TRIGGER)
 *
 * @trace            YTSDS_DmaTst_560
 */
DMATST_FUNC uint32 DmaTst_Lld_GetErrorStatus(void)
{
    volatile DMA_Type const *DmaRegBase = NULL_PTR;
    DmaRegBase = Dma_Base[0];
    return (DmaRegBase->ERS);
}

/**
 * @brief            Resets DMA registers to default state
 * @details          Clears CTRL configuration, disables all channel requests and interrupts
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_550
 */
DMATST_FUNC void DmaTst_Lld_ClearDmaRegister(void)
{
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    uint8 DmaInstance = 0U;
    DmaRegBase = Dma_Base[DmaInstance];
    DmaTst_Lld_InstanceClear(DmaRegBase);
}

/**
 * @brief            Gets error interrupt enable status
 * @details          Returns CHEIE register value indicating enabled error interrupts
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           uint32 Bitmask of enabled error interrupts (bits 0-15)
 *
 * @trace            YTSDS_DmaTst_558
 */
DMATST_FUNC uint32 DmaTst_Lld_GetErrorIntEnable(void)
{
    volatile DMA_Type const *DmaRegBase = NULL_PTR;
    DmaRegBase = Dma_Base[0];
    return (DmaRegBase->CHEIE);
}

/**
 * @brief            Checks transfer completion status
 * @details          Returns CHTLDIF register value indicating finished transfers
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           uint32 Bitmask of completed transfers (bits 0-15)
 *
 * @trace            YTSDS_DmaTst_561
 */
DMATST_FUNC uint32 DmaTst_Lld_GetLoopDoneFlag(void)
{
    volatile DMA_Type const *DmaRegBase = NULL_PTR;
    DmaRegBase = Dma_Base[0];
    return (DmaRegBase->CHTLDIF);
}

/**
 * @brief            Clears the DMA loop completion status flag
 * @details          Writes to the CHLPSTCLR register to reset the loop operation completion indicator.
 *                   This function directly manipulates hardware registers to acknowledge finished DMA loop tests.
 * @table            @service_id:       --          \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_838
 */
DMATST_FUNC void DmaTst_Lld_ClearLoopDoneFlag(void)
{
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    DmaRegBase = Dma_Base[0];
    DmaRegBase->CHTLDIF = DMA_CHTLDIF_CH_MASK;
}

/**
 * @brief            Processes DMA error interrupts
 * @details          Checks error status flags and calls registered callback function
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Instance DMA controller index (0 for single instance)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_576
 */
DMATST_FUNC static void DmaTst_Lld_Error_IrqHandler(uint8 Instance)
{
    volatile DMA_Type *DmaRegBase = NULL_PTR;
    uint32 ErrorStatus;
    DmaRegBase = Dma_Base[Instance];
    ErrorStatus = DmaTst_Lld_GetErrorIntStatusFlag(DmaRegBase); /*Get DMA Channel Error Interrupt Flag*/
    for (uint8 Channel = 0; Channel < DMATST_CHANNEL_HW_COUNT; ++Channel)
    {
        if ((ErrorStatus & (1UL << Channel)) != 0U)
        {
            DmaTst_Lld_SetDmaRequestCmd(Channel, FALSE);
            if (NULL_PTR != DmaTst_ChannelErrorCallbackFunction)
            {
                DmaTst_ChannelErrorCallbackFunction(Channel);
            }
            DmaTst_Lld_ClearDoneStatusFlag(DmaRegBase, Channel);
            DmaTst_Lld_ClearErrorIntStatusFlag(DmaRegBase, Channel);
            DmaTst_Lld_ClearIntStatusFlag(DmaRegBase, Channel);
            break;
        }
    }
}

/**
 * @brief            DMA error interrupts Handle
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_1013
 */
DMATST_FUNC void DmaTst_Error_IRQHandler(void)
{
    DmaTst_Lld_Error_IrqHandler(0U);
}

#define DMATST_STOP_SEC_CODE
#include "DmaTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file DmaTst_Lld.c */

