/**
* @file    DmaTst.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : DmaTst
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations List
 * PRQA S 0306 Rule 11.4: A conversion should not be performed between a pointer to object and an integer type.
 * PRQA S 0326 Rule 11.6: A cast shall not be performed between pointer to void and an arithmetic type.
 * PRQA S 3678 Rule 8.13: A pointer should point to a const-qualified type whenever possible.
 * PRQA S 3387 Rule 13.3: A full expression operator should have no potential side effects other than those caused by the operator itsel.
 * PRQA S 3206 Rule 2.7: A parameter is not used in a function.Redundant Constructs, Type Qualifiers, Security Problems.
 * PRQA S 4700: Metric value out fo range
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "DmaTst.h"
#include "DmaTst_Lld.h"
#include "Platform.h"
#include "pSIP_Dma.h"
#include "SchM_DmaTst.h"
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif // DMATST_ENABLE_USER_MODE_SUPPORT
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* DMATST_DEV_ERROR_DETECT == STD_ON */
#if (DMATST_ENABLE_DEM_REPORT_ERROR_STATUS == STD_ON)
#include "Dem.h"
#endif

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define DMATST_VENDOR_ID_C                            (180)
#define DMATST_AR_RELEASE_MAJOR_VERSION_C              (4)
#define DMATST_AR_RELEASE_MINOR_VERSION_C              (4)
#define DMATST_AR_RELEASE_REVISION_VERSION_C           (0)
#define DMATST_SW_MAJOR_VERSION_C                      (2)
#define DMATST_SW_MINOR_VERSION_C                      (0)
#define DMATST_SW_PATCH_VERSION_C                      (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and DMATST header file are of the same vendor */
#if (DMATST_VENDOR_ID_C != DMATST_VENDOR_ID)
#error "DmaTst.c and DmaTst.h have different vendor ids"
#endif

/* Check if source file and DMATST header file are of the same Autosar version */
#if (( DMATST_AR_RELEASE_MAJOR_VERSION_C != DMATST_AR_RELEASE_MAJOR_VERSION) || \
      ( DMATST_AR_RELEASE_MINOR_VERSION_C != DMATST_AR_RELEASE_MINOR_VERSION) || \
      ( DMATST_AR_RELEASE_REVISION_VERSION_C != DMATST_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of DmaTst.c and DmaTst.h are different"
#endif

/* Check if source file and DMATST header file are of the same Software version */
#if (( DMATST_SW_MAJOR_VERSION_C != DMATST_SW_MAJOR_VERSION) || \
      ( DMATST_SW_MINOR_VERSION_C != DMATST_SW_MINOR_VERSION) || \
      ( DMATST_SW_PATCH_VERSION_C != DMATST_SW_PATCH_VERSION))
#error "Software Version Numbers of DmaTst.c and DmaTst.h are different"
#endif

/* Check if source file and DMATST_LLD header are of the same vendor */
#if (DMATST_VENDOR_ID_C != DMATST_LLD_VENDOR_ID)
#error "DmaTst.c and DmaTst_Lld.h have different vendor ids"
#endif

/* Check if source file and DMATST_LLD header are of the same Autosar version */
#if (( DMATST_AR_RELEASE_MAJOR_VERSION_C != DMATST_LLD_AR_REL_MAJOR_VER) || \
      ( DMATST_AR_RELEASE_MINOR_VERSION_C != DMATST_LLD_AR_REL_MINOR_VER) || \
      ( DMATST_AR_RELEASE_REVISION_VERSION_C != DMATST_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of DmaTst.c and DmaTst_Lld.h are different"
#endif

/* Check if source file and DMATST_LLD header are of the same Software version */
#if (( DMATST_SW_MAJOR_VERSION_C != DMATST_LLD_SW_MAJOR_VER) || \
     ( DMATST_SW_MINOR_VERSION_C != DMATST_LLD_SW_MINOR_VER) || \
     ( DMATST_SW_PATCH_VERSION_C != DMATST_LLD_SW_PATCH_VER))
#error "Software Version Numbers of DmaTst.c and DmaTst_Lld.h are different"
#endif
/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
#define DMATST_START_SEC_VAR_INIT_UNSPECIFIED
#include "DmaTst_MemMap.h"
DMATST_VAR static const DmaTst_ConfigType *DmaTst_ConfigPtr = NULL_PTR;
#define DMATST_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "DmaTst_MemMap.h"
/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
#ifndef DOC
#define DMATST_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "DmaTst_MemMap.h"
/** @brief Global State to save information */
DMATST_VAR static DmaTst_InformationType DmaTst_Information;
DMATST_VAR static DmaTst_CheckType DmaTst_CheckItem[DMATST_TOTAL_SCENARIOS];
DMATST_VAR static DmaTst_ItemResultType *DmaTst_ItemResult[DMATST_TOTAL_SCENARIOS][DMATST_ERROR_INJECT_COUNT];
DMATST_VAR static DmaTst_ErrorCheckFunctionType DmaTst_ErrorCheckFunctionArray[9];
DMATST_VAR static uint8 ChannelPtr;

#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
DMATST_VAR  __attribute__((aligned(32))) static DmaTst_SoftwareCtsType DmaTst_RamReloadScts[2];
#endif /* !(CPU_YTM32B1HA0)*/
#define DMATST_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "DmaTst_MemMap.h"

#define DMATST_START_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "DmaTst_MemMap.h"
/* MR12 RULE 8.13 VIOLATION: A pointer should point to a const-qualified type whenever possible.
 * Tessy comp test need volatile value define, thus, couldn't adhere to M3CM Rule-8.13
 */
DMATST_VAR __attribute__((aligned(32))) static uint32 SourceData[3];
DMATST_VAR __attribute__((aligned(32))) static uint32  DestData[3];
#define DMATST_STOP_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "DmaTst_MemMap.h"

#define DMATST_START_SEC_VAR_INIT_BOOLEAN
#include "DmaTst_MemMap.h"
DMATST_VAR static volatile boolean ErrorCallbackFlag = FALSE;
#define DMATST_STOP_SEC_VAR_INIT_BOOLEAN
#include "DmaTst_MemMap.h"

extern const DmaTst_ConfigType DmaTst_InstanceConfig;
#endif
/*==================================================================================================
 *                                        LOCAL FUNCTIONS
==================================================================================================*/
#define DMATST_START_SEC_CODE
#include "DmaTst_MemMap.h"

/**
 * @brief            Updates Timeout status and error state when Timeout occurs
 * @details          Resets Timeout counter, sets test result to NOT_OK, and marks error state as FAILED
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Timeout Pointer to Timeout counter
 * @param[inout]     ErrorState Pointer to error state variable to update
 * @return           void
 *
 * @trace            YTSDS_DmaTst_588
 */
DMATST_FUNC static void DmaTst_TimeoutStatus_Update(uint32 *Timeout, DmaTst_ItemResultType *const ErrorState)
{
    *Timeout = 0U;
    DmaTst_Information.Result = DMATST_RESULT_NOT_OK;
    *ErrorState = DMATST_ITEM_FAILED;
}

#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
/**
 * @brief            Set the parameters required for RamReload Test
 * @details          Configures the parameters for the scatter-gather functionality in the DMA test.
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_993
 */
DMATST_FUNC static void DmaTst_SetSoftwareCts(void)
{
    /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
     * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
     */
    DmaTst_RamReloadScts[0].SADDR = (uint32)&SourceData[DMATST_RAMRELOAD_IDX]; /*PRQA S 0306*/
    DmaTst_RamReloadScts[0].DADDR = (uint32)&DestData[DMATST_RAMRELOAD_IDX]; /*PRQA S 0306*/
    DmaTst_RamReloadScts[0].ATTR = (uint16)(DMA_CTS_TCR_SMOD(DMA_SET_MODULO) | DMA_CTS_TCR_SSIZE(DMA_SET_TRANSFER_SIZE) | \
                                            DMA_CTS_TCR_DMOD(DMA_SET_MODULO) | DMA_CTS_TCR_DSIZE(DMA_SET_TRANSFER_SIZE));
    DmaTst_RamReloadScts[0].BCNT = DMA_SET_BCNT;
    DmaTst_RamReloadScts[0].CSR = (uint16)(DMA_CTS_CSR_TDINT(FALSE) | DMA_CTS_CSR_RLDEN(TRUE) | \
                                           DMA_CTS_CSR_DREQ(TRUE) | DMA_CTS_CSR_TLKCH(FALSE));
    DmaTst_RamReloadScts[0].SOFF = DMA_SET_OFFSET;
    DmaTst_RamReloadScts[0].DOFF = DMA_SET_OFFSET;
    DmaTst_RamReloadScts[0].STO = (sint16)0;
    DmaTst_RamReloadScts[0].TCNTRV = (uint16)DMA_CTS_TCNT_LKDIS_TCNT(1U);
    DmaTst_RamReloadScts[0].TCNT = (uint16)DMA_CTS_TCNT_LKDIS_TCNT(1U);
    DmaTst_RamReloadScts[0].RamLoadAddr = (sint32)&DmaTst_RamReloadScts[1]; /*PRQA S 0306*/
    DmaTst_RamReloadScts[1].SADDR = (uint32)&SourceData[0]; /*PRQA S 0306*/
    DmaTst_RamReloadScts[1].DADDR = (uint32)&DestData[0]; /*PRQA S 0306*/
    DmaTst_RamReloadScts[1].ATTR = (uint16)(DMA_CTS_TCR_SMOD(DMA_SET_MODULO) | DMA_CTS_TCR_SSIZE(DMA_SET_TRANSFER_SIZE) | \
                                            DMA_CTS_TCR_DMOD(DMA_SET_MODULO) | DMA_CTS_TCR_DSIZE(DMA_SET_TRANSFER_SIZE));
    DmaTst_RamReloadScts[1].BCNT = DMA_SET_BCNT;
    DmaTst_RamReloadScts[1].CSR = (uint16)(DMA_CTS_CSR_TDINT(FALSE) | DMA_CTS_CSR_RLDEN(TRUE) | DMA_CTS_CSR_DREQ(TRUE) | \
                                           DMA_CTS_CSR_TLKCH(DmaTst_ConfigPtr->RuntimeConfigPtr->LinkNextChannel) | \
                                           DMA_CTS_CSR_TLKEN(DmaTst_ConfigPtr->RuntimeConfigPtr->ChannelLinkCheck));
    DmaTst_RamReloadScts[1].SOFF = DMA_SET_OFFSET;
    DmaTst_RamReloadScts[1].DOFF = DMA_SET_OFFSET;
    DmaTst_RamReloadScts[1].STO = (sint16)0;
    DmaTst_RamReloadScts[1].TCNTRV = (uint16)DMA_CTS_TCNT_LKDIS_TCNT(1U);
    DmaTst_RamReloadScts[1].TCNT = (uint16)DMA_CTS_TCNT_LKDIS_TCNT(1U);
    DmaTst_RamReloadScts[1].RamLoadAddr = (sint32)&DmaTst_RamReloadScts[0]; /*PRQA S 0306*/
}
#endif

/**
 * @brief            Handles DMA channel error callback
 * @details          Processes error status register and updates error check states based on error type
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number passed as void pointer
 * @return           void
 *
 * @trace            YTSDS_DmaTst_578
 */
/* M3CM 4700 VIOLATION:This function is responsible for analyzing the error types of DMA-triggered
 * interrupts to determine whether the test has been passed. Therefore, the STCAL cannot follow the standard.
 */
DMATST_FUNC static void DmaTst_ChannelErrorCallback(DmaTst_ChannelType Channel) /*PRQA S 4700*/
{
    boolean ErrorFlag = TRUE;
    /*Before reading the error status register, it is necessary to check whether the DMA error interrupt enable register is enabled
    if an interrupt is triggered without being enabled, it is considered a security issue for that channel, and the test fails
    */
    uint32 ErrorIntEnable = DmaTst_Lld_GetErrorIntEnable();
    if (0U == (ErrorIntEnable & (1UL << Channel)))
    {
        ErrorCallbackFlag = FALSE;
    }
    else
    {
        uint32 ErrorStatus = DmaTst_Lld_GetErrorStatus();
        switch (DmaTst_CheckItem[Channel])
        {
        case TRANSFER_CANCEL_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_ECX_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCancelErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCancelErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case SOURCE_ADDRESS_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_SAE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceAddressErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceAddressErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case SOURCE_OFFSET_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_SOE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceOffsetErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceOffsetErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case DEST_ADDRESS_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_DAE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationAddressErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationAddressErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case DEST_OFFSET_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_DOE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationOffsetErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationOffsetErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case TRANSFER_COUNT_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_BTCE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCountErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCountErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case RAM_RELOAD_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_RLDE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.RamReloadErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.RamReloadErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case SOURCE_BUS_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_SBE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceBusErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceBusErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        case DEST_BUS_ERROR:
            if (0U != (ErrorStatus & DMATST_ERS_DBE_TRIGGER))
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationBusErrorState = DMATST_ITEM_PASS;
            }
            else
            {
                DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationBusErrorState = DMATST_ITEM_FAILED;
                ErrorFlag = FALSE;
            }
            break;
        default:
            ErrorFlag = FALSE;
            break;
        }
    }
    if (TRUE == ErrorFlag)
    {
        ErrorCallbackFlag = TRUE;
    }
    else
    {
        /* Do nothing */
    }
}

/**
 * @brief            Initializes error status tracking structure
 * @details          Sets all error check states to NOT_TESTED for all channels
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_582
 */
DMATST_FUNC static void DmaTst_ErrorStatusInit(void)
{
    for (uint8 Channel = 0; Channel < DMATST_CHANNEL_ERROR_TST_COUNT; ++Channel)
    {
        DmaTst_Information.ErrorDetail[Channel].Channel = Channel;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCancelErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceAddressErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceOffsetErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationAddressErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationOffsetErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCountErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.RamReloadErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceBusErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationBusErrorState = DMATST_ITEM_NOT_TESTED;
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferNormalState = DMATST_ITEM_NOT_TESTED;
    }
}

/**
 * @brief            Triggers DMA channel start via software
 * @details          Sends software trigger command to initiate DMA transfer
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        VirtualChannel Logical channel identifier to trigger
 * @return           void
 *
 * @trace            YTSDS_DmaTst_584
 */
DMATST_FUNC static void DmaTst_SoftTrigger(DmaTst_ChannelType VirtualChannel)
{
    /* Triggers a sw request for the current Channel. */
    SchM_Enter_DmaTst_DMATST_EXCLUSIVE_AREA_00();
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(DmaTst_Lld_TriggerChannelStart, VirtualChannel);
#else
    DmaTst_Lld_TriggerChannelStart(VirtualChannel);
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
    SchM_Exit_DmaTst_DMATST_EXCLUSIVE_AREA_00();
}

/**
 * @brief            Compares source and destination data buffers
 * @details          Performs full buffer comparison to verify DMA transfer accuracy
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        BaseData Pointer to source data buffer
 * @param[in]        CompareData Pointer to destination data buffer
 * @return           boolean Comparison result (TRUE = match, FALSE = mismatch)
 *
 * @trace            YTSDS_DmaTst_542
 */
DMATST_FUNC static boolean DmaTst_DataCompare(const uint32 *BaseData, const uint32 *CompareData)
{
    boolean Result = TRUE;
    for (uint8 Idx = 0U; Idx < DAMTST_TEST_ARRAY_SIZE; Idx++)
    {
        if (BaseData[Idx] != CompareData[Idx])
        {
            Result = FALSE;
            break;
        }
    }
    return Result;
}

/**
 * @brief            Initializes runtime test configuration
 * @details          Sets up DMA channel for runtime test, including transfer parameters and error check states
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_991
 */
DMATST_FUNC static void DmaTst_RunTimeTestInit(void)
{
    SchM_Enter_DmaTst_DMATST_EXCLUSIVE_AREA_01();
    DmaTst_Information.ErrorDetail[DMATST_TOTAL_SCENARIOS - 1U].Channel = DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel;
    if (TRUE == DmaTst_ConfigPtr->RuntimeConfigPtr->NormalTransferCheck)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
         */
        DmaTst_Lld_SetBasicChannelTransfer(DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
        for (uint32 Ptr = 0U; Ptr < DAMTST_TEST_ARRAY_SIZE; Ptr++)
        {
            DestData[Ptr] = 0U;
            SourceData[Ptr] = 0U;
        }
        SourceData[DMATST_NORMAL_IDX] = DMATST_NORMAL_IDX + 1U;
        if (TRUE == DmaTst_ConfigPtr->RuntimeConfigPtr->ChannelLinkCheck)
        {
            DmaTst_Lld_SetBasicChannelTransfer(DmaTst_ConfigPtr->RuntimeConfigPtr->LinkNextChannel, (uint32)&SourceData[DMATST_CHANNEL_LINK_IDX], (uint32)&DestData[DMATST_CHANNEL_LINK_IDX]); /*PRQA S 0306*/
            DmaTst_Lld_EnableChannelLink(DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel, DmaTst_ConfigPtr->RuntimeConfigPtr->LinkNextChannel);
            SourceData[DMATST_CHANNEL_LINK_IDX] = DMATST_CHANNEL_LINK_IDX;
        }
#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
        if (TRUE == DmaTst_ConfigPtr->RuntimeConfigPtr->RamReloadCheck)
        {
            DmaTst_SetSoftwareCts();
            DmaTst_Lld_EnableRamReload(DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel, (uint32)&DmaTst_RamReloadScts[0]); /*PRQA S 0306*/
            SourceData[DMATST_RAMRELOAD_IDX] = DMATST_RAMRELOAD_IDX;
        }
#endif
    }
    SchM_Exit_DmaTst_DMATST_EXCLUSIVE_AREA_01();
}

/**
 * @brief            Performs normal transfer validation test in startup
 * @details          Configures and executes standard DMA transfer, verifies data integrity
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_543
 */
DMATST_FUNC static Std_ReturnType DmaTst_NormalTransferCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    volatile uint32 LoopDoneFlagRegister = 0U;
    volatile uint32 Timeout = 0U;
    DmaTst_CheckItem[Channel] = TRANSFER_NORMAL;
    SourceData[DMATST_NORMAL_IDX] = DMATST_NORMAL_IDX + 1U;
    SourceData[DMATST_CHANNEL_LINK_IDX] = DMATST_CHANNEL_LINK_IDX;
    /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
     * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
     */
    DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
    DmaTst_ChannelType NextChannel = ((DMATST_CHANNEL_HW_COUNT - 1U) == Channel) ? 0U : (Channel + 1U);
    DmaTst_Lld_SetBasicChannelTransfer(NextChannel, (uint32)&SourceData[DMATST_CHANNEL_LINK_IDX], (uint32)&DestData[DMATST_CHANNEL_LINK_IDX]); /*PRQA S 0306*/
    DmaTst_Lld_EnableChannelLink(Channel, NextChannel);
#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
    DmaTst_SetSoftwareCts();
    DmaTst_Lld_EnableRamReload(Channel, (uint32)DmaTst_RamReloadScts); /*PRQA S 0306*/
#endif
    DmaTst_SoftTrigger(Channel); /* Trigger the channel */
    do
    {
        /**MR12 Rule 13.3 VIOLATION: Here, the logical AND operator incorporates a volatile register read with implicit status-clear side effect.
         * In this specific case, the hardware interface requires atomic read-clear during condition evaluation,
         * thus couldn't adhere to M3CM Rule-13.3.
         */
        ++Timeout; /*PRQA S 3387*/
        MCAL_QUA_FAULT_INJECTION(DMATST_LLD_DMA_UNLOCK_TIMEOUT);
        if (Timeout >= DMATST_ERROR_CHECK_TIMEOUT)
        {
            break;
        }
        LoopDoneFlagRegister = DmaTst_Lld_GetLoopDoneFlag();
    }
    while (0U == (LoopDoneFlagRegister & (1UL << Channel)));
    DmaTst_Lld_ClearLoopDoneFlag();
    Timeout = 0U;
#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
    SourceData[DMATST_RAMRELOAD_IDX] = DMATST_RAMRELOAD_IDX;
    DmaTst_SoftTrigger(Channel); /* Trigger the RamReload */
    do
    {
        /**MR12 Rule 13.3 VIOLATION: Here, the logical AND operator incorporates a volatile register read with implicit status-clear side effect.
         * In this specific case, the hardware interface requires atomic read-clear during condition evaluation,
         * thus couldn't adhere to M3CM Rule-13.3.
         */
        ++Timeout; /*PRQA S 3387*/
        if (Timeout >= DMATST_ERROR_CHECK_TIMEOUT)
        {
            break;
        }
        LoopDoneFlagRegister = DmaTst_Lld_GetLoopDoneFlag();
    }
    while (0U == (LoopDoneFlagRegister & (1UL << Channel)));
    DmaTst_Lld_ClearLoopDoneFlag();
    Timeout = 0U;
#endif
    if (TRUE == DmaTst_DataCompare(SourceData, DestData))
    {
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferNormalState = DMATST_ITEM_PASS;
    }
    else
    {
        DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferNormalState = DMATST_ITEM_FAILED;
    }
    for (uint32 Ptr = 0U; Ptr < DAMTST_TEST_ARRAY_SIZE; Ptr++)
    {
        SourceData[Ptr] = 0U;
        DestData[Ptr] = 0U;
    }
    return ReStatus;
}

/**
 * @brief            Performs normal transfer validation test in runtime mode
 * @details          Configures and executes standard DMA transfer, verifies data integrity
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_543
 */
DMATST_FUNC static Std_ReturnType DmaTst_RuntimeTransferCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = TRANSFER_NORMAL;
    volatile uint32 LoopDoneFlagRegister = 0U;
    volatile uint32 Timeout = 0U;
    if (TRUE == DmaTst_ConfigPtr->RuntimeConfigPtr->NormalTransferCheck)
    {
        DmaTst_SoftTrigger(DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel); /* Trigger the channel */
        do
        {
            /**MR12 Rule 13.3 VIOLATION: Here, the logical AND operator incorporates a volatile register read with implicit status-clear side effect.
             * In this specific case, the hardware interface requires atomic read-clear during condition evaluation,
             * thus couldn't adhere to M3CM Rule-13.3.
             */
            ++Timeout; /*PRQA S 3387*/
            MCAL_QUA_FAULT_INJECTION(DMATST_LLD_DMA_UNLOCK_TIMEOUT);
            if (Timeout >= DMATST_ERROR_CHECK_TIMEOUT)
            {
                break;
            }
            LoopDoneFlagRegister = DmaTst_Lld_GetLoopDoneFlag();
        }
        while (0UL == (LoopDoneFlagRegister & (1UL << Channel)));
    }
    if (TRUE == DmaTst_ConfigPtr->RuntimeConfigPtr->RamReloadCheck)
    {
        DmaTst_SoftTrigger(DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel); /* Trigger the RamReload */
        do
        {
            /**MR12 Rule 13.3 VIOLATION: Here, the logical AND operator incorporates a volatile register read with implicit status-clear side effect.
             * In this specific case, the hardware interface requires atomic read-clear during condition evaluation,
             * thus couldn't adhere to M3CM Rule-13.3.
             */
            ++Timeout; /*PRQA S 3387*/
            MCAL_QUA_FAULT_INJECTION(DMATST_LLD_DMA_UNLOCK_TIMEOUT);
            if (Timeout >= DMATST_ERROR_CHECK_TIMEOUT)
            {
                break;
            }
            LoopDoneFlagRegister = DmaTst_Lld_GetLoopDoneFlag();
        }
        while (0UL == (LoopDoneFlagRegister & (1UL << Channel)));
    }
    if (TRUE == DmaTst_DataCompare(SourceData, DestData))
    {
        DmaTst_Information.ErrorDetail[DMATST_TOTAL_SCENARIOS - 1U].ErrorCheckState.TransferNormalState = DMATST_ITEM_PASS;
    }
    else
    {
        DmaTst_Information.ErrorDetail[DMATST_TOTAL_SCENARIOS - 1U].ErrorCheckState.TransferNormalState = DMATST_ITEM_FAILED;
    }
    for (uint32 Ptr = 0U; Ptr < DAMTST_TEST_ARRAY_SIZE; Ptr++)
    {
        DestData[Ptr] = 0U;
    }
    return ReStatus;
}

/**
 * @brief            Validates transfer cancellation error handling
 * @details          Tests DMA's ability to handle transfer cancellation scenarios
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_589
 */
DMATST_FUNC static Std_ReturnType DmaTst_TransferCancelErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = TRANSFER_CANCEL_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
         */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
        DmaTst_Lld_SetTransferLoopbytes(Channel, 10000);
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
         * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
         */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
        DmaTst_Lld_TransferCancel();
    }
    return ReStatus;
}

/**
 * @brief            Validates source address error detection
 * @details          Tests DMA's ability to detect misaligned/invalid source addresses
 * @table            @service_id:       --        \n
 *                   @is_reentrant:     false      \n
 *                   @is_synchronous:   true       \n
 *                   @autosar_api:      false      \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_585
 */
DMATST_FUNC static Std_ReturnType DmaTst_SourceAddressErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = SOURCE_ADDRESS_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
         */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData + 1U, (uint32)DestData); /*PRQA S 0306*/
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
            * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
            */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/**
 * @brief            Validates source offset error detection
 * @details          Tests DMA's handling of invalid source offset configurations
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true \n
 *                   @autosar_api:      false \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_587
 */
DMATST_FUNC static Std_ReturnType DmaTst_SourceOffsetErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = SOURCE_OFFSET_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
         */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
        DmaTst_Lld_SetSrcOffset(Channel, 0x1);
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
         * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
         */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/**
 * @brief            Validates destination address error detection
 * @details          Tests DMA's ability to detect misaligned/invalid destination addresses
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true \n
 *                   @autosar_api:      false \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_579
 */
DMATST_FUNC static Std_ReturnType DmaTst_DestinationAddressErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = DEST_ADDRESS_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
        * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
        */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData + 1U); /*PRQA S 0306*/
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
         * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
         */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/**
 * @brief            Validates destination offset error detection
 * @details          Tests DMA's handling of invalid destination offset configurations
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true \n
 *                   @autosar_api:      false \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_581
 */
DMATST_FUNC static Std_ReturnType DmaTst_DestinationOffsetErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = DEST_OFFSET_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
        * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
        */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
        DmaTst_Lld_SetDestOffset(Channel, 0x1);
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
            * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
            */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/**
 * @brief            Validates transfer count error detection
 * @details          Tests DMA's handling of transfer count overflow/underflow conditions
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true \n
 *                   @autosar_api:      false \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_590
 */
DMATST_FUNC static Std_ReturnType DmaTst_TransferCountErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = TRANSFER_COUNT_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
        * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
        */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
        DmaTst_Lld_SetTransferLoopbytes(Channel, 5U);
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
            * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
            */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/**
 * @brief            Validates RAM reload error detection
 * @details          Tests scatter-gather functionality and RAM reload descriptor handling
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false \n
 *                   @is_synchronous:   true \n
 *                   @autosar_api:      false \n
 *                   @memory_map:       .mcal_text \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_583
 */
DMATST_FUNC static Std_ReturnType DmaTst_RamReloadErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
#ifndef CPU_YTM32B1HA0 /*HA0 doesn't support Scatter gather function*/
    DmaTst_CheckItem[Channel] = RAM_RELOAD_ERROR;
    uint32 CtsAddrWithOffset = (uint32)(&DmaTst_RamReloadScts[0]) + DMATST_CTS_ADDR_OFFSET; /*PRQA S 0306*/
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
         */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, (uint32)DestData); /*PRQA S 0306*/
        DmaTst_Lld_EnableRamReload(Channel,  CtsAddrWithOffset);
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type. In this specific case,
         * needs to convert the instance into a pointer type, thus, couldn't adhere to M3CM Rule-11.6
         */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
#else
    (void)Channel;
#endif /* !(CPU_YTM32B1HA0)*/
    return ReStatus;
}

/**
 * @brief            Validates source bus error detection
 * @details          Tests DMA's handling of inaccessible source memory regions
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false    \n
 *                   @is_synchronous:   true     \n
 *                   @autosar_api:      false    \n
 *                   @memory_map:       .mcal_text   \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_586
 */
DMATST_FUNC static Std_ReturnType DmaTst_SourceBusErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = SOURCE_BUS_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
        * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
        */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, DMATST_INVALIDED_SRC_ADDR, (uint32)DestData); /*PRQA S 0306*/
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
        * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
        */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/**
 * @brief            Validates destination bus error detection
 * @details          Tests DMA's handling of inaccessible destination memory regions
 * @table            @service_id:       --    \n
 *                   @is_reentrant:     false    \n
 *                   @is_synchronous:   true     \n
 *                   @autosar_api:      false    \n
 *                   @memory_map:       .mcal_text   \n
 * @sw_type          sw_detail
 * @param[in]        Channel Channel number to test
 * @return           Std_ReturnType Operation status (E_OK/E_NOT_OK)
 *
 * @trace            YTSDS_DmaTst_580
 */
DMATST_FUNC static Std_ReturnType DmaTst_DestinationBusErrorCheck(DmaTst_ChannelType Channel)
{
    Std_ReturnType ReStatus = E_OK;
    DmaTst_CheckItem[Channel] = DEST_BUS_ERROR;
    ReStatus = DmaTst_Lld_ChannelInit(Channel);
    if (E_OK == ReStatus)
    {
        /* MR12 RULE 11.4 VIOLATION: Here a cast is made between an object pointer and an integer type.
         * In this specific case, needs to obtain the integer value of the base address, thus, couldn't adhere to M3CM Rule-11.4
         */
        DmaTst_Lld_SetBasicChannelTransfer(Channel, (uint32)SourceData, DMATST_INVALIDED_DEST_ADDR); /*PRQA S 0306*/
        /* MR12 RULE 11.6 VIOLATION: Cast between a pointer to void and an integral type.
        * In this specific case, needs to convert the Parameter to instance, thus, couldn't adhere to M3CM Rule-11.6
        */
        DmaTst_Lld_InstallErrorCallback(DmaTst_ChannelErrorCallback);
        DmaTst_SoftTrigger(Channel);
    }
    return ReStatus;
}

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            This function initializes the run-time state structure to provide the DMA channel allocation
 *                   release, protect, and track the state for channels. This function also resets the DMA modules,
 *                   initializes the module to user-defined settings and default settings.
 * @details
 * @table            @service_id:       0x01        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @param[in]        ConfigPtr Pointer to DmaTst module configuration structure.
 * @return           void
 *
 * @trace            YTSDS_DmaTst_612
 */
/* MR12 RULE 2.7 VIOLATION: The pointer passed into the function is determined by the user whether to use it or not.
 * There may be cases where the pointer is not used at all. thus, couldn't adhere to M3CM Rule-2.7
 */
DMATST_FUNC void DmaTst_Init(const DmaTst_ConfigType *ConfigPtr) /*PRQA S 3206*/
{
    if (DMATST_UNINIT != DmaTst_Information.Status)
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_INIT, DMATST_E_INVALID_STATE);
#endif
    }
    else
    {
#if (DMATST_PRECOMPILE_SUPPORT == STD_ON)
        DmaTst_ConfigPtr = &DmaTst_InstanceConfig;
        {
#else
        if (NULL_PTR == ConfigPtr)
        {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
            (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                                  DMATST_SID_INIT, DMATST_E_PARAM_POINTER);
#endif
        }
        else
        {
            DmaTst_ConfigPtr = ConfigPtr;
#endif
            /*initialize error injection function into a array*/
            DmaTst_ErrorCheckFunctionArray[TRANSFER_CANCEL_ERROR] = DmaTst_TransferCancelErrorCheck;
            DmaTst_ErrorCheckFunctionArray[SOURCE_ADDRESS_ERROR] = DmaTst_SourceAddressErrorCheck;
            DmaTst_ErrorCheckFunctionArray[SOURCE_OFFSET_ERROR] = DmaTst_SourceOffsetErrorCheck;
            DmaTst_ErrorCheckFunctionArray[DEST_ADDRESS_ERROR] = DmaTst_DestinationAddressErrorCheck;
            DmaTst_ErrorCheckFunctionArray[DEST_OFFSET_ERROR] = DmaTst_DestinationOffsetErrorCheck;
            DmaTst_ErrorCheckFunctionArray[TRANSFER_COUNT_ERROR] = DmaTst_TransferCountErrorCheck;
            DmaTst_ErrorCheckFunctionArray[RAM_RELOAD_ERROR] = DmaTst_RamReloadErrorCheck;
            DmaTst_ErrorCheckFunctionArray[SOURCE_BUS_ERROR] = DmaTst_SourceBusErrorCheck;
            DmaTst_ErrorCheckFunctionArray[DEST_BUS_ERROR] = DmaTst_DestinationBusErrorCheck;
            for (uint8 Channel = 0; Channel < (DMATST_CHANNEL_ERROR_TST_COUNT + DMATST_ENABLE_RUNTIMETEST); ++Channel)
            {
                DmaTst_ItemResult[Channel][TRANSFER_CANCEL_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCancelErrorState;
                DmaTst_ItemResult[Channel][SOURCE_ADDRESS_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceAddressErrorState;
                DmaTst_ItemResult[Channel][SOURCE_OFFSET_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceOffsetErrorState;
                DmaTst_ItemResult[Channel][DEST_ADDRESS_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationAddressErrorState;
                DmaTst_ItemResult[Channel][DEST_OFFSET_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationOffsetErrorState;
                DmaTst_ItemResult[Channel][TRANSFER_COUNT_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.TransferCountErrorState;
                DmaTst_ItemResult[Channel][RAM_RELOAD_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.RamReloadErrorState;
                DmaTst_ItemResult[Channel][SOURCE_BUS_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.SourceBusErrorState;
                DmaTst_ItemResult[Channel][DEST_BUS_ERROR] = &DmaTst_Information.ErrorDetail[Channel].ErrorCheckState.DestinationBusErrorState;
            }
            DmaTst_Information.Status = DMATST_INIT;
            DmaTst_Information.Result = DMATST_RESULT_NOT_TESTED;
            /*Clear all DMA register */
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call(DmaTst_Lld_ClearDmaRegister);
#else
            DmaTst_Lld_ClearDmaRegister();
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
#if (DMATST_ENABLE_RUNTIMETEST == STD_ON)
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call(DmaTst_RunTimeTestInit);
#else
            DmaTst_RunTimeTestInit();
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
#endif
        }
    }
}

/**
 * @brief            De-initializes the DMA Test module and resets all configurations.
 * @details          Clears configuration pointers, resets module status to uninitialized state, and invalidates test results.
 *                   Performs error detection through the Development Error Tracer (DET) if invoked when already uninitialized.
 * @table            @service_id:       0x02        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @return           void
 *
 * @trace            YTSDS_DmaTst_609
 */
DMATST_FUNC void DmaTst_DeInit(void)
{
    if ((DMATST_TESTING == DmaTst_Information.Status) || (DMATST_UNINIT == DmaTst_Information.Status))
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_DEINIT, DMATST_E_UNINIT);
#endif /* #if (DMATST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        DmaTst_ConfigPtr = NULL_PTR;
        DmaTst_ErrorStatusInit();
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(DmaTst_Lld_ChannelDeInit);
#else
        DmaTst_Lld_ChannelDeInit();
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
        DmaTst_Information.Result = DMATST_RESULT_NOT_TESTED;
        DmaTst_Information.Status = DMATST_UNINIT;
    }
}

/**
 * @brief            Executes comprehensive DMA error detection test sequence.
 * @details          Performs channel initialization verification, error injection tests (source/destination address errors,
 *                   transfer count errors, etc.), and Timeout monitoring. Manages error callback registration and DMA channel
 *                   state transitions according to AUTOSAR specifications.
 * @table            @service_id:       0x03        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @return           Std_ReturnType
 * @retval E_OK      Test sequence completed (aggregate result available via GetResult)
 * @retval E_NOT_OK  Invalid module state or configuration error
 *
 * @trace            YTSDS_DmaTst_613
 */
/* M3CM 4700 VIOLATION:This function serves as the test startup function for this module,
 * responsible for initiating the sub-tests of each channel. It needs to determine whether
 * each sub-test item is enabled and also needs to verify whether the test is successful after completion.
 * Therefore, the STCAL cannot follow the standard.
 */
DMATST_FUNC Std_ReturnType DmaTst_Start(void) /*PRQA S 4700*/
{
    uint32 Timeout = 0U;    /* Timeout counter */
    Std_ReturnType ReStatus = E_NOT_OK;

    if (DMATST_INIT != DmaTst_Information.Status)
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_START, DMATST_E_UNINIT);
#endif /* #if (DMATST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else

    {
        DmaTst_Information.Status = DMATST_TESTING;
        DmaTst_Information.Result = DMATST_RESULT_NOT_TESTED;
        DmaTst_ErrorStatusInit();
        (void)Platform_SetIrq(DMA_Error_IRQn, TRUE);
        Platform_IrqHandlerType DmaTst_OldHandler;
        (void)Platform_InstallIrqHandler(DMA_Error_IRQn, DmaTst_Error_IRQHandler, &DmaTst_OldHandler);
        for (uint8 Idx = 0; Idx < DmaTst_ConfigPtr->ChannelCount; ++Idx)
        {
            ChannelPtr = Idx;
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(DmaTst_NormalTransferCheck, ChannelPtr);
#else
            (void)DmaTst_NormalTransferCheck(ChannelPtr);
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
            if (DMATST_ITEM_FAILED == DmaTst_Information.ErrorDetail[ChannelPtr].ErrorCheckState.TransferNormalState)
            {
                DmaTst_Information.Result = DMATST_RESULT_NOT_OK;
                if (DmaTst_ConfigPtr->FailedNotification != NULL_PTR)
                {
                    DmaTst_ConfigPtr->FailedNotification(DMATST_MODULE_ID);
                }
                continue;
            }
            for (uint8 ErrorInject = 0; ErrorInject < DMATST_ERROR_INJECT_COUNT; ++ErrorInject)
            {
                if (TRUE == (boolean)((DmaTst_ConfigPtr->ErrorTypeConfigPtr[ChannelPtr].Value >> ErrorInject) & 0x01U))
                {
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
                    OsIf_Trusted_Call1param(DmaTst_ErrorCheckFunctionArray[ErrorInject], ChannelPtr);
#else
                    (void)DmaTst_ErrorCheckFunctionArray[ErrorInject](ChannelPtr);
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
                    do
                    {
                        /* Wait for the error callback */
                        ++Timeout;
                        MCAL_QUA_FAULT_INJECTION(DMATST_LLD_DMA_UNLOCK_TIMEOUT);
                        if (Timeout > DMATST_ERROR_CHECK_TIMEOUT)
                        {
                            DmaTst_TimeoutStatus_Update(&Timeout, DmaTst_ItemResult[ChannelPtr][ErrorInject]);
                            if (DmaTst_ConfigPtr->FailedNotification != NULL_PTR)
                            {
                                DmaTst_ConfigPtr->FailedNotification(DMATST_MODULE_ID);
                            }
                            break;
                        }
                    }
                    while (FALSE == ErrorCallbackFlag);
                    Timeout = 0U;
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
                    OsIf_Trusted_Call(DmaTst_Lld_ChannelDeInit);
#else
                    DmaTst_Lld_ChannelDeInit();
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
                    ErrorCallbackFlag = FALSE;
                }
            }
        }
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(DmaTst_Lld_ClearDmaRegister);
#else
        DmaTst_Lld_ClearDmaRegister();
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
        (void)Platform_InstallIrqHandler(DMA_Error_IRQn, DmaTst_OldHandler, &DmaTst_OldHandler);
        (void)Platform_SetIrq(DMA_Error_IRQn, FALSE);
        if (DMATST_RESULT_NOT_OK != DmaTst_Information.Result)
        {
            DmaTst_Information.Result = DMATST_RESULT_OK;
        }
        ReStatus = E_OK;
        DmaTst_Information.Status = DMATST_TESTED;
#if (DMATST_ENABLE_RUNTIMETEST == STD_ON)
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(DmaTst_RunTimeTestInit);
#else
        DmaTst_RunTimeTestInit();
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
#endif
    }
    return ReStatus;
}

/**
 * @brief            Retrieves aggregated test execution status.
 * @details          Provides final test verdict combining all channel test results. Maintains state until next test cycle
 *                   initialization.
 * @table            @service_id:       0x04        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @return           DmaTst_ResultType
 * @retval DMATST_RESULT_NOT_TESTED  Pre-test state
 * @retval DMATST_RESULT_OK          All configured tests passed
 * @retval DMATST_RESULT_NOT_OK      One or more tests failed
 *
 * @trace            YTSDS_DmaTst_610
 */
DMATST_FUNC DmaTst_ResultType DmaTst_GetResult(void)
{
    if (DMATST_UNINIT == DmaTst_Information.Status)
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_GET_RESULT, DMATST_E_UNINIT);
#endif /* #if (DMATST_DEV_ERROR_DETECT == STD_ON)*/
    }
    return DmaTst_Information.Result;
}

/**
 * @brief            Retrieves per-channel detailed error status.
 * @details          Populates test result structure with individual error type statuses (9 error categories per channel)
 *                   including address errors, transfer count errors, and bus faults. Valid only after test execution.
 * @table            @service_id:       0x05        \n
 *                   @is_reentrant:     false       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type          sw_detail
 * @param[out]       DmaTstGetError  Pointer to result array (size DMATST_CHANNEL_ERROR_TST_COUNT)
 * @return           void
 *
 * @trace            YTSDS_DmaTst_611
 */
DMATST_FUNC void DmaTst_GetDetailsResult(DmaTst_GetErrorType *DmaTstGetError)
{
    if (NULL_PTR == DmaTstGetError)
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_GET_DETAILS_RESULT, DMATST_E_PARAM_POINTER);
#endif /* #if (DMATST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else if (DMATST_UNINIT == DmaTst_Information.Status)
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_GET_DETAILS_RESULT, DMATST_E_UNINIT);
#endif /* #if (DMATST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        for (uint8 Channel = 0; Channel < DMATST_CHANNEL_ERROR_TST_COUNT; ++Channel)
        {
            DmaTstGetError[Channel] = DmaTst_Information.ErrorDetail[Channel];
            DmaTstGetError[Channel].Channel = Channel;
        }
#if (DMATST_ENABLE_RUNTIMETEST == STD_ON)
        DmaTstGetError[DMATST_TOTAL_SCENARIOS - 1U].Channel = DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel;
        DmaTstGetError[DMATST_TOTAL_SCENARIOS - 1U] = DmaTst_Information.ErrorDetail[DMATST_TOTAL_SCENARIOS - 1U];
#endif
    }
}

#if (DMATST_ENABLE_RUNTIMETEST == STD_ON)
/**
 * @brief           Main function for the DMA test.
 * @details         In Runtime, this channel is tested by calling the function DmaTst_MainFunction(), focusing only on the normal transfer test.
 * @table           @service_id:       0x07        \n
 *                   @is_reentrant:     true       \n
 *                   @is_synchronous:   true        \n
 *                   @autosar_api:      false       \n
 *                   @memory_map:       .mcal_text  \n
 * @sw_type         sw_detail
 * @return           Std_ReturnType
 * @retval E_OK      Test sequence completed (aggregate result available via GetResult)
 * @retval E_NOT_OK  Invalid module state or configuration error
 *
 * @trace            YTSDS_DmaTst_847
 */
DMATST_FUNC Std_ReturnType DmaTst_MainFunction(void)
{
    Std_ReturnType ReStatus = E_NOT_OK;
    if ((DMATST_INIT != DmaTst_Information.Status) && (DMATST_TESTED != DmaTst_Information.Status))
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(DMATST_MODULE_ID, DMATST_INSTANCE_ID,
                              DMATST_SID_MAINFUNCTION, DMATST_E_UNINIT);
#endif /* #if (DMATST_DEV_ERROR_DETECT == STD_ON)*/
    }
    else
    {
        DmaTst_Information.Status = DMATST_TESTING;
        DmaTst_Information.Result = DMATST_RESULT_NOT_TESTED;
        DmaTst_ErrorStatusInit();
        ChannelPtr = DmaTst_ConfigPtr->RuntimeConfigPtr->TestChannel;
#ifdef DMATST_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(DmaTst_RuntimeTransferCheck, ChannelPtr);
#else
        (void)DmaTst_RuntimeTransferCheck(ChannelPtr);
#endif /* DMATST_ENABLE_USER_MODE_SUPPORT */
        if (DMATST_ITEM_FAILED == DmaTst_Information.ErrorDetail[DMATST_TOTAL_SCENARIOS - 1U].ErrorCheckState.TransferNormalState)
        {
            DmaTst_Information.Result = DMATST_RESULT_NOT_OK;
            if (DmaTst_ConfigPtr->FailedNotification != NULL_PTR)
            {
                DmaTst_ConfigPtr->FailedNotification(DMATST_MODULE_ID);
            }
        }
        if (DMATST_RESULT_NOT_OK != DmaTst_Information.Result)
        {
            DmaTst_Information.Result = DMATST_RESULT_OK;
#if (DMATST_ENABLE_DEM_REPORT_ERROR_STATUS == STD_ON)
            /* Call Dem_SetEventStatus. */
            (void)Dem_SetEventStatus(DmaTst_ConfigPtr->DmaTstErrorHardwareCfg.Id, DEM_EVENT_STATUS_PASSED);
        }
        else
        {
            /* Call Dem_SetEventStatus. */
            (void)Dem_SetEventStatus(DmaTst_ConfigPtr->DmaTstErrorHardwareCfg.Id, DEM_EVENT_STATUS_FAILED);
#endif /* #if (DMATST_ENABLE_DEM_REPORT_ERROR_STATUS == STD_ON) */
        }
        DmaTst_Information.Status = DMATST_TESTED;
    }
    return ReStatus;
}
#endif

#if (DMATST_VERSION_INFO_API == STD_ON)
/**
 * @brief       Service returns the version information of this module.
 * @details     This function retrieves the version information of the DmaTst module and stores it in the
 *              provided 'VersionInfo' structure. If the 'VersionInfo' pointer is NULL, it reports an error
 *              using the Det module (if enabled). If the pointer is valid, it populates the structure with
 *              the module's vendor ID, module ID, and software version information (major, minor, and patch versions).
 *              The version information includes: Module Id, Vendor Id, Vendor specific version number.\n
 *              The behavior of this function depends on the following compile-time switches:\n
 *              DMATST_DEV_ERROR_DETECT: If enabled, errors are reported using the Det module.
 * @param[out]  VersionInfo Pointer to a structure of type 'Std_VersionInfoType' where the
 *                         version information will be stored. This parameter must not be NULL.
 * @table       @service_id:       0x06        \n
 *              @is_reentrant:     true       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @return      void
 * @sw_type     sw_detail
 *
 * @trace YTSDS_DmaTst_1014
 */
DMATST_FUNC void DmaTst_GetVersionInfo(Std_VersionInfoType *VersionInfo)
{
    /* Validate the pointer to the version info */
    if (NULL_PTR == VersionInfo)
    {
#if (DMATST_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)DMATST_MODULE_ID,
            (uint8)DMATST_INSTANCE_ID,
            (uint8)DMATST_SID_GET_VERSION_INFO,
            (uint8)DMATST_E_PARAM_POINTER
        );
#endif
    }
    else
    {
        VersionInfo->vendorID = (uint16)DMATST_VENDOR_ID;
        VersionInfo->moduleID = (uint16)DMATST_MODULE_ID;
        VersionInfo->sw_major_version = (uint8)DMATST_SW_MAJOR_VERSION;
        VersionInfo->sw_minor_version = (uint8)DMATST_SW_MINOR_VERSION;
        VersionInfo->sw_patch_version = (uint8)DMATST_SW_PATCH_VERSION;
    }
}
#endif


#define DMATST_STOP_SEC_CODE
#include "DmaTst_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file DmaTst.c */

