/*
* @file    DmaTst_Lld.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : DmaTst_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef DMATST_LLD_H
#define DMATST_LLD_H

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "DmaTst_Types.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define DMATST_LLD_VENDOR_ID            (180)
#define DMATST_LLD_AR_REL_MAJOR_VER     (4)
#define DMATST_LLD_AR_REL_MINOR_VER     (4)
#define DMATST_LLD_AR_REL_REVISION_VER  (0)
#define DMATST_LLD_SW_MAJOR_VER         (2)
#define DMATST_LLD_SW_MINOR_VER         (0)
#define DMATST_LLD_SW_PATCH_VER         (0)
/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and DMATST_TYPES header file are of the same vendor */
#if (DMATST_LLD_VENDOR_ID != DMATST_TYPES_VENDOR_ID)
#error "DmaTst_Lld.h and DmaTst_Types.h have different vendor ids"
#endif

/* Check if source file and DMATST_TYPES header file are of the same Autosar version */
#if ((DMATST_LLD_AR_REL_MAJOR_VER != DMATST_TYPES_AR_REL_MAJOR_VER) || \
     (DMATST_LLD_AR_REL_MINOR_VER != DMATST_TYPES_AR_REL_MINOR_VER) || \
     (DMATST_LLD_AR_REL_REVISION_VER != DMATST_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of DmaTst_Lld.h and DmaTst_Types.h are different"
#endif

/* Check if source file and DMATST_TYPES header file are of the same software version */
#if ((DMATST_LLD_SW_MAJOR_VER != DMATST_TYPES_SW_MAJOR_VER) || \
     (DMATST_LLD_SW_MINOR_VER != DMATST_TYPES_SW_MINOR_VER) || \
     (DMATST_LLD_SW_PATCH_VER != DMATST_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of DmaTst_Lld.h and DmaTst_Types.h are different"
#endif
/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
#define DMA_SET_BCNT                    (uint16)4U
#define DMA_SET_OFFSET                  (sint16)4
#define DMA_SET_MODULO                  (uint16)2U
#define DMA_SET_TRANSFER_SIZE           (uint16)2U
#define DMA_SET_CYCLES                  (uint16)2U
/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
/**
 * @brief               Configures DMA channel transfer parameters
 * @details             Sets up source/destination addresses and transfer attributes with alignment constraints
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Virtual channel identifier
 * @param[in]           SrcAddr Physical source address (32-bit aligned)
 * @param[in]           DestAddr Physical destination address (32-bit aligned)
 * @return              void
 */
void DmaTst_Lld_SetBasicChannelTransfer(DmaTst_ChannelType Channel, uint32 SrcAddr, uint32 DestAddr);

/**
 * @brief               Registers error interrupt callback handler
 * @details             Stores callback function pointer in global state with exclusive area protection
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Callback Function pointer for error events
 * @return              void
 */
void DmaTst_Lld_InstallErrorCallback(DmaTst_CallbackType Callback);

/**
 * @brief               Initializes DMA channel hardware configuration
 * @details             Performs instance reset, enables debug mode and continuous trigger
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Virtual channel identifier
 * @return              Std_ReturnType Initialization status (E_OK/E_NOT_OK)
 */
Std_ReturnType DmaTst_Lld_ChannelInit(const DmaTst_ChannelType Channel);

/**
 * @brief               Activates DMA channel transfer
 * @details             Sets START bit in Channel's CSR register through protected access
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Target channel number
 * @return              void
 */
void DmaTst_Lld_TriggerChannelStart(DmaTst_ChannelType Channel);

/**
 * @brief               Aborts active DMA transfers
 * @details             Sets ECX bit in CTRL register to cancel all ongoing transfers
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              void
 */
void DmaTst_Lld_TransferCancel(void);

/**
 * @brief               Enables scatter-gather descriptor reload
 * @details             Configures RLDEN flag and next CTS address for chained transfers
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Target channel number
 * @param[in]           NextCtsAddr Physical address of next CTS descriptor (32-byte aligned)
 * @return              void
 */
void DmaTst_Lld_EnableRamReload(DmaTst_ChannelType Channel, uint32 NextCtsAddr);

/**
 * @brief               Resets DMA hardware to default state
 * @details             Performs software reset via IPC register with protection
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              void
 */
void DmaTst_Lld_ChannelDeInit(void);

/**
 * @brief               Configures destination address offset
 * @details             Sets DOFF register for post-transfer address adjustment
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Target channel number
 * @param[in]           Offset Signed 16-bit offset value (-32768~32767)
 * @return              void
 */
void DmaTst_Lld_SetDestOffset(DmaTst_ChannelType Channel, sint16 Offset);

/**
 * @brief               Configures source address offset
 * @details             Sets SOFF register for post-transfer address adjustment
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Target channel number
 * @param[in]           Offset Signed 16-bit offset value (-32768~32767)
 * @return              void
 */
void DmaTst_Lld_SetSrcOffset(DmaTst_ChannelType Channel, sint16 Offset);

/**
 * @brief               Sets transfer data quantity
 * @details             Configures BCNT register with bytes-per-loop-cycle value
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Channel Target channel number
 * @param[in]           Nbytes Transfer bytes per loop cycle (0-4095)
 * @return              void
 */
void DmaTst_Lld_SetTransferLoopbytes(DmaTst_ChannelType Channel, uint32 Nbytes);

/**
 * @brief               Retrieves DMA error status register value
 * @details             Returns contents of ERS register containing error type flags
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              uint32 Bitmask of active error status flags (DMATST_ERS_*_TRIGGER)
 */
uint32 DmaTst_Lld_GetErrorStatus(void);

/**
 * @brief               Resets DMA registers to default state
 * @details             Clears CTRL configuration, disables all channel requests and interrupts
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              void
 */
void DmaTst_Lld_ClearDmaRegister(void);

/**
 * @brief               Enables DMA channel linking for chained transfers
 * @details             Configures the CHCTRL.LINKEN bit and sets the LINKCH field to specify
 *                      the next channel in the transfer chain. This enables automatic
 *                      triggering of subsequent channels upon current transfer completion.
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           ActiveChannel Currently active DMA channel index (0-7)
 * @param[in]           NextChannel Next DMA channel index to trigger (0-7)
 * @return              void
 */
void DmaTst_Lld_EnableChannelLink(uint32 Channel, uint32 NextChannel);

/**
 * @brief               Clears the DMA loop completion status flag
 * @details             Writes to the CHLPSTCLR register to reset the loop operation completion indicator.
 *                      This function directly manipulates hardware registers to acknowledge finished DMA loop tests.
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              void
 */
void DmaTst_Lld_ClearLoopDoneFlag(void);

/**
 * @brief               Gets error interrupt enable status
 * @details             Returns CHEIE register value indicating enabled error interrupts
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              uint32 Bitmask of enabled error interrupts (bits 0-15)
 */
uint32 DmaTst_Lld_GetErrorIntEnable(void);

/**
 * @brief               Checks transfer completion status
 * @details             Returns CHTLDIF register value indicating finished transfers
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @return              uint32 Bitmask of completed transfers (bits 0-15)
 */
uint32 DmaTst_Lld_GetLoopDoneFlag(void);

/**
 * @brief               Processes DMA error interrupts
 * @details             Checks error status flags and calls registered callback function
 * @table               @service_id: --                      \n
 *                      @is_reentrant: false                \n
 *                      @is_synchronous: true               \n
 *                      @autosar_api: false                 \n
 *                      @memory_map: .mcal_text             \n
 * @sw_type             sw_detail
 * @param[in]           Instance DMA controller index (0 for single instance)
 * @return              void
 */
void DmaTst_Error_IRQHandler(void);

#ifdef __cplusplus
}
#endif

#endif /* End of file DmaTst_Lld.h */
