/**
* @file    CrcTst_Lld_Crc.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : CrcTst_Lld_Crc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 2986 Rule 2.2: There shall be no dead code. This operation is redundant. The value of the result is always that of the right-hand operand.
 * PRQA S 2985 Rule 2.2: There shall be no dead code. This operation is redundant. The value of the result is always that of the left-hand operand.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "CrcTst_Lld_Crc.h"
#include "Platform.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRCTST_LLD_CRC_VENDOR_ID_C                      (180)
#define CRCTST_LLD_CRC_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CRCTST_LLD_CRC_AR_RELEASE_MINOR_VERSION_C       (4)
#define CRCTST_LLD_CRC_AR_RELEASE_REVISION_VERSION_C    (0)
#define CRCTST_LLD_CRC_SW_MAJOR_VERSION_C               (2)
#define CRCTST_LLD_CRC_SW_MINOR_VERSION_C               (0)
#define CRCTST_LLD_CRC_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CrcTst_Lld_Crc header file are of the same vendor */
#if (CRCTST_LLD_CRC_VENDOR_ID_C != CRCTST_LLD_CRC_VENDOR_ID)
#error "CrcTst_Lld_Crc.c and CrcTst_Lld_Crc.h have different vendor ids"
#endif

/* Check if source file and CrcTst_Lld_Crc header file are of the same Autosar version */
#if (( CRCTST_LLD_CRC_AR_RELEASE_MAJOR_VERSION_C != CRCTST_LLD_CRC_AR_RELEASE_MAJOR_VERSION) || \
      ( CRCTST_LLD_CRC_AR_RELEASE_MINOR_VERSION_C != CRCTST_LLD_CRC_AR_RELEASE_MINOR_VERSION) || \
      ( CRCTST_LLD_CRC_AR_RELEASE_REVISION_VERSION_C != CRCTST_LLD_CRC_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CrcTst_Lld_Crc.c and CrcTst_Lld_Crc.h are different"
#endif

/* Check if source file and CrcTst_Lld_Crc header file are of the same Software version */
#if (( CRCTST_LLD_CRC_SW_MAJOR_VERSION_C != CRCTST_LLD_CRC_SW_MAJOR_VERSION) || \
      ( CRCTST_LLD_CRC_SW_MINOR_VERSION_C != CRCTST_LLD_CRC_SW_MINOR_VERSION) || \
      ( CRCTST_LLD_CRC_SW_PATCH_VERSION_C != CRCTST_LLD_CRC_SW_PATCH_VERSION))
#error "Software Version Numbers of CrcTst_Lld_Crc.c and CrcTst_Lld_Crc.h are different"
#endif

/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
#define CRC_START_SEC_CODE
#include "Crc_MemMap.h"

/**
 * @brief       Get the CRC control command.
 * @details     Generates the CRC control command based on the CRC configuration and polynomial mode setting.
 * @param[in]   CrcConfig   Pointer to the CRC configuration structure.
 * @param[in]   PolyModSet  Polynomial mode setting.
 * @return      uint32      Generated CRC control command.
 * @private
 */
CRC_FUNC uint32 CrcTst_Lld_CrcGetCtlCmd(const CrcTst_PolConfigType *CrcConfig, uint8 PolyModSet)
{
    uint32 TempCtrlCmd = 0;
    /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the
     * left-hand/right-hand operand. This line of code has been maintained for the sake of code clarity and readability.
     */
    TempCtrlCmd = CRC_CTRL_INV_OUT(0) |  /*PRQA S 2986*/
                  CRC_CTRL_SWAP_OUT(CrcConfig->OutputReflect) |
                  CRC_CTRL_SWAP_IN(CrcConfig->InputReflect) |
                  CRC_CTRL_MODE(PolyModSet);   /*PRQA S 2985*/
    return TempCtrlCmd;
}

/**
 * @brief       Write the CRC control command.
 * @param[in]   crc - Pointer to the CRC register structure.
 * @param[in]   CtrlCmd - The CRC control command.
 * @private
 */
CRC_FUNC void CrcTst_Lld_CrcWriteCtlCmd(volatile CRC_Type *BaseAddr, uint32 CtrlCmd)
{
    BaseAddr->CTRL = CtrlCmd;
}

/**
 * @brief       Write the CRC Initial value.
 * @param[in]   crc - Pointer to the CRC register structure.
 * @param[in]   CrcInitialValue - The CRC Initial value.
 * @private
 */
CRC_FUNC void CrcTst_Lld_CrcWriteInitialValue(volatile CRC_Type *BaseAddr, uint32 CrcInitialValue)
{
    BaseAddr->INIT = CrcInitialValue;
}

/**
 * @brief       Write the CRC Input data in 4 bytes.
 * @param[in]   crc - Pointer to the CRC register structure.
 * @param[in]   CrcData - The CRC Input data in 4 bytes.
 * @private
 */
CRC_FUNC void CrcTst_Lld_CrcWriteDataU32(volatile CRC_Type *BaseAddr, uint32 CrcData)
{
    BaseAddr->DATA.IN32 = CrcData;
    MCAL_QUA_FAULT_INJECTION(CRCTST_LLD_WRITE_AGAIN);
}

/**
 * @brief       Write the CRC Input data in byte.
 * @param[in]   crc - Pointer to the CRC register structure.
 * @param[in]   CrcData - The CRC Input data in byte.
 * @private
 */
CRC_FUNC void CrcTst_Lld_CrcWriteDataU8(volatile CRC_Type *BaseAddr, uint8 CrcData)
{
    BaseAddr->DATA.IN8 = CrcData;
}

/**
 * @brief       Read the CRC result.
 * @param[in]   crc - Pointer to the CRC register structure.
 * @return      uint32 - The CRC result.
 * @private
 */
CRC_FUNC uint32 CrcTst_Lld_CrcReadCrcResult(const volatile CRC_Type *BaseAddr)
{
    return BaseAddr->RESULT;
}

#define CRC_STOP_SEC_CODE
#include "Crc_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file CrcTst_Lld_Crc.c */

