/**
* @file    CrcTst_Lld_Crc.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : CrcTst_Lld_Crc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CRCTST_LLD_CRC_H
#define CRCTST_LLD_CRC_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "pSIP_Crc.h"
#include "CrcTst_Types.h"

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CRCTST_LLD_CRC_VENDOR_ID                      (180)
#define CRCTST_LLD_CRC_AR_RELEASE_MAJOR_VERSION       (4)
#define CRCTST_LLD_CRC_AR_RELEASE_MINOR_VERSION       (4)
#define CRCTST_LLD_CRC_AR_RELEASE_REVISION_VERSION    (0)
#define CRCTST_LLD_CRC_SW_MAJOR_VERSION               (2)
#define CRCTST_LLD_CRC_SW_MINOR_VERSION               (0)
#define CRCTST_LLD_CRC_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if CrcTst_Lld_Crc.h file and CrcTst_Types.h file are of the same vendor */
#if (CRCTST_LLD_CRC_VENDOR_ID != CRCTST_TYPES_VENDOR_ID)
#error "CrcTst_Lld_Crc.h and CrcTst_Types.h have different vendor ids"
#endif

/* Check if CrcTst_Lld_Crc.h and CrcTst_Types.h file are of the same Autosar version */
#if ((CRCTST_LLD_CRC_AR_RELEASE_MAJOR_VERSION != CRCTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (CRCTST_LLD_CRC_AR_RELEASE_MINOR_VERSION != CRCTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (CRCTST_LLD_CRC_AR_RELEASE_REVISION_VERSION != CRCTST_TYPES_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CrcTst_Lld_Crc.h and CrcTst_Types.h are different"
#endif

/* Check if CrcTst_Lld_Crc.h and CrcTst_Types.h file are of the same software version */
#if ((CRCTST_LLD_CRC_SW_MAJOR_VERSION != CRCTST_TYPES_SW_MAJOR_VERSION) || \
     (CRCTST_LLD_CRC_SW_MINOR_VERSION != CRCTST_TYPES_SW_MINOR_VERSION) || \
     (CRCTST_LLD_CRC_SW_PATCH_VERSION != CRCTST_TYPES_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CrcTst_Lld_Crc.h and CrcTst_Types.h are different"
#endif

/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#define CRCTST_START_SEC_CODE
#include "CrcTst_MemMap.h"

/**
 * @brief       Get the CRC control command.
 * @details     This function generates the CRC control command based on the CRC configuration and polynomial mode setting.
 * @param[in]   CrcConfig   Pointer to the CRC configuration structure.
 * @param[in]   PolyModSet  Polynomial mode setting.
 * @return      uint32      Generated CRC control command.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_CrcTst_507
 */
CRCTST_FUNC uint32 CrcTst_Lld_CrcGetCtlCmd(const CrcTst_PolConfigType *CrcConfig, uint8 PolyModSet);

/**
 * @brief       Write the CRC control command.
 * @details     This function writes the CRC control command.
 * @param[in]   BaseAddr    Pointer to the CRC register structure.
 * @param[in]   CtrlCmd     The CRC control command.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_CrcTst_510
 */
CRCTST_FUNC void CrcTst_Lld_CrcWriteCtlCmd(volatile CRC_Type *BaseAddr, uint32 CtrlCmd);

/**
 * @brief       Write the CRC initial value.
 * @details     This function writes the CRC initial value.
 * @param[in]   BaseAddr        Pointer to the CRC register structure.
 * @param[in]   CrcInitialValue The CRC initial value.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_CrcTst_513
 */
CRCTST_FUNC void CrcTst_Lld_CrcWriteInitialValue(volatile CRC_Type *BaseAddr, uint32 CrcInitialValue);

/**
 * @brief       Write the CRC input data in 4 bytes.
 * @details     This function writes the CRC input data in 4 bytes.
 * @param[in]   BaseAddr    Pointer to the CRC register structure.
 * @param[in]   CrcData     The CRC input data in 4 bytes.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_CrcTst_511
 */
CRCTST_FUNC void CrcTst_Lld_CrcWriteDataU32(volatile CRC_Type *BaseAddr, uint32 CrcData);

/**
 * @brief       Write the CRC input data in 1 byte.
 * @details     This function writes the CRC input data in 1 byte.
 * @param[in]   BaseAddr    Pointer to the CRC register structure.
 * @param[in]   CrcData     The CRC input data in 1 byte.
 * @return      void
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_CrcTst_512
 */
CRCTST_FUNC void CrcTst_Lld_CrcWriteDataU8(volatile CRC_Type *BaseAddr, uint8 CrcData);

/**
 * @brief       Read the CRC result.
 * @details     This function reads the CRC result.
 * @param[in]   BaseAddr    Pointer to the CRC register structure.
 * @return      uint32      The CRC result.
 * @table       @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 *              @memory_map:       .mcal_text  \n
 * @sw_type     sw_detail
 *
 * @trace YTSDS_CrcTst_509
 */
CRCTST_FUNC uint32 CrcTst_Lld_CrcReadCrcResult(const volatile CRC_Type *BaseAddr);

#define CRCTST_STOP_SEC_CODE
#include "CrcTst_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* End of file CrcTst_Lld_Crc.h */
