/**
 * @file    CrcTst.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR CrcTst module interface
 * @details API implementation for CrcTst driver
 *
 * @addtogroup CRCTST_MODULE
 * @{
 */

/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Crc
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
 *=================================================================================================*/

#ifndef CRCTST_H
#define CRCTST_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
 *=================================================================================================*/
#include "CrcTst_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRCTST_VENDOR_ID                      (180)
#define CRCTST_MODULE_ID                      (515)
#define CRCTST_AR_RELEASE_MAJOR_VERSION       (4)
#define CRCTST_AR_RELEASE_MINOR_VERSION       (4)
#define CRCTST_AR_RELEASE_REVISION_VERSION    (0)
#define CRCTST_SW_MAJOR_VERSION               (2)
#define CRCTST_SW_MINOR_VERSION               (0)
#define CRCTST_SW_PATCH_VERSION               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if CrcTst.h file and CrcTst_Types.h file are of the same vendor */
#if (CRCTST_VENDOR_ID != CRCTST_TYPES_VENDOR_ID)
#error "CrcTst.h and CrcTst_Types.h have different vendor ids"
#endif

/* Check if CrcTst.h and CrcTst_Types.h file are of the same Autosar version */
#if ((CRCTST_AR_RELEASE_MAJOR_VERSION != CRCTST_TYPES_AR_RELEASE_MAJOR_VERSION) || \
     (CRCTST_AR_RELEASE_MINOR_VERSION != CRCTST_TYPES_AR_RELEASE_MINOR_VERSION) || \
     (CRCTST_AR_RELEASE_REVISION_VERSION != CRCTST_TYPES_AR_RELEASE_REVISION_VERSION) \
    )
#error "AutoSar Version Numbers of CrcTst.h and CrcTst_Types.h are different"
#endif

/* Check if CrcTst.h and CrcTst_Types.h file are of the same software version */
#if ((CRCTST_SW_MAJOR_VERSION != CRCTST_TYPES_SW_MAJOR_VERSION) || \
     (CRCTST_SW_MINOR_VERSION != CRCTST_TYPES_SW_MINOR_VERSION) || \
     (CRCTST_SW_PATCH_VERSION != CRCTST_TYPES_SW_PATCH_VERSION) \
    )
#error "Software Version Numbers of CrcTst.h and CrcTst_Types.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
 *=================================================================================================*/
/** @brief CrcTst MODULE INSTANCE ID */
#define CRCTST_INSTANCE_ID ((uint8)0U)

/**
 * @defgroup CrcTst_ID_Info_Macro
 * @brief    Service ID number for all CRCTST driver services
 * @sw_type  sw_detail
 * @{
 */
/** @brief API Service ID for CrcTst_Init */
#define CRCTST_SID_INIT                     (0x00U)
/** @brief API Service ID for CrcTst_DeInit */
#define CRCTST_SID_DEINIT                   (0x01U)
/** @brief API Service ID for CrcTst_GetState */
#define CRCTST_SID_GET_STATE                (0x02U)
/** @brief API Service ID for CrcTst_StartFgnd */
#define CRCTST_SID_START_FGND               (0x03U)
/** @brief API Service ID for CrcTst_GetFgndResult */
#define CRCTST_SID_GET_FGND_RESULT          (0x04U)
/** @brief API Service ID for CrcTst_GetFgndDetailsResult */
#define CRCTST_SID_GET_FGND_RESULT_DETAILS  (0x05U)
/** @brief API Service ID for CrcTst_GetRegisterSignature */
#define CRCTST_SID_GET_REG_SIGN             (0x06U)
/** @brief API Service ID for CrcTst_StartBgnd */
#define CRCTST_SID_START_BGND               (0x07U)
/** @brief API Service ID for CrcTst_MainFunction */
#define CRCTST_SID_MAIN_FUNCTION            (0x08U)
/** @brief API Service ID for CrcTst_Abort */
#define CRCTST_SID_ABORT                    (0x09U)
/** @brief API Service ID for CrcTst_GetBgndResult */
#define CRCTST_SID_GET_BGND_RESULT          (0x0AU)
/** @brief API Service ID for CrcTst_GetBgndDetailsResult */
#define CRCTST_SID_GET_BGND_RESULT_DETAILS  (0x0BU)
/** @brief API Service ID for CrcTst_GetVersionInfo */
#define CRCTST_SID_GET_VERSION_INFO         (0x0CU)
/** @} */

/**
 * @defgroup CrcTst_Dev_Error_Macro
 * @brief    CRCTST development errors
 * @sw_type  sw_detail
 * @{
 */
/** @brief API CrcTst_Init service called with wrong parameter. */
#define CRCTST_E_INIT_FAILED            (0x00U)
/** @brief A particular API is called in an unexpected state. */
#define CRCTST_E_STATUS_FAILURE         (0x01U)
/** @brief API called without Crc Test initialization. */
#define CRCTST_E_UNINIT                 (0x02U)
/** @brief API service CrcTst_Init() called again without a CrcTst_DeInit() in between. */
#define CRCTST_E_ALREADY_INITIALIZED    (0x03U)
/** @brief API service called with a NULL pointer. */
#define CRCTST_E_PARAM_POINTER          (0x04U)
/** @} */


/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
 *==================================================================================================*/
#define CRCTST_START_SEC_CODE
#include "CrcTst_MemMap.h"

/**
 * @brief       Initialize and change the state of the CRC Test module.
 * @details     This function initializes the CRC Test module by setting up the configuration
 *              and resetting the internal state variables. It also validates the input configuration.
 * @param[in]   ConfigPtr Pointer to the selected configuration set.
 * @table       @service_id:       0x00        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      void
 * @sw_type     sw_arch
 */
CRCTST_FUNC void CrcTst_Init(const CrcTst_ConfigType *ConfigPtr);

/**
 * @brief       De-initialize the CRC Test module.
 * @details     De-initialize the structure, variables, change the state from or CRCTST_INIT to CRCTST_UNINIT.
 * @param[in]   void
 * @table       @service_id:       0x01        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      void
 * @sw_type     sw_arch
 */
CRCTST_FUNC void CrcTst_DeInit(void);

/**
 * @brief       Get the current state of the CRC Test module.
 * @details     This function returns the current state of the CRC Test module.
 * @param[in]   void
 * @table       @service_id:       0x02        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      CrcTst_StateType Module CrcTst state
 * @sw_type     sw_arch
 */
CRCTST_FUNC CrcTst_StateType CrcTst_GetState(void);

#if (CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON)
/**
 * @brief       Main function of the CRC Test module to process the foreground test.
 * @details     This function performs the foreground CRC test for the configured CRC types (CRC8, CRC16, CRC32).
 * @param[in]   void
 * @table       @service_id:       0x03        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      Std_ReturnType
 * @retval      E_OK function is executed successfully
 * @sw_type     sw_arch
 */
CRCTST_FUNC Std_ReturnType CrcTst_StartFgnd(void);

/**
 * @brief       Service to get the last result of the foreground test.
 * @details     This function retrieves the result of the last executed foreground CRC test.
 *              It checks the module's state and returns the appropriate result.
 * @param[in]   void
 * @table       @service_id:       0x04        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      CrcTst_TestResultType
 * @retval      CRCTST_E_NOT_TESTED Not tested yet.
 * @retval      CRCTST_E_OKAY No errors detected.
 * @retval      CRCTST_E_NOT_OK At least one error detected.
 * @sw_type     sw_arch
 */
CRCTST_FUNC CrcTst_TestResultType CrcTst_GetFgndResult(void);

/**
 * @brief       Service to get the detail result of the foreground test.
 * @details     This function retrieves detailed results of the foreground CRC test.
 * @param[out]  FgndResults Pointer to where to store the detailed result.
 * @table       @service_id:       0x05        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      void
 * @sw_type     sw_arch
 */
CRCTST_FUNC void CrcTst_GetFgndDetailsResult(CrcTst_CrcFuncTestDetailsType *FgndResults);
#endif /* CRCTST_CRC_FUNCTION_TEST_ENABLE == STD_ON */

#if (CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON)
/**
 * @brief        Get the register signature.
 * @details      This function calculates and retrieves the register signature for the CRC Test module.
 *               The signature is computed based on the background configuration data provided in the module's configuration.
 * @param[in]   void
 * @table       @service_id:       0x06        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      Std_ReturnType     E_OK if successful, E_NOT_OK otherwise.
 * @retval      E_OK function is executed successfully
 * @retval      E_NOT_OK parameter or state is not valid
 * @sw_type     sw_arch
 */
CRCTST_FUNC Std_ReturnType CrcTst_GetRegisterSignature(void);

/**
* @brief        Service to start the background test.
* @details      This function starts the background CRC test process. It checks the module's state
*               and ensures that the module is properly initialized before starting the test.
 * @param[in]   void
 * @table       @service_id:       0x07        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @sw_type     sw_arch
 */
CRCTST_FUNC Std_ReturnType CrcTst_StartBgnd(void);

/**
 * @brief       Main function of the CRC Test module to process the background test.
 * @details     Cyclically called by scheduler to perform processing of atomic testcase in background mode.
 * @param[in]   void
 * @table       @service_id:       0x08        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      Std_ReturnType
 * @retval      E_OK function is executed successfully
 * @retval      E_NOT_OK parameter or state is not valid
 * @sw_type     sw_arch
 */
CRCTST_FUNC Std_ReturnType CrcTst_MainFunction(void);

/**
 * @brief       Abort the CRC Test in background test.
 * @details     This function aborts the ongoing CRC test by changing the module state from `CRCTST_RUNNING_BGND`
 *              to `CRCTST_INIT`. If the module is in the `CRCTST_UNINIT` state, an error is reported
 *              using the Development Error Tracer (DET).
 * @param[in]   void
 * @table       @service_id:       0x09        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      void
 * @sw_type     sw_arch
 */
CRCTST_FUNC void CrcTst_Abort(void);

/**
 * @brief       Get the result of the background test.
 * @param[in]   void
 * @table       @service_id:       0x0A        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      CrcTst_TestResultType
 * @retval      CRCTST_E_NOT_TESTED Not tested yet
 * @retval      CRCTST_E_OKAY No errors detected
 * @retval      CRCTST_E_NOT_OK At least one error detected
 * @sw_type     sw_arch
 */
CRCTST_FUNC CrcTst_TestResultType CrcTst_GetBgndResult(void);

/**
 * @brief       Service to get the detail result of the background test.
 * @details     This function retrieves detailed results of the background CRC test.
 * @param[out]  BgndResults Pointer to where to store the detailed result.
 * @table       @service_id:       0x0B        \n
 *              @is_reentrant:     false       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @return      void
 * @sw_type     sw_arch
 */
CRCTST_FUNC void CrcTst_GetBgndDetailsResult(CrcTst_TestResultType *BgndResults);

#endif /* CRCTST_REGISTER_INTEGRITY_TEST_ENABLE == STD_ON */

#if (CRCTST_VERSION_INFO_API == STD_ON)
/**
 * @brief       This service returns the version information of this module
 * @param[out]  Versioninfo Pointer to where to store the version information of this module.
 * @table       @service_id:       0x0C        \n
 *              @is_reentrant:     true       \n
 *              @is_synchronous:   true        \n
 *              @autosar_api:      false       \n
 * @sw_type     sw_arch
 */
CRCTST_FUNC void CrcTst_GetVersionInfo(Std_VersionInfoType *Versioninfo);
#endif

#define CRCTST_STOP_SEC_CODE
#include "CrcTst_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* CRCTST_H */

/** @} */
